/*************************************** 
*
* Freescale Semiconductor Inc. 
* (c) Copyright 2012 Freescale Semiconductor, Inc. 
* ALL RIGHTS RESERVED 
*
****************************************//*
*
* @file etimer_api.c 
* 
* @author B37643
* 
* @brief  Description about file
*/ 

#include "..\inc\etimer_api.h"
#include "..\inc\etimer_defines.h"
#include "..\inc\etimer_soc_config.h"
#include "common.h"


uint16_t	IP_BUS_DIVIDER[] = {24, 25, 26, 27 ,28 ,29 ,30 ,31};//1,2,4,8,16,32,64,128

void set_module			(uint8_t timer);
void eTimer_CONFIG_PINS	(uint8_t module, uint8_t	channel, uint8_t direct, uint8_t slew_rate);

//**********Initializing the IP instances so that they can be used in structural approach as per coding guidelines rule no 1***************/
//**********No arguments**********/
int ETIMER_INIT(){
//Initializing the structs
return 1;
}


//**********API to reset the IP instance to it's reset values***********/
//**********IPNAME_instance is the instance of IP whose values are to be set to reset state values*********/
int ETIMER_RESET(int ETIMER_instance){
//resetting the IP instance to it's default reset values
return 1;
}


/********************************************************************************/
/*	NAME	: Generate_signal													*/
/*	INPUTS	: 	uint8_t timer		which timer module							*/
/*			  	uint8_t channel		which channel of timer						*/
/*				uint32_t frequency	frequency of output signal	[Hz]			*/
/*				uint16_t duty		duty of output signal,in %					*/
/*				uint32_t motor_freq	module frequency			[kHz]			*/
/*	OUTPUTS	: possibilities, =0 no problem, = 1 it is not possible to generate 	*/
/*				signal with these parameters - frequency is uot of range		*/
/*	DESCRIPT: generate output signal											*/
/*  ACCURACY: Accuracy of duty cycle for motor frequency 160 MHz				*/
/*				frequency range				maximum variance					*/
/*				>0		Hz	-	<= 160	kHz			0.1	%						*/
/*				>160	kHz	-	<= 1.6	MHz			1	%						*/
/*				>1.6	MHz	-	<= 16	MHz			10	%						*/
/*				>16		MHz -	<= 80	MHz			50	%						*/	 	
/*																				*/	
/********************************************************************************/
uint8_t Generate_Signal(uint8_t timer, uint8_t channel, uint32_t frequency, uint16_t duty, uint32_t motor_freq)
{
  uint32_t	range 		= 0;	//this range define the frequency of signal in counter value
  uint8_t 	i 			= DIV1;	//divider of channel which generate signal
 
  set_module(timer);
  motor_freq = motor_freq*1000; //Hz

  if ((frequency<=(motor_freq/2))&&(frequency>motor_freq/(128*MIN_FREQ)))
  {
	//select input clock divider, depending on the generating frequency 
	while ((motor_freq/((1<<i)*frequency)>=MIN_FREQ)&&(i<7))
	{
	  i++;
	}
		
	range = motor_freq/((1<<i)*frequency);
	eTimer->CHANNEL[channel].CH_CTRL1.B.PRISRC	= IP_BUS_DIVIDER[i];

	eTimer->CHANNEL[channel].CH_COMP1.R			= MAX_COUNT - (range*duty)/1000;
	eTimer->CHANNEL[channel].CH_COMP2.R			= MAX_COUNT;
	eTimer->CHANNEL[channel].CH_CMPLD1.R		= MAX_COUNT - range+1;
	eTimer->CHANNEL[channel].CH_CCCTRL.B.CLC1	= 0x7;	//reinicialize CNTR by CMPLD1	

	eTimer->CHANNEL[channel].CH_CTRL1.B.CNTMODE	= 1;
	eTimer->CHANNEL[channel].CH_CTRL1.B.LENGTH	= 1;

	//definition of output
	eTimer->CHANNEL[channel].CH_CTRL2.B.OEN		= 1;	//enable output
	eTimer->CHANNEL[channel].CH_CTRL2.B.OUTMODE	= 8;	//Set on successful compare on COMP1, clear on successful compare on COMP2
  }
  else return 1;

  
  return 0;
}

/********************************************************************************/
/*	NAME	: Generate_signal2													*/
/*	INPUTS	: 	uint8_t timer		which timer module							*/
/*			  	uint8_t channel4period_b - base channel for period				*/
/*				uint8_t channel4period_o - output channel for period			*/
/*				uint8_t channel4pulse - output channel which create the signal	*/
/*				uint32_t period	    period of output signal		[ms]			*/
/*				uint32_t width		width of pulze				[ns]			*/
/*				uint32_t motor_freq	module frequency			[kHz]			*/
/*	OUTPUTS	: possibilities, =0 no problem, = 1 it is not possible to generate 	*/
/*				signal with these parameters - frequency is uot of range		*/
/*	DESCRIPT: generate output signal with long period, channel4pulse generate 	*/
/*				pulse with width, the period is given channel4perio_b and 		*/
/*				channel4period_o, perid channel trigger the channel which create*/
/*				the pulse														*/
/*	LIMITS:		2/motor_freq[kHz]	<	period[ms]	<	65000*65000*128/motor_freq[kHz]	*/
/*				1000000/motor_freq[kHz]	<	width[ns]	<	65000*1000000/motor_freq[kHz]*/
/********************************************************************************/
uint8_t Generate_Signal2(uint8_t timer, uint8_t channel4period_b,  uint8_t channel4period_o, uint8_t channel4pulse, uint32_t period, uint32_t width, uint32_t motor_freq)
{
  uint32_t	range_b		= 0;		//range for channel of timer which defines the period
  uint32_t	range_o		= 0;		//range for channel of timer which defines the period, this is used only for longer period	
  uint32_t	i_range_b	= 0; 		//ideal range for period >= 65000*128/motor_freq
  uint32_t	i_range_o	= 0; 		//ideal range for period >= 65000*128/motor_freq
  uint32_t	i_div_period= 0; 		//ideal range for period >= 65000*128/motor_freq 
  uint32_t	error		= -1;		
  uint8_t 	div_pulse	= DIV1;		//divider of channel which is used for pulse
  uint8_t	div_period	= DIV1;		//divider of channel which is used for period


  //check meaningfulness of settings
  if ((2*period*1000000)<width) return 1;				//check ratio of period to width 
  if (period>(65000*((65000*128)/motor_freq))) return 1;//check the maximum of period
  if ((period*1000000)<(2*1000000/motor_freq)) return 1;//check minimum period
  if (width<1000000/motor_freq) return 1;				//check the minimum pulse width
  if (width>(65000*1000*128/motor_freq)*1000) return 1;		//check maximum width

  //set timer
  set_module(timer);

  //set the period---
  if (period>=(65000*128/motor_freq))	
  {//this is valid for longer period - for this is used 2 channel of timer, channel4period_b and channel4period_o
    //select input clock divider, depending on the generating frequency
	div_period = 0;
	while ((((MIN_FREQ*(1<<div_period))/motor_freq)<period)&&(div_period<7))
	{
	  div_period++;
	}
	range_b = period*motor_freq/(1<<div_period);

	//find the best values of div_1_lev, range_b and range_o for given period 
	error = -1;
	for (div_period = 0;div_period<8;div_period++)
	{
	  for (range_b = 500;range_b<MIN_FREQ+500;range_b = range_b + 500)
	  {
		for (range_o = 100;range_o<MIN_FREQ+100;range_o = range_o +100)
		{
		  if (error>abs(period-((1<<div_period)*range_b*range_o)/motor_freq))
		  {
			error = abs(period-((1<<div_period)*range_b*range_o)/motor_freq);
			i_range_b 		= range_b;
			i_range_o 		= range_o;
			i_div_period	= div_period;
		  }
		}
	  }
	}
	
	eTimer->CHANNEL[channel4period_o].CH_CTRL1.B.PRISRC	= 16 + channel4period_b;		//primary source is output of channel4period_b channel

	eTimer->CHANNEL[channel4period_o].CH_COMP1.R			= MAX_COUNT - i_range_o/2;	//duty cycle is always 50 % - it is only trigger for channel which generate the pulse
	eTimer->CHANNEL[channel4period_o].CH_COMP2.R			= MAX_COUNT;				//this is second board of period, this is clear output signal				
	eTimer->CHANNEL[channel4period_o].CH_CMPLD1.R			= MAX_COUNT - i_range_o+1;	//this is given by period, create the period of signal with eTimer->CHANNEL[channel4period_b].CH_CMPLD1.R 
	eTimer->CHANNEL[channel4period_o].CH_CCCTRL.B.CLC1		= 0x7;						//reinicialize CNTR by CMPLD1
	
	eTimer->CHANNEL[channel4period_o].CH_CTRL1.B.CNTMODE	= 1;						//count rising edge of primary source
	eTimer->CHANNEL[channel4period_o].CH_CTRL1.B.LENGTH		= 1;						//count until compare , then reinitialize
	  
    eTimer->CHANNEL[channel4period_o].CH_CTRL2.B.OUTMODE	= 8;						//Set on successful compare on COMP1, clear on successful compare on COMP2
  }

  if (period<(65000*128/motor_freq))
  {//for shorter period is used only one channel of timer - channel4period_b
	//select input clock divider, depending on the generating frequency
	i_div_period = 0;
	while ((((MIN_FREQ*(1<<div_period))/motor_freq)<period)&&(i_div_period<7))
	{
	  i_div_period++;
	}
	i_range_b = period*motor_freq/(1<<i_div_period);
  }

	eTimer->CHANNEL[channel4period_b].CH_CTRL1.B.PRISRC	= IP_BUS_DIVIDER[i_div_period];	//primary source is IP bus, divide is given by period

	eTimer->CHANNEL[channel4period_b].CH_COMP1.R			= MAX_COUNT - i_range_b/2;	//duty cycle is always 50 % - it is only trigger for channel which generate the pulse
	eTimer->CHANNEL[channel4period_b].CH_COMP2.R			= MAX_COUNT;				//this is second board of period, this is clear output signal
	eTimer->CHANNEL[channel4period_b].CH_CMPLD1.R			= MAX_COUNT - i_range_b+1;	//this is given by period, create the period of signal with eTimer->CHANNEL[channel4period_o].CH_CMPLD1.R  
	eTimer->CHANNEL[channel4period_b].CH_CCCTRL.B.CLC1		= 0x7;						//reinicialize CNTR by CMPLD1	

	eTimer->CHANNEL[channel4period_b].CH_CTRL1.B.CNTMODE	= 1;						//count rising edge of primary source
	eTimer->CHANNEL[channel4period_b].CH_CTRL1.B.LENGTH		= 1;						//count until compare , then reinitialize

	eTimer->CHANNEL[channel4period_b].CH_CTRL2.B.OUTMODE	= 8;						//Set on successful compare on COMP1, clear on successful compare on COMP2
  
  //---
  
  //set the pulse
  //select input clock divider, depending on the generating frequency
  div_pulse=0;
  while ((((MIN_FREQ*10000000*(1<<div_pulse))/motor_freq)<width)&&(div_pulse<7))
  {
	div_pulse++;
  }
  eTimer->CHANNEL[channel4pulse].CH_COMP1.R			= ((motor_freq/1000)*(width/(1<<div_pulse)))/1000;//set the width of signal
  eTimer->CHANNEL[channel4pulse].CH_COMP2.R			= MAX_COUNT;					//here is not use comparator 2, set uot of working area
  
  eTimer->CHANNEL[channel4pulse].CH_CTRL1.B.PRISRC	= IP_BUS_DIVIDER[div_pulse];	//primary source is IP bus, divide is given by width
  
  if (period<65000*128/motor_freq) eTimer->CHANNEL[channel4pulse].CH_CTRL1.B.SECSRC	= 16 + channel4period_b;//secondary source is output of channel4period_b channel	
  else eTimer->CHANNEL[channel4pulse].CH_CTRL1.B.SECSRC	= 16 + channel4period_o;	//secondary source is output of channel4period_o channel	
  
  eTimer->CHANNEL[channel4pulse].CH_CTRL1.B.CNTMODE	= 0x6;							//edge of secondary source triggers primary count till compare
  eTimer->CHANNEL[channel4pulse].CH_CTRL1.B.LENGTH	= 1;							//count until compare , then reinitialize

  //definition of output
  eTimer->CHANNEL[channel4pulse].CH_CTRL2.B.OEN		= 1;							//enable output
  eTimer->CHANNEL[channel4pulse].CH_CTRL2.B.OUTMODE	= 5;							//set on compare with COMP1, clear on seccondary source input edge
  eTimer->CHANNEL[channel4pulse].CH_CTRL2.B.OPS		= 1;							//inverted output

  return 0;
}

/********************************************************************************/
/*	NAME	: Generate_OneShot_signal_set										*/
/*	INPUTS	: 	uint8_t timer		which timer module							*/
/*			  	uint8_t channel		which channel of timer						*/
/*				uint32_t delay		delay of pulse	[us]						*/
/*				uint16_t width		width of pulse  [us]						*/
/*				uint8_t	active_level HIGH or LOW								*/
/*				uint32_t motor_freq	module frequency			[kHz]			*/
/*	OUTPUTS	: possibilities, =0 no problem, = 1 it is not possible to generate 	*/
/*				signal with these parameters 									*/
/*	DESCRIPT: setting output oneshot signal, for start use function 			*/
/*				start_Generate_OneShot_signal									*/
/********************************************************************************/
uint8_t Generate_OneShot_signal_set(uint8_t timer, uint8_t channel, uint32_t delay, uint32_t width, uint32_t motor_freq, uint8_t active_level)
{
  uint32_t	range 		= 0;	//this range is for generate the delay
  uint8_t 	i 			= DIV1;	//divider of channel which generate signal
 
  set_module(timer);
  motor_freq = motor_freq*1000; //Hz

  if ((delay>((2*1000000)/motor_freq))&&(delay<(((12800*MIN_FREQ)/motor_freq)*10000)))
  {
	if (delay<=width) return 1; // if is possible to create signal with given delay and width
	
	//select input clock divider, depending on the generating frequency 
	while ((delay*motor_freq/((1<<i)*1000000)>=MIN_FREQ)&&(i<7))
	{
	  i++;
	}
		
	range = (delay*(motor_freq/1000))/(1000*(1<<i));
	eTimer->CHANNEL[channel].CH_CTRL1.B.PRISRC	= IP_BUS_DIVIDER[i];
	
	eTimer->CHANNEL[channel].CH_CNTR.R			= 0x0;	//clear counter for the new use
	eTimer->CHANNEL[channel].CH_COMP1.R			= range;//delay
	eTimer->CHANNEL[channel].CH_COMP2.R			= range + (width*motor_freq)/(1000000*(1<<i));//delay + width
	eTimer->CHANNEL[channel].CH_CTRL1.B.ONCE	= 1;

	//definition of output
	eTimer->CHANNEL[channel].CH_CTRL2.B.OEN		= 1;	//enable output
	eTimer->CHANNEL[channel].CH_CTRL2.B.OPS		= active_level;	//output polarity
	eTimer->CHANNEL[channel].CH_CTRL2.B.OUTMODE	= 4;	//Toggle OFLAG output using alternating compare registers

  }
  else return 1;

  return 0;
}

/********************************************************************************/
/*	NAME	: start_Generate_OneShot_signal										*/
/*	INPUTS	: 	uint8_t timer		which timer module							*/
/*			  	uint8_t channel		which channel of timer						*/
/*	OUTPUTS	: none																*/
/*	DESCRIPT: start generateing the one shot mode, for setting parameters of 	*/
/*				oneshot signal use Generate_OneShot_signal_set					*/
/********************************************************************************/
void start_Generate_OneShot_signal (uint8_t timer, uint8_t channel)
{
  set_module(timer);
  eTimer->CHANNEL[channel].CH_CNTR.R			= 0x0;	//clear counter for new use
  eTimer->CHANNEL[channel].CH_CTRL1.B.CNTMODE	= 1;	//start counting
}

/********************************************************************************/
/*	NAME	: Measure_signal_parameters_set										*/
/*	INPUTS	: 	uint8_t timer		which timer module							*/
/*			  	uint8_t channel		which channel of timer						*/
/*				uint32_t frequency	frequency of output signal	[Hz]			*/
/*				uint16_t duty		duty of output signal,in %					*/
/*				uint32_t motor_freq	module frequency			[kHz]			*/
/*	OUTPUTS	: possibilities, =0 no problem, = 1 it is not possible to generate 	*/
/*				signal with these parameters - frequency is uot of range		*/
/*	DESCRIPT: setting the measure input signal									*/
/********************************************************************************/
uint8_t Measure_signal_parameters_set (uint8_t timer, uint8_t channel)
{
  set_module(timer);									//set etimer 
  
  eTimer->CHANNEL[channel].CH_CTRL1.B.PRISRC	= IP_BUS_DIVIDER[0];//maximum resolution
  
  eTimer->CHANNEL[channel].CH_CCCTRL.B.CPT1MODE	= 0x2;	//rising edge
  eTimer->CHANNEL[channel].CH_CCCTRL.B.CPT2MODE	= 0x1;	//falling edge
  eTimer->CHANNEL[channel].CH_CCCTRL.B.CFWM		= 0x2;  //capture flag set as soon as more than 3 values will be in FIFOs

  eTimer->CHANNEL[channel].CH_CTRL1.B.LENGTH	= 0x0;	//continue counting to roll over
  eTimer->CHANNEL[channel].CH_CTRL1.B.ONCE		= 0x0;	//count repeatedly
  eTimer->CHANNEL[channel].CH_CTRL1.B.SECSRC	= channel;	//counter "channel" input pin is use for trigger the capturing - measuring signal is connect to this pin
  eTimer->CHANNEL[channel].CH_CTRL1.B.CNTMODE	= 0x1;	//count rising edge of primary source

  return 0;
}

/********************************************************************************/
/*	NAME	: start_Measure_signal_parameters									*/
/*	INPUTS	: 	uint8_t timer		which timer module							*/
/*			  	uint8_t channel		which channel of timer						*/
/*				uint32_t frequency	frequency of the measure signal	[Hz]			*/
/*				uint16_t duty		duty of the measure signal,in %					*/
/*				uint32_t motor_freq	module frequency			[kHz]			*/
/*	OUTPUTS	: possibilities, =0 no problem, = 1 it is not possible to generate 	*/
/*				signal with these parameters - frequency is uot of range		*/
/*	DESCRIPT: measure input signal												*/
/********************************************************************************/
uint8_t start_Measure_signal_parameters (uint8_t timer, uint8_t channel,uint32_t motor_freq, uint32_t *frequency, uint32_t *duty)
{
  static uint16_t	measure[4]	= {0, 0, 0, 0};
 
  set_module(timer);									//set etimer 
  //measure
  eTimer->CHANNEL[channel].CH_CCCTRL.B.ARM		= 0x1;	//enable capturing
  while ((eTimer->CHANNEL[channel].CH_STS.B.ICF1 == 0x0)||(eTimer->CHANNEL[channel].CH_STS.B.ICF2 == 0x0));//wait for capture 2 cap1 values and 2 capt2 values
  eTimer->CHANNEL[channel].CH_CCCTRL.B.ARM		= 0x0;	//disable capturing

  eTimer->CHANNEL[channel].CH_STS.B.ICF1	= 0x1;		//clear capture 1 flag
  eTimer->CHANNEL[channel].CH_STS.B.ICF2 	= 0x1;		//clear capture 2 flag

  measure[0]	= eTimer->CHANNEL[channel].CH_CAPT1.R;	//read first capture1 value
  measure[1]	= eTimer->CHANNEL[channel].CH_CAPT1.R;	//read second capture 1 value 
  measure[2]	= eTimer->CHANNEL[channel].CH_CAPT2.R;	//read first capture2 value
  measure[3]	= eTimer->CHANNEL[channel].CH_CAPT2.R;	//read second capture2 value

  //calculate frequency
  *frequency	= motor_freq/(uint16_t)((measure[1] - measure[0]));

  //calculate duty
  *duty			= (uint16_t)((measure[2] - measure[0]))*1000/(uint16_t)((measure[1] - measure[0]));
  
  return 0;
}


/********************************************************************************/
/*	NAME	: set_module														*/
/*	INPUTS	:	uint8_t timer		- which module will be use					*/
/*	OUTPUTS	: void																*/
/*	DESCRIPT: set the addresses of etimer module								*/					
/********************************************************************************/
void set_module(uint8_t timer)
{
	switch (timer)
	{
		case 0:eTimer = &ETIMER_0;break;
		case 1:eTimer = &ETIMER_1;break;
		case 2:eTimer = &ETIMER_2;break;
	}
}


/********************************************************************************/
/*	NAME	: Change_mode														*/
/*	INPUTS	: uint8_t mode		- new mode										*/
/*			uint8_t sys_clock	- source of system clock						*/
/*			uint16_t freq		- it value of frequency PLL if sys_clock are	*/
/*									PLL0 or PLL1								*/
/*	OUTPUTS	: void																*/
/*	DESCRIPT: change mode and source of system clock							*/
/********************************************************************************/
void Change_mode(uint8_t mode, uint8_t sys_clock, uint16_t freq)
{
  uint32_t *pr_mode,*actual, pom = MC_ME.GS.B.S_CURRENT_MODE;

   
  switch(pom)
  {//store address of mode configuration register of actual mode to variable actual
	case ME_TEST: actual	=(uint32_t*) &MC_ME.TEST_MC.R;break;
	case ME_SAFE: actual	=(uint32_t*) &MC_ME.SAFE_MC.R;break;
	case ME_DRUN: actual	=(uint32_t*) &MC_ME.DRUN_MC.R;break;
	case ME_RUN0: actual	=(uint32_t*) &MC_ME.RUN_MC[0].R;break;
	case ME_RUN1: actual	=(uint32_t*) &MC_ME.RUN_MC[1].R;break;
	case ME_RUN2: actual	=(uint32_t*) &MC_ME.RUN_MC[2].R;break;
	case ME_RUN3: actual	=(uint32_t*) &MC_ME.RUN_MC[3].R;break;
	case ME_HALT0: actual	=(uint32_t*) &MC_ME.HALT0_MC.R;break;
	case ME_STOP0: actual	=(uint32_t*) &MC_ME.STOP0_MC.R;break;
  }
  switch(mode)
  {//store address of mode configuration register of new mode to variable pr_mode
	case ME_TEST: pr_mode	=(uint32_t*) &MC_ME.TEST_MC.R;break;
	case ME_SAFE: pr_mode	=(uint32_t*) &MC_ME.SAFE_MC.R;break;
	case ME_DRUN: pr_mode	=(uint32_t*) &MC_ME.DRUN_MC.R;break;
	case ME_RUN0: pr_mode	=(uint32_t*) &MC_ME.RUN_MC[0].R;break;
	case ME_RUN1: pr_mode	=(uint32_t*) &MC_ME.RUN_MC[1].R;break;
	case ME_RUN2: pr_mode	=(uint32_t*) &MC_ME.RUN_MC[2].R;break;
	case ME_RUN3: pr_mode	=(uint32_t*) &MC_ME.RUN_MC[3].R;break;
	case ME_HALT0: pr_mode	=(uint32_t*) &MC_ME.HALT0_MC.R;break;
	case ME_STOP0: pr_mode	=(uint32_t*) &MC_ME.STOP0_MC.R;break;
  }
  
  //********** set mode **********
//  if (MC_ME.GS.B.S_CURRENT_MODE!=mode)
//  {
	if (MC_ME.GS.B.S_SYSCLK!=SYS_RCO)
	{
	  //allow RC oscillator as system clock for change mode if the different source is used for this (External osc, PLL0, PLL1) 
	  *actual = ((*actual&0xFFFFFFF0)|SYS_RCO);
	  MC_ME.MCTL.R =(pom<<28)|ME_KEY;
	  MC_ME.MCTL.R =(pom<<28)|ME_IKEY;
	  while(MC_ME.GS.B.S_MTRANS);
	  while(MC_ME.GS.B.S_CURRENT_MODE != pom);
	  while (!MC_ME.GS.B.S_IRC);
	}
	//change mode
	MC_ME.ME.R|=1<<mode;
	MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	MC_ME.MCTL.R =(mode<<28)|ME_IKEY;

	while(MC_ME.GS.B.S_MTRANS);
	while(MC_ME.GS.B.S_CURRENT_MODE != mode);

//  }

  //********** set sys_clock **********
  switch (sys_clock)
  {
	case SYS_RCO:
	{
	  *pr_mode = (*pr_mode|(1<<4));					//enable RC oscillator for new mode
	  	  
	  MC_ME.MCTL.R =(mode<<28)|ME_KEY;				
	  MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	  while(MC_ME.GS.B.S_MTRANS);
      while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	  while (!MC_ME.GS.B.S_IRC);
	  
	  *pr_mode = ((*pr_mode&0xFFFFFFF0)|sys_clock);	//set RCO as sysstem clock
	  MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	  MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	  while(MC_ME.GS.B.S_MTRANS);
	  while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	  
	  //set new frequency for LinFlex - UART
	  MC_CGM.SC_DC0.R = 0x80010000;
	  sci_init(7);
	}break;
	case SYS_OSC:
	{
	  *pr_mode = (*pr_mode|(1<<5));					//enable external oscillator for new mode
	  
	  MC_ME.MCTL.R =(mode<<28)|ME_KEY;				
	  MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	  while(MC_ME.GS.B.S_MTRANS);
	  while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	  while (!MC_ME.GS.B.S_XOSC);

	  *pr_mode = ((*pr_mode&0xFFFFFFF0)|sys_clock);	//set external oscillator as system clock
	  MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	  MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	  while(MC_ME.GS.B.S_MTRANS);
	  while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	  //set new frequency for LinFlex - UART
	  MC_CGM.SC_DC0.R = 0x80030000;
	  sci_init(10);
	 }break;
	case SYS_PL0:
	{
 	  *pr_mode = (*pr_mode|(1<<5));					//enable External oscilator
	  
	  MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	  MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	  while(MC_ME.GS.B.S_MTRANS);
	  while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	  while (!MC_ME.GS.B.S_XOSC);
	  
	  *pr_mode = (*pr_mode|(1<<6));					//enable PLL0
	  Set_pll(freq,0,mode);							//configuration PLL0
	  Upgrate_PLL(pr_mode,0,mode);
	  
	  *pr_mode = ((*pr_mode)&0xFFFFFFF0)|0x2;		//set PLL0 as system clock
      MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	  MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	  while(MC_ME.GS.B.S_MTRANS);
	  while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	 	  
	  //set new frequency for LinFlex - UART
	  MC_CGM.SC_DC0.R = 0x80030000;
	  sci_init(freq/4);
	}break;
	case SYS_PL1:
	{
 	  *pr_mode = (*pr_mode|(1<<5));					//enable External oscilator
	  
	  MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	  MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	  while(MC_ME.GS.B.S_MTRANS);
	  while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	  
	  while (!MC_ME.GS.B.S_XOSC);

	  *pr_mode = (*pr_mode|(1<<7));					//enable PLL0
	  Set_pll(freq,1,mode);							//configuration PLL1	
	  Upgrate_PLL(pr_mode,1,mode);
	  
	  *pr_mode = ((*pr_mode)&0xFFFFFFF0)|0x4;		//set PLL1 as sysytem clock
      MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	  MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	  while(MC_ME.GS.B.S_MTRANS);
	  while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	 	  
	  //set new frequency for LinFlex - UART
	  MC_CGM.SC_DC0.R = 0x80030000;
	  sci_init(freq/4);
	  
	}break;
  }
}

/********************************************************************************/
/*	NAME	: Upgrate_PLL														*/
/*	INPUTS	: uint32_t* uk_mode			- address of control registr of mode	*/
/*			uint8_t which				- which PLL								*/
/*			uint8_t mode				- mode of mikrocrontoler				*/
/*	OUTPUTS	: void																*/
/*	DESCRIPT: upgrate PLL, it means turn off and switch on the PLL				*/
/********************************************************************************/
void Upgrate_PLL(uint32_t *uk_mode,uint8_t which, uint8_t mode)
{
  if (which == 0)
  {//PLL0
	*uk_mode = *uk_mode&(~(1<<6));				//disable PLL0
	MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	while(MC_ME.GS.B.S_MTRANS);
	while(MC_ME.GS.B.S_CURRENT_MODE != mode);

	*uk_mode = *uk_mode|(1<<6);					//enable PLL0
	MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	while(MC_ME.GS.B.S_MTRANS);
	while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	while(PLLDIG.PLL0SR.B.LOCK ==0);
  }
  else
  {//PLL1
	*uk_mode = *uk_mode&(~(1<<7));				//disable PLL1
	MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	while(MC_ME.GS.B.S_MTRANS);
	while(MC_ME.GS.B.S_CURRENT_MODE != mode);

	*uk_mode = *uk_mode|(1<<7);					//enable PLL1
	MC_ME.MCTL.R =(mode<<28)|ME_KEY;
	MC_ME.MCTL.R =(mode<<28)|ME_IKEY;
	while(MC_ME.GS.B.S_MTRANS);
	while(MC_ME.GS.B.S_CURRENT_MODE != mode);
	while(PLLDIG.PLL0SR.B.LOCK ==0);
  }
}

/********************************************************************************/
/*	NAME	: Set_pll															*/
/*	INPUTS	: uint16_t PLL_value		- value of PLL in MHz					*/
/*			uint8_t which_PLL			- which PLL								*/
/*			uint8_t mode				- for which mode						*/
/*	OUTPUTS	: void																*/
/*	DESCRIPT: configuration digital PLL											*/
/********************************************************************************/
void Set_pll(uint16_t PLL_value, uint8_t which_PLL, uint8_t mode)
{
  uint8_t mfd = 0, prediv = 0, rfdphi = 0;			
  uint8_t b_mfd = 0, b_prediv = 0, b_rfdphi = 0;	//b=best value - with minimal error
  uint16_t vco = 0,error = -1;

  if (which_PLL ==0)
  {//PLL0
	for (mfd = 8;mfd<128;mfd++)
	{
	  for (prediv = 1;prediv<8;prediv++)
	  {
		for (rfdphi = 1;rfdphi<64;rfdphi++)
		{
		  vco = 40*mfd/prediv;
		  if ((vco>600)&&(vco<1250))
		  { //vco must be in range 600-1250 MHz
			if ((error>abs(PLL_value-vco/rfdphi))||((error==abs(PLL_value-vco/rfdphi))&&(vco>40*b_mfd/b_prediv)&&((40*mfd%prediv)==0)&&((vco%rfdphi)==0)))
			{ //better values for PLL or the same but with higher VCO - better jitter 
			  b_mfd = mfd;
			  b_prediv = prediv;
			  b_rfdphi = rfdphi;
			  error = abs (PLL_value-vco/rfdphi);
			}
		  }
		}
	  }
	}
	MC_CGM.AC3_SC.B.SELCTL = 1; 		// Select XOSC as source of PLL
	PLLDIG.PLL0DV.B.RFDPHI = b_rfdphi;	// PHI output: VCO divide by 
	PLLDIG.PLL0DV.B.RFDPHI1 = b_rfdphi;	// PHI1 output: VCO divide by  
	PLLDIG.PLL0DV.B.PREDIV = b_prediv;	// Divide by 
	PLLDIG.PLL0DV.B.MFD = b_mfd;		// Multiply by 
  }
  else
  {//PLL1
	for (mfd = 16;mfd<34;mfd++)
	{
	  for (rfdphi = 1;rfdphi<64;rfdphi++)
	  {
		vco = 40*mfd;
		if ((vco>600)&&(vco<1250))
		{//vco must be in range 600-1250 MHz
		  if ((error>abs(PLL_value-vco/(2*rfdphi)))||((error==abs(PLL_value-vco/(2*rfdphi)))&&(vco>40*b_mfd)))
		  { //better values for PLL or the same but with higher VCO - better jitter
			b_mfd = mfd;
			b_rfdphi = rfdphi;
			error = abs (PLL_value-vco/(2*rfdphi));
		  }
		}	
	  }
	}
	MC_CGM.AC3_SC.B.SELCTL = 1; 		// Select XOSC as source of PLL
	PLLDIG.PLL1DV.B.RFDPHI = b_rfdphi;	// PHI output: VCO divide by 2
	PLLDIG.PLL1DV.B.MFD = b_mfd;		// Multiply by 30
  }
  
  MC_ME.MCTL.R =(mode<<28)|ME_KEY;
  MC_ME.MCTL.R =(mode<<28)|ME_IKEY;  
  while(MC_ME.GS.B.S_MTRANS);
  while(MC_ME.GS.B.S_CURRENT_MODE != mode);

  //wait for locked PLL loop
  if (which_PLL == 0) while(!MC_ME.GS.B.S_PLL0);
  else while(!MC_ME.GS.B.S_PLL1);
}


void eTimer_CONFIG_PINS(uint8_t module, uint8_t	channel, uint8_t direct, uint8_t slew_rate)
{
  switch (module)
  {
	case 0://eTimer_0
	{
	  switch (channel)
	  {
		case 0:
		{
		  if (direct == DIRECTION_OUT)//ETC0 - OUT
		  {
		  SIUL2.MSCR[0].B.SSS	= 0x1;	//A[0]- ETC0
		  SIUL2.MSCR[0].B.OBE 	= 0x1;
		  SIUL2.MSCR[0].B.IBE 	= 0x0;
		  SIUL2.MSCR[0].B.SRC 	= slew_rate;	
		  }
		  else //ETC0 - IN
		  {
		  SIUL2.IMCR[59].B.SSS = 0x1;	//D[10]- ETC0
		  //SIUL2.IMCR[59].B.SSS = 0x2;	//A[0]- ETC0
	  	  SIUL2.MSCR[58].B.IBE = 0x1;
		  SIUL2.MSCR[58].B.OBE = 0x0;
		  SIUL2.MSCR[58].B.SRC = slew_rate;
		  }
		}break;
		case 1:
		{
		  if (direct == DIRECTION_OUT)//ETC1 - OUT
		  {
		  SIUL2.MSCR[1].B.SSS 	= 0x1;	//A[1]- ETC1
		  SIUL2.MSCR[1].B.OBE 	= 0x1;
		  SIUL2.MSCR[1].B.IBE 	= 0x0;
		  SIUL2.MSCR[1].B.SRC 	= slew_rate;	
		  }
		  else
		  {
		  //ETC1 - IN
		  SIUL2.IMCR[60].B.SSS = 0x1;	//D[11]- ETC1
		  //SIUL2.IMCR[60].B.SSS = 0x2;	//A[1]- ETC1
    	  SIUL2.MSCR[59].B.IBE = 0x1;
		  SIUL2.MSCR[59].B.OBE = 0x0;
		  SIUL2.MSCR[59].B.SRC = slew_rate;
		  }
		}break;
		case 2:
		{
		  if (direct == DIRECTION_OUT)//ETC2 - OUT
		  {
		  SIUL2.MSCR[2].B.SSS 	= 0x1;	//A[2]- ETC2
		  SIUL2.MSCR[2].B.OBE 	= 0x1;
		  SIUL2.MSCR[2].B.IBE 	= 0x0;
		  SIUL2.MSCR[2].B.SRC 	= slew_rate;	
		  }
		  else
		  {
		  //ETC2 - IN
		  SIUL2.IMCR[61].B.SSS = 0x1;	//F[0]- ETC2
		  //SIUL2.IMCR[61].B.SSS = 0x2;	//A[2]- ETC2
		  SIUL2.MSCR[80].B.IBE = 0x1;
		  SIUL2.MSCR[80].B.OBE = 0x0;
		  SIUL2.MSCR[80].B.SRC = slew_rate;
		  }
		}break;
		case 3:
		{
		  if (direct == DIRECTION_OUT)//ETC3 - OUT
		  {	
		  SIUL2.MSCR[3].B.SSS 	= 0x1;	//A[3]- ETC3
		  SIUL2.MSCR[3].B.OBE 	= 0x1;
		  SIUL2.MSCR[3].B.IBE 	= 0x0;
		  SIUL2.MSCR[3].B.SRC 	= slew_rate;
		  }
		  else
		  {
		  //ETC3 - IN
		  SIUL2.IMCR[62].B.SSS = 0x1;	//D[14]- ETC3
		  //SIUL2.IMCR[62].B.SSS = 0x2;	//A[3]- ETC3
		  SIUL2.MSCR[62].B.IBE = 0x1;
		  SIUL2.MSCR[62].B.OBE = 0x0;
		  SIUL2.MSCR[62].B.SRC = slew_rate;
		  }
		}break;
		case 4:
		{
		  if (direct == DIRECTION_OUT)//ETC4 - OUT
		  {
		  SIUL2.MSCR[4].B.SSS 	= 0x3;	//A[4]- ETC4
		  SIUL2.MSCR[4].B.OBE 	= 0x1;
		  SIUL2.MSCR[4].B.IBE 	= 0x0;
		  SIUL2.MSCR[4].B.SRC 	= slew_rate;
		  /*SIUL2.MSCR[43].B.SSS 	= 0x1;	//C[11]- ETC4
	  	  SIUL2.MSCR[43].B.OBE 	= 0x1;
		  SIUL2.MSCR[43].B.IBE 	= 0x0;
		  SIUL2.MSCR[43].B.SRC 	= slew_rate;*/
		  }
		  else
		  {
		  //ETC4 - IN
		  SIUL2.IMCR[63].B.SSS = 0x1;	//B[14]- ETC4
		  /*SIUL2.IMCR[63].B.SSS = 0x2;	//G[3]- ETC4
		  SIUL2.IMCR[63].B.SSS = 0x3;	//A[4]- ETC4
		  SIUL2.IMCR[63].B.SSS = 0x4;	//C[11]- ETC4*/
		  SIUL2.MSCR[30].B.IBE = 0x1;
		  SIUL2.MSCR[30].B.OBE = 0x0;
		  SIUL2.MSCR[30].B.SRC = slew_rate;
		  }
		}break;
		case 5:
		{
		  if (direct == DIRECTION_OUT)//ETC5 - OUT
		  {
		  SIUL2.MSCR[44].B.SSS 	= 0x1;	//C[12]- ETC5
		  SIUL2.MSCR[44].B.OBE 	= 0x1;
		  SIUL2.MSCR[44].B.IBE	= 0x0;
		  SIUL2.MSCR[44].B.SRC	= slew_rate;
		  /*SIUL2.MSCR[77].B.SSS 	= 0x1;	//E[13]- ETC5
	  	  SIUL2.MSCR[77].B.OBE 	= 0x1;
		  SIUL2.MSCR[77].B.IBE 	= 0x0;
		  SIUL2.MSCR[77].B.SRC 	= slew_rate;*/
		  }
		  else
		  {
		  //ETC5 - IN
		  SIUL2.IMCR[64].B.SSS = 0x1;	//B[8]- ETC5
		  /*SIUL2.IMCR[64].B.SSS = 0x2;	//G[4]- ETC5
		  SIUL2.IMCR[64].B.SSS = 0x3;	//C[12]- ETC5
		  SIUL2.IMCR[64].B.SSS = 0x4;	//E[13]- ETC5*/
		  SIUL2.MSCR[24].B.IBE = 0x1;
		  SIUL2.MSCR[24].B.OBE = 0x0;
		  SIUL2.MSCR[24].B.SRC = slew_rate;	
		  }
		}break;
	  }
	}break;
  }
}

/********************************************************************************/
/*	NAME	: init_peripheral_eTimer											*/
/*	INPUTS	: void																*/
/*	OUTPUTS	: void																*/
/*	DESCRIPT: allow peripherals which are used for CRC validation				*/
/********************************************************************************/
void init_peripheral_eTimer(void)
{
  	MC_ME.RUN_PC[1].R 		= 0xFF;		//enable PCT[1] in all modes
	MC_ME.PCTL247.R			= 0x1;		//eTimer 0
    MC_ME.PCTL137.R			= 0x1;		//eTImer 1
    MC_ME.PCTL245.R			= 0x1;		//eTimer 2
	MC_ME.PCTL204.R 		= 0x1;		//LIN_0 - UART communication
  
	//source clock for UART - RC oscillator without divider
	MC_CGM.AC0_SC.B.SELCTL 	= 0x0;
	MC_CGM.AC0_DC0.R 		= 0x80000000;
	
	MC_ME.RUN_PC[0].R 		= 0x0;	//disable clock to other unused peripherals
}
