/*
 * Main.c
 *
 *  Created on: Mar 11, 2013
 *      Author: B34981
 */

/**********************************************************************************************
* Includes
**********************************************************************************************/
#include <hidef.h> 
#include "derivative.h"

#include "main.h"
#include "GPIO.h"
#include "CPMU.h"
#include "FTMRZ.h"
#include "Comms.h"

/**********************************************************************************************
* External objects
**********************************************************************************************/


/**********************************************************************************************
* Constants and macros
**********************************************************************************************/
//#define DEBUG


/**********************************************************************************************
* Global variables
**********************************************************************************************/
UINT8 u8flashErased = 0;
UINT16 u16timeoutCounter = 0;

/* Used as external variables on other files */
UINT8 u8BootPhraseRcvd = 0;			/* Flag signaling that a complete Boot Phrase has been received */
BootPhraseStruct BP;				/* Boot Phrase structure to receive data from Comms */
ProgramStruct PS;					/* Data structure used by the programming routine */

/* Function pointers to RAM */
pt2Func FTMRZ_Erase_Flash_RAM;
pt2Func FTMRZ_Prog_Phrase_RAM;

/**********************************************************************************************
* Local types
**********************************************************************************************/


/**********************************************************************************************
* Local function prototypes
*********************************************************************************************/
UINT8 Verify_Phrase_Checksum(void);
void Copy_Shadow(void);
pt2Func Get_RAM_Ptr (pt2Func ptr);
void Jump_Main_Application(void);


/**********************************************************************************************
* Local variables
**********************************************************************************************/


/**********************************************************************************************
* Local functions
**********************************************************************************************/


/**********************************************************************************************
* Global functions
**********************************************************************************************/

/***********************************************************************************************
*
* @brief    main - Program entry
* @param    none
* @return   none
*
************************************************************************************************/  
void main(void) {
	UINT16 u16BP_ptr = 0, u16PS_ptr = 0;		/* Boot Phrase and Program Structure pointers */  	
		
	PLL_Init();							/* Configure clock to 32 MHz from 4 MHz oscillator */
	GPIO_Init();						/* Configure GPIOs and sets its initial values */
	Comms_Init();						/* Configure SCI to receive and transmit LIN data */
	FTMRZ_Init();						/* Configure FTMRZ clocking */
	RTI_Init();							/* Configure RTI module to count each 100 ms */
	
	Copy_Shadow();						/* Move shadow functions (Flash related only) to RAM */
		
	/* Get RAM pointer of function after being copied */
	FTMRZ_Erase_Flash_RAM = Get_RAM_Ptr(FTMRZ_Erase_Flash);		
	FTMRZ_Prog_Phrase_RAM = Get_RAM_Ptr(FTMRZ_Prog_Phrase);
		
	#if ACTIVITY_LED_ENABLE
		ACTIVITY_LED = LED_ON;										/* LED ON to indicate bootloader in timeout mode */
	#endif
	
  for(;;) {

	  	Comms_Rx_Data();										/* Check for received serial data, must return */
	  
	  	if(u8BootPhraseRcvd)									/* If a complete booth phrase was received */
		{
	  		u8BootPhraseRcvd = 0;	  		
			#if ACTIVITY_LED_ENABLE
	  			ACTIVITY_LED = ~ACTIVITY_LED;						/* Toggle LED to indicate reception activity */	  	
			#endif
			if(Verify_Phrase_Checksum())						/* Check that phrase checksum matches */
			{
				if(!u8flashErased)								/* If the flash hasn't been erased, erase it */
				{					
					RTI_Disable();								/* Stop the RTI timer, no timeout required anymore */
					#if ACTIVITY_LED_ENABLE
						ACTIVITY_LED = LED_ON;
					#endif
					FTMRZ_Erase_Flash_RAM();					/* Jump to RAM and Erase Flash, return to Flash when done. */
					#if ACTIVITY_LED_ENABLE
						ACTIVITY_LED = LED_OFF;
					#endif
					u8flashErased = 1;							/* Indicate that flash has been erased */	
				}											
												
				u16PS_ptr = BP.F.PhraseAddress[2] & 0x7;		/* Store the last three bits that represent the shift of the write within the Flash write boundary */ 				
				
				PS.Add.Byte[1] = BP.F.PhraseAddress[0];
				PS.Add.Byte[2] = BP.F.PhraseAddress[1];
				PS.Add.Byte[3] = BP.F.PhraseAddress[2] & 0xF8;	/* Remove the shift eliminating the last three bits */
														
				BP.F.PhraseSize -= 4; 							/* Remove address bytes (3) and CRC (1) from the actual data counter */
				for(u16BP_ptr = 0; u16BP_ptr < BP.F.PhraseSize; u16BP_ptr++, u16PS_ptr++)	/* Loop through the boot phrase data moving it to the program structure */		
				{											
					PS.Data[u16PS_ptr % 8] = BP.F.PhraseData[u16BP_ptr];			/* Store only in groups of 8 bytes on the program structure, consider shift pointer */
					
					if((u16PS_ptr % 8 == 7) || u16BP_ptr == (BP.F.PhraseSize - 1))	/* If the last PS.Data element has been updated or this is the last element of the BP */
					{
						if(PS.Add.DWord == 0xFFFFF8)		 	/* If attempt to write the startup vector, DONE, jump to user application */
						{
							Comms_Tx_Ack(OK);					/* Indicate correct reception and flashing */
							Jump_Main_Application();
						}
						else if((PS.Add.DWord < BOOTLOADER_START_ADD) || (PS.Add.DWord >= FLASH_VECTOR_SECTOR_ADD)) /* Only program data if it is NOT on the bootloader FLASH area */
						{
							FTMRZ_Prog_Phrase_RAM();				/* Program the current PS */																				
						}						
						
						PS.Add.DWord += 8;						/* Advance the program pointer 8 bytes */
						
					}
				}															
				Comms_Tx_Ack(OK);								/* Indicate correct reception and flashing */				
			}
			else
			{
				Comms_Tx_Ack(CRC_Error);						/* Indicate boot phrase was incorrect, expect retry */
			}
									
		}	  	    
		
	  	if(RTI_CheckTimer())									/* If 100ms have elapsed */
	  	{
	  		u16timeoutCounter++;
	  		if(u16timeoutCounter >= TIMEOUT)					/* If timeout */
	  		{
	  			Jump_Main_Application();
	  		}
	  	}
  }  
}


/***********************************************************************************************
*
* @brief    Verify_Phrase_Checksum - Calculate S19 phrase checksum and compare it with the received value
* @param    none
* @return   1 - checksum matches, 0 - checksum error
*
************************************************************************************************/  
UINT8 Verify_Phrase_Checksum()
{
#if CHECK_PHRASE_CHECKSUM
	
	UINT16 u8cntr = 0;
	UINT16 u16checksum = 0;
	
	for(u8cntr = 0; u8cntr < BP.F.PhraseSize; u8cntr++)		/* Add all the elements on the phrase except the checksum itself */
	{		
		u16checksum += BP.Byte[u8cntr];
	}	
	
	u16checksum = ~u16checksum;								/* Ones complement on checksum */
	
	if((UINT8)(u16checksum) == BP.Byte[BP.F.PhraseSize])	/* If checksum matches the last byte on the prhase (CRC), send a match */
	{
		return 1;		/* Report that checksum matches */
	}
	else
	{
		return 0;		/* Report checksum error */
	}
#else					/* If checksum revision is not required */	
	return 1;			/* Report that checksum matches */
#endif
}


/***********************************************************************************************
*
* @brief    Copy_Shadow - Copy functions on shadow segment to RAM (only Flash related funcions that must run on RAM)
* @param    none
* @return   none
*
************************************************************************************************/  
void Copy_Shadow()
{
   UINT16 i;
   for (i = 0; i < SHADOW_SIZE_IN_WORDS; i++)
   {
	  ((UINT16 *)SHADOW_RAM_ADD)[i] = ((UINT16 *)SHADOW_ROM_ADD)[i];
   }
}


/***********************************************************************************************
*
* @brief    Get_RAM_Ptr - Get the function pointer on the copied Shadow segment on RAM
* @param    pt2Func ptr - Pointer of the function we want to get the RAM pointer of (must be located on shadow ROM)
* @return   Pointer of the function on the RAM segment
*
************************************************************************************************/  
pt2Func Get_RAM_Ptr (pt2Func ptr)
{  
   UINT8 * aux=((UINT8 *)SHADOW_RAM_ADD)+(((UINT8 *)ptr)-((UINT8 *)SHADOW_ROM_ADD));
   if (aux < ((UINT8 *)SHADOW_RAM_ADD) || aux>((UINT8*)SHADOW_RAM_ADD)+ SHADOW_SIZE_IN_WORDS*2)
   {
      aux = 0;
   }
   return (pt2Func)aux; 
}


/***********************************************************************************************
*
* @brief    Jump_Main_Application - Restore all modified registers to default values and jump to the main application
* @param    none
* @return   none
*
************************************************************************************************/  
void Jump_Main_Application()
{
	pt2Func MainApplication = (void*)APPLICATION_START_ADD;		/* Create application pointer */	
	#if ACTIVITY_LED_ENABLE
		ACTIVITY_LED = LED_OFF;
	#endif	
	
	CPMUCLKS = 0x80;
	CPMUREFDIV = 0x0F;
	CPMUSYNR = 0x58;
	CPMUPOSTDIV = 0x03;
	CPMUOSC = 0x00;
	CPMUINT = 0x00;
	#if ACTIVITY_LED_ENABLE	
		ACTIVITY_LED_DDR = 0;
	#endif
	FCLKDIV = 0x00;
	Comms_Reset_Reg();					/* Reset communications driver registers to default values */
	MainApplication();					/* Jump to user application */
}
