

/*
 * File:		UartToIIC.c
 * Purpose:		Main process
 *
 */

#include "common.h"
#include "iic.h"
#include "uart.h"

#define FC_CMD_ACK					0xfc
#define FC_CMD_NACK					0x03
#define FC_CMD_IDENT				0x49
#define FC_CMD_ERASE				0x45
#define FC_CMD_WRITE				0X57
#define FC_CMD_READ					0x52
#define FC_CMD_QUIT					0x51
#define FC_CMD_ERROR				0x01
#define FC_CMD_HOOK					0x02


#define IIC_SLAVE_ADDRESS			0xB0


unsigned char m_ucIIC_TxBuff[300];
unsigned char m_ucIIC_RxBuff[300];
uint32_t m_uiIIC_TxIndex;
uint32_t m_uiIIC_RxIndex;
uint32_t *m_pFrameLength;

/********************************************************************/
// local function 
/********************************************************************/
uint8_t UartSendData(uint8_t *pData,uint32_t uiLength);
uint8_t IIC_SendData(uint8_t *pData,uint32_t uiLength);
uint8_t IIC_ReadData(uint8_t *pData,uint32_t uiLength);
void IIC_GenerateVarifySum(uint8_t *pData,uint32_t uiLength);
uint8_t IIC_CheckResponse(uint8_t Command,uint32_t Interval,uint32_t OverTime);
void Delay_uS(uint32_t Delay_uS);

/********************************************************************/

uint8_t UartSendData(uint8_t *pData,uint32_t uiLength)
{
	uint32_t i;
	for(i=0;i<uiLength;i++)
	{
		uart0_putchar(UART0_BASE_PTR,pData[i]);
	}
	return 1;
}

uint8_t IIC_SendData(uint8_t *pData,uint32_t uiLength)
{
	uint32_t i;
    uint32_t ucState;
    if( I2C0_S & I2C_S_ARBL_MASK )
    {
        I2C0_S |= I2C_S_ARBL_MASK;
    }
	ucState = I2C_Start();
	ucState = I2C_CycleWrite(IIC_SLAVE_ADDRESS);
	for(i=0;i<uiLength;i++)
	{
		ucState = I2C_CycleWrite(pData[i]);
	}
	ucState = I2C_Stop();
	return ucState;
}

uint8_t IIC_ReadData(uint8_t *pData,uint32_t uiLength)
{
	uint32_t i;
	uint32_t ucState;
	uint8_t Dummy;
	uiLength = uiLength - 1;
    if( I2C0_S & I2C_S_ARBL_MASK )
    {
        I2C0_S |= I2C_S_ARBL_MASK;
    }
	ucState = I2C_Start();
	ucState = I2C_CycleWrite(IIC_SLAVE_ADDRESS|0x01);

	// dummy read
	Dummy = I2C_CycleRead(0);
	for(i=0;i<uiLength;i++)
	{
		pData[i] = I2C_CycleRead(0);
	}

	// read the last byte, don't send ACK
	pData[i] = I2C_CycleRead(1);
	ucState = I2C_Stop();
	return ucState;
}

void IIC_GenerateVarifySum(uint8_t *pData,uint32_t uiLength)
{
	uint32_t i;
	uint8_t Sum;
	Sum = 0;
	for(i=0;i<uiLength;i++)
	{
		Sum += pData[i]; 
	}
	pData[uiLength] = Sum;
}

void Delay_uS(uint32_t Delay_uS)
{
	uint32_t j,i;
    for(i=0;i<Delay_uS;i++)
    {
      for(j=0;j<50;j++);
      {
        asm("nop");
      }
    }
}

uint8_t IIC_CheckResponse(uint8_t Command,uint32_t Interval,uint32_t OverTime)
{
	uint32_t i,j;
	for(j=0;j<OverTime;j++)
	{
		for(i=0;i<Interval;i++)
		{
			Delay_uS(65);
		}
		IIC_ReadData(&m_ucIIC_RxBuff[0],2);
		if( (Command|0x80) == m_ucIIC_RxBuff[0] )
		{
			return 1;
		}
	}
	return 0;
}


void Memcpy_Byte(uint8_t * Dest,uint8_t * Src,uint32_t Size)
{
	while(Size--)
	{
		*Dest++ = *Src++;
	}
}

int main (void)
{
	uint32_t i;
	uint32_t uiFrameLength;

    // delay some to wait slave to complete initialization
    for(i=0;i<10000;i++);
    Init_I2C();
    m_pFrameLength = (uint32_t *)&m_ucIIC_TxBuff[0];
    //printf("start to get response from slave!\n");
    do{
    	m_uiIIC_RxIndex = 0;
        *m_pFrameLength = 0x06;
        m_ucIIC_TxBuff[4] = FC_CMD_HOOK;
        IIC_GenerateVarifySum(&m_ucIIC_TxBuff[0],*m_pFrameLength-1);
        if( IIC_SendData(&m_ucIIC_TxBuff[0],*m_pFrameLength))
        {
			// I2c error
			Init_I2C();
			continue;
        }
        if( IIC_ReadData(&m_ucIIC_RxBuff[0],2) )
        {
        	// I2c error
			Init_I2C();
			continue;
		}
		
	}while( ( m_ucIIC_RxBuff[0] != (FC_CMD_HOOK|0x80) )||
			( m_ucIIC_RxBuff[1] != 0xfc ) );
    
	// hook success
    do{
     	uart0_putchar(UART0_BASE_PTR,0xfc);
     	Delay_uS(500000);
     }while( (!( UART0_S1 & UART0_S1_RDRF_MASK ))&&( UART0_D != 0xfc));
     UART0_D;
     uart0_putchar(UART0_BASE_PTR,0xfc);
    while(1)
    {
		m_uiIIC_TxIndex = 4;
		m_ucIIC_TxBuff[m_uiIIC_TxIndex++] = uart0_getchar(UART0_BASE_PTR);
		switch(m_ucIIC_TxBuff[4])
		{
		case FC_CMD_ACK:
			uart0_putchar(UART0_BASE_PTR,FC_CMD_ACK);
			break;
	   	case FC_CMD_IDENT:
	   		m_uiIIC_TxIndex ++;
	   		*m_pFrameLength = m_uiIIC_TxIndex;
	   		IIC_GenerateVarifySum(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex-1);
	   		IIC_SendData(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex);

	   		if(IIC_CheckResponse(FC_CMD_IDENT,10,10))
	   		{
				IIC_ReadData(&m_ucIIC_RxBuff[0],42);
				Memcpy_Byte((uint8_t *)&uiFrameLength,
							(uint8_t *)&m_ucIIC_RxBuff[1],sizeof(uint32_t));
				UartSendData(&m_ucIIC_RxBuff[5],uiFrameLength-5);
			}
			else
			{
				// no response 
				uart0_putchar(UART0_BASE_PTR,FC_CMD_NACK);
			}
			break;
		case FC_CMD_ERASE:
			do{
				m_ucIIC_TxBuff[m_uiIIC_TxIndex++] = uart0_getchar(UART0_BASE_PTR);
			}while( m_uiIIC_TxIndex < (sizeof(uint32_t)+5) );
			m_uiIIC_TxIndex ++;
	   		*m_pFrameLength = m_uiIIC_TxIndex;
	   		IIC_GenerateVarifySum(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex-1);
			IIC_SendData(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex);

			if(IIC_CheckResponse(FC_CMD_ERASE,20000,10))
	   		{
				IIC_ReadData(&m_ucIIC_RxBuff[0],2);
				UartSendData(&m_ucIIC_RxBuff[1],1);
			}
			else
			{
				// no response 
				uart0_putchar(UART0_BASE_PTR,FC_CMD_NACK);
			}
			break;
		case FC_CMD_WRITE:
			do{
				m_ucIIC_TxBuff[m_uiIIC_TxIndex++] = uart0_getchar(UART0_BASE_PTR);
				
			}while(!(( m_uiIIC_TxIndex > 10 )&&(m_uiIIC_TxIndex > (m_ucIIC_TxBuff[9] + 9))) );
			m_uiIIC_TxIndex ++;
	   		*m_pFrameLength = m_uiIIC_TxIndex;
	   		IIC_GenerateVarifySum(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex-1);
			IIC_SendData(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex);

			if(IIC_CheckResponse(FC_CMD_WRITE,(m_ucIIC_TxBuff[9]/4)*70,10))
	   		{
				IIC_ReadData(&m_ucIIC_RxBuff[0],2);
				UartSendData(&m_ucIIC_RxBuff[1],1);
			}
			else
			{
				// no response 
				uart0_putchar(UART0_BASE_PTR,FC_CMD_NACK);
			}
			break;
		case FC_CMD_READ:
			do{
				m_ucIIC_TxBuff[m_uiIIC_TxIndex++] = uart0_getchar(UART0_BASE_PTR);
			}while( m_uiIIC_TxIndex <= (sizeof(uint32_t) + 5 ) );
			m_uiIIC_TxIndex ++;
	   		*m_pFrameLength = m_uiIIC_TxIndex;
	   		IIC_GenerateVarifySum(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex-1);
			IIC_SendData(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex);

			if(IIC_CheckResponse(FC_CMD_READ,50,10))
	   		{
				IIC_ReadData(&m_ucIIC_RxBuff[0],1+m_ucIIC_TxBuff[m_uiIIC_TxIndex-2]);
				UartSendData(&m_ucIIC_RxBuff[1],m_ucIIC_TxBuff[m_uiIIC_TxIndex-2]);
			}
			else
			{
				// no response 
				uart0_putchar(UART0_BASE_PTR,FC_CMD_NACK);
			}
			break;
		case FC_CMD_QUIT:
			m_uiIIC_TxIndex ++;
	   		*m_pFrameLength = m_uiIIC_TxIndex;
	   		IIC_GenerateVarifySum(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex-1);
	   		IIC_SendData(&m_ucIIC_TxBuff[0],m_uiIIC_TxIndex);
	   		break;
		default:
		break;
		}
	}
}
/********************************************************************/
