

#include <mqx.h>
#include <bsp.h>
#include <io_prv.h>
#include <fio_prv.h>
#include "i2c.h"
#include "i2c_int_prv.h"
#include "i2c_ki2c_prv.h"
#include <string.h>

typedef struct fb_ki2c_info_struct
{  
    KI2C_INIT_STRUCT init;
    I2C_MemMapPtr    i2c_ptr;
    void             (_CODE_PTR_ old_isr)(pointer);
    pointer          old_isr_data;
    uint_32          vector;
    uint_8           mode;

    uint_32      index;
    // master
    LWSEM_STRUCT lwsem;
    uchar_ptr    data;
    uint_32      data_len;
    uint_32      index_rx_tx;
    uint_32      index_repeat_start;
    uchar        buf_x_stop[2];

    // slave
    LWSEM_STRUCT lwsem_slave_rx;
    LWSEM_STRUCT lwsem_slave_tx;
    uint_8       reg_addr;
    uchar_ptr    buf_rx;
    uchar_ptr    buf_tx;
    uint_8       buf_rx_empty;
    uint_8       buf_tx_empty;
    uint_8       buf_rx_index; // need it as index can not keep, to get 0xff as stop command
    uint_8       task_block;
} FB_KI2C_INFO_STRUCT, _PTR_ FB_KI2C_INFO_STRUCT_PTR; 

typedef volatile struct fb_ki2c_info_struct * FB_VKI2C_INFO_STRUCT_PTR; 

extern uint_32 _ki2c_int_fb_init(IO_I2C_INT_DEVICE_STRUCT_PTR, char_ptr);
extern uint_32 _ki2c_int_fb_deinit (IO_I2C_INT_DEVICE_STRUCT_PTR, FB_VKI2C_INFO_STRUCT_PTR);
extern uint_32 _ki2c_int_fb_rx(IO_I2C_INT_DEVICE_STRUCT_PTR, uchar_ptr, uint_32);
extern uint_32 _ki2c_int_fb_tx(IO_I2C_INT_DEVICE_STRUCT_PTR, uchar_ptr, uint_32);
static void    _ki2c_isr(pointer);
uint_32        _ki2c_int_fb_ioctl(FB_VKI2C_INFO_STRUCT_PTR, uint_32, uint_32_ptr);

static const uint_32 BAUDRATE_MULT[] = { 1, 2, 4 };
static const uint_32 BAUDRATE_ICR[]= {  20, 22, 24, 26, 28, 30, 34, 40, 28, 32, 
    36, 40, 44, 48, 56, 68, 48, 56, 64, 72, 80, 88, 104, 128, 80, 96, 112, 128,
    144, 160, 192, 240, 160, 192, 224, 256, 288, 320, 384, 480, 320, 384, 448, 
    512, 576, 640, 768, 960, 640, 768, 896, 1024, 1152, 1280, 1536, 1920, 1280, 
    1536, 1792, 2048, 2304, 2560, 3072, 2840 };

#define REPEAT_INDEX_TX 0xffffffff
#define REPEAT_INDEX_RX 2
#define I2C_FB_DATA_LEN(len) len+2
#define REG_ADDR_SW_STOP 0xff
//#define DEBUG_ENABLE



uint_32 _ki2c_int_fb_install
    (
      char_ptr              identifier,
      KI2C_INIT_STRUCT_CPTR init_data_ptr
    )
{
   return _io_i2c_int_install(identifier,
      (uint_32 (_CODE_PTR_)(pointer, char_ptr))_ki2c_int_fb_init,
      (uint_32 (_CODE_PTR_)(pointer, pointer))_ki2c_int_fb_deinit,
      (int_32 (_CODE_PTR_)(pointer, char_ptr, int_32))_ki2c_int_fb_rx,
      (int_32 (_CODE_PTR_)(pointer, char_ptr, int_32))_ki2c_int_fb_tx,
      (int_32 (_CODE_PTR_)(pointer, uint_32, uint_32_ptr))_ki2c_int_fb_ioctl, 
      (pointer)init_data_ptr);
}
static uint_32 _ki2c_set_baudrate 
    (
        uint_32 clock, 
        uint_32 baudrate
    ) 
{
    #define MAX_ICR 64
    uint_32 mult, icr, min, minmult = 0, minicr = 0;
    int_32  val;
    
    min = (uint_32)-1;
    /* We will find the nearest smallest diference in desired and real baudrate.
    ** This is transformed to find smallest clock diference.
    ** IIC baud rate = bus speed (Hz)/(mul * SCL divider) =>
    ** IIC baud rate * mul * SCL divider <= clock
    */
    for (mult = 0; mult < 3; mult++) 
    {
        for (icr = 0; icr < MAX_ICR; icr++) 
        {
            val = BAUDRATE_MULT[mult] * BAUDRATE_ICR[icr] * baudrate - clock;
            if (val < 0) val = -val;
            if (min > val) 
            {
                min = val;
                minmult = mult;
                minicr = icr;
            }
        }
    }
    return I2C_F_MULT(minmult) | I2C_F_ICR(minicr);
}

static uint_32 _info_struct_init
    (
      KI2C_INIT_STRUCT_PTR io_init_ptr,
      pointer _PTR_        io_info_ptr_ptr,
      char_ptr             open_name_ptr
    )
{
    I2C_MemMapPtr           i2c_ptr;
    FB_VKI2C_INFO_STRUCT_PTR   io_info_ptr;

    i2c_ptr = _bsp_get_i2c_base_address (io_init_ptr->CHANNEL);
    if (NULL == i2c_ptr)
    {
        return I2C_ERROR_CHANNEL_INVALID;
    }

    io_info_ptr = (FB_VKI2C_INFO_STRUCT_PTR)_mem_alloc_system_zero ((uint_32)sizeof (FB_KI2C_INFO_STRUCT));
    if (NULL == io_info_ptr) 
    {
        return MQX_OUT_OF_MEMORY;
    }
    _mem_set_type ((pointer)io_info_ptr, MEM_TYPE_IO_I2C_INFO_STRUCT);

    _bsp_i2c_io_init (io_init_ptr->CHANNEL);

    *io_info_ptr_ptr                              = (pointer)io_info_ptr;
    io_info_ptr->init                             = *io_init_ptr;
    io_info_ptr->i2c_ptr                          = i2c_ptr;
    io_info_ptr->old_isr                          = NULL;
    io_info_ptr->old_isr_data                     = NULL;
    io_info_ptr->vector                           = 0;
    io_info_ptr->mode                             = io_init_ptr->MODE;

    io_info_ptr->buf_rx_empty = FALSE;
    io_info_ptr->buf_tx_empty = TRUE;
    io_info_ptr->task_block   = 0;
    io_info_ptr->buf_x_stop[1]= 0xff;
    _lwsem_create((LWSEM_STRUCT_PTR)&(io_info_ptr->lwsem)         , 0);
    _lwsem_create((LWSEM_STRUCT_PTR)&(io_info_ptr->lwsem_slave_rx), 0);
    _lwsem_create((LWSEM_STRUCT_PTR)&(io_info_ptr->lwsem_slave_tx), 0);

    i2c_ptr->C1 = 0x00;
    i2c_ptr->S = 0x00;
    i2c_ptr->A1 = io_init_ptr->STATION_ADDRESS << 1;
    i2c_ptr->F = _ki2c_set_baudrate( BSP_I2C_CLOCK, io_init_ptr->BAUD_RATE ); 
    printf("i2c_ptr->F:%x\n", i2c_ptr->F);

    i2c_ptr->C1 |= I2C_C1_IICEN_MASK;
    return I2C_OK;
}

uint_32 _ki2c_int_fb_init
    (
      IO_I2C_INT_DEVICE_STRUCT_PTR int_io_dev_ptr,
      char_ptr                     open_name_ptr
    )
{
    I2C_MemMapPtr                   i2c_ptr;
    FB_VKI2C_INFO_STRUCT_PTR        io_info_ptr;
    KI2C_INIT_STRUCT_PTR            i2c_init_ptr;
    uint_32                         vector, result;

    i2c_init_ptr = (KI2C_INIT_STRUCT_PTR)(int_io_dev_ptr->DEV_INIT_DATA_PTR);
    result = _info_struct_init (i2c_init_ptr,
                                &(int_io_dev_ptr->DEV_INFO_PTR),
                                open_name_ptr);
    if (result)
    {
        return result;
    }

    io_info_ptr = int_io_dev_ptr->DEV_INFO_PTR;
    i2c_ptr = io_info_ptr->i2c_ptr;
    vector = _bsp_get_i2c_vector (i2c_init_ptr->CHANNEL);
    if (0 == vector)
    {
        return I2C_ERROR_CHANNEL_INVALID;
    }
    io_info_ptr->vector = vector;
    io_info_ptr->old_isr_data = _int_get_isr_data (vector);
    io_info_ptr->old_isr = _int_install_isr (vector, _ki2c_isr, (pointer)io_info_ptr);
    _bsp_int_init((IRQInterruptIndex)vector, i2c_init_ptr->LEVEL, i2c_init_ptr->SUBLEVEL, TRUE);

    // b35970{ 
    //i2c_ptr->FLT = 0x1f;
    // }
    i2c_ptr->C1 |= I2C_C1_IICIE_MASK;

    return result;
}
uint_32 _ki2c_int_fb_deinit
    (
      IO_I2C_INT_DEVICE_STRUCT_PTR int_io_dev_ptr,
      FB_VKI2C_INFO_STRUCT_PTR        io_info_ptr
    )
{
    I2C_MemMapPtr                   i2c_ptr;
    KI2C_INIT_STRUCT_PTR            i2c_init_ptr;

    if ((NULL == io_info_ptr) || (NULL == int_io_dev_ptr)) 
    {
        return I2C_ERROR_INVALID_PARAMETER;
    }

    i2c_ptr = io_info_ptr->i2c_ptr;
    if (i2c_ptr->S & I2C_S_BUSY_MASK) 
    {
        return I2C_ERROR_DEVICE_BUSY;
    }

    i2c_ptr->C1 = 0x00;
    i2c_ptr->S = 0xFF; 

    i2c_init_ptr = (KI2C_INIT_STRUCT_PTR)(int_io_dev_ptr->DEV_INIT_DATA_PTR);
    _int_install_isr (io_info_ptr->vector, io_info_ptr->old_isr, io_info_ptr->old_isr_data);
    _mem_free (int_io_dev_ptr->DEV_INFO_PTR);
    int_io_dev_ptr->DEV_INFO_PTR = NULL;

    return I2C_OK;
}

uint_32 _ki2c_int_fb_ioctl
    (
      FB_VKI2C_INFO_STRUCT_PTR io_info_ptr,
      uint_32               cmd,
      uint_32_ptr           param_ptr
    )
{
    I2C_MemMapPtr            i2c_ptr;
    uint_32                  result = MQX_OK;
    uint_32                  icr, mult;

    i2c_ptr = io_info_ptr->i2c_ptr;

    switch (cmd) 
    {
        case IO_IOCTL_I2C_SET_MASTER_MODE:
            io_info_ptr->mode = I2C_MODE_MASTER;
            break;

        case IO_IOCTL_I2C_SET_SLAVE_MODE:
            io_info_ptr->mode = I2C_MODE_SLAVE;
            i2c_ptr->A1 = I2C_A1_AD(*param_ptr);
            break;

        case IO_IOCTL_I2C_SET_BAUD:
            if (NULL == param_ptr)
                result = I2C_ERROR_INVALID_PARAMETER;
            else if (i2c_ptr->S & I2C_S_BUSY_MASK) 
                result = I2C_ERROR_DEVICE_BUSY;
            else
                i2c_ptr->F = _ki2c_set_baudrate( BSP_I2C_CLOCK, *param_ptr);
            break;

        default:
            break;
    }
    return result;
}
static void _start(I2C_MemMapPtr i2c_ptr)
{
    while(i2c_ptr->S & I2C_S_BUSY_MASK);

    do
    {
        i2c_ptr->S  |= I2C_S_ARBL_MASK;
        i2c_ptr->C1 |= (I2C_C1_MST_MASK|I2C_C1_TX_MASK);
    }while(i2c_ptr->S & I2C_S_ARBL_MASK);
}
static void _stop(I2C_MemMapPtr i2c_ptr)
{
    i2c_ptr->C1 &= (~ (I2C_C1_TX_MASK | I2C_C1_TXAK_MASK));
    i2c_ptr->S  |= (I2C_S_ARBL_MASK | I2C_S_IICIF_MASK);
    i2c_ptr->C1 &= ~I2C_C1_MST_MASK;

    while(i2c_ptr->S & I2C_S_BUSY_MASK);
}
static void _give_ack(I2C_MemMapPtr i2c_ptr)
{
    i2c_ptr->C1 &= ~I2C_C1_TXAK_MASK;
}
static void _give_nack(I2C_MemMapPtr i2c_ptr)
{
    i2c_ptr->C1 |= I2C_C1_TXAK_MASK;
}
static void _to_rx_mode(I2C_MemMapPtr i2c_ptr)
{
    i2c_ptr->C1 &= ~(I2C_C1_TX_MASK);
}
static void _to_tx_mode(I2C_MemMapPtr i2c_ptr)
{
    i2c_ptr->C1 |= I2C_C1_TX_MASK;
}
static void _repeat_start(I2C_MemMapPtr i2c_ptr)
{
    do
    {
        i2c_ptr->C1 |= I2C_C1_RSTA_MASK;
    }while(i2c_ptr->S & I2C_S_ARBL_MASK);
}
static void _clear_int_flag(I2C_MemMapPtr i2c_ptr)
{
    i2c_ptr->S  |= I2C_S_IICIF_MASK;
}
static uint_32 _fb_rx_master
    (
      FB_VKI2C_INFO_STRUCT_PTR        io_info_ptr,
      I2C_MemMapPtr                i2c_ptr,
      uchar_ptr                    buffer,
      uint_32                      length
    )
{
    if(i2c_ptr->S & I2C_S_BUSY_MASK)
        return 0;

    length = I2C_FB_DATA_LEN(length);
    io_info_ptr->data     = buffer;
    io_info_ptr->data_len = length;
    io_info_ptr->index    = 0;
    io_info_ptr->index_rx_tx = 0;
    io_info_ptr->index_repeat_start = REPEAT_INDEX_RX;

    _start(i2c_ptr);
    io_info_ptr->index++;
    i2c_ptr->D = (io_info_ptr->data[io_info_ptr->index-1] << 1) | I2C_OPERATION_WRITE; // send address

    _lwsem_wait((LWSEM_STRUCT_PTR)(&(io_info_ptr->lwsem)));

    if(io_info_ptr->index_rx_tx < 1)
        return 0;
    else
        return io_info_ptr->index_rx_tx - 1;
}
static uint_32 _fb_tx_master
    (
      FB_VKI2C_INFO_STRUCT_PTR        io_info_ptr,
      I2C_MemMapPtr                i2c_ptr,
      uchar_ptr                    buffer,
      uint_32                      length
    )
{
    i2c_ptr = io_info_ptr->i2c_ptr;
    
    io_info_ptr->data       = buffer;
    io_info_ptr->data_len   = I2C_FB_DATA_LEN(length);
    io_info_ptr->index      = 0;
    io_info_ptr->index_rx_tx = 0;
    io_info_ptr->index_repeat_start = REPEAT_INDEX_TX;
    io_info_ptr->buf_x_stop[0] = io_info_ptr->data[0];

    _start(i2c_ptr);
    io_info_ptr->index++;
    i2c_ptr->D = (io_info_ptr->data[io_info_ptr->index-1] << 1) | I2C_OPERATION_WRITE; // send address

    _lwsem_wait((LWSEM_STRUCT_PTR)(&(io_info_ptr->lwsem)));

    if(io_info_ptr->index_rx_tx < 2)
        return 0;
    else
        return io_info_ptr->index_rx_tx - 2;
}
static uint_32 _fb_rx_slave
    (
      FB_VKI2C_INFO_STRUCT_PTR     io_info_ptr,
      I2C_MemMapPtr                i2c_ptr,
      uchar_ptr                    buffer,
      uint_32                      length
    )
{
    io_info_ptr->buf_rx       = buffer;
    io_info_ptr->buf_rx_empty = TRUE;
    
    _lwsem_wait((LWSEM_STRUCT_PTR)(&(io_info_ptr->lwsem_slave_rx)));
    return io_info_ptr->buf_rx_index;
}
static uint_32 _fb_tx_slave
    (
      FB_VKI2C_INFO_STRUCT_PTR     io_info_ptr,
      I2C_MemMapPtr                i2c_ptr,
      uchar_ptr                    buffer,
      uint_32                      length
    )
{
    io_info_ptr->buf_tx       = buffer;
    io_info_ptr->buf_tx_empty = FALSE;
    _lwsem_wait((LWSEM_STRUCT_PTR)(&(io_info_ptr->lwsem_slave_tx)));
    return io_info_ptr->index - 1;
}
uint_32 _ki2c_int_fb_rx
    (
      IO_I2C_INT_DEVICE_STRUCT_PTR int_io_dev_ptr,
      uchar_ptr                    buffer,
      uint_32                      length
    )
{
    FB_VKI2C_INFO_STRUCT_PTR           io_info_ptr;
    io_info_ptr = int_io_dev_ptr->DEV_INFO_PTR;

    if(io_info_ptr->mode == I2C_MODE_MASTER)
        return _fb_rx_master(io_info_ptr, io_info_ptr->i2c_ptr, buffer, length);
    else
        return _fb_rx_slave(io_info_ptr, io_info_ptr->i2c_ptr, buffer, length);
}
uint_32 _ki2c_int_fb_tx
    (
      IO_I2C_INT_DEVICE_STRUCT_PTR int_io_dev_ptr,
      uchar_ptr                    buffer,
      uint_32                      length
    )
{
    FB_VKI2C_INFO_STRUCT_PTR io_info_ptr;
    io_info_ptr  = int_io_dev_ptr->DEV_INFO_PTR;
    if(io_info_ptr->mode == I2C_MODE_MASTER)
        return _fb_tx_master(io_info_ptr, io_info_ptr->i2c_ptr, buffer, length);
    else
        return _fb_tx_slave(io_info_ptr, io_info_ptr->i2c_ptr, buffer, length);
}
static void _isr_ii2c_master(FB_VKI2C_INFO_STRUCT_PTR io_info_ptr)
{
    I2C_MemMapPtr i2c_ptr = io_info_ptr->i2c_ptr;
    int i;

    _clear_int_flag(i2c_ptr);
    
    if (i2c_ptr->C1 & I2C_C1_TX_MASK)
    {
        // tx
        if (i2c_ptr->S & I2C_S_RXAK_MASK)
        {
            // nack
            #ifdef DEBUG_ENABLE
            printf("nack stop.\n");
            #endif
            io_info_ptr->index_rx_tx = io_info_ptr->index;
            _stop(i2c_ptr);
            _lwsem_post((LWSEM_STRUCT_PTR)(&(io_info_ptr->lwsem)));
        }
        else
        {
            // ack
            if((io_info_ptr->index == io_info_ptr->data_len) && (io_info_ptr->index_repeat_start == REPEAT_INDEX_TX))
            {
                #ifdef DEBUG_ENABLE
                printf("ack, sent all, stop -> 0xff, %d \n", io_info_ptr->index);
                #endif
                io_info_ptr->index_rx_tx = io_info_ptr->index;
                io_info_ptr->index = 0;
                io_info_ptr->data = (uchar_ptr)(io_info_ptr->buf_x_stop);
                _repeat_start(i2c_ptr);
                i2c_ptr->D = (io_info_ptr->data[io_info_ptr->index] << 1) | I2C_OPERATION_WRITE;
            }
            else
            {
                if(io_info_ptr->index == io_info_ptr->index_repeat_start)
                {
                    #ifdef DEBUG_ENABLE
                    printf("repeat start, address, %x\n", (io_info_ptr->data[0] << 1) );
                    #endif
                    _repeat_start(i2c_ptr);
                    i2c_ptr->D = (io_info_ptr->data[0] << 1) | I2C_OPERATION_READ;
                }
                else if(io_info_ptr->index == (io_info_ptr->index_repeat_start + 1) )
                {
                    #ifdef DEBUG_ENABLE
                    printf("change to r mode, %x \n", io_info_ptr->data[io_info_ptr->index]);
                    #endif
                    _to_rx_mode(i2c_ptr);

                    if(io_info_ptr->index == io_info_ptr->data_len)
                        _give_nack(i2c_ptr);
                    else
                        _give_ack(i2c_ptr);

                    // dummy read to start a reading operation
                    io_info_ptr->data[io_info_ptr->index] = i2c_ptr->D;
                }
                else
                {
                    if(io_info_ptr->index == 2)
                    {
                        if(io_info_ptr->data[1] == 0xff)
                        {
                            // end
                            _stop(i2c_ptr);
                            _lwsem_post((LWSEM_STRUCT_PTR)(&(io_info_ptr->lwsem)));
                        }
                    }
                    #ifdef DEBUG_ENABLE
                    printf("continue sending, index = %d, v = %x\n", io_info_ptr->index, io_info_ptr->data[io_info_ptr->index]);
                    #endif
                    i2c_ptr->D = io_info_ptr->data[io_info_ptr->index];
                }
            }
        }
    }
    else
    {
        // rx
        io_info_ptr->index_rx_tx = io_info_ptr->index - 2;
        if(io_info_ptr->index_rx_tx == io_info_ptr->data_len-1)
        {
            _stop(i2c_ptr);
            io_info_ptr->data[io_info_ptr->index_rx_tx] = i2c_ptr->D;
            _lwsem_post((LWSEM_STRUCT_PTR)(&(io_info_ptr->lwsem)));
            #ifdef DEBUG_ENABLE
            printf("read finished, index = %d, v = %x.\n", io_info_ptr->index_rx_tx, io_info_ptr->data[io_info_ptr->index_rx_tx]);
            #endif
        }
        else
        {
            if(io_info_ptr->index_rx_tx == (io_info_ptr->data_len-2) )
                _give_nack(i2c_ptr);
            else
                _give_ack(i2c_ptr);

            io_info_ptr->data[io_info_ptr->index_rx_tx] = i2c_ptr->D;
            #ifdef DEBUG_ENABLE
            printf("continue reading, index = %d, v = %x\n", io_info_ptr->index_rx_tx, io_info_ptr->data[io_info_ptr->index_rx_tx]);
            #endif
        }
    }
    io_info_ptr->index++;
}
static void _slave_one_frame_received(FB_VKI2C_INFO_STRUCT_PTR io_info_ptr)
{
    io_info_ptr->buf_rx_empty = FALSE;
    _lwsem_post((LWSEM_STRUCT_PTR)(&(io_info_ptr->lwsem_slave_rx)));
}
static void _slave_one_frame_sent(FB_VKI2C_INFO_STRUCT_PTR io_info_ptr)
{
    io_info_ptr->buf_tx_empty = TRUE;
    _lwsem_post((LWSEM_STRUCT_PTR)(&(io_info_ptr->lwsem_slave_tx)));
}
static void _isr_ii2c_slave(FB_VKI2C_INFO_STRUCT_PTR io_info_ptr)
{
    I2C_MemMapPtr i2c_ptr = io_info_ptr->i2c_ptr;
    uint_8  reg_addr;
    uint_32 len;
    uint_8  *buf;

    _clear_int_flag(i2c_ptr);

    if(i2c_ptr->S & I2C_S_ARBL_MASK)
    {
        #ifdef DEBUG_ENABLE
        printf("ARBL\n");
        #endif
        i2c_ptr->S |= I2C_S_ARBL_MASK;
        if(i2c_ptr->S & I2C_S_IAAS_MASK)
            goto SWR;
    }
    else
    {
        // no ARBL
        if(i2c_ptr->S & I2C_S_IAAS_MASK)
        {
SWR:
            // IAAS
            io_info_ptr->index = 0;
            _give_ack(i2c_ptr);
            if(i2c_ptr->S & I2C_S_SRW_MASK)
            {
                #ifdef DEBUG_ENABLE
                printf("IAAS slave tx\n");
                #endif
				if(io_info_ptr->buf_tx_empty == FALSE)
				{
					// have data to send
                    _to_tx_mode(i2c_ptr);
                    #ifdef DEBUG_ENABLE
                    printf("send d:%x\n", io_info_ptr->buf_tx[io_info_ptr->index]);
                    #endif
					i2c_ptr->D = io_info_ptr->buf_tx[io_info_ptr->index];
				}
				else
                    _give_nack(i2c_ptr);
            }
            else
            {
                #ifdef DEBUG_ENABLE
                printf("IAAS slave rx\n");
                printf("s:%x, c1:%x\n", i2c_ptr->S, i2c_ptr->C1);
                #endif
                _to_rx_mode(i2c_ptr);
                reg_addr = i2c_ptr->D; // dummy read
            }
        }
        else
        {
            // NONE IAAS
            if(i2c_ptr->C1 & I2C_C1_TX_MASK)
            {
                // slave tx
                #ifdef DEBUG_ENABLE
                printf("normal slave tx, ");
                #endif
                if(i2c_ptr->S & I2C_S_RXAK_MASK) 
                {
                    #ifdef DEBUG_ENABLE
                    printf("get nack, end\n");
                    #endif
                    _to_rx_mode(i2c_ptr);
                    _slave_one_frame_sent(io_info_ptr);
                    reg_addr = i2c_ptr->D; // dummy read, release bus
                }
                else
                {
                    #ifdef DEBUG_ENABLE
                    printf("get ack, continue, d = %x%\n", io_info_ptr->buf_tx[io_info_ptr->index]);
                    #endif
                    i2c_ptr->D = io_info_ptr->buf_tx[io_info_ptr->index];
                }
            }
            else
            {
                // slave rx
                #ifdef DEBUG_ENABLE
                printf("slave normal rx ");
                #endif
                if(io_info_ptr->index == 1)
                {
                    reg_addr = i2c_ptr->D;
                    if(reg_addr == REG_ADDR_SW_STOP)
                    {
                        // get STOP command
                        _give_nack(i2c_ptr);
                        #ifdef DEBUG_ENABLE
                        printf("end, get stop\n");
                        #endif
                        _slave_one_frame_received(io_info_ptr);
                    }
                    else
                    {
                        // get other command
                        #ifdef DEBUG_ENABLE
                        printf("get reg addr: %x\n", reg_addr);
                        #endif
                        io_info_ptr->reg_addr = reg_addr;
                    }
                }
                else if(io_info_ptr->index == 2)
                {
                    // data r
                    io_info_ptr->buf_rx_index = 0;
                    if(io_info_ptr->buf_rx_empty)
                    {
                        io_info_ptr->buf_rx[io_info_ptr->buf_rx_index] = i2c_ptr->D;
                        #ifdef DEBUG_ENABLE
                        printf("got rx buf, give ack, %x\n", io_info_ptr->buf_rx[io_info_ptr->buf_rx_index]);
                        #endif
                        io_info_ptr->buf_rx_index++;
                    }
                    else
                    {
                        #ifdef DEBUG_ENABLE
                        printf("not got rx buf, give nack\n");
                        #endif
                        _give_nack(i2c_ptr);
                        reg_addr = i2c_ptr->D; // dummy read
                    }
                }
                else
                {
                    // just receive data here
                    io_info_ptr->buf_rx[io_info_ptr->buf_rx_index] = i2c_ptr->D;
                    #ifdef DEBUG_ENABLE
                    printf("get data %d,  %x\n", 
                           io_info_ptr->buf_rx_index,
                           io_info_ptr->buf_rx[io_info_ptr->buf_rx_index]);
                    #endif
                    io_info_ptr->buf_rx_index++;
                }
            }
        }
        io_info_ptr->index++;
    }
}

static void _ki2c_isr
(
      pointer            parameter
)
{
    FB_VKI2C_INFO_STRUCT_PTR io_info_ptr = parameter;

    //_int_disable();
    #ifdef DEBUG_ENABLE
    printf("i2c isr, s: %x\n", io_info_ptr->i2c_ptr->S);
    #endif

    if (io_info_ptr->mode == I2C_MODE_MASTER)
        _isr_ii2c_master(io_info_ptr);
    else
        _isr_ii2c_slave(io_info_ptr);
    
    //_int_enable();
}



