
/*
 *
 * FILE : PXR4040_HWInit.c
 *
 * DESCRIPTION:
 *  This file contains all PXR40xx derivative needed initializations, 
 *  and all initializations for the PXR40xx boards which are supported.
 *  This includes setting up the External Bus Interface to allow access
 *  to memory on the external bus, and ensuring there is a valid entry in
 *  the MMU for the external memory access.
 */

#include "PXR4040.h"       /*platform development header*/
#include "PXR4040_HWInit.h"

#ifdef __cplusplus
extern "C" {
#endif

	
#ifndef INIT_USED_BOARD
#pragma error INIT_USED_BOARD should be defined !
#endif
	

/**********************************************************************/
/* Device derivative specific hardware initialization				  */
/**********************************************************************/
 
/* This macro allows to use C defined address with the inline assembler */
#define MAKE_HLI_COMPATIBLE(hli_name, c_expr) enum { hli_name=/*lint -e30*/((int)(c_expr)) };

/* All these functions must be located in the initial 4KB memory window (.init) 
   and implemented "nofralloc" so as to not use the stack 
*/

/* Memory initialization */
__declspec(section ".init") void INIT_Derivative(void);

/* Write one MMU Table Entry */
__declspec(section ".init") void WriteMMUTableEntry(void);

/* 
  Gives access to all memory map. This is required in order to do the device
  initialization disable the watch-dog timer or initialize internal ECC SRAM. 
*/
__declspec(section ".init") void __initMMUAll(void);

/* 
   Initialize the MMU in order to access the external memory.
   Since external memory is a board feature the page length is binded to
   the actual implementation. Also, to enable this page the 
   INIT_EXTERNAL_BUS_INTERFACE_SETUP flag must be on.
   The start page address is 0x20000000.
 */
void __initMMUExternalMemory(void);

/* 
   Initialize the device MMU table:
   FLASH: 			0x0000_0000 - 0x00FF_FFFF
   SRAM:  			0x4000_0000 - 0x4003_FFFF
   Per. A Modules:  0xC3F0_0000 - 0xC3FF_FFFF
   Per. B modules:  0xFFF0_0000-0xFFFF_FFFF.   	
 */
void __initMMU(void);

/*----------------------------------------------------------------------------*/
/* Initialization of external bus to access external memory   						  */
/*----------------------------------------------------------------------------*/
/* Initialize a set of contiguous PCRs */
void InitPCRs(void);

/* Initialize the SIU External Bus Interface */
void __initSIUExtBusInterface(void);

/* Initialize the used EBI Chip Selects */
void __initEBIChipSelects(void);

void INIT_ExternalBusAndMemory(void);

/*----------------------------------------------------------------------------*/
/* Function implementations                                                   */
/*----------------------------------------------------------------------------*/

/* Symbol L2SRAM_LOCATION is defined in the application linker command file (.lcf) 
   It is defined to the start of the L2SRAM of the PXR40xx. 
*/
/*lint -esym(752, L2SRAM_LOCATION) */
extern long L2SRAM_LOCATION;  

/*----------------------------------------------------------------------------*/
/* External Memory Locations from lcf file                                    */
/*----------------------------------------------------------------------------*/

#if INIT_USED_BOARD==ADAT516_AXM_0472
/* 
 Symbol AXM_0472_EXTERNAL_RAM is defined in the application linker command file (.lcf) 
 it is defined to the start of the external memory on the ADAT516 AXM-0472 board. 
*/
extern long AXM_0472_EXTERNAL_RAM[];
#endif

__asm void INIT_Derivative(void) 
{
nofralloc
	
	/* Don't have a stack yet, save the return address in a register */
	mflr     r26;
	bl __initMMUAll;
	mtlr r26;
	
	// Disable WatchDog
	//*(volatile unsigned int*)0xfff38010 = 0x0000C520; 
	lis r4,1
	subi r4,r4,15072
	lis r3,-12
	stw r4,-32752(r3)
	//*(volatile unsigned int*) 0xfff38010 = 0x0000D928; 
	lis r4,1
	subi r4,r4,9944
	lis r3,-12
	stw r4,-32752(r3)
	//*(volatile unsigned int*) 0xfff38000 = 0x8000010A; 
	lis r4,-32768
	addi r4,r4,266
	lis r3,-12
	stw r4,-32768(r3)
	
#if defined(ROM_VERSION)	
	/* PXR40xx L2SRAM */
    lis r11,L2SRAM_LOCATION@h /* Base address of the L2SRAM, 64-bit word aligned */
    ori r11,r11,L2SRAM_LOCATION@l

    li r12, 2048  /* Loop counter to get all of low L2SRAM; 256*1024/4 bytes/32 GPRs = 2048 */
    mtctr r12

    init_l2sram_loop:
        stmw r0,0(r11)        /* Write all 32 GPRs to L2SRAM */
        addi r11,r11,128      /* Inc the ram ptr; 32 GPRs * 4 bytes = 128 */
    bdnz init_l2sram_loop 	  /* Loop for all L2SRAM */
#endif
    
    /*
      Finally initialize the MMU table 
     */
	mflr     r26;
	bl __initMMU;
	mtlr r26;
	
    blr
}

__asm void __initMMUAll(void)
{
	/* Set up MMU for internal memory : a single window, 4GB wide
	   Base address = 0x0000_0000, 4 GByte Memory Space, Not Guarded, Cache inhibit, BigEndian, All Access
	*/
	MAKE_HLI_COMPATIBLE(ALL_MAS0_VALUE, MAS0_VALUE(0))
	MAKE_HLI_COMPATIBLE(ALL_MAS1_VALUE, MAS1_VALUE(V_VALID, IPROT_PROTECTED, TID_GLOBAL, 0, TSIZE_4GB))
	#if __option(vle) 
	MAKE_HLI_COMPATIBLE(ALL_MAS2_FLAGS,MAS2_FLAGS(VLE_MODE, WRITE_BACK, CACHE_INHIBIT, MEM_COHERENCE_NREQ, NOT_GUARDED, BIG_ENDIAN ))
	#else
	MAKE_HLI_COMPATIBLE(ALL_MAS2_FLAGS,MAS2_FLAGS(BOOK_E_MODE, WRITE_BACK, CACHE_INHIBIT, MEM_COHERENCE_NREQ, NOT_GUARDED, BIG_ENDIAN ))
	#endif
	MAKE_HLI_COMPATIBLE(ALL_MAS3_FLAGS,MAS3_FLAGS(READ_WRITE_EXECUTE))

	nofralloc
	mflr     r27

    /* load r3 with MAS0 value */
	lis r3,ALL_MAS0_VALUE@h
	ori r3,r3,ALL_MAS0_VALUE@l
	
	/* load r4 with MAS1 value */
	lis r4,ALL_MAS1_VALUE@h
	ori r4,r4,ALL_MAS1_VALUE@l
	
	/* clear r5 to get the base address 0x0000_0000 */       
	xor r5,r5,r5
	
	/* copy the base ram address to r6 also */
	mr r6,r5
	
	ori r5,r5,ALL_MAS2_FLAGS@l
	ori r6,r6,ALL_MAS3_FLAGS@l
	
	bl WriteMMUTableEntry 
	mtlr r27
	
    blr
}

/*
   Initialize the device MMU table:
   ----------------------------------------------------------
   Name:			Range							TLB entry
   ----------------------------------------------------------
   FLASH 			0x0000_0000 - 0x00FF_FFFF		0
   SRAM  			0x4000_0000 - 0x4003_FFFF		1
   Per. A Modules  	0xC3F0_0000 - 0xC3FF_FFFF		2
   Per. B modules  	0xFFF0_0000	- 0xFFFF_FFFF 		3
   *EXT bus			0x2000_0000 - ?					4
   ----------------------------------------------------------
   * created upon initializing the external bus
 */

/* 
  Optimization on creating the MMU entry:
  specify only the first 16-bit of the page address.
*/
#define FLASH_PAGE_START 		0
#define CAM_FLASH_PAGE_START 	0

/* 0x4000 */
#define SRAM_PAGE_START  		16384
#define CAM_SRAM_PAGE_START  	1

/* 0xC3F0 */
#define PERA_PAGE_START			50160
#define CAM_PERA_PAGE_START 	2

/* 0xFFF0 */
#define PERB_PAGE_START			65520
#define CAM_PERB_PAGE_START		3

/*
  The external RAM start is configured from an external symbol 
  defined in linker command file.
*/ 
#define CAM_EXT_RAM_START 		4

__asm void __initMMU(void)
{
	/* FLASH: 16 MByte Memory Space, Not Protected, Guarded, Cachable, All Access */

	MAKE_HLI_COMPATIBLE(FLASH_MAS0_VALUE, MAS0_VALUE(CAM_FLASH_PAGE_START))
	MAKE_HLI_COMPATIBLE(FLASH_MAS1_VALUE, MAS1_VALUE(V_VALID, IPROT_PROTECTED, TID_GLOBAL, 0, TSIZE_16MB))
	#if __option(vle) 
	MAKE_HLI_COMPATIBLE(FLASH_MAS2_VALUE,MAS2_FLAGS(VLE_MODE, WRITE_BACK, CACHEABLE, MEM_COHERENCE_NREQ, NOT_GUARDED, BIG_ENDIAN ))
	#else
	MAKE_HLI_COMPATIBLE(FLASH_MAS2_VALUE,MAS2_FLAGS(BOOK_E_MODE, WRITE_BACK, CACHEABLE, MEM_COHERENCE_NREQ, NOT_GUARDED, BIG_ENDIAN ))
	#endif
	MAKE_HLI_COMPATIBLE(FLASH_MAS3_VALUE,MAS3_FLAGS(READ_WRITE_EXECUTE))
	
	/* Internal SRAM: 256 KByte Memory Space, Not Protected, Guarded, Cache inhibit, All Access */

	MAKE_HLI_COMPATIBLE(SRAM_MAS0_VALUE, MAS0_VALUE(CAM_SRAM_PAGE_START))
	MAKE_HLI_COMPATIBLE(SRAM_MAS1_VALUE, MAS1_VALUE(V_VALID, IPROT_PROTECTED, TID_GLOBAL, 0, TSIZE_256KB))
	#if __option(vle) 
	MAKE_HLI_COMPATIBLE(SRAM_MAS2_VALUE,MAS2_FLAGS(VLE_MODE, WRITE_BACK, CACHE_INHIBIT, MEM_COHERENCE_NREQ, NOT_GUARDED, BIG_ENDIAN ))
	#else
	MAKE_HLI_COMPATIBLE(SRAM_MAS2_VALUE,MAS2_FLAGS(BOOK_E_MODE, WRITE_BACK, CACHE_INHIBIT, MEM_COHERENCE_NREQ, NOT_GUARDED, BIG_ENDIAN ))
	#endif
	MAKE_HLI_COMPATIBLE(SRAM_MAS3_VALUE,MAS3_FLAGS(READ_WRITE_EXECUTE))	
	
	/* Peripheral 0xC3F0_xxxx Modules,1 MByte Memory Space, Guarded, Don't Cache, All Access */
	
	MAKE_HLI_COMPATIBLE(PERA_MAS0_VALUE, MAS0_VALUE(CAM_PERA_PAGE_START))
	MAKE_HLI_COMPATIBLE(PERA_MAS1_VALUE, MAS1_VALUE(V_VALID, IPROT_PROTECTED, TID_GLOBAL, 0, TSIZE_1MB))
	#if __option(vle) 
	MAKE_HLI_COMPATIBLE(PERA_MAS2_VALUE,MAS2_FLAGS(VLE_MODE, WRITE_BACK, CACHE_INHIBIT, MEM_COHERENCE_NREQ, GUARDED, BIG_ENDIAN ))
	#else
	MAKE_HLI_COMPATIBLE(PERA_MAS2_VALUE,MAS2_FLAGS(BOOK_E_MODE, WRITE_BACK, CACHE_INHIBIT, MEM_COHERENCE_NREQ, GUARDED, BIG_ENDIAN ))
	#endif
	MAKE_HLI_COMPATIBLE(PERA_MAS3_VALUE,MAS3_FLAGS(READ_WRITE_EXECUTE))	

	/* Peripheral 0xFFF0_xxxx Modules,1 MByte Memory Space, Guarded, Don't Cache, All Access */
	
	MAKE_HLI_COMPATIBLE(PERB_MAS0_VALUE, MAS0_VALUE(CAM_PERB_PAGE_START))
	MAKE_HLI_COMPATIBLE(PERB_MAS1_VALUE, MAS1_VALUE(V_VALID, IPROT_PROTECTED, TID_GLOBAL, 0, TSIZE_1MB))
	#if __option(vle) 
	MAKE_HLI_COMPATIBLE(PERB_MAS2_VALUE,MAS2_FLAGS(VLE_MODE, WRITE_BACK, CACHE_INHIBIT, MEM_COHERENCE_NREQ, GUARDED, BIG_ENDIAN ))
	#else
	MAKE_HLI_COMPATIBLE(PERB_MAS2_VALUE,MAS2_FLAGS(BOOK_E_MODE, WRITE_BACK, CACHE_INHIBIT, MEM_COHERENCE_NREQ, GUARDED, BIG_ENDIAN ))
	#endif
	MAKE_HLI_COMPATIBLE(PERB_MAS3_VALUE,MAS3_FLAGS(READ_WRITE_EXECUTE))	

	nofralloc
	
	/* configure SRAM MMU entry */
	
	lis r3,SRAM_MAS0_VALUE@h
	ori r3,r3,SRAM_MAS0_VALUE@l
	
	lis r4,SRAM_MAS1_VALUE@h
	ori r4,r4,SRAM_MAS1_VALUE@l
	
	/* reset register */
	xor r5,r5,r5
    
	/* load r5 with the page start address */       
    lis r5,SRAM_PAGE_START

    /* copy the address to r6 */
    mr r6,r5
    
    /* end MAS2 setup in r3 */
    ori r5,r5,SRAM_MAS2_VALUE@l
    
    /* end MAS3 setup in r4 */
    ori r6,r6,SRAM_MAS3_VALUE@l
	
	mflr     r27
	bl WriteMMUTableEntry 
	mtlr r27

	/* configure peripheral 0xC3F0_xxxx modules MMU entry */
	
	lis r3,PERA_MAS0_VALUE@h
	ori r3,r3,PERA_MAS0_VALUE@l
	
	lis r4,PERA_MAS1_VALUE@h
	ori r4,r4,PERA_MAS1_VALUE@l
	
	/* reset register */
	xor r5,r5,r5
    
	/* load r5 with the page start address */       
    lis r5,PERA_PAGE_START

    /* copy the address to r6 */
    mr r6,r5
    
    /* end MAS2 setup in r3 */
    ori r5,r5,PERA_MAS2_VALUE@l
    
    /* end MAS3 setup in r4 */
    ori r6,r6,PERA_MAS3_VALUE@l
	
	mflr     r27
	bl WriteMMUTableEntry 
	mtlr r27
	
	/* configure peripheral 0xFFF0_xxxx modules MMU entry */
	lis r3,PERB_MAS0_VALUE@h
	ori r3,r3,PERB_MAS0_VALUE@l
	
	lis r4,PERB_MAS1_VALUE@h
	ori r4,r4,PERB_MAS1_VALUE@l
	
	/* reset register */
	xor r5,r5,r5
    
	/* load r5 with the page start address */       
    lis r5,PERB_PAGE_START

    /* copy the address to r6 */
    mr r6,r5
    
    /* end MAS2 setup in r3 */
    ori r5,r5,PERB_MAS2_VALUE@l
    
    /* end MAS3 setup in r4 */
    ori r6,r6,PERB_MAS3_VALUE@l
	
	mflr     r27
	bl WriteMMUTableEntry 
	mtlr r27	
	
	/* configure FLASH MMU entry */
	
	lis r3,FLASH_MAS0_VALUE@h
	ori r3,r3,FLASH_MAS0_VALUE@l

	lis r4,FLASH_MAS1_VALUE@h
	ori r4,r4,FLASH_MAS1_VALUE@l
    
	/* reset register */
	xor r5,r5,r5
    
	/* load r5 with the page start address */       
    lis r5,FLASH_PAGE_START

    /* copy the address to r6 */
    mr r6,r5
    
    /* end MAS2 setup in r3 */
    ori r5,r5,FLASH_MAS2_VALUE@l
    
    /* end MAS3 setup in r4 */
    ori r6,r6,FLASH_MAS3_VALUE@l
	
	mflr     r27
	bl WriteMMUTableEntry 
	mtlr r27
	
	blr
}

/* Write one MMU Table Entry:               */
/* r3, r4, r5 and r6 must hold              */
/* the values of MAS0, MAS1, MAS2 and MAS3  */
__asm void WriteMMUTableEntry( void )
{ 
nofralloc

    /* Write MMU Assist Register 0 (MAS0); SPR 624 */
    mtspr   624, r3
    /* Write MMU Assist Register 1 (MAS1); SPR 625 */
    mtspr   625, r4
    /* Write MMU Assist Register 2 (MAS2); SPR 626 */
    mtspr   626, r5
    /* Write MMU Assist Register 3 (MAS3); SPR 627 */
    mtspr   627, r6
    /* Write the table entry */
    tlbwe

    blr
}

/*
  Initializes the external bus to the the external memory 
  and creates a TLB entry for it. 
 */
__asm void INIT_ExternalBusAndMemory(void) 
{
nofralloc

    mflr     r28
    
    /* Initialize the SIU External Bus Interface */
    bl __initSIUExtBusInterface
    /* Initialize the used EBI Chip Selects */
    bl __initEBIChipSelects
    /* Initialize the needed MMU Table entries */
    bl __initMMUExternalMemory
    
    mtlr     r28
    
    blr
}


/*----------------------------------------------------------------------------*/
/* SIU External Bus Interface                                                 */
/*----------------------------------------------------------------------------*/

/* Initialize a set of contiguous PCRs:               */
/* r3: the firts PCR to initialize                    */
/* r4: the value to write in the PCRs                 */ 
/* r5: the number of PCRs to initialize               */
__asm void InitPCRs(void) 
{
nofralloc

    mtctr r5                   /* initialize ctr with the number of PCRs to initialize */
    pcr_init_loop:
        sth r4,0(r3)           /* Write r4 to current PCR address */
        addi r3,r3, 2          /* Inc the memory ptr by 2 to point to the next PCR */
        bdnz pcr_init_loop     /* Loop for ctr PCRs */

    blr
}

/* Initialize the SIU External Bus Interface */
__asm void __initSIUExtBusInterface(void)
{
	
#if INIT_USED_BOARD==ADAT516_AXM_0472
	/* PCR_FLAGS(pa, obe, ibe, dsc, ode, hys, src, wpe, wps)*/
	
/* The following pin assignments work only for AXM-0472 REV C.*/	
/* Note the RM revision 3/2010 starts counting PCR from 75 */

/* Default jumper set is chip select 0 */
/* PCR no. 256, offset 0x240, D_CS0 181*/
MAKE_HLI_COMPATIBLE(SIU_PCR_256,&SIU.PCR[256].R)
MAKE_HLI_COMPATIBLE(SIU_PCR_256_VAL,PCR_FLAGS(PA_Primary,OBE_DISABLE,IBE_DISABLE,DSC_20PF,ODE_DISABLE,HYS_DISABLE,SRC_MIN,WPE_ENABLE,WPS_ENABLE))

/* Data and address PCRs */

/* Note AXM-0472 REV C. multiplexes the address with data lines */
/* PCR no. 257, offset 0x242, alternate function D_ADD_DAT31 182*/
MAKE_HLI_COMPATIBLE(SIU_PCR_257,&SIU.PCR[257].R)
MAKE_HLI_COMPATIBLE(SIU_PCR_257_VAL, PCR_FLAGS(PA_Alt1,OBE_DISABLE,IBE_DISABLE,DSC_20PF,ODE_DISABLE,HYS_DISABLE,SRC_MIN,WPE_DISABLE,WPS_DISABLE))

/* PCR no. 263 - 277, offset 0x24E, alternate function D_ADD_DAT16 - 30 188*/
MAKE_HLI_COMPATIBLE(SIU_PCR_263_277,&SIU.PCR[263].R)
MAKE_HLI_COMPATIBLE(SIU_PCR_263_277_VAL, PCR_FLAGS(PA_Alt1,OBE_DISABLE,IBE_DISABLE,DSC_20PF,ODE_DISABLE,HYS_DISABLE,SRC_MIN,WPE_DISABLE,WPS_DISABLE))

/* PCR no. 278 - 293, offset 0x26C, primary function D_ADD_DAT0 - 15 203*/
MAKE_HLI_COMPATIBLE(SIU_PCR_278_293,&SIU.PCR[278].R)
MAKE_HLI_COMPATIBLE(SIU_PCR_278_293_VAL, PCR_FLAGS(PA_Primary,OBE_DISABLE,IBE_DISABLE,DSC_20PF,ODE_DISABLE,HYS_DISABLE,SRC_MIN,WPE_DISABLE,WPS_DISABLE))

/* Control PCRs */
/* PCR no. 294 - 305, offset 0x28C, primary functions D_RD_WR, D_WE0, D_WE1, D_OE, D_TS, D_ALE,D_TA, D_CS1, D_BDIP, D_WE2, D_WE3, D_ADD9 219*/
MAKE_HLI_COMPATIBLE(SIU_PCR_294_305,&SIU.PCR[294].R)
MAKE_HLI_COMPATIBLE(SIU_PCR_294_305_VAL, PCR_FLAGS(PA_Primary,OBE_DISABLE,IBE_DISABLE,DSC_20PF,ODE_DISABLE,HYS_DISABLE,SRC_MIN,WPE_ENABLE,WPS_ENABLE))
#endif 
	nofralloc

#if INIT_USED_BOARD==ADAT516_AXM_0472	
    mflr r27

    /* 
        Configure address and data bus pins.
    */
    
    /* PCR no. 257, offset 0x242, alternate function D_ADD_DAT31 */
    lis r3,SIU_PCR_257@h
    ori r3,r3,SIU_PCR_257@l
    lis r4,SIU_PCR_257_VAL@h
    ori r4,r4,SIU_PCR_257_VAL@l
    li r5,1
    bl InitPCRs
    
    /* PCR no. 263 - 277, offset 0x24E, alternate function D_ADD_DAT16 - 30 */
    lis r3,SIU_PCR_263_277@h
    ori r3,r3,SIU_PCR_263_277@l
    lis r4,SIU_PCR_263_277_VAL@h
    ori r4,r4,SIU_PCR_263_277_VAL@l
    li r5,15
    bl InitPCRs

    /* PCR no. 278 - 293, offset 0x26C, primary function D_ADD_DAT0 - 15 */
    lis r3,SIU_PCR_278_293@h
    ori r3,r3,SIU_PCR_278_293@l
    lis r4,SIU_PCR_278_293_VAL@h
    ori r4,r4,SIU_PCR_278_293_VAL@l
    li r5,16
    bl InitPCRs

    /* Configure control lines */
    /* PCR no. 256, offset 0x240, D_CS0 */
    lis r3,SIU_PCR_256@h
    ori r3,r3,SIU_PCR_256@l
    lis r4,SIU_PCR_256_VAL@h
    ori r4,r4,SIU_PCR_256_VAL@l
    li r5,1
    bl InitPCRs    
    
    /*
		PCR no. 294 - 305, offset 0x28C, 
		primary functions D_RD_WR, D_WE0, D_WE1, D_OE, D_TS,
		D_ALE,D_TA, D_CS1, D_BDIP, D_WE2, D_WE3, D_ADD9
	*/
    lis r3,SIU_PCR_294_305@h
    ori r3,r3,SIU_PCR_294_305@l
    lis r4,SIU_PCR_294_305_VAL@h
    ori r4,r4,SIU_PCR_294_305_VAL@l
    li r5,11
    bl InitPCRs 

    mtlr r27
#endif
    
    blr
}

/* Initialize the used EBI Chip Selects */
__asm void __initEBIChipSelects(void)
{
#if INIT_USED_BOARD==ADAT516_AXM_0472
MAKE_HLI_COMPATIBLE(EBI_CS0_CAL_BR0,&EBI.CS[0].BR.R)
MAKE_HLI_COMPATIBLE(EBI_CS0_CAL_OR0,&EBI.CS[0].OR.R)
#endif

nofralloc

#if INIT_USED_BOARD==ADAT516_AXM_0472
    /*  CY7C1338 512K External SRAM - burst inhibit, 0 wait, multiplexed AD
        Set up Memory Controller CS0 @ ADAT516_AXM_0472
    */

    /* EBI.CS[0].BR.R = (unsigned long) AXM_0472_EXTERNAL_RAM | 0x83UL; */
	/* Sets the base address to AXM_0472_EXTERNAL_RAM - defined in LCF file*/	
    lis      r3,AXM_0472_EXTERNAL_RAM@h
    addi     r0,r3,AXM_0472_EXTERNAL_RAM@l
    
    /* 32-bit wide bus, Address/Data multiplexed mode, burst inhibit*/
    ori      r0,r0,0x83 
    lis      r3,EBI_CS0_CAL_BR0@h
    ori      r3,r3,EBI_CS0_CAL_BR0@l
    
    stw      r0,0(r3)
    
    /* 	Set address mask for 512KB range,
		0-clock cycle wait states, 0 - cycle length in clocks
    */
    /*  EBI.CS[0].OR.R = 0xfff80000; */
    lis      r0,0xFFF8
    lis      r3,EBI_CS0_CAL_OR0@h
    ori      r3,r3,EBI_CS0_CAL_OR0@l

    stw      r0,0(r3)
    
#endif

    blr
}
	
/* Initialize the MMU to allow access to the external RAM*/
__asm void __initMMUExternalMemory(void)
{
#if INIT_USED_BOARD==ADAT516_AXM_0472
	/* 
		Set up MMU for External Memory for the CY7C1338 SRAM module of 512KB.
		Base address = 0x2000_0000, 4 MByte Memory Space, Not Guarded, Cachable, All Access
	 */
	MAKE_HLI_COMPATIBLE(EXT_RAM_MAS0_VALUE,MAS0_VALUE(CAM_EXT_RAM_START))
	MAKE_HLI_COMPATIBLE(EXT_RAM_MAS1_VALUE,MAS1_VALUE(V_VALID, IPROT_PROTECTED, TID_GLOBAL, 0, TSIZE_512KB))
	#if __option(vle) 
	MAKE_HLI_COMPATIBLE(EXT_RAM_MAS2_FLAGS,MAS2_FLAGS(VLE_MODE,WRITE_BACK,CACHE_INHIBIT,MEM_COHERENCE_NREQ,NOT_GUARDED,BIG_ENDIAN))
	#else
	MAKE_HLI_COMPATIBLE(EXT_RAM_MAS2_FLAGS,MAS2_FLAGS(BOOK_E_MODE,WRITE_BACK,CACHE_INHIBIT,MEM_COHERENCE_NREQ,NOT_GUARDED,BIG_ENDIAN))
	#endif
	MAKE_HLI_COMPATIBLE(EXT_RAM_MAS3_FLAGS,MAS3_FLAGS(READ_WRITE_EXECUTE))
#endif
	
	nofralloc
	
#if INIT_USED_BOARD==ADAT516_AXM_0472
    /* load r3 with MAS0 value */
    lis r3,EXT_RAM_MAS0_VALUE@h
    ori r3,r3,EXT_RAM_MAS0_VALUE@l
     
    /* load r4 with MAS1 value */
    lis r4,EXT_RAM_MAS1_VALUE@h
    ori r4,r4,EXT_RAM_MAS1_VALUE@l
    
    /* load r5 with the external RAM address from the lcf file */       
    lis r5,AXM_0472_EXTERNAL_RAM@h
    ori r5,r5,AXM_0472_EXTERNAL_RAM@l
         
    /*
     clrrwi rA,rS,n (n < 32)
     mask with 0xffffc000 since EPN=MAS2[0:21] and RPN=MAS3[0:21]  
    */
    clrrwi   r5,r5,10
    /* copy the masked external ram address to r6 also */
    mr r6,r5
    /* end MAS2 setup in r3 */
    ori r5,r5,EXT_RAM_MAS2_FLAGS@l
    /* end MAS3 setup in r4 */
    ori r6,r6,EXT_RAM_MAS3_FLAGS@l
     
    /* write the MMU entry defined through r3, r4, r5 and r6 */
    mflr     r27
    bl WriteMMUTableEntry
    mtlr r27
    
#endif
    
    blr
}


#ifdef __cplusplus
}
#endif
