/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     spi.c
*
* @author   B06050
* 
* @version  1.0.3.0
* 
* @date     Mar-28-2012
* 
* @brief    Serial Peripheral Interface SW module source file.
*
*******************************************************************************/
#include "spi.h"

/*******************************************************************************
*
* Function: void SPI_0_Init(void)
*
* Description:  This function initializes the SPI0 module.
*
* Note:         This function must be called first, before a call to any other
*               SPI0 related function within this SW module.
*
*******************************************************************************/
void SPI_0_Init(void)
{
    uint8_t u8temp;

    SPI0CR1_SPE = 0;    /* SPI module disabled */
    SPI0CR2 = 0x00;

    u8temp = SPI0SR;    /* Read the status register and */
    u8temp = SPI0DRL;   /* Read the data register to clear the status flags */

    SPI0BR_SPPR = SPI_0_SPPR;   /* Baud rate divider */
    SPI0BR_SPR = SPI_0_SPR;
    
    SPI0CR2_MODFEN = 0; /* Slave select output pin not used by SPI module */
    SPI0CR1_SSOE = 0;

    SPI0CR1_MSTR = 1;   /* SPI in Master mode */
    SPI0CR1_CPHA = 1;   /* Data sampling at falling edges */
    SPI0CR1_SPE = 1;    /* SPI module enabled */
}

/*******************************************************************************
*
* Function: uint8_t SPI_0_SendByte(uint8_t data)
*
* Description:  This function transmits 8-bit data on the SPI0 bus
*               and returns 8-bit data received from the slave device.
*
* Param[in]:    data        8-bit data to be transmitted
*
* Return:       uint8_t     received 8-bit data from the slave device
*
*******************************************************************************/
uint8_t SPI_0_SendByte(uint8_t data)
{
    SPI0CR2_XFRW = 0;       /* 8-bit transfer width */
    
    while (!SPI0SR_SPTEF)   /* Wait for empty data register */
    {
    }
    SPI0DRL = data;         /* Load byte in data transmission register */
    while (!SPI0SR_SPIF)    /* Wait for the end of the transfer */
    {
    }

    return SPI0DRL;
}

/*******************************************************************************
*
* Function: uint16_t SPI_0_SendWord(uint16_t data)
*
* Description:  This function transmits 16-bit data on the SPI0 bus
*               and returns 16-bit data received from the slave device.
*
* Param[in]:    data        16-bit data to be transmitted
*
* Return:       uint16_t    received 16-bit data from the slave device
*
*******************************************************************************/
uint16_t SPI_0_SendWord(uint16_t data)
{
    SPI0CR2_XFRW = 1;       /* 16-bit transfer width */

    while (!SPI0SR_SPTEF)   /* Wait for empty data register */
    {
    }
    SPI0DR = data;          /* Load 16-bit word into data transmission
                               register */
    while (!SPI0SR_SPIF)    /* Wait for the end of the transfer */   
    {
    }
    data |= SPI0DR;         /* Store the 16-bit word */

    return data;
}