/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     adc.c
*
* @author   B06050
* 
* @version  1.0.9.0
* 
* @date     Jun-29-2012
* 
* @brief    Analog-to-Digital Converter SW module source file.
*
*******************************************************************************/
#include "adc.h"

/*******************************************************************************
*
* Function: void ADC_Init(void)
*
* Description:  This function initializes the ADC module.
*
*******************************************************************************/
void ADC_Init(void)
{
    ATDCTL0 = 0x0F; /* Wrap around channel 15 */
    ATDCTL1 = 0xA1; /* ETRIG1 selected as external trig. source, 10-bit
                       resolution */
    ATDCTL2 = 0x00; /* External trigger disabled, sequence complete interrupt
                       disabled */
    ATDCTL3 = 0x0B; /* Left justified, single conversion seq. length, FIFO off,
                       freeze immediately */
    ATDCTL4 = 0x01; /* 4 ADC clock sample time, 6.25MHz ADC clock
                       @BUSclk=25MHz (4+15 ADC clocks sampling+conversion) */
    ATDSTAT0 = 0xB0;/* Clear status register 0 flags */
    ATDCMPE = 0x00; /* ATD compare disabled on all channels */
    ATDDIEN = 0x0000;   /* Digital input buffer disabled on all channel pins */
}

/*******************************************************************************
*
* Function: void ADC_StartSingleConversion(uint8_t channel)
*
* Description:  This function disables triggered conversions and starts single
*               channel conversion (10-bit). Sequence complete interrupt are
*               disabled.
*
* Param[in]:    u8channel   ADC channel number
*
*******************************************************************************/
void ADC_StartSingleConversion(uint8_t u8channel)
{
    ATDCTL2 = 0x00;         /* Disable external trigger and interrupts */
    ATDCTL3 &= ~ADC_S_MASK; /* Reset conversion sequence length */
    ATDCTL3_S1C = 1;        /* Set conversion sequence length to 1 */
    ATDSTAT0 = ATDSTAT0_SCF_MASK;   /* Clear sequence complete flag */
    ATDCTL5 = u8channel;    /* Start conversion */
}

/*******************************************************************************
*
* Function: void ADC_EnableTrigSeq(uint8_t channel, uint8_t length)
*
* Description:  This function enables conversion sequence triggered by external
*               trigger input (ETRIG0, rising edge). Sequence complete interrupt
*               enabled.
*
* Param[in]:    u8channel   ADC channel number (first in sequence)
*               u8length    ADC conversion sequence length
*
*******************************************************************************/
void ADC_EnableTrigSeq(uint8_t u8channel, uint8_t u8length)
{
    ATDCTL3 &= ~ADC_S_MASK;             /* Reset conversion sequence length */
    ATDCTL3 |= ((u8length & 0x0F) << 3);/* Set sequence length */
    ATDSTAT0 = ATDSTAT0_SCF_MASK;       /* Clear sequence complete flag */
    ATDCTL2 = 0x0E;                     /* External trigger enabled (rising
                                           edge), sequence complete interrupt
                                           enabled */ 
    ATDCTL5 = (ADC_MULT_MODE | u8channel);
}

/*******************************************************************************
*
* Function: void ADC_ModTrigSeqCh(uint8_t channel)
*
* Description:  This function function sets the first channel to be converted
*               in the conversion sequence.
*
* Param[in]:    u8channel   ADC channel number (first in sequence)
*
*******************************************************************************/
#pragma INLINE
void ADC_ModTrigSeqCh(uint8_t u8channel)
{
    ATDCTL2_ETRIGE = 0;
    ATDCTL2_ETRIGE = 1;
    ATDSTAT0 = ATDSTAT0_SCF_MASK;       /* Clear sequence complete flag */
    ATDCTL2_ASCIE = 1;                  /* Enable Sequence Complete interrupt */
    ATDCTL5 = (ADC_MULT_MODE | u8channel);
}
    