;*******************************************************************************
;
; Freescale Semiconductor Inc.
; (c) Copyright 2012 Freescale Semiconductor, Inc.
; ALL RIGHTS RESERVED.
;
;*******************************************************************************
;
; $File Name: math.asm$
;
; Author: B06050
; 
; $Version: 1.0.6.0$
; 
; $Date: Apr-18-2012$
; 
; Description: Mathematical function assembly file.
;
;*******************************************************************************
 XDEF AddSat_F16
 XDEF SubSat_F16
 XDEF AddSat_F32
 XDEF SubSat_F32
 XDEF Mul_F16
 XDEF Div_F16
 XDEF Div_U32U16U16

;*******************************************************************************
;
; Function: tFrac16 AddSat_F16(tFrac16 f16In1, tFrac16 f16In2)
;
; Description:  This function performs 16-bit fractional addition with
;               saturation.
;
; Param[in]:    f16In1      First 16-bit signed fractional operand
;               f16In2      Second 16-bit signed fractional operand
;
; Return:       tFrac16     16-bit fractional sum of two input arguments
;
;*******************************************************************************
AddSat_F16:
    ; f16In2 in D, f16In1 on SP+3
    ADDD 3, SP          ; Add the first operand value to accumulator
                        ; second operand value is already in acc.
    BLT  AddSat_F16low  ; Branch if N or V set - too low?
    BPL  AddSat_F16ret1 ; Branch if N=0 (V=0)
    LDD  #32767         ; Positive saturate (N=1, V=1)
AddSat_F16ret1:
    RTC
AddSat_F16low:
    BMI  AddSat_F16ret2 ; Branch if N=1 (V=0)
    LDD  #32768         ; Negative saturate (N=0, V=1)
AddSat_F16ret2:
    RTC                 ; Return accumulator value

;*******************************************************************************
;
; Function: tFrac32 AddSat_F32(tFrac32 f32In1, tFrac32 f32In2)
;
; Description:  This function performs 32-bit fractional addition with
;               saturation.
;
; Param[in]:    f32In1      First 32-bit signed fractional operand
;               f32In2      Second 32-bit signed fractional operand
;
; Return:       tFrac32     32-bit fractional sum of two input arguments
;
;*******************************************************************************
AddSat_F32:
    ; f32In2_H in X, f32In2_L in D, f32In1_H on SP+3, f32In1_L on SP+5
    ADDD 5,SP           ; f32In2_L + f32In1_L => D
    EXG  D,X            ; Exchange D & X: f32In2_H => D, D => X
    ADCB 4,SP           ; f32In2_H + f32In1_H => D (with carry)
    ADCA 3,SP
    BLT  AddSat_F32tlow ; Branch if N or V set - too low?
    BPL  AddSat_F32ret1 ; Branch if N=0 (V=0)
    LDD  #32767         ; Positive saturate (N=1, V=1)
    LDX  #65535
AddSat_F32ret1:
    EXG  D,X
    RTC
AddSat_F32tlow:
    BMI  AddSat_F32ret2 ; Branch if N=1 (V=0)
    LDD  #32768         ; Negative saturate (N=0, V=1)
AddSat_F32ret2:
    EXG  D,X            ; Result X:D
    RTC
        
;/******************************************************************************
;*
;* Function: tFrac16 SubSat_F16(tFrac16 f16In1, tFrac16 f16In2)
;*
;* Description: This function performs 16-bit fractional subtraction
;*              with saturation.
;*
;* Param[in]:   f16In1      First 16-bit signed fractional operand
;*              f16In2      Second 16-bit signed fractional operand
;*
;* Return:      tFrac16     16-bit fractional difference of the second operand
;*                           from the first operand
;*
;******************************************************************************/
SubSat_F16:
    ; f16In2 in D, f16In1 on SP+3
    COMA                ; Provide one's complement of D
    COMB
    ADDD #1             ; D to two's complement
    BRA  AddSat_F16     ; Branch to 16-bit addition with saturation function

;*******************************************************************************
;
; Function: tFrac32 SubSat_F32(tFrac32 f32In1, tFrac32 f32In2)
;
; Description:  This function performs 32-bit fractional subtraction
;               with saturation.
;
; Param[in]:    f32In1      First 32-bit signed fractional operand
;               f32In2      Second 32-bit signed fractional operand
;
; Return:       tFrac32     32-bit fractional difference of the second operand
;                           from the first operand
;
; Notes:    This function uses function AddSat_F32.
;
;*******************************************************************************
SubSat_F32:
    ; f32In2_H in X, f32In2_L in D, f32In1_H on SP+3, f32In1_L on SP+5
    COMA                ; Convert D to one's complement
    COMB
    EXG  D,X            ; Exchange D & X
    COMA                ; Convert D to one's complement
    COMB
    EXG  D,X            ; Exchange D & X
    IBNE D,AddSat_F32   ; Increment D, branch if D=0
    INX                 ; Increment X (D to two's complement)
    BRA  AddSat_F32     ; Branch to 32-bit addition with saturation function

;*******************************************************************************
;
; Function: tFrac16 Mul_F16(tFrac16 f16In1,tFrac16 f16In2)
;
; Description:  This function performs 16-bit fractional multiplication without
;               saturation.
;
; Param[in]:    f16In1      First 16-bit signed fractional operand
;               f16In2      Second 16-bit signed fractional operand
;
; Return:       tFrac16     16-bit fractional multiplication of input arguments
;
;*******************************************************************************
Mul_F16:
    ; f32In2 in D, f32In1 on SP+3
    LDY   3, SP         ; Load the first operand into index reg. Y
    EMULS               ; Multiply D by Y
    LSLD                ; Shift D left to produce carry if MSB set
    XGDY                ; Swap D and Y
    BCC   Mul_F16ret    ; Branch if carry cleared by previous LSLD
    LSLD                ; Shift D left (get rid of duplicate sign bit)
    ADDB  #1            ; Add 1
    RTC                 ; Return D
Mul_F16ret:
    LSLD                ; Shift D left (get rid of duplicate sign bit)
    RTC                 ; Return D

;*******************************************************************************
;
; Function: tFrac16 Div_F16(tFrac16 f16In1,tFrac16 f16In2)
;
; Description:  This function performs 16-bit fractional division without
;               saturation.
;
; Param[in]:    f16In1      16-bit signed fractional dividend
;               f16In2      16-bit signed fractional divisor
;
; Return:       tFrac16     16-bit fractional quotient of input arguments
;
;*******************************************************************************
Div_F16:
    ; f16In2 in D, f16In1 on SP+3
    XGDX                ; Exchange D with X (f16In2 => X)
    CLRA                ; Clear A
    CLRB                ; Clear B
    XGDY                ; Exchange D and Y (0 => Y)
    LDD   3, SP         ; Load f16In1 to D
    ASRA                ; Arithmetic shift D right
    RORB                
    XGDY                ; Exchange D and Y
    RORA                ; Rotate A (set D MSB with carry)
    EDIVS               ; (Y:D) / X => Y; (int32_t)f16In1 << 15
    XGDY                ; Result of the division in D
    RTC                 ; Return D

;*******************************************************************************
;
; Function: uint16_t Div_U32U16U16(uint32_t u32In1,uint16_t u16In2)
;
; Description:  This function performs unsigned 32-bit by 16-bit division
;               without saturation.
;
; Param[in]:    u32In1      32-bit unsigned integer dividend
;               u16In2      16-bit unsigned integer divisor
;
; Return:       uint16_t    16-bit integer quotient of input arguments
;
;*******************************************************************************
Div_U32U16U16:
    ; u16In2 in D, u32In1_H on SP+3, u32In1_L on SP+5
    XGDX                ; Exchange D with X (f16In2 => X)
    LDY   3, SP         ; Load u32In1_H index reg. Y
    LDD   5, SP         ; Load u32In1_L index reg. Y
    EDIV                ; u32In1 / u16In2
    XGDY                ; Result of the division to D
    RTC                 ; Return D
    