/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     BLDCSensorless.c
*
* @author   B06050
*
* @version  1.0.8.0
*
* @date     Jul-27-2012
*
* @brief    BLDC Motor Control Sensorless Application for MC9S12G128.
*
*******************************************************************************/
#include <hidef.h>      /* Common defines and macros */
#include "derivative.h" /* Derivative-specific definitions */
#include "sys.h"
#include "pim.h"
#include "spi.h"
#include "sci.h"
#include "tim.h"
#include "adc.h"
#include "pwm.h"
#include "can.h"
#include "can_cfg.h"
#include "MC33905_routines.h"
#include "MC33937_routines.h"
#include "math.h"
#include "S12G128_appconfig.h"
#include "freemaster.h"

#define APP_INIT                0   /* Application states */
#define APP_ALIGNMENT           1
#define APP_START               2
#define APP_RUN                 3
#define APP_STOP                4
#define APP_FAULT               5

/*******************************************************************************
*
* Type definition
*
*******************************************************************************/
typedef void (*tPointerFcn)(void);  /* pointer to a function */

typedef union {
    uint16_t R;
    struct {
        uint16_t Alignment:1;
        uint16_t Sensorless:1;
        uint16_t StallCheckReq:1;
        uint16_t EnableCMT:1;
        uint16_t AfterCMT:1;
        uint16_t CloseLoop:1;
        uint16_t NewZC:1;
        uint16_t AdcSaved:1;
        uint16_t CurrentLimiting:1;
        uint16_t Fault:1;
        uint16_t Reserved:6;
    }B;
}tDriveStatus;

typedef union {
    uint8_t R;
    struct {
        uint8_t OverDCBusCurrent:1;
        uint8_t OverDCBusVoltage:1;
        uint8_t UnderDCBusVoltage:1;
        uint8_t PreDriverError:1;
        uint8_t Reserved:4;
    }B;
}tFaultStatus;

typedef struct {
    tFrac16 BEMFVoltage;
    tFrac16 DCBVVoltage;
    tFrac16 DCBIVoltage;
    tFrac16 DCBIOffset;
}tADCresults;


/*******************************************************************************
*
* Function prototypes
*
*******************************************************************************/
static void AppInit(void);
static void AppAlignment(void);
static void AppStart(void);
static void AppRun(void);
static void AppStop(void);
static void AppFault(void);

static void AppStopToAlignment(void);
static void AppAlignmentToStart(void);
static void AppStartToRun(void);

static void FaultDetection(void);
static void LED_Indication(void);
static void Encoder_RotCtrl(void);
static void Encoder_SwCtrl(void);

static void PWM_Alignment(void);
static void PWM_Vector_0(void);
static void PWM_Vector_1(void);
static void PWM_Vector_2(void);
static void PWM_Vector_3(void);
static void PWM_Vector_4(void);
static void PWM_Vector_5(void);

static void ZCdetectPhCfalling(void);
static void ZCdetectPhBraising(void);
static void ZCdetectPhAfalling(void);
static void ZCdetectPhCraising(void);
static void ZCdetectPhBfalling(void);
static void ZCdetectPhAraising(void);

static void Control_Loop(void);

#pragma CODE_SEG __NEAR_SEG NON_BANKED
void interrupt ADC_SC_ISR(void);
void interrupt TIM1_OC_ISR(void);
#pragma CODE_SEG DEFAULT

static uint8_t StallCheck(void);

void main(void);

/*******************************************************************************
*
* Variables definition
*
*******************************************************************************/
Can_ReturnType ret_val;

static vuint16_t NextCmtPeriod, duty_cycle;
static vuint16_t NextCmtSector, ActualCmtSector;
static uint16_t  alignmentTimer, startCMTcounter, speedLoopCounter = SPEED_LOOP_PRESCALER, PICounter = 20;

MC33937_SR_T        mc33937_Status;
static uint8_t      appState, appSwitchState, faultSwitchClear;
static tDriveStatus driveStatus;
static tFaultStatus faultStatus, faultStatusLatched;
static tADCresults  ADCResults;

static int16_t  bemfVoltage, bemfVoltageOld;
static uint16_t timeBackEmf, timeOldBackEmf, timeCommutation, timeZCToff;
static uint16_t timeZC, lastTimeZC;
static uint16_t actualPeriodZC;
static uint16_t periodZC_F_PhA, periodZC_R_PhA;
static uint16_t periodZC_F_PhB, periodZC_R_PhB;
static uint16_t periodZC_F_PhC, periodZC_R_PhC;
static uint16_t period6ZC;
static uint16_t advanceAngle;

static tFrac16 torque_F_PhA, torque_R_PhA;
static tFrac16 torque_F_PhB, torque_R_PhB;
static tFrac16 torque_F_PhC, torque_R_PhC;
static tFrac16 torqueErr, torque, torque6ZC;

static uint16_t debugPeriodMin, debugPeriodMax, stallCheckCounter;
static uint8_t  StallError;

static tFrac16 speedErr, requiredSpeed = 3000;
static tFrac16 actualSpeed;
static tFrac16 speedPIOut, currentPIOut;
static CONTROLLER_PIAW_R_T_F16 speedPIPrms, currentPIPrms;
static FILTER_MA_T_F16 Udcb_filt;
static tFrac16 u_dc_bus_filt;

static tFrac16 requiredSpeed;

static uint8_t  rotSwitchA, rotSwitch;
static uint8_t  LEDcounter;
static uint16_t switchCounter;

/* Array with pointers to the state machine functions */
static tPointerFcn AppStateMachine[] = \
{
    AppInit,
    AppAlignment,
    AppStart,
    AppRun,
    AppStop,
    AppFault
};

/* Array with pointers to the commutation functions */
static tPointerFcn PWMCommutationFcn[] = \
{
    PWM_Vector_0,
    PWM_Vector_1,
    PWM_Vector_2,
    PWM_Vector_3,
    PWM_Vector_4,
    PWM_Vector_5
};

/* Array with pointers to the zero-cross detection functions */
static tPointerFcn ZCdetection[] = \
{
    ZCdetectPhCfalling,
    ZCdetectPhBraising,
    ZCdetectPhAfalling,
    ZCdetectPhCraising,
    ZCdetectPhBfalling,
    ZCdetectPhAraising
};

/* Array with ADC channel numbers to start the conversion sequence with */
static uint8_t startADCchannel[] = \
{
    (ADC_INPUT_AN4),
    (ADC_INPUT_AN2),
    (ADC_INPUT_AN0),
    (ADC_INPUT_AN4),
    (ADC_INPUT_AN2),
    (ADC_INPUT_AN0)
};

/* FreeMASTER constants (need to be referenced in ENTRIES section)
   of the Project.prm to avoid deadstripping */
const tFrac16 fm_voltage = FM_U_SCALE;
const tFrac16 fm_current = FM_I_SCALE;

int16_t test;

/*******************************************************************************
*
* Function: void main(void)
*
* Description: main function
*
*******************************************************************************/
void main(void)
{
    uint8_t ret;

    _DISABLE_COP(); /* Disable COP watchdog */
    DISABLE_INTERRUPTS();

    /* Init system clock to 50MHz, BUSclk=25MHz */
    PLL_Init();
    /* Initialize pin states */
    PIM_Init();
    /* Initialize SPI0 */
    SPI_0_Init();
    /* Initialize SCI0 */
    SCI_0_Init();
    /* initialize FreeMASTER */
    ret = FMSTR_Init();
    /* Initialize MC33905 System Basis Chip */
    MC33905_Config();
    /* Initialize MSCAN driver */
    ret_val = Can_Init(&Can_ModuleConfig[0]);
    /* Set MSCAN to RUN mode */
    ret_val = Can_SetMode(CAN_G128_CAN0,CAN_MODE_RUN);
    /* Initialize ADC */
    ADC_Init();
    /* Initialize PWM */
    PWM_Init();
    /* Configure MC33937A MOSFET pre-driver */
    do {
        ret = MC33937_Config();
    }while(ret == 0);

    /* Reset fault logic */
    FLT_RST = SET;
    FLT_RST = CLEAR;

    /* Enable MC33937A MOSFET Pre-driver */
    MC33937_Enable();

    /* Initialize moving average filter */
    FilterMAInit_F16(&Udcb_filt);
    Udcb_filt.u16NSamples = 2;

    /* Save actual encoder switch position */
    rotSwitchA = ROT_A;

    /* TEST */
    DDRA_DDRA7 = 1;
    DDRA_DDRA6 = 1;

    /* Wait 10ms for power stage 3.3V power supply voltage to settle down */
    API_Delay(API_5MS);
    API_Delay(API_5MS);
    
    ENABLE_INTERRUPTS();
    /* Initialize and start TIM counter:
       tTIM1 = 4ms, tTIM2 = 1ms (fTIM=BUSclk/32) */
    TIM_Init(TIM_PERIOD_4MS, TIM_PERIOD_1MS);   
    
    for(;;)
    {
        /* Fault detection */
        FaultDetection();
        /* ADC sequence complete flag polling - only if ADC interrupts were
           disabled after the zero-cross event was detected                  */
        if((driveStatus.B.NewZC == 1) && (ATDSTAT0_SCF == 1))
        {
            /* Clear conversion complete flag */
            ATDSTAT0 = ATDSTAT0_SCF_MASK;
            /* Save conversion result of Back-EMF voltage and DC-BUS current */
            ADCResults.DCBIVoltage = ((tFrac16)(ATDDR0 >> 1) - ADCResults.DCBIOffset);
            ADCResults.DCBVVoltage = (tFrac16)(ATDDR1 >> 1);
        }
        /* Speed, current and user input control loop */
        if(TFLG1_C2F == 1)
        {
            /* Clear Clear output compare event flag */
            TFLG1 = TFLG1_C2F_MASK;
            TC2 = TCNT + TIM_PERIOD_1MS;
            Control_Loop();
        }
        /* FreeMASTER polling function */
        FMSTR_Poll();
        /* Call BLDC application state machine function */
        AppStateMachine[appState]();
        /* Clear SBC watchdog */
        MC33905_ClearWDT();
    }
}

/******************************************************************************
*
* Function: void AppInit(void)
*
* Description: BLDC application init. function
*
*******************************************************************************/
static void AppInit(void)
{
    driveStatus.B.Alignment = 0;
    driveStatus.B.EnableCMT = 0;
    driveStatus.B.CloseLoop = 0;
    driveStatus.B.Sensorless = 0;
    driveStatus.B.NewZC = 0;

    PWME = PWM_DISABLE;         /* Disable all PWMs */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    timeZCToff = TIME_TOFF;
    duty_cycle = 0;
    PWMDTY23 = duty_cycle;
    PWMDTY45 = duty_cycle;
    PWMDTY67 = duty_cycle;
    PWMDTY01 = 1;

    advanceAngle = ADVANCE_ANGLE;

    NextCmtPeriod = TIM_PERIOD_1MS;
    
    /* Wait for DC-BUS current to settle down (after motor stops) */
    API_Delay(API_100US);

    /* DC-Bus current calibration */
    ADC_StartSingleConversion(ADC_DCBI_AN);

    /* Wait until the DC-Bus current conversion is complete */
    while(ATDSTAT0_SCF == 0)
    {
    }

    ADCResults.DCBIOffset = (tFrac16)(ATDDR0 >> 1);

    PWME = PWM_TRIG_EN;         /* Enable PWM1 (ADC trigger signal) */

    ADC_EnableTrigSeq(ADC_INPUT_AN0,2);

    appState = APP_STOP;
}

/*******************************************************************************
*
* Function: void AppStop(void)
*
* Description: BLDC application Stop state function
*
*******************************************************************************/
static void AppStop(void)
{
    if(appSwitchState)
    {
        AppStopToAlignment();
    }
}

/*******************************************************************************
*
* Function: void AppStopToAlignment(void)
*
* Description:  BLDC application Stop to Alignment transition function
*
*******************************************************************************/
static void AppStopToAlignment(void)
{
    driveStatus.B.Alignment = 1;
    driveStatus.B.EnableCMT = 0;
    driveStatus.B.CloseLoop = 0;
    driveStatus.B.Sensorless = 0;
    driveStatus.B.NewZC = 0;

    NextCmtPeriod = TIM_PERIOD_1MS;

    TC1 = TCNT + NextCmtPeriod;
    timeZCToff = TIME_TOFF;
    alignmentTimer = ALIGNMENT_TIME;
    duty_cycle = ALIGNMENT_DUTY_CYCLE;
    
    /* Update PWM duty cycle */
    PWMDTY23 = duty_cycle;
    PWMDTY45 = duty_cycle;
    PWMDTY67 = duty_cycle;
    PWMDTY01 = (duty_cycle >> 1);

    /* Apply PWM settings for motor alignment */
    PWM_Alignment();

    appState = APP_ALIGNMENT;
}

/*******************************************************************************
*
* Function: void AppAlignment(void)
*
* Description:  BLDC application Alignment state function
*
*******************************************************************************/
static void AppAlignment(void)
{
    if(alignmentTimer == 0)
    {
        AppAlignmentToStart();
    }
}

/*******************************************************************************
*
* Function: void AppAlignmentToStart(void)
*
* Description:  BLDC application Alignment to Start state transition function
*
*******************************************************************************/
static void AppAlignmentToStart(void)
{
    NextCmtSector = 0;  /* starting sector */

    driveStatus.B.Alignment = 0;
    driveStatus.B.EnableCMT = 1;
    driveStatus.B.AfterCMT  = 0;

    NextCmtPeriod = START_FIRST_PERIOD;
    startCMTcounter = START_CMT_CNT;

    TC1 = TCNT + NextCmtPeriod;
    PWMCommutationFcn[NextCmtSector](); /* Apply first commutation sector */

    /* Setup analog channel to start conversion sequence on */
    ATDCTL2_ETRIGE = 0;
    ATDCTL2_ETRIGE = 1;
    ATDCTL5 = startADCchannel[NextCmtSector];

    NextCmtSector++;
    NextCmtPeriod = Mul_F16(NextCmtPeriod,START_CMT_ACCELERATION);

    appState = APP_START;
}

/*******************************************************************************
*
* Function: void AppStart(void)
*
* Description:  BLDC application Start state function
*
*******************************************************************************/
static void AppStart(void)
{
    if(driveStatus.B.AfterCMT == 1)
    {
        timeZC = TCNT - (NextCmtPeriod >> 1);

        startCMTcounter--;
        if(startCMTcounter > 0)
        {
            driveStatus.B.AfterCMT = 0;

            NextCmtPeriod = Mul_F16(NextCmtPeriod,START_CMT_ACCELERATION);

            duty_cycle += START_DUTY_CYCLE_INC;

            /* Update PWM duty cycle */
            PWMDTY23 = duty_cycle;
            PWMDTY45 = duty_cycle;
            PWMDTY67 = duty_cycle;
            PWMDTY01 = Mul_F16(duty_cycle,BEMF_SENSING_POINT_FRAC) + DELAY;
        }
    }

    if(startCMTcounter == 0)
    {
        AppStartToRun();
    }
}

/*******************************************************************************
*
* Function: void AppStartToRun(void)
*
* Description:  BLDC application Start to Run state transition function
*
*******************************************************************************/
static void AppStartToRun(void)
{
    /* Speed PI controller initialization */
    speedPIPrms.f16UpperLimit = SPEED_PI_UPPER_LIMIT;
    speedPIPrms.f16LowerLimit = SPEED_PI_LOWER_LIMIT;
    speedPIPrms.f16CC1sc = SPEED_PI_CC1;
    speedPIPrms.f16CC2sc = SPEED_PI_CC2;
    speedPIPrms.u16NShift = 0;
    speedPIPrms.f32Acc = (((tFrac32)duty_cycle << 15)/(tFrac32)PWM_MODULO) << 16;
    speedPIPrms.f16InErrK1 = 0;
    
    /* Current PI controller initialization */
    currentPIPrms.f16UpperLimit = CURRENT_PI_UPPER_LIMIT;
    currentPIPrms.f16LowerLimit = CURRENT_PI_LOWER_LIMIT;
    currentPIPrms.f16CC1sc = CURRENT_PI_CC1;
    currentPIPrms.f16CC2sc = CURRENT_PI_CC2;
    currentPIPrms.u16NShift = 0;
    currentPIPrms.f32Acc = speedPIPrms.f32Acc;
    currentPIPrms.f16InErrK1 = 0;

    appState = APP_RUN;
    stallCheckCounter = 0;
    StallError = 0;

    periodZC_F_PhA = NextCmtPeriod;
    periodZC_F_PhB = NextCmtPeriod;
    periodZC_F_PhC = NextCmtPeriod;
    periodZC_R_PhA = NextCmtPeriod;
    periodZC_R_PhB = NextCmtPeriod;
    periodZC_R_PhC = NextCmtPeriod;
    actualPeriodZC = NextCmtPeriod;

    driveStatus.B.Sensorless = 1;
    driveStatus.B.CloseLoop = 1;
}

/*******************************************************************************
*
* Function: void AppRun(void)
*
* Description:  BLDC application Run state function
*
*******************************************************************************/
static void AppRun(void)
{
    if(appSwitchState == 0)
    {
        appState = APP_INIT;
    }

    if(StallError)
    {
        AppStopToAlignment();
    }
}

/*******************************************************************************
*
* Function: void AppFault(void)
*
* Description: BLDC application Fault state function
*
*******************************************************************************/
static void AppFault(void)
{
    driveStatus.B.Alignment = 0;
    driveStatus.B.EnableCMT = 0;
    driveStatus.B.CloseLoop = 0;
    driveStatus.B.Sensorless = 0;
    driveStatus.B.NewZC = 0;

    PWME = PWM_DISABLE;         /* Disable all PWMs */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWME = PWM_TRIG_EN;         /* Enable PWM1 (ADC trigger signal) */

    if(faultSwitchClear == 1)
    {
        /* Clear MC33937A status register */
        MC33937_ClearFaults();

        /* Reset fault logic */
        FLT_RST = SET;
        FLT_RST = CLEAR;

        /* Re-enable MC33937A */
        MC33937_Enable();

        driveStatus.B.Fault = 0;
        faultStatusLatched.R = 0;

        faultSwitchClear = 0;
        appState = APP_INIT;
    }
}

/*******************************************************************************
*
* Function: void FaultDetection(void)
*
* Description:  BLDC application Fault detection function
*
*******************************************************************************/
static void FaultDetection(void)
{
    faultStatus.B.UnderDCBusVoltage = FAULT0;
    faultStatus.B.OverDCBusCurrent = FAULT1;
    
    faultStatus.B.PreDriverError = MC33937_INT;
    
    /* Reading faults from MC33937 pre-driver */
    if(faultStatus.B.PreDriverError)
    {
        mc33937_Status = MC33937_ReadStatus();
    }

    faultStatus.B.OverDCBusVoltage = (ADCResults.DCBVVoltage > FRAC16(DCBUS_OVERVOLT_THRESH/VOLT_RANGE_MAX)) ? SET:CLEAR;

    faultStatusLatched.R |= faultStatus.R;

    if(faultStatusLatched.R != 0)
    {
        driveStatus.B.Fault = 1;
        appState = APP_FAULT;
    }
}

/*******************************************************************************
*
* Function: void LED_Indication(void)
*
* Description:  LED indication control function
*
*******************************************************************************/
static void LED_Indication(void)
{
    /* State LED indication */
    if(LEDcounter-- == 0)
    {
        /* Default value */
        LEDcounter = PERIOD_STOP;
        switch(appState)
        {
            case APP_RUN:
            {
                LED0 = CLEAR;
                break;
            }
            case APP_INIT:
            case APP_ALIGNMENT:
            case APP_START:
            case APP_STOP:
            {
                LED0 = ~LED0;
                break;
            }
            case APP_FAULT:
            {
                LED0 = ~LED0;
                LEDcounter = PERIOD_FAULT;
                break;
            }
        }
    }
}

/*******************************************************************************
*
* Function: void Encoder_RotCtrl(void)
*
* Description:  Encoder speed control function
*
*******************************************************************************/
static void Encoder_RotCtrl(void)
{
    if(ROT_A != rotSwitchA)
    {
        rotSwitchA = ROT_A;
        if(rotSwitchA == ROT_B)
        {
            requiredSpeed -= SPEED_STEP;

            if(requiredSpeed < MIN_SPEED)
            {
                requiredSpeed = MIN_SPEED;
            }
        }
        else
        {
            requiredSpeed += SPEED_STEP;

            if(requiredSpeed > MAX_SPEED)
            {
                requiredSpeed = MAX_SPEED;
            }
        }
    }
}

/*******************************************************************************
*
* Function: void Encoder_SwCtrl(void)
*
* Description:  Encoder switch control function
*
*******************************************************************************/
static void Encoder_SwCtrl(void)
{
    if((ROT_SW == 0) && (ROT_SW == rotSwitch))
    {
        switchCounter++;

        if((switchCounter == SW_PRESS_LONG) && (appState == APP_FAULT))
        {
            faultSwitchClear = 1;
        }

        if(switchCounter > SW_PRESS_LONG)
        {
            switchCounter = SW_PRESS_LONG + 1;
        }
    }
    else
    {
        if((switchCounter > SW_PRESS_DEBOUNCE) && (switchCounter < SW_PRESS_SHORT_MAX))
        {
            appSwitchState = appSwitchState ^ 0x01;
        }

        switchCounter = 0;
    }

    rotSwitch = ROT_SW;
}

/*******************************************************************************
*
* Function: void PWM_Alignment(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the BLDC motor alignment.
*
* Notes:        Phase A TOP off, BOTTOM on
*               Phase B TOP off, BOTTOM on
*               Phase C complementary PWM (zero dead time)
*
*******************************************************************************/
static void PWM_Alignment(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT3 = 0x0000;           /* Reset PWM3 counter (phase A high-side PWM) */

    PORTA |= PWM_LS_ALIGN;      /* Enable all low-side switches to be driven
                                   by high-side PWM signals */
    PWME |= PWM_HS_ALIGN;       /* Enable phase C high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_0(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 0.
*
* Notes:        Phase A complementary PWM (zero dead time)
*               Phase B TOP off, BOTTOM on
*               Phase C TOP off, BOTTOM off
*
*******************************************************************************/
static void PWM_Vector_0(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT3 = 0x0000;           /* Reset PWM3 counter (phase A high-side PWM) */

    PORTA |= PWM_LS_SECT0;      /* Enable phase A & B low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT0;       /* Enable phase A high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_1(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 1.
*
* Notes:        Phase A complementary PWM (zero dead time)
*               Phase B TOP off, BOTTOM off
*               Phase C TOP off, BOTTOM on
*
*******************************************************************************/
static void PWM_Vector_1(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT3 = 0x0000;           /* Reset PWM3 counter (phase A high-side PWM) */

    PORTA |= PWM_LS_SECT1;      /* Enable phase A & C low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT1;       /* Enable phase A high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_2(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 2.
*
* Notes:        Phase A TOP off, BOTTOM off
*               Phase B complementary PWM (zero dead time)
*               Phase C TOP off, BOTTOM on
*
*******************************************************************************/
static void PWM_Vector_2(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT5 = 0x0000;           /* Reset PWM5 counter (phase B high-side PWM) */

    PORTA |= PWM_LS_SECT2;      /* Enable phase B & C low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT2;       /* Enable phase B high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_3(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 3.
*
* Notes:        Phase A TOP off, BOTTOM on
*               Phase B complementary PWM (zero dead time)
*               Phase C TOP off, BOTTOM off
*
*******************************************************************************/
void PWM_Vector_3(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT5 = 0x0000;           /* Reset PWM5 counter (phase B high-side PWM) */

    PORTA |= PWM_LS_SECT3;      /* Enable phase A & B low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT3;       /* Enable phase B high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_4(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 4.
*
* Notes:        Phase A TOP off, BOTTOM on
*               Phase B TOP off, BOTTOM off
*               Phase C complementary PWM (zero dead time)
*
*******************************************************************************/
static void PWM_Vector_4(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT7 = 0x0000;           /* Reset PWM7 counter (phase C high-side PWM) */

    PORTA |= PWM_LS_SECT4;      /* Enable phase A & C low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT4;       /* Enable phase C high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_5(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 5.
*
* Notes:        Phase A TOP off, BOTTOM off
*               Phase B TOP off, BOTTOM on
*               Phase C complementary PWM (zero dead time)
*
*******************************************************************************/
static void PWM_Vector_5(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT7 = 0x0000;           /* Reset PWM7 counter (phase C high-side PWM) */

    PORTA |= PWM_LS_SECT5;      /* Enable phase B & C low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT5;       /* Enable phase C high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void ZCdetectPhCfalling(void)
*
* Description:  Phase C zero-cross detection (falling Back-EMF).
*
*******************************************************************************/
static void ZCdetectPhCfalling(void)
{
    tFrac16 delta;

    if(bemfVoltage <= 0)
    {
        PORTA_PA7 = ~PORTA_PA7;
        /* Falling interpolation */
        delta = bemfVoltage - bemfVoltageOld;
        if((driveStatus.B.AdcSaved == 1) && (delta < bemfVoltage))
        {
            timeBackEmf -= Mul_F16(Div_F16(bemfVoltage, delta), (timeBackEmf - timeOldBackEmf));
        }
        else
        {
            timeBackEmf -= ((timeBackEmf - timeOldBackEmf) >> 1);
        }

        lastTimeZC = timeZC;
        timeZC = timeBackEmf;

        periodZC_F_PhC = timeZC - lastTimeZC;
        actualPeriodZC = (actualPeriodZC + periodZC_F_PhC) >> 1;

        NextCmtPeriod = Mul_F16(actualPeriodZC, advanceAngle);
        TC1 = timeZC + NextCmtPeriod;
        /* Current shifted right to fit sum of 6 currents in 16-bits */
        torque_F_PhC = ADCResults.DCBIVoltage >> 3;

        driveStatus.B.NewZC = 1;
    }
}

/*******************************************************************************
*
* Function: void ZCdetectPhBraising(void)
*
* Description:  Phase B zero-cross detection (raising Back-EMF).
*
*******************************************************************************/
static void ZCdetectPhBraising(void)
{
    tFrac16 delta;

    if(bemfVoltage >= 0)
    {
        PORTA_PA7 = ~PORTA_PA7;
        /* Raising interpolation */
        delta = bemfVoltage - bemfVoltageOld;
        if((driveStatus.B.AdcSaved == 1) && (delta > bemfVoltage))
        {
            timeBackEmf -= Mul_F16(Div_F16(bemfVoltage, delta), (timeBackEmf - timeOldBackEmf));
        }
        else
        {
            timeBackEmf -= ((timeBackEmf - timeOldBackEmf) >> 1);
        }

        lastTimeZC = timeZC;
        timeZC = timeBackEmf;

        periodZC_R_PhB = timeZC - lastTimeZC;
        actualPeriodZC = (actualPeriodZC + periodZC_R_PhB) >> 1;

        NextCmtPeriod = Mul_F16(actualPeriodZC, advanceAngle);
        TC1 = timeZC + NextCmtPeriod;
        /* Current shifted right to fit sum of 6 currents in 16-bits */
        torque_R_PhB = ADCResults.DCBIVoltage >> 3;

        driveStatus.B.NewZC = 1;
    }
}

/*******************************************************************************
*
* Function: void ZCdetectPhAfalling(void)
*
* Description:  Phase A zero-cross detection (falling Back-EMF).
*
*******************************************************************************/
static void ZCdetectPhAfalling(void)
{
    tFrac16 delta;

    if(bemfVoltage <= 0)
    {
        PORTA_PA7 = ~PORTA_PA7;
        /* Falling interpolation */
        delta = bemfVoltage - bemfVoltageOld;
        if((driveStatus.B.AdcSaved == 1) && (delta < bemfVoltage))
        {
            timeBackEmf -= Mul_F16(Div_F16(bemfVoltage, delta), (timeBackEmf - timeOldBackEmf));
        }
        else
        {
            timeBackEmf -= ((timeBackEmf - timeOldBackEmf) >> 1);
        }

        lastTimeZC = timeZC;
        timeZC = timeBackEmf;

        periodZC_F_PhA = timeZC - lastTimeZC;
        actualPeriodZC = (actualPeriodZC + periodZC_F_PhA) >> 1;

        NextCmtPeriod = Mul_F16(actualPeriodZC, advanceAngle);
        TC1 = timeZC + NextCmtPeriod;
        /* Current shifted right to fit sum of 6 currents in 16-bits */
        torque_F_PhA = ADCResults.DCBIVoltage >> 3;

        driveStatus.B.NewZC = 1;
    }
}

/*******************************************************************************
*
* Function: void ZCdetectPhCraising(void)
*
* Description:  Phase C zero-cross detection (raising Back-EMF).
*
*******************************************************************************/
static void ZCdetectPhCraising(void)
{
    tFrac16 delta;

    if(bemfVoltage >= 0)
    {
        PORTA_PA7 = ~PORTA_PA7;
        /* Raising interpolation */
        delta = bemfVoltage - bemfVoltageOld;
        if((driveStatus.B.AdcSaved == 1) && (delta > bemfVoltage))
        {
            timeBackEmf -= Mul_F16(Div_F16(bemfVoltage, delta), (timeBackEmf - timeOldBackEmf));
        }
        else
        {
            timeBackEmf -= ((timeBackEmf - timeOldBackEmf) >> 1);
        }

        lastTimeZC = timeZC;
        timeZC = timeBackEmf;

        periodZC_R_PhC = timeZC - lastTimeZC;
        actualPeriodZC = (actualPeriodZC + periodZC_R_PhC) >> 1;

        NextCmtPeriod = Mul_F16(actualPeriodZC, advanceAngle);
        TC1 = timeZC + NextCmtPeriod;
        /* Current shifted right to fit sum of 6 currents in 16-bits */
        torque_R_PhC = ADCResults.DCBIVoltage >> 3;

        driveStatus.B.NewZC = 1;
    }
}

/*******************************************************************************
*
* Function: void ZCdetectPhBfalling(void)
*
* Description:  Phase B zero-cross detection (falling Back-EMF).
*
*******************************************************************************/
static void ZCdetectPhBfalling(void)
{
    tFrac16 delta;

    if(bemfVoltage <= 0)
    {
        PORTA_PA7 = ~PORTA_PA7;
        /* Falling interpolation */
        delta = bemfVoltage - bemfVoltageOld;
        if((driveStatus.B.AdcSaved == 1) && (delta < bemfVoltage))
        {
            timeBackEmf -= Mul_F16(Div_F16(bemfVoltage, delta), (timeBackEmf - timeOldBackEmf));
        }
        else
        {
            timeBackEmf -= ((timeBackEmf - timeOldBackEmf) >> 1);
        }

        lastTimeZC = timeZC;
        timeZC = timeBackEmf;

        periodZC_F_PhB = timeZC - lastTimeZC;
        actualPeriodZC = (actualPeriodZC + periodZC_F_PhB) >> 1;

        NextCmtPeriod = Mul_F16(actualPeriodZC, advanceAngle);
        TC1 = timeZC + NextCmtPeriod;
        /* Current shifted right to fit sum of 6 currents in 16-bits */
        torque_F_PhB = ADCResults.DCBIVoltage >> 3;

        driveStatus.B.NewZC = 1;
    }
}

/*******************************************************************************
*
* Function: void ZCdetectPhAraising(void)
*
* Description:  Phase A zero-cross detection (raising Back-EMF).
*
*******************************************************************************/
static void ZCdetectPhAraising(void)
{
    tFrac16 delta;

    if(bemfVoltage >= 0)
    {
        PORTA_PA7 = ~PORTA_PA7;
        /* Raising interpolation */
        delta = bemfVoltage - bemfVoltageOld;
        if((driveStatus.B.AdcSaved == 1) && (delta > bemfVoltage))
        {
            timeBackEmf -= Mul_F16(Div_F16(bemfVoltage, delta), (timeBackEmf - timeOldBackEmf));
        }
        else
        {
            timeBackEmf -= ((timeBackEmf - timeOldBackEmf) >> 1);
        }

        lastTimeZC = timeZC;
        timeZC = timeBackEmf;

        periodZC_R_PhA = timeZC - lastTimeZC;
        actualPeriodZC = (actualPeriodZC + periodZC_R_PhA) >> 1;

        NextCmtPeriod = Mul_F16(actualPeriodZC, advanceAngle);
        TC1 = timeZC + NextCmtPeriod;
        /* Current shifted right to fit sum of 6 currents in 16-bits */
        torque_R_PhA = ADCResults.DCBIVoltage >> 3;

        driveStatus.B.NewZC = 1;
    }
}

/*******************************************************************************
*
* Function: void Control_Loop(void)
*
* Description:  Speed control, current limitation and user control loop
*               function.
*
*******************************************************************************/
void Control_Loop(void)
{
    if(driveStatus.B.StallCheckReq == 1)
    {
        driveStatus.B.StallCheckReq = 0;
        StallError = StallCheck();
    }

    if(driveStatus.B.CloseLoop == 1)
    {
        if(duty_cycle > DC_THRESHOLD)
        {
            /* Current limitation calculation (all currents shifted >> 3) */
            torque6ZC = torque_R_PhA + torque_R_PhB + torque_R_PhC + \
                        torque_F_PhA + torque_F_PhB + torque_F_PhC;
        }
        else
        {
            /* Ignore current measurement if duty cycle is too small to measure
               the DC-Bus current properly */
            torque6ZC = 0;
        }
        /* Shift left to scale back to <-1,1) */
        torque = (Mul_F16(torque6ZC,TORQUE_FILT_CONST) << 3);
        torqueErr = SubSat_F16(MAX_TORQUE, torque);
        currentPIOut = ControllerPIrAW_F16(torqueErr, &currentPIPrms);

        if(--speedLoopCounter == 0)
        {
            speedLoopCounter = SPEED_LOOP_PRESCALER;

            /* Speed control */
            period6ZC = periodZC_F_PhA + periodZC_R_PhA + periodZC_F_PhB + \
                        periodZC_R_PhB + periodZC_F_PhC + periodZC_R_PhC;
            actualSpeed = (tFrac16)Div_U32U16U16(SPEED_CALC_NUMERATOR, period6ZC);
            speedErr = requiredSpeed - (tFrac16)actualSpeed;
            speedPIOut = ControllerPIrAW_F16(speedErr, &speedPIPrms);

            if(currentPIOut >= speedPIOut)
            {
                /* If max torque not achieved, use speed PI output */
                currentPIPrms.f32Acc = speedPIPrms.f32Acc;
                /* PWM duty cycle update <- speed PI */
                duty_cycle = Mul_F16(speedPIOut, PWM_MODULO);
                driveStatus.B.CurrentLimiting = 0;
            }
            else
            {
                /* Limit speed PI output by current PI if max. torque achieved */
                speedPIPrms.f32Acc = currentPIPrms.f32Acc;
                /* PWM duty cycle update <- current PI */
                duty_cycle = Mul_F16(currentPIOut, PWM_MODULO);
                driveStatus.B.CurrentLimiting = 1;
            }

            /* Duty cycle 0-1 -> 0-PWM_MODULO */
            DISABLE_INTERRUPTS();
            PWMDTY23 = duty_cycle;
            PWMDTY45 = duty_cycle;
            PWMDTY67 = duty_cycle;
            PWMDTY01 = Mul_F16(duty_cycle,BEMF_SENSING_POINT_FRAC) + DELAY;
            ENABLE_INTERRUPTS();
        }

    }
    else
    {
        actualSpeed = 0;
        speedErr = 0;
        torque = 0;
    }

    /* User input control and LED indication */
    LED_Indication();
    Encoder_SwCtrl();
    Encoder_RotCtrl();
}

#pragma CODE_SEG __NEAR_SEG NON_BANKED
/*******************************************************************************
*
* Function: void interrupt ADC_SC_ISR(void)
*
* Description:  ADC conversion sequence complete interrupt service function.
*
*******************************************************************************/
void interrupt ADC_SC_ISR(void)
{
    PORTA_PA6 = 1;
    
    timeOldBackEmf = timeBackEmf;
    timeBackEmf = TC0;
    
    /* Save conversion result of Back-EMF voltage and DC-BUS current */
    ADCResults.DCBIVoltage = ((tFrac16)(ADC_DCBI_RES >> 1) - ADCResults.DCBIOffset);
    ADCResults.BEMFVoltage = (tFrac16)(ADC_BEMF_RES >> 1);

    /* Disable external trigger and start single conversion of DC-Bus voltage */
    ADC_StartSingleConversion(ADC_DCBV_AN);

    if(driveStatus.B.AfterCMT == 1)
    {
        if((timeBackEmf - timeCommutation) > timeZCToff)
        {
            driveStatus.B.AfterCMT = 0;
        }
    }

    /* Wait until the conversion DC-BUS is complete */
    while(ATDSTAT0_SCF == 0)
    {
    }
    
    /* Save conversion result of DC-BUS voltage */
    ADCResults.DCBVVoltage = (tFrac16)(ATDDR0 >> 1);
    
    if(driveStatus.B.Sensorless == 1)
    {
        /* DC-BUS resistor braking */
        UNI3_BRAKE = (ADCResults.DCBVVoltage > FRAC16(U_DCB_TRIP/VOLT_RANGE_MAX)) ? SET : CLEAR;
    }
    else
    {
        UNI3_BRAKE = CLEAR;
    }

    if((driveStatus.B.AfterCMT == 0) && (driveStatus.B.NewZC == 0) && (driveStatus.B.Sensorless == 1))
    {
        u_dc_bus_filt = ADCResults.DCBVVoltage;
        //u_dc_bus_filt = (tFrac16)(FilterMA_F16(ADCResults.DCBVVoltage, &Udcb_filt));

        bemfVoltage = ADCResults.BEMFVoltage - (u_dc_bus_filt >> 1);
        
        ZCdetection[ActualCmtSector]();

        bemfVoltageOld = bemfVoltage;   /* Save previous Back-EMF voltage
                                           (for ADC samples interpolation) */
        driveStatus.B.AdcSaved = 1;
    }

    if(driveStatus.B.NewZC == 1)
    {
        /* Re-enable externally triggered conversions (clears SCF flag) */
        /* DCBI -> DCBV */
        ADC_EnableTrigSeq(ADC_INPUT_AN5,2);
        /* Disable ADC interrupts for the rest of the commutation period
           to not collide with TIM1 (commutation) interrupt (SCF flag needs
           to be polled in the main endless loop to read DCBI & DCBV conversion
           results) */
        ATDCTL2_ASCIE = 0;
        ATDSTAT0 = ATDSTAT0_SCF_MASK;
    }
    else
    {
        /* Re-enable externally triggered conversions (clears SCF flag) */
        ADC_EnableTrigSeq(startADCchannel[ActualCmtSector],2);
    }

    PORTA_PA6 = 0;
}

/*******************************************************************************
*
* Function: void interrupt TIM1_OC_ISR(void)
*
* Description:  TIM1 output compare interrupt service routine.
*               Commutation function.
*
*******************************************************************************/
void interrupt TIM1_OC_ISR(void)
{
//    PORTA_PA7 = 1;
    
    timeCommutation = TCNT;

    if(driveStatus.B.EnableCMT == 1)
    {
        /* Commutation */
        PWMCommutationFcn[NextCmtSector]();
        /* Re-configure externally triggered conversions, enable interrupts */
        ADC_ModTrigSeqCh(startADCchannel[NextCmtSector]);

        driveStatus.B.AfterCMT = 1;
    }
    else
    {
        if(alignmentTimer > 0)
        {
            alignmentTimer--;
        }
    }

    if(driveStatus.B.Sensorless == 1)
    {
        if(driveStatus.B.NewZC == 0)
        {
            /* In the middle between two commutations */
            timeZC = timeCommutation - (actualPeriodZC >> 1);
        }
        TC1 = timeCommutation + (actualPeriodZC << 1);
        timeZCToff = actualPeriodZC >> 2;       /* 0.25 * actualPeriodZC */
        driveStatus.B.StallCheckReq = 1;
    }
    else
    {
        TC1 = timeCommutation + NextCmtPeriod;  /* Alignment + open loop */
    }

    ActualCmtSector = NextCmtSector;
    if(driveStatus.B.EnableCMT)
    {
        NextCmtSector++;
        if(NextCmtSector > 5)
        {
            NextCmtSector = 0;
        }
    }

    driveStatus.B.NewZC = 0;
    driveStatus.B.AdcSaved = 0;

    TFLG1 = TFLG1_C1F_MASK;     /* Clear output compare event flag */
//    PORTA_PA7 = 0;
}
#pragma CODE_SEG DEFAULT


/*******************************************************************************
*
* Function: int8_t StallCheck(void)
*
* Description:  Stall detection function
*
*******************************************************************************/
static uint8_t StallCheck(void)
{
    uint16_t max = 0, min = 65535;

    if(periodZC_F_PhA>max)
    {
        max = periodZC_F_PhA;
    }
    if(periodZC_F_PhA<min)
    {
        min = periodZC_F_PhA;
    }
    if(periodZC_F_PhB>max)
    {
        max = periodZC_F_PhB;
    }
    if(periodZC_F_PhB<min)
    {
        min = periodZC_F_PhB;
    }
    if(periodZC_F_PhC>max)
    {
        max = periodZC_F_PhC;
    }
    if(periodZC_F_PhC<min)
    {
        min = periodZC_F_PhC;
    }
    if(periodZC_R_PhA>max)
    {
        max = periodZC_R_PhA;
    }
    if(periodZC_R_PhA<min)
    {
        min = periodZC_R_PhA;
    }
    if(periodZC_R_PhB>max)
    {
        max = periodZC_R_PhB;
    }
    if(periodZC_R_PhB<min)
    {
        min = periodZC_R_PhB;
    }
    if(periodZC_R_PhC>max)
    {
        max = periodZC_R_PhC;
    }
    if(periodZC_R_PhC<min)
    {
        min = periodZC_R_PhC;
    }

    /* Save min and max commutation periods for tuning purposes */
    debugPeriodMin = min;
    debugPeriodMax = max;

    /* 0.33 * period6ZC = 2 * CMT period; 0.0833 * period6ZC = 0.5 * CMT period */
    if((max > Mul_F16(period6ZC, FRAC16(0.3333))) || (min < (Mul_F16(period6ZC, FRAC16(0.0833)))))
    {
        if(stallCheckCounter < STALLCHECK_MAX_ERRORS)
        {
            stallCheckCounter++;
        }
    }
    else
    {
        /* 115 = 115 us, .....(duty_cycle <= MIN_DUTY_CYCLE) */
        if(min < 115)
        {
            if(stallCheckCounter < STALLCHECK_MAX_ERRORS)
            {
                stallCheckCounter++;
            }
        }
        else
        {
            if(stallCheckCounter > 0)
            {
                stallCheckCounter--;
            }
        }
    }

    if(stallCheckCounter >= STALLCHECK_MAX_ERRORS)
    {
        return(1);
    }
    else
    {
        return(0);
    }
}

/*******************************************************************************
*
* Function: void Can_RxNotification
*
* Description:  CAN receive notification function
*
*******************************************************************************/
void Can_RxNotification
(
    Can_ModuleIDType    Can_ModuleID,
    Can_IDType          Can_ID,
    Can_DataLengthType  Can_DataLength,
    uint8               *Can_Data
)
{
    asm
    {
        nop
    }
}