/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     MC33905_routines.c
*
* @author   B06050
* 
* @version  1.0.2.0
* 
* @date     Jul-10-2012
* 
* @brief    MC33905 System Basis Chip configuration function source file with
*           routines for SBC setup and diagnostics.
*
*******************************************************************************/
#include "MC33905_routines.h"

/*******************************************************************************
*
* Function: void MC33905_Write(uint16_t u16SBC_Address, uint8_t u8SBC_Data)
*
* Description:  Writes 8-bit value into SBC register via SPI0.
*
* Param[in]:    u16SBC_Address      SBC register address shifted left by 8 bits
*               u8SBC_Data          8-bit register value
*
* Notes:        SPI_0_Init (spi.c) function must be already called prior to this
*               function call.
*
*******************************************************************************/
void MC33905_Write(uint16_t u16SBC_Address, uint8_t u8SBC_Data)
{    
    uint16_t u16SBC_Command;    /* Data word to be sent to the SBC */
    
    u16SBC_Command = (u16SBC_Address | SBC_WR | u8SBC_Data);
          
    MC33905_CS = 0;    /* CS low */
    (void)SPI_0_SendWord(u16SBC_Command);  /* Send data to SBC */      
    MC33905_CS = 1;    /* CS high */
}

/*******************************************************************************
*
* Function: uint16_t MC33905_Read(uint16_t u16SBC_Address, uint8_t u8SBC_Data)
*
* Description:  Reads 8-bit value from SBC register via SPI0.
*
* Param[in]:    u16SBC_Address      SBC register address shifted left by 8 bits
*               u8SBC_Data          8-bit register sub-address
*
* Return:       uint16_t            Register value
*
* Notes:        SPI_0_Init (spi.c) function must be already called prior to this
*               function call.
*
*******************************************************************************/
uint16_t MC33905_Read(uint16_t u16SBC_Address, uint8_t u8SBC_Data)
{      
    uint16_t u16SBC_Command;    /* Data word to be sent to the SBC */
    uint16_t u16RxData;         /* Variable to store received word */
    
    u16SBC_Command = (u16SBC_Address | u8SBC_Data);
    
    MC33905_CS = 0;    /* CS low */ 
    u16RxData = SPI_0_SendWord(u16SBC_Command); 
    MC33905_CS = 1;    /* CS high */          
        
    return(u16RxData);
}

/*******************************************************************************
*
* Function: void MC33905_Config(void)
*
* Description:  Performs initial SBC configuration.
*
* Notes:        SPI_0_Init (spi.c) function must be already called prior to this
*               function call.
*
*******************************************************************************/
void MC33905_Config(void)
{
    uint16_t u16SBC_Status;
    uint16_t u16Batfail_flag;
   
    /* Read 16 bit flags */    
    u16SBC_Status = MC33905_Read(SBC_REGISTER|SBC_WR|SBC_REGULATOR|SBC_NEXT, 0x00);

    /* Clear flags if needed */

    /* Clear VREG flags */
    u16Batfail_flag = MC33905_Read(SBC_FLAG|SBC_REGULATOR|SBC_NEXT, SBC_HL);    
    MC33905_Write(SBC_FLAG|SBC_REGULATOR|SBC_NEXT, 0x00);

    /* Clear SAFE flag */
    MC33905_Write(SBC_FLAG|SBC_MODE_RM|SBC_NEXT, SBC_HL);

    /* Clear IO flags */
    MC33905_Write(SBC_FLAG|SBC_IO|SBC_NEXT, 0x00);    
//!    MC33905_Write(SBC_FLAG|SBC_IO|SBC_NEXT, SBC_HL);
    
    /* Clear LIN flags */
    MC33905_Write(SBC_FLAG|SBC_LIN1|SBC_NEXT, 0x00);    
//!    MC33905_Write(SBC_FLAG|SBC_LIN1|SBC_NEXT, SBC_HL);

    MC33905_Write(SBC_FLAG|SBC_LIN2|SBC_NEXT, 0x00);    
//!    MC33905_Write(SBC_FLAG|SBC_LIN2|SBC_NEXT, SBC_HL);

    /* Clear CAN flags */
    MC33905_Write(SBC_FLAG|SBC_CAN|SBC_NEXT, 0x00);    
//!    MC33905_Write(SBC_FLAG|SBC_CAN|SBC_NEXT, SBC_HL);
  
    /* LIN1 and LIN2 */           
    MC33905_Write(SBC_REGISTER|SBC_INIT_LIN_IO, SBC_LIN_INITVAL);
    
    /* Watchdog settings */
    MC33905_Write(SBC_REGISTER|SBC_INIT_WDOG, SBC_WDT_INITVAL);   
    
    /* VREG */
    MC33905_Write(SBC_REGISTER|SBC_INIT_VREG, SBC_VREG_INITVAL);
    
    /* MISC */
    MC33905_Write(SBC_REGISTER|SBC_INIT_MISC, SBC_MISC_INITVAL);      
      
    /* Wait at least 50 microseconds before writing to timers registers */  
    API_Delay(API_100US);
    
    /* Set watchdog default period */
    MC33905_SetWDTperiod(SBC_WDT_TIMEOUT);
            
    MC33905_ClearWDT();  /* Transition to normal mode */
    
    /* Resources initial configuration */
    /* Interrupt sources */
    MC33905_Write(SBC_REGISTER|SBC_INTERRUPT, SBC_INT_INITVAL);
    
    /* Regulator */ 
    MC33905_Write(SBC_REGISTER|SBC_REGULATOR, SBC_REG_INITVAL); 
            
    /* CAN transceiver */
    MC33905_CAN_Config(SBC_CAN_INITV);  
    
    /* LIN transceivers */
    MC33905_Write(SBC_REGISTER|SBC_LIN1,SBC_LIN1_INITV);   
    MC33905_Write(SBC_REGISTER|SBC_LIN2,SBC_LIN2_INITV); 
        
    /* I/O configuration */    
    MC33905_Write(SBC_REGISTER|SBC_IO,SBC_IO_INITV); 
            
    /* MUX output */
    MC33905_Write(SBC_REGISTER|SBC_MUX, SBC_MUX_INITV); 
    
    /* Read MODE flag register to clear SAFE flag */
    MC33905_Write(SBC_FLAG|SBC_MODE_RM|SBC_NEXT, 0x80);                              
}

/*******************************************************************************
*
* Function: void MC33905_ClearWDT(void)
*
* Description:  Resets SBC watchdog.
*
* Notes:        SPI_0_Init (spi.c) function must be already called prior to this
*               function call.
*
*******************************************************************************/
void MC33905_ClearWDT(void)
{       
    /* Read MODE flag register to clear SAFE flag */
    //MC33905_Write(SBC_FLAG|SBC_MODE_RM|SBC_NEXT, 0x80);       
    
    /* Clear watchdog */
    MC33905_Write(SBC_REGISTER|SBC_WDOG_REFRESH, 0x00);        
}

/*******************************************************************************
*
* Function: void MC33905_SetWDTperiod(uint8_t u8WDT_Period)
*
* Description:  Sets SBC watchdog period.
*
* Param[in]:    u8WDT_Period        Watchdog period (see SBC_WDT_TIMEOUT macro
*                                   in the MC33905_routines.h initial
*                                   configurating section)
*
* Notes:        SPI_0_Init (spi.c) function must be already called prior to this
*               function call.
*
*******************************************************************************/
void MC33905_SetWDTperiod(uint8_t u8WDT_Period)
{
    MC33905_Write(SBC_REGISTER|SBC_IMCUSTP_WDNOR, u8WDT_Period);     
}

/*******************************************************************************
*
* Function: void MC33905_CAN_Config(uint8_t u8CANMode)
*
* Description:  Performs SBC CAN transceiver configuration.
*
* Param[in]:    u8CANMode           SBC CAN configuration (see SBC_CAN_INITV
*                                   macro in the MC33905_routines.h initial
*                                   configuration section)
*
* Notes:        SPI_0_Init (spi.c) function must be already called prior to this
*               function call.
*
*******************************************************************************/
void MC33905_CAN_Config(uint8_t u8CANMode)
{
    /* CAN configuration */
    MC33905_Write(SBC_REGISTER|SBC_CAN, u8CANMode);
}

/*******************************************************************************
*
* Function: void MC33905_SetCyclicSense(uint8_t u8CycSens_Period)
*
* Description:  Configures SBC cyclic sense and cyclic sense interrupt period.
*
* Param[in]:    u8CycSens_Period    SBC cyclic sense and cyclic interrupt
*                                   settings (see MC33905_routines.h for
*                                   predefined macros)
*
* Notes:        SPI_0_Init (spi.c) function must be already called prior to this
*               function call.
*
*******************************************************************************/
void MC33905_SetCyclicSense(uint8_t u8CycSens_Period)
{ 
     /* Update timer register */                 
    MC33905_Write(SBC_REGISTER|SBC_CYC_SENSE_INT,u8CycSens_Period);  
}

/*******************************************************************************
*
* Function: void MC33905_StopMode(uint8_t u8StopMode)
*
* Description:  Sets the SBC Stop mode.
*
* Param[in]:    u8StopMode          SBC Stop mode settings (see
*                                   MC33905_routines.h for predefined macros)
*
* Notes:        SPI_0_Init (spi.c) function must be already called prior to this
*               function call.
*
*******************************************************************************/
void MC33905_StopMode(uint8_t u8StopMode)
{                                                            
    /* Send Stop mode command */
    MC33905_Write(SBC_REGISTER|SBC_MODE_RM, SBC_MODE_4|u8StopMode);                      
}

/*******************************************************************************
*
* Function: uint16_t MC33905_Read(uint16_t u16SBC_Address, uint8_t u8SBC_Data)
*
* Description:  Sets the SBC Sleep mode.
*
* Param[in]:    u8SleepMode         SBC Sleep mode settings (see
*                                   MC33905_routines.h for predefined macros)
*
* Notes:        SPI_0_Init (spi.c) function must be already called prior to this
*               function call.
*
*******************************************************************************/
void MC33905_SleepMode(uint8_t u8SleepMode)
{                                                            
    /* Send Sleep mode command */
    MC33905_Write(SBC_REGISTER|SBC_MODE_RM, u8SleepMode);                      
}