/*******************************************************************************
* FILE NAME: etpu_sc.h              COPYRIGHT (c) Freescale Semiconductor 2004
*                                               All Rights Reserved
* DESCRIPTION:
* This file contains the prototypes and definitions for the eTPU Speed
* Controller (SC) API.
*===============================================================================
* REV      AUTHOR      DATE        DESCRIPTION OF CHANGE
* ---   -----------  ----------    ---------------------
* 1.0   A. Butok     26/Oct/04     Updated after merge of PID to SC,
*       M. Brejl                   ready for release.
*       M. Princ
* 2.0   M. Brejl     17/Dec/04     Updated for new build structure.
* 2.2   M. Brejl     18/Feb/05     PWMM_chan init parameter replaced by
*                                  output_chan, output_offset and link_chan
*                                  parameters, in order to support update of
*                                  any function parameter, not only the PWMM
*                                  duty-cycle.
* 2.4   M. Brejl     21/Jan/12     Resolver support added.
*******************************************************************************/
#ifndef _FS_ETPU_SC_H_
#define _FS_ETPU_SC_H_

#include "etpu_sc_auto.h"    /* Auto generated header file */
#include "etpu_hd_auto.h"    /* Auto generated header file */
#include "etpu_qd_auto.h"    /* Auto generated header file */
#include "etpu_rslv_auto.h"  /* Auto generated header file */
#include "etpu_mc_ctrl_auto.h" /* Auto generated header file */

/*******************************************************************************
*                              Types
*******************************************************************************/
typedef struct {
   fract24_t P_gain;        /* P-portion gain in fractional format (9.15) */
   fract24_t I_gain;        /* I-portion gain in fractional format (9.15) */
   fract24_t D_gain;        /* D-portion gain in fractional format (9.15) */
   int16_t positive_limit;  /* positive limit threashold in fractional format (1.15) */
   int16_t negative_limit;  /* negative limit threashold in fractional format (1.15) */
} sc_pid_params_t;

typedef struct {
   fract24_t ramp_incr_up;    /* ramp increment in up direction in fractional format (1.23) */
   fract24_t ramp_incr_down;  /* ramp increment in down direction in fractional format (1.23) */
} sc_ramp_params_t;

/*******************************************************************************
*                            Definitions
*******************************************************************************/
/* mode options. */
#define FS_ETPU_SC_SLAVE              (0)    /* Slave mode */
#define FS_ETPU_SC_MASTER             (1)    /* Master mode */

/* decoder_type options */
#define FS_ETPU_SC_QD                 (0)    /* Quadrature Decoder */
#define FS_ETPU_SC_HD_REV_PERIOD      (1)    /* Hall Decoder - Revolution period is measured */
#define FS_ETPU_SC_HD_SEC_PERIOD_1    (2)    /* 1-phase Hall Decoder - Sector period is measured */
#define FS_ETPU_SC_HD_SEC_PERIOD_2    (4)    /* 2-phase Hall Decoder - Sector period is measured */
#define FS_ETPU_SC_HD_SEC_PERIOD_3    (6)    /* 3-phase Hall Decoder - Sector period is measured */
#define FS_ETPU_SC_HD_SEC_PERIOD_4    (8)    /* 4-phase Hall Decoder - Sector period is measured */
#define FS_ETPU_SC_RSLV               (9)    /* Resolver */

/* configuration options. */
#define FS_ETPU_SC_RAMP_OFF_PID_OFF   (FS_ETPU_SC_FM_RAMP_OFF + FS_ETPU_SC_FM_PID_OFF)  /* Ramp OFF, PID OFF */
#define FS_ETPU_SC_RAMP_OFF_PID_ON    (FS_ETPU_SC_FM_RAMP_OFF + FS_ETPU_SC_FM_PID_ON)   /* Ramp OFF, PID ON */
#define FS_ETPU_SC_RAMP_ON_PID_OFF    (FS_ETPU_SC_FM_RAMP_ON  + FS_ETPU_SC_FM_PID_OFF)  /* Ramp ON,  PID OFF */
#define FS_ETPU_SC_RAMP_ON_PID_ON     (FS_ETPU_SC_FM_RAMP_ON  + FS_ETPU_SC_FM_PID_ON)   /* Ramp ON,  PID ON */

/* HD_QD_etpu_tcr_freq options */
#define FS_ETPU_SC_ETPU_A_TCR1_FREQ   etpu_a_tcr1_freq   /* frequecy of eTPU engine A - TCR1 */
#define FS_ETPU_SC_ETPU_A_TCR2_FREQ   etpu_a_tcr2_freq   /* frequecy of eTPU engine A - TCR2 */
#define FS_ETPU_SC_ETPU_B_TCR1_FREQ   etpu_b_tcr1_freq   /* frequecy of eTPU engine B - TCR1 */
#define FS_ETPU_SC_ETPU_B_TCR2_FREQ   etpu_b_tcr2_freq   /* frequecy of eTPU engine B - TCR2 */

/* saturation flag */
#define FS_ETPU_SC_SATURATION_NO      (0)
#define FS_ETPU_SC_SATURATION_POS     (1)
#define FS_ETPU_SC_SATURATION_NEG     (2)

/*******************************************************************************
*                       Function Prototypes
*******************************************************************************/
/* SC channel initialization */
int32_t fs_etpu_sc_init( uint8_t channel,
                         uint8_t priority,
                         uint8_t mode,
                         uint8_t decoder_type,
                         uint8_t configuration,
                        uint24_t period,
                        uint24_t start_offset,
                        uint24_t services_per_irq,
                sc_pid_params_t* p_pid_params,
               sc_ramp_params_t* p_ramp_params,
                         uint8_t HD_QD_RSLV_chan,
                         uint8_t output_chan,
                        uint16_t output_offset,
                         uint8_t link_chan,
                        uint24_t omega_max_rpm,
                        uint24_t omega_min_rpm,
                         uint8_t pole_pairs,
                        uint32_t HD_QD_etpu_tcr_freq,
                        uint32_t rslv_freq,
                        uint24_t qd_pc_per_rev );

/* Force controller update */
int32_t fs_etpu_sc_update(uint8_t channel);

/* Change confguration */
int32_t fs_etpu_sc_set_configuration(uint8_t  channel,
                                     uint8_t configuration);

/* Change PID parameters */
int32_t fs_etpu_sc_set_pid_params(uint8_t  channel,
                          sc_pid_params_t* p_pid_params);

/* Change RAMP parameters */
int32_t fs_etpu_sc_set_ramp_params(uint8_t  channel,
                          sc_ramp_params_t* p_ramp_params);

/* Set omega_desired */
int32_t fs_etpu_sc_set_omega_desired(uint8_t   channel,
                                     fract24_t omega_desired);

/* Set controller integral portion */
int32_t fs_etpu_sc_set_integral_portion(uint8_t channel, fract24_t i_k1);

/* Get omega_actual */
fract24_t fs_etpu_sc_get_omega_actual( uint8_t channel);

/* Get controller saturation flag */
uint8_t fs_etpu_sc_get_saturation_flag(uint8_t channel);

/* Get ramp output */
fract24_t fs_etpu_sc_get_ramp_output(uint8_t channel);

/* Get controller output */
fract24_t fs_etpu_sc_get_output(uint8_t channel);

/* Get controller error */
fract24_t fs_etpu_sc_get_error(uint8_t channel);

/* Get controller integral portion */
fract24_t fs_etpu_sc_get_integral_portion(uint8_t channel);

/* Get SC configuration */
uint8_t fs_etpu_sc_get_configuration(uint8_t channel);

/* Get QD position difference */
int24_t fs_etpu_sc_get_pc_diff(uint8_t channel);

/* Get QD time difference */
uint24_t fs_etpu_sc_get_time_diff(uint8_t channel);

/*******************************************************************************
*        Information for eTPU Graphical Configuration Tool 
*******************************************************************************/
/* full function name: SC - Speed Controller */ 
/* channel assignment when (mode=FS_ETPU_SC_MASTER); load: sc_master
   channel=channel
*/
/* channel assignment when (mode=FS_ETPU_SC_SLAVE); load: sc_slave
   channel=channel
*/
/* load expression sc_master
base = 100*etpu_tcr1_freq/(period*etpu_clock_freq)
channel = if(decoder_type==FS_ETPU_SC_QD, 332*base, 254*base)
*/
/* load expression sc_slave
var update_frequency {1..100000} [Hz]
base = 100*update_frequency/etpu_clock_freq
channel = if(decoder_type==FS_ETPU_SC_QD, 320*base, 242*base)
*/

#endif

/*********************************************************************
 *
 * Copyright:
 *	Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ********************************************************************/
