/*******************************************************************************
* FILE NAME: etpu_pmsmvc.h          COPYRIGHT (c) Freescale Semiconductor 2006
*                                               All Rights Reserved
* DEACIMVHZRIPTION:
* This file contains the prototypes and definitions for the eTPU Permanent 
* Magnet Synchronous Motor Vector Control (PMSMVC) API.
*===============================================================================
* REV      AUTHOR      DATE        DESCRIPTION OF CHANGE
* ---   -----------  ----------    ---------------------
* 0.1   M. Brejl     22/Aug/05     Initial file version.
* 2.0   M. Brejl     22/Aug/05     Clean up of compiler warnings.
* 2.1   M. Princ     17/Mar/06     Corrections due to the new implementation
*                                  of ripple elimination.
* 2.2   M. Brejl     27/Jan/12     Support of RSLV usage added.
* 2.3   M. Brejl     30/Mar/12     Compensation delay added.
* 2.4   M. Brejl     25/Apr/12     Addition of fs_etpu_pmsmvc_get_u_dq_clim()
*******************************************************************************/
#ifndef _FS_ETPU_PMSMVC_H_
#define _FS_ETPU_PMSMVC_H_

#include "etpu_pmsmvc_auto.h"  /* Auto generated header file */
#include "etpu_sc_auto.h"      /* Auto generated header file */
#include "etpu_qd_auto.h"      /* Auto generated header file */
#include "etpu_rslv_auto.h"    /* Auto generated header file */
#include "etpu_mc_ctrl_auto.h" /* Auto generated header file */

/*******************************************************************************
*                              Types
*******************************************************************************/
/* Permanent Magnet Synchronous Motor parameters */
typedef struct {
   fract24_t Ke;  /* motor electrical constant in fractional format (3.21) */
   fract24_t Ld;  /* motor induction in D-coordinates in fractional format (3.21) */
   fract24_t Lq;  /* motor induction in Q-coordinates in fractional format (3.21) */
} pmsmvc_motor_params_t;

/* ABC coordinates */
typedef struct {
   fract24_t a;
   fract24_t b;
   fract24_t c;
} pmsmvc_abc_t;

/* AB coordinates */
typedef struct {
   fract24_t a;
   fract24_t b;
} pmsmvc_ab_t;

/* DQ coordinates */
typedef struct {
   fract24_t d;
   fract24_t q;
} pmsmvc_dq_t;

/* PID controller parameters */
typedef struct {
   fract24_t P_gain;        /* P-portion gain in fractional format (9.15) */
   fract24_t I_gain;        /* I-portion gain in fractional format (9.15) */
   fract24_t D_gain;        /* D-portion gain in fractional format (9.15) */
   int16_t positive_limit;  /* positive limit threshold in fractional format (1.15) */
   int16_t negative_limit;  /* negative limit threshold in fractional format (1.15) */
} pmsmvc_pid_params_t;


/*******************************************************************************
*                            Definitions
*******************************************************************************/
/* mode options. */
#define FS_ETPU_PMSMVC_SLAVE              (0)   /* Slave mode */
#define FS_ETPU_PMSMVC_MASTER             (1)   /* Master mode */

/* circle_limitation_config options. */
#define FS_ETPU_PMSMVC_CIRCLE_LIMITATION_OFF (FS_ETPU_PMSMVC_FM_CIRCLE_LIMITATION_OFF)  /* Circle limitation OFF */
#define FS_ETPU_PMSMVC_CIRCLE_LIMITATION_ON  (FS_ETPU_PMSMVC_FM_CIRCLE_LIMITATION_ON)   /* Circle limitation ON */

/* inv_mod_index options */
#define FS_ETPU_PMSMVC_INVMODINDEX_SINE   (0x7FFFFF)  /* Sine Wave Modulation */
#define FS_ETPU_PMSMVC_INVMODINDEX_SIN3H  (0x6ED9EC)  /* Sine Wave with 3-rd Harmonic */
#define FS_ETPU_PMSMVC_INVMODINDEX_SVM    (0x6ED9EC)  /* Space Vector Modulation */

/* mod_index options */
#define FS_ETPU_PMSMVC_MODINDEX_SINE      (0x400000)  /* Sine Wave Modulation */
#define FS_ETPU_PMSMVC_MODINDEX_SIN3H     (0x49E69F)  /* Sine Wave with 3-rd Harmonic */
#define FS_ETPU_PMSMVC_MODINDEX_SVM       (0x49E69F)  /* Space Vector Modulation */

/* saturation flag */
#define FS_ETPU_PMSMVC_SATURATION_NO      (0)
#define FS_ETPU_PMSMVC_SATURATION_POS     (1)
#define FS_ETPU_PMSMVC_SATURATION_NEG     (2)

/*******************************************************************************
*                       Function Prototypes
*******************************************************************************/
/* PMSMVC channel initialization */
int32_t fs_etpu_pmsmvc_init( uint8_t channel,
                             uint8_t priority,
                             uint8_t mode,
                             uint8_t circle_limitation_config,
                            uint24_t period,
                            uint24_t start_offset,
                            uint24_t services_per_irq,
                             uint8_t SC_chan,
                             uint8_t QD_RSLV_chan,
                            uint24_t qd_pc_per_rev,
                             uint8_t rslv_pole_pairs,
                             uint8_t motor_pole_pairs,
              pmsmvc_motor_params_t* p_motor_params,
                pmsmvc_pid_params_t* p_pid_d_params,
                pmsmvc_pid_params_t* p_pid_q_params,
                             int24_t inv_mod_index,
                             uint8_t output_chan,
                            uint16_t output_offset,
                             uint8_t link_chan,
                            uint24_t compensation_delay);

/* Force controller update */
int32_t fs_etpu_pmsmvc_update(uint8_t channel);

/* Set desired D-current */
int32_t fs_etpu_pmsmvc_set_i_d_desired(uint8_t  channel,
                                     fract24_t  i_d_desired);

/* Set desired Q-current */
int32_t fs_etpu_pmsmvc_set_i_q_desired(uint8_t  channel,
                                     fract24_t  i_q_desired);

/* Set i_dq_desired - desired phase currents in 2-phase orthogonal rotating
   reference frame */
int32_t fs_etpu_pmsmvc_set_i_dq_desired(uint8_t  channel,
                                        pmsmvc_dq_t * p_i_dq_desired);

/* Set u_dc_bus_measured */
int32_t fs_etpu_pmsmvc_set_u_dc_bus_measured(uint8_t  channel,
                                          ufract24_t  u_dc_bus_measured);

/* Set i_abc - phase currents in 3-phase stationary reference frame */
int32_t fs_etpu_pmsmvc_set_i_abc(uint8_t  channel,
                                 pmsmvc_abc_t * p_i_abc);

/* Get i_abc - phase currents in 3-phase stationary reference frame */
int32_t fs_etpu_pmsmvc_get_i_abc(uint8_t channel,
                                 pmsmvc_abc_t * p_i_abc);

/* Get i_ab - phase currents in 2-phase orthog. stationary ref. frame */
int32_t fs_etpu_pmsmvc_get_i_ab(uint8_t channel,
                                pmsmvc_ab_t * p_i_ab);

/* Get i_dq - phase currents in 2-phase orthogonal rotating ref. frame */
int32_t fs_etpu_pmsmvc_get_i_dq(uint8_t channel,
                                pmsmvc_dq_t * p_i_dq);

/* Get i_dq_desired - desired phase currents in 2-phase orthogonal 
   rotating ref. frame */
int32_t fs_etpu_pmsmvc_get_i_dq_desired(uint8_t channel,
                                        pmsmvc_dq_t * p_i_dq_desired);

/* Get u_dq - stator voltages in 2-phase orthogonal rotating reference frame
   - before circle limitation*/
int32_t fs_etpu_pmsmvc_get_u_dq(uint8_t channel,
                                pmsmvc_dq_t * p_u_dq);

/* Get u_dq - stator voltages in 2-phase orthogonal rotating reference frame
   - after circle limitation*/
int32_t fs_etpu_pmsmvc_get_u_dq_clim(uint8_t channel,
                                     pmsmvc_dq_t * p_u_dq);

/* Get u_ab - stator voltages in 2-phase orthogonal stationary reference frame*/
int32_t fs_etpu_pmsmvc_get_u_ab(uint8_t channel,
                                pmsmvc_ab_t * p_u_ab);

/* Get D PID controller saturation flags */
uint8_t fs_etpu_pmsmvc_get_saturation_flag_d(uint8_t channel);

/* Get Q PID controller saturation flags */
uint8_t fs_etpu_pmsmvc_get_saturation_flag_q(uint8_t channel);

/* Set D PID controller integral portion */
int32_t fs_etpu_pmsmvc_set_integral_portion_d( uint8_t channel, fract24_t i_k1);

/* Set Q PID controller integral portion */
int32_t fs_etpu_pmsmvc_set_integral_portion_q( uint8_t channel, fract24_t i_k1);

/* Set PMSMVC configuration - PID on or PID off */
int32_t fs_etpu_pmsmvc_set_configuration(uint8_t  channel, uint8_t configuration);

/*******************************************************************************
*        Information for eTPU Graphical Configuration Tool 
*******************************************************************************/
/* full function name: PMSMVC - PMSM Vector Control */
/* channel assignment when (mode=FS_ETPU_PMSMVC_MASTER); load: pmsmvc_master
   channel=channel
*/
/* channel assignment when (mode=FS_ETPU_PMSMVC_SLAVE); load: pmsmvc_slave
   channel=channel
*/
/* load expression pmsmvc_master
base = 100*etpu_tcr1_freq/(period*etpu_clock_freq)
calc_thread1 = 546
calc_thread2 = if(circle_limitation_config==FS_ETPU_PMSMVC_CIRCLE_LIMITATION_OFF, 208, 452)
channel = (12+calc_thread1+calc_thread2)*base
*/
/* load expression pmsmvc_slave
var update_frequency {1..100000} [Hz]
base = 100*update_frequency/etpu_clock_freq
calc_thread1 = 546
calc_thread2 = if(circle_limitation_config==FS_ETPU_PMSMVC_CIRCLE_LIMITATION_OFF, 208, 452)
channel = (calc_thread1+calc_thread2)*base
*/

#endif

/*********************************************************************
 *
 * Copyright:
 *	Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ********************************************************************/
