/*******************************************************************************
* FILE NAME: etpu_asac.h            COPYRIGHT (c) Freescale Semiconductor 2005
*                                               All Rights Reserved
* DESCRIPTION:
* This file contains the prototypes and defines for the eTPU Analog Sensing
* for AC motors (ASAC) function API.
*===============================================================================
* REV      AUTHOR      DATE        DESCRIPTION OF CHANGE
* ---   -----------  ----------    ---------------------
* 2.0   M.Princ      02/Sep/05     Initial version derived from ASDC function.
* 2.1   M.Princ      01/Dec/05     eQADC command queue definition added,
*                                  DMA request and interrupt to CPU on the first
*                                  and the second edge option added.
* 2.4   M.Brejl      18/Jan/12     Option negate_phase_currents added. 
********************************************************************************/
#ifndef _ETPU_ASAC_H_
#define _ETPU_ASAC_H_

#include "etpu_asac_auto.h"    /* Auto generated header file for eTPU ASAC */
#include "etpu_pwmmac_auto.h"  /* Auto generated header file for eTPU PWMMAC */
#include "etpu_pwmf_auto.h"    /* Auto generated header file for eTPU PWMF */

/*******************************************************************************
*                              Types
*******************************************************************************/
/* ABC coordinates */
typedef struct {
   fract24_t a;
   fract24_t b;
   fract24_t c;
} asac_abc_t;

/* table of eQADC conversion commands for current measurement - structure type definition. */
typedef struct {
  uint32_t ADC0_iA_cmd; /* conversion command - measurement of phaseA current on ADC0 */
  uint32_t ADC1_iA_cmd; /* conversion command - measurement of phaseA current on ADC1 */
  uint32_t ADC0_iB_cmd; /* conversion command - measurement of phaseB current on ADC0 */
  uint32_t ADC1_iB_cmd; /* conversion command - measurement of phaseB current on ADC1 */
  uint32_t ADC0_iC_cmd; /* conversion command - measurement of phaseC current on ADC0 */
  uint32_t ADC1_iC_cmd; /* conversion command - measurement of phaseC current on ADC1 */
} etpu_asac_eQADC_cmds_t;

/*******************************************************************************
*                            Definitions
*******************************************************************************/
/* DTC_option options */
#define FS_ETPU_ASAC_DTC_OFF  FS_ETPU_ASAC_FM_DTC_OFF  /* Dead Time Compensation off */
#define FS_ETPU_ASAC_DTC_ON   FS_ETPU_ASAC_FM_DTC_ON   /* Dead Time Compensation on */

/* phase_current_option options */
#define FS_ETPU_ASAC_PHASE_CURR_USE_ALL        0  /* use all 3 measured currents */
#define FS_ETPU_ASAC_PHASE_CURR_USE_AB_CALC_C  4  /* calc current C from currents A and B */
#define FS_ETPU_ASAC_PHASE_CURR_USE_BC_CALC_A  3  /* calc current A from currents B and C */
#define FS_ETPU_ASAC_PHASE_CURR_USE_CA_CALC_B  5  /* calc current B from currents C and A */
#define FS_ETPU_ASAC_PHASE_CURR_USE_SECTOR     1  /* calc the 3rd current from 2 selected by sector */

/* negate_phase_currents */
#define FS_ETPU_ASAC_PHASE_CURRENTS_POS    0
#define FS_ETPU_ASAC_PHASE_CURRENTS_NEG    FS_ETPU_ASAC_OPTION_INVERT_CURRENTS

/* cfifo_update options */
#define FS_ETPU_ASAC_CFIFO_UPDATE_ON   1  /* update ADC command queue according to sector value */
#define FS_ETPU_ASAC_CFIFO_UPDATE_OFF  0  /* no operation */

/* DMA_interrupt_option options */
#define FS_ETPU_ASAC_DMA_INTR_NONE            0                                      /* DMA request and interrupt to CPU are not generated */
#define FS_ETPU_ASAC_DMA_INTR_ON_FIRST_EDGE   FS_ETPU_ASAC_OPTION_DMA_INTR_1ST_EDGE  /* DMA request and interrupt to CPU are generated on the first ASAC edge */
#define FS_ETPU_ASAC_DMA_INTR_ON_SECOND_EDGE  FS_ETPU_ASAC_OPTION_DMA_INTR_2ND_EDGE  /* DMA request and interrupt to CPU are generated on the second ASAC edge */
#define FS_ETPU_ASAC_DMA_INTR_ON_BOTH_EDGES   (FS_ETPU_ASAC_OPTION_DMA_INTR_1ST_EDGE + FS_ETPU_ASAC_OPTION_DMA_INTR_2ND_EDGE)  /* DMA request and interrupt to CPU are generated on both ASAC edges */

/* polarity options */
#define FS_ETPU_ASAC_PULSE_HIGH         0  /* Pulse high, low-high edge triggers */
#define FS_ETPU_ASAC_PULSE_LOW          1  /* Pulse low, high-low edge triggers */

/* mode options */
#define FS_ETPU_ASAC_MODE_PERIODIC      0  /* Periodic mode */
#define FS_ETPU_ASAC_MODE_SYNC          1  /* PWM synchronized mode */

/* bit_shift options */
#define FS_ETPU_ASAC_SHIFT_LEFT_BY_8    0  /* left shift by 8 */
#define FS_ETPU_ASAC_SHIFT_LEFT_BY_10   1  /* left shift by 10 */
#define FS_ETPU_ASAC_SHIFT_LEFT_BY_12   2  /* left shift by 12 */
#define FS_ETPU_ASAC_SHIFT_LEFT_BY_16   3  /* left shift by 16 */

/* measure_dc_offsets_mask options */
#define FS_ETPU_ASAC_DC_OFFSET_SAMPLE_A        1  /* assign dc_offset_a */
#define FS_ETPU_ASAC_DC_OFFSET_SAMPLE_B        2  /* assign dc_offset_b */
#define FS_ETPU_ASAC_DC_OFFSET_SAMPLE_C        4  /* assign dc_offset_c */
#define FS_ETPU_ASAC_DC_OFFSET_SAMPLE_D        8  /* assign dc_offset_d */
#define FS_ETPU_ASAC_DC_OFFSET_SAMPLE_A_B      3  /* assign dc_offset_a_b */
#define FS_ETPU_ASAC_DC_OFFSET_SAMPLE_A_B_C    7  /* assign dc_offset_a_b_c */
#define FS_ETPU_ASAC_DC_OFFSET_SAMPLE_A_B_C_D 15  /* assign dc_offset_a_b_c_d */

/* measure_samples_mask options */
#define FS_ETPU_ASAC_SAMPLE_A_B_C        1  /* process samples_a_b_c (phase currents) */
#define FS_ETPU_ASAC_SAMPLE_D            8  /* process sample_d (DC_BUS voltage) */
#define FS_ETPU_ASAC_SAMPLE_A_B_C_D      9  /* process samples_a_b_c_d */

/*******************************************************************************
*                       Function Prototypes
*******************************************************************************/
/* This routine is used to initialize the eTPU channel for the ASAC function. */
int32_t fs_etpu_asac_init( uint8_t    channel,
                           uint8_t    priority,
                           uint8_t    polarity,
                           uint8_t    mode,
                           uint8_t    measure_samples_mask,
                           uint8_t    DTC_option,
                           uint8_t    phase_current_option,
                           uint8_t    negate_phase_currents,
                           uint8_t    cfifo_update,
                           uint8_t    DMA_interrupt_option,
                           uint24_t   period,
                           uint24_t   start_offset,
                           int24_t    egde_offset,
                           uint8_t    PWMMAC_chan,
                           uint24_t   measure_time,
                           uint8_t    periods_per_outerloop,
                           uint8_t    SC_BC_outerloop_chan,
                           uint8_t    PMSMVC_ACIMVC_innerloop_chan,
                           uint32_t * result_queue,
                           uint8_t    queue_offset_a,
                           uint8_t    queue_offset_b,
                           uint8_t    queue_offset_c,
                           uint8_t    queue_offset_d,
                           fract24_t  forget_factor_a,
                           fract24_t  forget_factor_b,
                           fract24_t  forget_factor_c,
                           fract24_t  forget_factor_d,
                           uint8_t    bit_shift,
                           fract24_t  dtc_threshold,
                           uint8_t    outputA_chan,
                           uint16_t   outputA_offset,
                           uint8_t    outputB_chan,
                           uint16_t   outputB_offset,
                           uint8_t    outputC_chan,
                           uint16_t   outputC_offset,
                           uint8_t    outputD1_chan,
                           uint16_t   outputD1_offset,
                           uint8_t    outputD2_chan,
                           uint16_t   outputD2_offset,
                           uint8_t    eQADC_cmds_number,
             etpu_asac_eQADC_cmds_t * eQADC_cmds_current_measurements,
                           uint32_t * eQADC_cmds_other_measurements);

/* This function enables to set the dc_offsets by measuring. */
int32_t fs_etpu_asac_measure_dc_offsets(uint8_t channel,
                                        int8_t  measure_dc_offsets_mask);

/* This function enables to set the dc_offsets of samples A,B,C,D by values. */
int32_t fs_etpu_asac_set_dc_offsets_ABCD(uint8_t    channel,
                                         ufract24_t dc_offset_a,
                                         ufract24_t dc_offset_b,
                                         ufract24_t dc_offset_c,
                                         ufract24_t dc_offset_d);

/* This function enables to set the dc_offsets of samples A,B,C by values. */
int32_t fs_etpu_asac_set_dc_offsets_ABC(uint8_t    channel,
                                        ufract24_t dc_offset_a,
                                        ufract24_t dc_offset_b,
                                        ufract24_t dc_offset_c);

/* This function enables to set the dc_offset of sample D by value. */
int32_t fs_etpu_asac_set_dc_offset_D(uint8_t    channel,
                                     ufract24_t dc_offset_d);

/* This function gets outputA (after filtration). */
fract24_t fs_etpu_asac_get_outputA(uint8_t channel);

/* This function gets outputB (after filtration). */
fract24_t fs_etpu_asac_get_outputB(uint8_t channel);

/* This function gets outputC (after filtration). */
fract24_t fs_etpu_asac_get_outputC(uint8_t channel);

/* This function gets outputD (after filtration). */
ufract24_t fs_etpu_asac_get_outputD(uint8_t channel);

/* This function gets phase currents (outputs A,B,C after filtration). */
int32_t fs_etpu_asac_get_phase_currents(uint8_t channel,
                                        asac_abc_t * p_i_abc);

/* This function gets sampleA (prior to filtration). */
fract24_t fs_etpu_asac_get_sampleA(uint8_t channel);

/* This function gets sampleB (prior to filtration). */
fract24_t fs_etpu_asac_get_sampleB(uint8_t channel);

/* This function gets sampleC (prior to filtration). */
fract24_t fs_etpu_asac_get_sampleC(uint8_t channel);

/* This function gets sampleD (prior to filtration). */
fract24_t fs_etpu_asac_get_sampleD(uint8_t channel);

/* This function gets phase currents samples (samples A,B,C prior to filtration). */
int32_t fs_etpu_asac_get_phase_currents_samples(uint8_t channel,
                                                asac_abc_t * p_i_abc);

/* This function gets sampleA DC offset. */
ufract24_t fs_etpu_asac_get_dc_offsetA(uint8_t channel);

/* This function gets sampleB DC offset. */
ufract24_t fs_etpu_asac_get_dc_offsetB(uint8_t channel);

/* This function gets sampleC DC offset. */
ufract24_t fs_etpu_asac_get_dc_offsetC(uint8_t channel);

/* This function gets sampleD DC offset. */
ufract24_t fs_etpu_asac_get_dc_offsetD(uint8_t channel);

/* This function gets address of the eQADC command queue beginning. */
uint32_t fs_etpu_asac_get_eQADC_cmds_addr( uint8_t channel);

/*******************************************************************************
*        Information for eTPU Graphical Configuration Tool 
*******************************************************************************/
/* full function name: ASAC - Analog Sensing for AC Motors */
/* channel assignment when (mode=FS_ETPU_ASAC_MODE_PERIODIC); load: asac_periodic
   channel=channel
*/
/* channel assignment when (mode=FS_ETPU_ASAC_MODE_SYNC); load: asac_sync
   channel=channel
*/
/* load expression asac_periodic
base = 100*etpu_tcr1_freq/(period*etpu_clock_freq)

calc_first_edge = if(phase_current_option==FS_ETPU_ASAC_PHASE_CURRENTS_ON_SAMPLE_2_SELECTED_CURRENTS, 112, 54)
calc_second_edge_ABC = if(phase_current_option==FS_ETPU_ASAC_PHASE_CURRENTS_ON_SAMPLE_2_SELECTED_CURRENTS, \
                         if(DTC_option==FS_ETPU_ASAC_DTC_OFF,262,318), \
                         if(DTC_option==FS_ETPU_ASAC_DTC_OFF,248,304))
calc_second_edge_ABCD = if(phase_current_option==FS_ETPU_ASAC_PHASE_CURRENTS_ON_SAMPLE_2_SELECTED_CURRENTS, \
                         if(DTC_option==FS_ETPU_ASAC_DTC_OFF,310,364), \
                         if(DTC_option==FS_ETPU_ASAC_DTC_OFF,294,350))
channel = \
    if(measure_samples_mask==FS_ETPU_ASAC_SAMPLE_D, \
      (54+96)*base, \
    if(measure_samples_mask==FS_ETPU_ASAC_SAMPLE_A_B_C, \
      (calc_first_edge+calc_second_edge_ABC)*base, \
      (calc_first_edge+calc_second_edge_ABCD)*base))
*/
/* load expression asac_sync
var update_frequency {1..100000} [Hz]
base = 100*update_frequency/etpu_clock_freq

calc_first_edge = if(phase_current_option==FS_ETPU_ASAC_PHASE_CURRENTS_ON_SAMPLE_2_SELECTED_CURRENTS, 112, 54)
calc_second_edge_ABC = if(phase_current_option==FS_ETPU_ASAC_PHASE_CURRENTS_ON_SAMPLE_2_SELECTED_CURRENTS, \
                         if(DTC_option==FS_ETPU_ASAC_DTC_OFF,262,318), \
                         if(DTC_option==FS_ETPU_ASAC_DTC_OFF,248,304))
calc_second_edge_ABCD = if(phase_current_option==FS_ETPU_ASAC_PHASE_CURRENTS_ON_SAMPLE_2_SELECTED_CURRENTS, \
                         if(DTC_option==FS_ETPU_ASAC_DTC_OFF,310,364), \
                         if(DTC_option==FS_ETPU_ASAC_DTC_OFF,294,350))
channel = \
    if(measure_samples_mask==FS_ETPU_ASAC_SAMPLE_D, \
      (54+96)*base, \
    if(measure_samples_mask==FS_ETPU_ASAC_SAMPLE_A_B_C, \
      (calc_first_edge+calc_second_edge_ABC)*base, \
      (calc_first_edge+calc_second_edge_ABCD)*base))
*/


#endif

/*********************************************************************
 *
 * Copyright:
 *	Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ********************************************************************/
