/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2010 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
********************************************************************************
*
* File Name:  iic.h
*
* $Date:      Oct-12-2010$
*
* $Version:   2.5.7.0$
*
* Description: Header file for the I2C driver
*
* This file only supports two hardware versions of the module (IIC_VERSION_3 and IIC_VERSION_4). 
* A version  of module used in your project is defined in the "arch.h" 
* processor architecture definition file.
*
*  IIC_VERSION_1 .. 56F801x devices, see MC56F8000RM.pdf
*  IIC_VERSION_2 .. 56F802x/3x devices, see MC56F802x3xRM.pdf
*  IIC_VERSION_3 .. 56F800x devices, see MC56F8006RM.pdf
*  IIC_VERSION_4 .. 56F824x/5x devices, see MC56F82xxRM.pdf
*
*******************************************************************************/

#ifndef __IIC_H
#define __IIC_H

/* qs.h is a master header file, which must be included */
#if !defined(__ARCH_H) || !defined(__PERIPH_H) || !defined(__APPCONFIG_H)
#error Please include qs.h before iic.h
#endif

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
*
*                      General Interface Description
*
*
* The Inter-IC Bus (IIC or I2C) is a two-wire, bidirectional serial bus that provides 
* a simple, efficient method of data exchange between devices. Being a two-wire device, 
* the IIC Bus minimizes the need for large numbers of connections between devices, 
* and eliminates the need for an address decoder.
* 
* This bus is suitable for applications requiring occasional communications over a 
* short distance between a number of devices. It also provides flexibility, allowing 
* additional devices to be connected to the bus for further expansion and system 
* development.
*
* For more information, please refer to the user manual for the particular chip that you
* are using, such as the MC56F80xx Peripheral Manual, for a description
* of the I2C device.
*
* The I2C device driver interface uses "ioctl" call to control specific I2C functions
* during operation. For details see "ioctl" call.
*
******************************************************************************/

/************************************************************
* I2C module identifier
*************************************************************/

#ifdef IIC_BASE
#define IIC (&ArchIO.Iic)
#define I2C IIC
#endif

#ifdef IIC0_BASE
#define IIC_0   (&ArchIO.Iic0)
#endif

#ifdef IIC1_BASE
#define IIC_1   (&ArchIO.Iic1)
#endif


/*************************************************************
* arch.h should define device-specific information about IIC
**************************************************************/

#if !defined(IIC_VERSION)
#error IIC not properly described in arch.h (old Quick_Start version?)
#endif

#if !(defined(IIC_VERSION_3) || defined(IIC_VERSION_4))
#error This file implements IIC version 3 and 4 only (56F800x and 56F82xx devices or later)
#endif


/****************************************************
* Configurable items, i.e. defines for appconfig.h
*****************************************************

  void i2cISR( void );           prototype for user ISR 
  #define INT_VECTOR_ADDR_23     interrupt vector (e.g. i2cISR)

  #define IIC_IICF_INIT 0x0000
  #define IIC_IICA1_INIT 0x0000
  #define IIC_IICA2_INIT 0x0000
  #define IIC_IICC1_INIT 0x0000
  #define IIC_IICC2_INIT 0x0000
  #define IIC_IICSMB_INIT 0x0000
  #define IIC_IICSLTH_INIT 0x0000
  #define IIC_IICSLTL_INIT 0x0000

NEW IN VERSION 4
  #define IIC_FILT_INIT 0x0000

*/

/*****************************************************************
* Single-Instruction ioctl() commands
*
*  Word16 ioctl( IIC_module_identifier, Command_name, Parameter );
*
******************************************************************

 COMMAND NAME                         PARAMETERS, COMMENTS
----------------------------------------------------------------*/
#define IIC_SET_ADDRESS                 /* UWord16, Write to the IIC Bus Address Register (0-127) */
#define IIC_SET_PRESCALER               /* UWord16, Write to the IIC Bus Frequency Divider Register */
#define IIC_I_BUS                       /* IIC_ENABLE / IIC_DIABLE, Enable/Disable IIC module */
#define IIC_I_BUS_INT                   /* IIC_ENABLE / IIC_DIABLE, Enable/Disable IIC Interrupt */
#define IIC_MASTER_SLAVE_MODE           /* IIC_MASTER / IIC_SLAVE, Set IIC Master or Slave */
#define IIC_TX_RX_MODE                  /* IIC_TRANSMIT / IIC_RECEIVE, Select the direction of data transfers */
#define IIC_TX_ACK                      /* IIC_NO_ACK / IIC_ACK, Enable/Disable Transmiting of Acknowledge */
#define IIC_REPEAT_START                /* NULL, Generate a repeated START condition on the bus */
#define IIC_WRITE_CONTROL_REG           /* UWord16, Write to IIC-Bus Control Register  */
#define IIC_READ_CONTROL_REG            /* NULL, Read the IIC-Bus Control Register as UWord16 */
#define IIC_READ_DATA                   /* NULL, Read IIC Bus Data I/O Register as UWord16 */
#define IIC_WRITE_DATA                  /* UWord16, Write IIC Bus Data I/O Register  */

#define IIC_GET_MASTER_MODE             /* NULL, return non-zero if IIC Master mode is set */
#define IIC_GET_TX_MODE                 /* NULL, return non-zero if transmit mode is set */

#define IIC_READ_STATUS_REG             /* NULL, Read IIC Bus Status Register as UWord16 */
#define IIC_TEST_STATUS_REG             /* IIC_TRANSFER_COMPLETE | IIC_ADDRESSED_AS_SLAVE | IIC_BUS_BUSY | 
                                           IIC_ARBITRATION_LOST | IIC_SLAVE_TRANSMIT | IIC_IBUS_INT | IIC_RX_ACK 
                                           Test IIC Bus Status Register for selected bits */

#define IIC_GET_TRANSFER_COMPLETE       /* NULL, Get Data transferring status */
#define IIC_GET_ADDRESSED_AS_SLAVE      /* NULL, Get the Addressed as a slave flag */
#define IIC_GET_BUS_BUSY                /* NULL, Get the Bus busy status */
#define IIC_CLEAR_ARBITRATION_LOST      /* NULL, Clear Arbitration Lost */
#define IIC_GET_ARBITRATION_LOST        /* NULL, Get Arbitration Lost status */
#define IIC_GET_SLAVE_TRANSMIT          /* NULL, Get Slave Read/Write status */
#define IIC_CLEAR_I_BUS_INT             /* NULL, Clear I-Bus Interrupt flag */
#define IIC_GET_I_BUS_INT               /* NULL, Get I-Bus Interrupt flag */
#define IIC_GET_RX_ACK                  /* NULL, Get Received Acknowledge flag, */

#if IIC_VERSION >=3
/* IICC2 */
#define IIC_SET_GENERAL_CALL_ADDRESS     /* IIC_ENABLE / IIC_DIABLE, enables or disables general call address (IICC2->GCAEN) */
#define IIC_SET_ADDRESS_EXTENSION_MODE  /* IIC_7BIT / IIC_10BIT, sets the number of bits used for the slave address (IICC2->ADEXT) */
/* IICSMB */
#define IIC_CLEAR_LOW_TIMEOUT_FLAG      /* NULL, Clears low timeout flags (IICSMB->SLTF) */
/* IICA2 */
#define IIC_SET_SMBUS_ADDRESS           /* UWord16, Writes to the Secondary IIC Bus Address Register (0-127) */
#endif

#if IIC_VERSION == 3
#define IIC_TEST_TIMEOUT_FLAGS          /* IIC_LOW_TIMEOUT | IIC_HIGH_TIMEOUT, tests state of timeout flags (IICSMB->SLTF,SHTF) */
#endif

#if IIC_VERSION >=4
#define IIC_SET_WAKEUP_IN_STOP          /* IIC_ENABLE / IIC_DIABLE, Enable/Disable wake up function in stop mode (CR1->WUEN) */
#define IIC_TEST_TIMEOUT_FLAGS          /* IIC_LOW_TIMEOUT | IIC_HIGH_TIMEOUT | IIC_HIGH_TIMEOUT2, tests state of timeout flags (IICSMB->SLTF,SHTF,SHTF2) */
#define IIC_CLEAR_HIGH_TIMEOUT_FLAG2    /* NULL, Clears High Timeout flag 2 (IICSMB->SHFT2) */
#define IIC_CLEAR_TIMEOUT_FLAGS         /* IIC_LOW_TIMEOUT | IIC_HIGH_TIMEOUT2, Clears selected timeout flags (IICSMB->SLRF,SHFT2) */
#define IIC_SET_GLITCH_FILTER           /* UWord16, value (0-7), disables glitch filter, when 0 is paramterer or set glitch filter when 1 to 7 is parameter (FILT->FLT) */
#endif


/*****************************************************************
* Multi-Instruction ioctl() commands
*
*  Word16 ioctl( IIC_module_identifier, Command_name, Parameter );
*
******************************************************************

 COMMAND NAME                         PARAMETERS, COMMENTS
----------------------------------------------------------------*/
#define IIC_GET_ADDRESS                 /* NULL, Read IIC Address (0-127) */
#if IIC_VERSION >=3
/* IICC2 */
#define IIC_SET_ADDRESS_EXTENSION       /* UWord16,High part of IIC address (0-7) - AD[8-10],  (IICC2->AD[10-8]) */
#define IIC_SET_10BIT_ADDRESS           /* UWord16, Write to the IIC Bus Address Register (0-1023) (IICC2->AD[10-8],(IICA1->AD[7-1]) */
#define IIC_GET_10BIT_ADDRESS           /* NULL, Reads IIC Address (0-1023) (IICC2->AD[10-8],(IICA1->AD[7-1]) */
/* IICSMB */
#define IIC_SET_FAST_ACK_NACK           /* IIC_ENABLE / IIC_DIABLE, enables or disables fast set up ACK/NACK response (IICSMB->FACK) */
#define IIC_SET_SMBUS_RESPONSE_ADDRESS  /* IIC_ENABLE / IIC_DIABLE, enables or disables SMBUS alert response address (IICSMB->ALERTEN) */
#define IIC_SET_SECOND_IIC_ADDRESS      /* IIC_ENABLE / IIC_DIABLE, enables or disables SMBus device default address (IICSMB->SIICAEN) */
#define IIC_SET_TIME_OUT_CLOCK          /* IIC_DIV64 / IIC_DIV1, selects the clock sources of Time Out Counter  (IICSMB->TCKSEL) */
/* IICA2 */
#define IIC_GET_SMBUS_ADDRESS           /* NULL, Reads Secondary IIC Address (0-127) */

#define IIC_WRITE_SCL_LOW_TIMEOUT       /* UWord16, Writes to the SCL LowTime Out Register */
#define IIC_READ_SCL_LOW_TIMEOUT        /* NULL, Reads the SCL LowTime Out Register */
#endif

#if IIC_VERSION >=4
#define IIC_SET_HIGH_TIMEOUT2_INTERRUPT /* IIC_ENABLE / IIC_DIABLE, Enable/Disable high timeout 2 interrupt (IICSMB->SHTF2IE) */
#endif


/*****************************************************************
* Software Layer ioctl() commands
*
*  Word16 ioctl( IIC_module_identifier, Command_name, Parameter );
*
******************************************************************

 COMMAND NAME                         PARAMETERS, COMMENTS
----------------------------------------------------------------*/
#define IIC_INIT                        /* NULL */


/***********************************************************************************************************
** IIC PARAMETERS DEFINITION                                                                        **
***********************************************************************************************************/

#define IIC_ON                      1
#define IIC_OFF                     0
#define IIC_ENABLE                  1
#define IIC_DISABLE                 0
#define IIC_MASTER                  1
#define IIC_SLAVE                   0
#define IIC_TRANSMIT                1
#define IIC_RECEIVE                 0
#define IIC_NO_ACK                  1
#define IIC_ACK                     0
#define IIC_TRANSFER_COMPLETE       IIC_IBSR_TCF
#define IIC_ADDRESSED_AS_SLAVE      IIC_IBSR_IAAS
#define IIC_BUS_BUSY                IIC_IBSR_IBB
#define IIC_ARBITRATION_LOST        IIC_IBSR_IBAL
#define IIC_SLAVE_TRANSMIT          IIC_IBSR_SRW
#define IIC_IBUS_INT                IIC_IBSR_IBIF
#define IIC_RX_ACK                  IIC_IBSR_RXAK

#define IIC_10BIT                   1
#define IIC_7BIT                    0
#define IIC_DIV1                    1
#define IIC_DIV64                   0

/* IICSMB register */
#define IIC_LOW_TIMEOUT             IIC_IICSMB_SLTF
#define IIC_HIGH_TIMEOUT            IIC_IICSMB_SHTF

#if IIC_VERSION >=4
#define IIC_HIGH_TIMEOUT2           IIC_IICSMB_SHTF2

#endif

/****************************************************************************
* register and bit names used in the ioctl functions below
*****************************************************************************/

/* IBAD register */
#define IIC_IBAD_MASK               0xFE 
#define IIC_IBAD_ADR6               0x0080
#define IIC_IBAD_ADR5               0x0040
#define IIC_IBAD_ADR4               0x0020
#define IIC_IBAD_ADR3               0x0010
#define IIC_IBAD_ADR2               0x0008
#define IIC_IBAD_ADR1               0x0004
#define IIC_IBAD_ADR0               0x0002
#define IIC_IBAD_ADR                0x00FE

/* IBFD register */
#define IIC_IBFD_MASK               0xFF 
#define IIC_IBFD_IBC7               0x0080
#define IIC_IBFD_IBC6               0x0040
#define IIC_IBFD_IBC5               0x0020
#define IIC_IBFD_IBC4               0x0010
#define IIC_IBFD_IBC3               0x0008
#define IIC_IBFD_IBC2               0x0004
#define IIC_IBFD_IBC1               0x0002
#define IIC_IBFD_IBC0               0x0001
#define IIC_IBFD_IBC                0x00FF

/* IBCR register */
#define IIC_IBCR_MASK               0xFC 
#define IIC_IBCR_IBEN               0x0080
#define IIC_IBCR_IBIE               0x0040
#define IIC_IBCR_MS_SL              0x0020
#define IIC_IBCR_TX_RX              0x0010
#define IIC_IBCR_TXAK               0x0008
#define IIC_IBCR_RSTA               0x0004

#if IIC_VERSION >=4
#define IIC_IBCR_WUEN               0x0002
#endif

/* IBSR register */
#define IIC_IBSR_MASK               0xF7 
#define IIC_IBSR_TCF                0x0080
#define IIC_IBSR_IAAS               0x0040
#define IIC_IBSR_IBB                0x0020
#define IIC_IBSR_IBAL               0x0010
#define IIC_IBSR_SRW                0x0004
#define IIC_IBSR_IBIF               0x0002
#define IIC_IBSR_RXAK               0x0001

/* IBDR register */
#define IIC_IBDR_MASK               0xFF 
#define IIC_IBDR_D7                 0x0080
#define IIC_IBDR_D6                 0x0040
#define IIC_IBDR_D5                 0x0020
#define IIC_IBDR_D4                 0x0010
#define IIC_IBDR_D3                 0x0008
#define IIC_IBDR_D2                 0x0004
#define IIC_IBDR_D1                 0x0002
#define IIC_IBDR_D0                 0x0001
#define IIC_IBDR_D                  0x00FF

/* IICC2 register */
#define IIC_IICC2_GCAEN             0x0080
#define IIC_IICC2_ADEXT             0x0040
#define IIC_IICC2_ADR_MASK          0x0007

/* IICSMB register */
#define IIC_IICSMB_FACK             0x0080
#define IIC_IICSMB_ALERTEN          0x0040
#define IIC_IICSMB_SIICAEN          0x0020
#define IIC_IICSMB_TCKSEL           0x0010
#define IIC_IICSMB_SLTF             0x0008
#define IIC_IICSMB_SHTF             0x0004

#if IIC_VERSION >=4
#define IIC_IICSMB_SHTF2            0x0002
#define IIC_IICSMB_SHTF2IE          0x0001

/* FILT register */
#define IIC_FILT_FLT_MASK           0x0007
#endif


/****************************************************************************
* I2C ioctl macro implementation
*****************************************************************************/

/***********************************************************************
* I2C init
***********************************************************************/

void iicInit(arch_sIIC *pI2cBase);
#define ioctlIIC_INIT(pIicBase, bParam) iicInit(pIicBase)

/***********************************************************************
* I2C control register
***********************************************************************/

/* Enable/Disable IIC module */
#define ioctlIIC_I_BUS(pIicBase, param) \
  if (param) periphBitSet(IIC_IBCR_IBEN, &((pIicBase)->ctrl1)); \
  else periphBitClear(IIC_IBCR_IBEN, &((pIicBase)->ctrl1))

/* Enable/Disable IIC Interrupt */
#define ioctlIIC_I_BUS_INT(pIicBase, param) \
  if (param) periphBitSet(IIC_IBCR_IBIE, &((pIicBase)->ctrl1)); \
  else periphBitClear(IIC_IBCR_IBIE, &((pIicBase)->ctrl1))

/* Set IIC Master or Slave */
#define ioctlIIC_MASTER_SLAVE_MODE(pIicBase, param) \
  if (param) periphBitSet(IIC_IBCR_MS_SL, &((pIicBase)->ctrl1)); \
  else periphBitClear(IIC_IBCR_MS_SL, &((pIicBase)->ctrl1))

/* Select the direction of data transfers */
#define ioctlIIC_TX_RX_MODE(pIicBase, param) \
  if (param) periphBitSet(IIC_IBCR_TX_RX, &((pIicBase)->ctrl1)); \
  else periphBitClear(IIC_IBCR_TX_RX, &((pIicBase)->ctrl1))

/* Enable/Disable Transmiting of Acknowledge */
#define ioctlIIC_TX_ACK(pIicBase, param) \
  if (param) periphBitSet(IIC_IBCR_TXAK, &((pIicBase)->ctrl1)); \
  else periphBitClear(IIC_IBCR_TXAK, &((pIicBase)->ctrl1))

/* Generate a repeated START condition on the bus */
#define ioctlIIC_REPEAT_START(pIicBase, param) \
  periphBitSet(IIC_IBCR_RSTA, &((pIicBase)->ctrl1))

/* Write to IIC-Bus Control Register  */
#define ioctlIIC_WRITE_CONTROL_REG(pIicBase, param) \
  periphMemWrite(param, &((pIicBase)->ctrl1))

/* Read the IIC-Bus Control Register as UWord16 */
#define ioctlIIC_READ_CONTROL_REG(pIicBase, param)  \
  periphMemRead(&((pIicBase)->ctrl1))

#define ioctlIIC_GET_MASTER_MODE(pIicBase, param)  \
  periphBitTest(IIC_IBCR_MS_SL, &((pIicBase)->ctrl1))

#define ioctlIIC_GET_TX_MODE(pIicBase, param)  \
  periphBitTest(IIC_IBCR_TX_RX, &((pIicBase)->ctrl1))

/***********************************************************************
* I2C address
***********************************************************************/

/* UWord16, Write to the IIC Bus Address Register (0-127) */
#define ioctlIIC_SET_ADDRESS(pIicBase, param) \
  periphMemWrite((param) << 1, &((pIicBase)->addr1))

/* NULL, read IIC Address (0-127) */
#define ioctlIIC_GET_ADDRESS(pIicBase, param) \
  (periphMemRead(&((pIicBase)->addr1)) >> 1)

/***********************************************************************
* I2C baudrate register
***********************************************************************/

/* UWord16, Write to the IIC Bus Frequency Divider Register */
#define ioctlIIC_SET_PRESCALER(pIicBase, param) \
  periphMemWrite(param, &((pIicBase)->fdiv))

/***********************************************************************
* I2C status register
***********************************************************************/

/* Read IIC Bus Status Register as UWord16 */
#define ioctlIIC_READ_STATUS_REG(pIicBase, param)  \
  periphMemRead(&((pIicBase)->stat))

/* Test IIC Bus Status Register for selected bits */
#define ioctlIIC_TEST_STATUS_REG(pIicBase, param) \
  periphBitTest(param, &((pIicBase)->stat))

/* Get Data transferring status */
#define ioctlIIC_GET_TRANSFER_COMPLETE(pIicBase, param) \
  periphBitTest(IIC_IBSR_TCF, &((pIicBase)->stat))

/* Get the Addressed as a slave flag */
#define ioctlIIC_GET_ADDRESSED_AS_SLAVE(pIicBase, param) \
  periphBitTest(IIC_IBSR_IAAS, &((pIicBase)->stat))

/* Get the Bus busy status */
#define ioctlIIC_GET_BUS_BUSY(pIicBase, param) \
  periphBitTest(IIC_IBSR_IBB, &((pIicBase)->stat))

/* Clear Arbitration Lost */
#define ioctlIIC_CLEAR_ARBITRATION_LOST(pIicBase, param) \
    periphMemWrite(IIC_IBSR_IBAL, &((pIicBase)->stat))

/* Get Arbitration Lost status */
#define ioctlIIC_GET_ARBITRATION_LOST(pIicBase, param) \
  periphBitTest(IIC_IBSR_IBAL, &((pIicBase)->stat)) 

/* Get Slave Read/Write status */
#define ioctlIIC_GET_SLAVE_TRANSMIT(pIicBase, param) \
  periphBitTest(IIC_IBSR_SRW, &((pIicBase)->stat)) 

/* Clear I-Bus Interrupt flag */
#define ioctlIIC_CLEAR_I_BUS_INT(pIicBase, param) \
    periphMemWrite(IIC_IBSR_IBIF, &((pIicBase)->stat))

/* Get I-Bus Interrupt flag */
#define ioctlIIC_GET_I_BUS_INT(pIicBase, param) \
  periphBitTest(IIC_IBSR_IBIF, &((pIicBase)->stat))

/* Get Received Acknowledge flag */
#define ioctlIIC_GET_RX_ACK(pIicBase, param) \
  periphBitTest(IIC_IBSR_RXAK, &((pIicBase)->stat))

/***********************************************************************
* I2C data register
***********************************************************************/

/* Read IIC Bus Data I/O Register as UWord16, never optimize out */
#define ioctlIIC_READ_DATA(pIicBase, param)  \
  periphMemForcedRead(&((pIicBase)->data))

/* Write IIC Bus Data I/O Register  */
#define ioctlIIC_WRITE_DATA(pIicBase, param) \
  periphMemWrite(param, &((pIicBase)->data))

/***********************************************************************
* I2C Control register 2
***********************************************************************/

/* IICC2 */
/* Enable/Disable general call address */
#define ioctlIIC_SET_GENERAL_CALL_ADDRESS(pIicBase, param) \
  if (param) periphBitSet(IIC_IICC2_GCAEN, &((pIicBase)->iicc2)); \
  else periphBitClear(IIC_IICC2_GCAEN, &((pIicBase)->iicc2))

/* Set the number of bits used for the slave address. */
#define ioctlIIC_SET_ADDRESS_EXTENSION_MODE(pIicBase, param) \
  if (param) periphBitSet(IIC_IICC2_ADEXT, &((pIicBase)->iicc2)); \
  else periphBitClear(IIC_IICC2_ADEXT, &((pIicBase)->iicc2))

/* Set upper 3 bits in 10bit adress mode */
#define ioctlIIC_SET_ADDRESS_EXTENSION(pIicBase, param) \
  periphBitGrpSet(IIC_IICC2_ADR_MASK, param, &((pIicBase)->iicc2))

/* Set slave address in 10bit adress mode (0-1023) */
#define ioctlIIC_SET_10BIT_ADDRESS(pIicBase, param) \
  periphBitGrpSet(IIC_IICC2_ADR_MASK, (param) >> 7, &((pIicBase)->iicc2)) \
  periphMemWrite((param) << 1, &((pIicBase)->addr1))


/* NULL, read IIC Address (0-1023) */
#define ioctlIIC_GET_10BIT_ADDRESS(pIicBase, param) \
  ( (periphMemRead(&((pIicBase)->addr1)) >> 1) | \
  ( (periphMemRead(&((pIicBase)->iicc2))&IIC_IICC2_ADR_MASK) << 7 ) )


/* IICSMB register */
#if IIC_VERSION <=3
/* Enable/Disable sending ACK/NACK automatically */
#define ioctlIIC_SET_FAST_ACK_NACK(pIicBase, param) \
  if (param) {periphSafeBitSet(IIC_IICSMB_SLTF, IIC_IICSMB_FACK, &((pIicBase)->iicsmb));} \
  else periphSafeBitClear(IIC_IICSMB_SLTF, IIC_IICSMB_FACK, &((pIicBase)->iicsmb))

/* Enable/Disable SMBus alert response address */
#define ioctlIIC_SET_SMBUS_RESPONSE_ADDRESS(pIicBase, param) \
  if (param) {periphSafeBitSet(IIC_IICSMB_SLTF, IIC_IICSMB_ALERTEN, &((pIicBase)->iicsmb));} \
  else periphSafeBitClear(IIC_IICSMB_SLTF, IIC_IICSMB_ALERTEN, &((pIicBase)->iicsmb))

/* Enable/Disable SMBus device default address */
#define ioctlIIC_SET_SECOND_IIC_ADDRESS(pIicBase, param) \
  if (param) {periphSafeBitSet(IIC_IICSMB_SLTF, IIC_IICSMB_SIICAEN, &((pIicBase)->iicsmb));} \
  else periphSafeBitClear(IIC_IICSMB_SLTF, IIC_IICSMB_SIICAEN, &((pIicBase)->iicsmb))

/* Enable/Disable SMBus device default address */
#define ioctlIIC_SET_TIME_OUT_CLOCK(pIicBase, param) \
  if (param) {periphSafeBitSet(IIC_IICSMB_SLTF, IIC_IICSMB_TCKSEL, &((pIicBase)->iicsmb));} \
  else periphSafeBitClear(IIC_IICSMB_SLTF, IIC_IICSMB_TCKSEL, &((pIicBase)->iicsmb))

/* clear low timeout flags */
#define ioctlIIC_CLEAR_LOW_TIMEOUT_FLAG(pIicBase, param) \
  periphBitSet(IIC_IICSMB_SLTF, &((pIicBase)->iicsmb)) 

#endif /* IIC_VERSION <=3 */

/* test state of timeout flags */
#define ioctlIIC_TEST_TIMEOUT_FLAGS(pIicBase, param) \
  periphBitTest(param, &((pIicBase)->iicsmb)) 

/* UWord16, Write to the SMBUS IIC Bus Address Register (0-127) */
#define ioctlIIC_SET_SMBUS_ADDRESS(pIicBase, param) \
  periphMemWrite((param) << 1, &((pIicBase)->iica2))

/* NULL, read SMBUS IIC Address (0-127) */
#define ioctlIIC_GET_SMBUS_ADDRESS(pIicBase, param) \
  (periphMemRead(&((pIicBase)->iica2)) >> 1)

/* Write to the SCL LowTime Out Register */
#define ioctlIIC_WRITE_SCL_LOW_TIMEOUT(pIicBase, param) \
  periphMemWrite( (param) >> 8, &((pIicBase)->iicslth)); \
  periphMemWrite(param, &((pIicBase)->iicsltl))

/* NULL, Read the SCL LowTime Out Register */
#define ioctlIIC_READ_SCL_LOW_TIMEOUT(pIicBase, param) \
  ( periphMemRead(&((pIicBase)->iicsltl)) | \
  (periphMemRead(&((pIicBase)->iicslth)) << 8) )


#if IIC_VERSION >=4
/* correct old commands*/
/* Enable/Disable sending ACK/NACK automatically */
#define ioctlIIC_SET_FAST_ACK_NACK(pIicBase, param) \
  if (param) {periphSafeBitSet(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_FACK, &((pIicBase)->iicsmb));} \
  else periphSafeBitClear(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_FACK, &((pIicBase)->iicsmb))

/* Enable/Disable SMBus alert response address */
#define ioctlIIC_SET_SMBUS_RESPONSE_ADDRESS(pIicBase, param) \
  if (param) {periphSafeBitSet(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_ALERTEN, &((pIicBase)->iicsmb));} \
  else periphSafeBitClear(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_ALERTEN, &((pIicBase)->iicsmb))

/* Enable/Disable SMBus device default address */
#define ioctlIIC_SET_SECOND_IIC_ADDRESS(pIicBase, param) \
  if (param) {periphSafeBitSet(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_SIICAEN, &((pIicBase)->iicsmb));} \
  else periphSafeBitClear(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_SIICAEN, &((pIicBase)->iicsmb))

/* Enable/Disable SMBus device default address */
#define ioctlIIC_SET_TIME_OUT_CLOCK(pIicBase, param) \
  if (param) {periphSafeBitSet(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_TCKSEL, &((pIicBase)->iicsmb));} \
  else periphSafeBitClear(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_TCKSEL, &((pIicBase)->iicsmb))

/* clear low timeout flags */
#define ioctlIIC_CLEAR_LOW_TIMEOUT_FLAG(pIicBase, param) \
  periphBitClear(IIC_IICSMB_SHTF2, &((pIicBase)->iicsmb)) 
  /*instruction bfclear reads, masks and returns data, if IIC_IICSMB_SLTF flag is set, returns back 1 and clears the flag, 
  flag IIC_IICSMB_SHTF2 is not cleared, because instruction sets this bit to 0 and returns 0.*/   


/* new commands in version 4*/
/* Enables/Disables wakeup function in stop mode */
#define ioctlIIC_SET_WAKEUP_IN_STOP(pIicBase, param) \
  if (param) periphBitSet(IIC_IBCR_WUEN, &((pIicBase)->ctrl1)); \
  else periphBitClear(IIC_IBCR_WUEN, &((pIicBase)->ctrl1))

/* clears high timeout flag 2 */
#define ioctlIIC_CLEAR_HIGH_TIMEOUT_FLAG2(pIicBase, param) \
  periphBitClear(IIC_IICSMB_SLTF, &((pIicBase)->iicsmb)) 
  /*instruction bfclear reads, masks and returns data, if IIC_IICSMB_SHTF2 flag is set, returns back 1 and clears the flag, 
  flag IIC_IICSMB_SLTF is not cleared, because instruction sets this bit to 0 and returns 0.*/   

/* clears selected timeout flags  */
#define ioctlIIC_CLEAR_TIMEOUT_FLAGS(pIicBase, param) \
  periphSafeAckByOne(IIC_IICSMB_SLTF|IIC_IICSMB_SHTF2, (param), &((pIicBase)->iicsmb)) 

/* Enable/Disable High Timeout 2 Interrupt */
#define ioctlIIC_SET_HIGH_TIMEOUT2_INTERRUPT(pIicBase, param) \
  if (param) {periphSafeBitSet(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_SHTF2IE, &((pIicBase)->iicsmb));} \
  else periphSafeBitClear(IIC_IICSMB_SLTF | IIC_IICSMB_SHTF2, IIC_IICSMB_SHTF2IE, &((pIicBase)->iicsmb))

/* UWord16, Writes to the glitch filter register  (0-7) */
#define ioctlIIC_SET_GLITCH_FILTER(pIicBase, param) \
  periphMemWrite( (param), &((pIicBase)->iicfilt))


#endif /* IIC_VERSION >=4 */

#ifdef __cplusplus
}
#endif

#endif                                      
