/*
 * File:		uart_lon_tx.c
 * Purpose:		LON mode basic transmit.
 *
 */

#include "common.h"
#include "uart.h"

/********************************************************************/
/* UART module LON mode basic transmit.
 *
 * Tests basic LON mode transmit operation for a single UART. The 
 * function will transmit specific a byte of data using CEA709.1
 *
 */
int 
uart_lon_basic_tx(UART_MemMapPtr module, uint8 ch)
{
    /* Configure the UART for LON mode */
    UART_C6_REG(module) = UART_C6_EN709_MASK;
    
    /* Test 4-bit preamble. The minimum legal value */
    UART_PRE_REG(module) = 0x4;
    
    /*Number of data bytes per packet*/
    UART_TPL_REG(module) = 0x01;    
    
    /* Set beta1 timer to 1 as 0 seems to max out delay */
    UART_B1T_REG(module) = 1;

    /*Enable xmit portion*/
    UART_C6_REG(module) |= UART_C6_TX709_MASK;
    
    /*send character*/
    uart_putchar (module, ch); 
    
    /* Wait for transmit to complete */
    while( !(UART_S3_REG(module) & UART_S3_PTXF_MASK))
    {;}

    /* Disable transmit to prevent a second packet from attempting to go out */
    UART_C6_REG(module) &= ~UART_C6_TX709_MASK;
    
    /* Clear the PTXF flag */
    UART_S3_REG(module) = UART_S3_PTXF_MASK;

    /*disable CEA709.1 controller*/
    UART_C6_REG(module) = 0;   

    return 0;    
} 

/********************************************************************/
/* UART module LON mode transmit packet multiple bytes.
 *
 * LON mode transmit packet length for a single UART.
 *
 */
int 
uart_lon_tx_array(UART_MemMapPtr module, uint8 *data, uint8 tx_len)
{
    uint8 i;
    
    /* Configure the UART for LON mode */
    UART_C6_REG(module) = UART_C6_EN709_MASK;
    
    /* Set the preamble length */
    UART_PRE_REG(module) = 0x4;
    
    /* Set beta1 timer to 1 as 0 seems to max out delay */
    UART_B1T_REG(module) = 1;

    /* In order to avoid possible underflows, enable the TxFIFO */
    UART_PFIFO_REG(module) |= UART_PFIFO_TXFE_MASK;
    
    /* Set the watermark to 4 */
    UART_TWFIFO_REG(module) = 4;   
    
    /* max packet length (255) */

    /* Set transmit packet length */
    UART_TPL_REG(module) = tx_len;    
    
    UART_C6_REG(module) |= UART_C6_TX709_MASK;

    /* Transmit bytes up to the packet length being tested */
    for(i=0; i < tx_len; i++)
      uart_putchar(module, data[i]);
                
    /* Wait for transmit to complete */
    while( !(UART_S3_REG(module) & UART_S3_PTXF_MASK))
    {;}

    /* Disable transmit to prevent a second packet from attempting to go out */
    UART_C6_REG(module) &= ~UART_C6_TX709_MASK;
    
    /* Clear the PTXF flag */
    UART_S3_REG(module) = UART_S3_PTXF_MASK;

    /* All tests complete. Return registers to default values */
    UART_C6_REG(module) = 0;      
        
    return 0;    
} 
/********************************************************************/   
