/*
 * File:		uart_lon_rx.c
 * Purpose:		LON mode basic receive.
 *
 */

#include "common.h"
#include "uart.h"

/********************************************************************/
/* UART module LON mode basic receive.
 *
 * Basic LON mode receive operation for a single UART.
 */
int 
uart_lon_basic_rx(UART_MemMapPtr module, uint8 *ch)
{           
    /* Disable the receiver on the UART that is being used for Rx to avoid
     * any issues with the beginning signal state while wires are being
     * connected.
     */
    UART_C2_REG(module) &= ~UART_C2_RE_MASK;
    
    /* Configure the UART for LON mode */
    UART_C6_REG(module) = UART_C6_EN709_MASK;
    
    /* Test 4-bit preamble. The minimum legal value */
    UART_PRE_REG(module) = 0x4;
    
    /*Number of data bytes per packet*/
    UART_TPL_REG(module) = 0x01;    
    
    /* Re-enable the receiver for the test */
    UART_C2_REG(module) |= UART_C2_RE_MASK;
    
    /* Wait for a receive packet to be detected */
    while( !(UART_S3_REG(module) & UART_S3_PRXF_MASK))
    {;}
                
    /* Get the received data */
    *ch = UART_D_REG(module);

    /* Clear the packet received flag */
    UART_S3_REG(module) = UART_S3_PRXF_MASK;
#if 0
    /* Test the receive packet length */
    if (UART_RPL_REG(module) != 0x1)
    {
      error++;      
      printf("ERR! Incorrect receive packet length. rx_pkt_len =  0x%02X\n", UART_RPL_REG(module));
    }
#endif

    UART_C6_REG(module) = 0;   
        
    return 0;    
} 

/********************************************************************/   
/* UART module LON mode receive packet length.
 *
 * Tests LON mode receive packet length for a single UART.
 *
 */
int 
uart_lon_rx_array(UART_MemMapPtr module, uint8 *data, uint8 rx_len)
{   
    uint8 i;
    
    /* Disable the receiver on the UART that is being used for Rx to avoid
     * any issues with the beginning signal state while wires are being
     * connected.
     */
    UART_C2_REG(module) &= ~UART_C2_RE_MASK;    
    
    /* Configure the UART for LON mode */
    UART_C6_REG(module) = UART_C6_EN709_MASK;
    
    /* Test 4-bit preamble. The minimum legal value */
    UART_PRE_REG(module) = 0x4;
    
    /* In order to avoid possible overflows, enable the RxFIFO */
    UART_PFIFO_REG(module) |= UART_PFIFO_RXFE_MASK;
    
    /* Set the watermark to 1 */
    UART_RWFIFO_REG(module) = 1;
    
    /* Re-enable the receiver for the test */
    UART_C2_REG(module) |= UART_C2_RE_MASK;
    
    /* Test max packet length (255) */
        
    /* Check bytes as they are recieved */
    for(i=0; i < rx_len; i++)
    {          
      data[i] = uart_getchar(module);
    }
        
    /* Wait for a receive packet to be detected */
    while( !(UART_S3_REG(module) & UART_S3_PRXF_MASK))
    {;}
 
    /* Wait a bit before testing RPL */
    //for( i=0; i < 0x100; i++);        
    while(!(UART_RPL_REG(module)))
    {;}

#if 0
    /* All expected bytes have been received now test the recieve packet length */
    if (UART_RPL_REG(module) != rx_len)
    {    
      printf("ERR! Incorrect receive packet length. RPL: 0x%02X expected: 0x%02X\n", UART_RPL_REG(module), rx_len);
    }
#endif    
                
    /* Clear the packet received flag */
    UART_S3_REG(module) = UART_S3_PRXF_MASK;
    
    /* All tests complete. Return registers to default values */
    UART_C6_REG(module) = 0;
        
    return 0;    
}
/********************************************************************/