#*************************************************************************
#* FILE: init.s   
#*                                                                        
#* DESCRIPTION:                                                           
#* Example init code for MPC5676R. Performs following setup tasks:        
#*     1) Sets FMPLL to 180MHz. (may require modification for desired     
#*        operating frequency)
#*     2) Configure MMU                                              
#*     3) Add MMU entry to support C stack in cache.         
#*     4) Invalidate and enable both instruction and data caches.         
#*     5) Enable SPE instructions (GHS compiler will use SPE by default)  
#*     6) Intialize ECC bits on all 384K of internal SRAM.                
#*     7) Reduce flash wait states. (may require modification of wait     
#*        state parameters for desired operating frequency)               
#*     8) Enables branch target buffer for performance increase.          
#*     9) Lock the stack in cache memory. (included linker file required) 
#*     10)Branch to _start in GHS provided crt0.s file to finish setup    
#*        of the C environment. _start in crt0.s will call main().        
#*========================================================================
#* UPDATE HISTORY                                                         
#* Revision    Author      Date          Description of change            
#* 1.0         B. Terry    11/12/2009    Initial version for MPC5674F.
#* 1.1         D. Erazmus  12/14/2010    Ported to MPC5676R.
#* 1.2         D. Erazmus  07/27/2011    Converted to VLE instruction set.
#* 1.3         D. Erazmus  11/08/2011    Fixed error in BIUCR value.
#*========================================================================
#* COPYRIGHT (c) Freescale Semiconductor, Inc. 2011                      
#* All Rights Reserved                                                   
#*************************************************************************
	
	.vle
	.globl __start
	.section .rcw, ax
	.long 0x015a0000
	.long __start
  
	.section .init,avx     		# The "ax" generates symbols for debug

__start:

	mfpir	r5					# Check core id
	se_cmpi	r5,0
	se_bne  pll_end				# Skip pll init if this is core 1

#******************************
# configure FMPLL to 180MHz (40MHz crystal)
#******************************

# ESYNCR1
    e_lis   r3, 0xC3F8
    e_lis   r4, 0x7004    # EPREDIV = 4
    e_or2i  r4, 0x004A    # EMFD = 74
    e_stw   r4, 8(r3)

# ESYNCR2
    e_li    r4, 0x0003    # ERFD = 3
    e_stw   r4, 12(r3)         

wait_for_lock:
    e_lwz   r5, 4(r3)     # load SYNSR
    e_andi. r5, r5, 0x8
    se_beq wait_for_lock
    
# PLL is now at 180MHz

# Select PLL as system clock
    e_lis   r3,0xC3F9
    e_or2i  r3,0x09A0
    e_li    r4,0x2010
    e_stw   r4,0(r3)

pll_end:

#*****************************
# configure the MMU
#*****************************
# Note 1: Explicitly configure MMU here because MPC5676R BAM does not cover
# all available SRAM or PBRIDGE spaces when it sets up the MMU.  Also, when
# core 1 is running this same code it may not have executed BAM at all.
#
# Note 2: configure TLB1 and TLB3 first before TLB0 since core 1 may be running
# from RAM or Flash with the default 4k TLB0 page out of reset.
#
#TLB1 = internal flash @ 0x0000_0000, VLE
    e_lis   r3,0x1001
    mtspr   mas0,r3
    e_lis   r3,0xC000
    e_or2i  r3,0x0680
    mtspr   mas1,r3
    e_lis   r3,0x0000
    e_or2i  r3,0x0020
    mtspr   mas2,r3
    e_lis   r3,0x0000
    e_or2i  r3,0x003F
    mtspr   mas3,r3
    msync     # Synchronize in case running from flash
    tlbwe
    se_isync  # Synchronize in case running from flash

#TLB3 = internal SRAM @ 0x4000_0000, VLE, Write-Through Cache
    e_lis   r3,0x1003
    mtspr   mas0,r3
    e_lis   r3,0xC000
    e_or2i  3,0x0480
    mtspr   mas1,r3
    e_lis   r3,0x4000
    e_or2i  r3,0x0030
    mtspr   mas2,r3
    e_lis   r3,0x4000
    e_or2i  r3,0x003F
    mtspr   mas3,r3
    msync     # Synchronize in case running from SRAM
    tlbwe
    se_isync  # Synchronize in case running from SRAM

#TLB0 = pbridgeB @ 0xFFE0_0000, Cache inhibited, Guarded
    e_lis   r3,0x1000
    mtspr   mas0,r3
    e_lis   r3,0xC000
    e_or2i  r3,0x0580
    mtspr   mas1,r3
    e_lis   r3,0xFFE0
    e_or2i  r3,0x000A
    mtspr   mas2,r3
    e_lis   r3,0xFFE0
    e_or2i  r3,0x003F
    mtspr   mas3,r3
    tlbwe

#TLB2 = external bus @ 0x2000_0000, VLE
    e_lis   r3,0x1002
    mtspr   mas0,r3
    e_lis   r3,0xC000
    e_or2i  r3,0x0700
    mtspr   mas1,r3
    e_lis   r3,0x2000
    e_or2i  r3,0x0020
    mtspr   mas2,r3
    e_lis   r3,0x0000
    e_or2i  r3,0x003F
    mtspr   mas3,r3
    tlbwe

#TLB4 = pbridgeA @ 0xC3E0_0000, Cache inhibited, Guarded
    e_lis   r3,0x1004
    mtspr   mas0,r3
    e_lis   r3,0xC000
    e_or2i  r3,0x0580
    mtspr   mas1,r3
    e_lis   r3,0xC3E0
    e_or2i  r3,0x000A
    mtspr   mas2,r3
    e_lis   r3,0xC3E0
    e_or2i  r3,0x003F
    mtspr   mas3,r3
    tlbwe

#TLB5 = 4k stack for each core (will be locked in cache)
# (Note: located just after 512k TLB3 entry for SRAM)
    e_lis   r3,0x1005
    mtspr   mas0,r3
    e_lis   r3,0xC000
    e_or2i  r3,0x0180
    mtspr   mas1,r3
    e_lis   r3,0x4008
    e_or2i  r3,0x0000
    mtspr   mas2,r3
    e_lis   r3,0x4008
    e_or2i  r3,0x003F
    mtspr   mas3,r3
    tlbwe

#******************************
# invalidate and enable the data and instruction caches
#******************************
# data cache
    e_lis   r3,0x0010
    e_or2i  r3,0x0003
    mtspr   l1csr0,r3
# inst cache
    e_lis   r3,0x0
    e_or2i  r3,0x0003
    mtspr   l1csr1,r3

#*****************************
# Enable SPE
#*****************************
	mfmsr	r6
	e_or2is	r6, 0x0200
	mtmsr	r6

#*****************************
# initialize 384k SRAM
# (core 0 only)
#*****************************
	mfpir	r5					    # Check core id
	se_cmpi	r5,0
	se_bne	flashopt_end			# Skip sram init and flash optimization if this is core 1

# Store number of 128Byte (32GPRs) segments in Counter
    e_lis   r5, _SRAM_SIZE@h   # Initialize r5 to size of SRAM (Bytes)
    e_or2i  r5, _SRAM_SIZE@l
    e_srwi  r5, r5, 0x7        # Divide SRAM size by 128
    mtctr   r5                 # Move to counter for use with "bdnz"

# Base Address of the internal SRAM
    e_lis   r5, _SRAM_BASE_ADDR@h
    e_or2i  r5, _SRAM_BASE_ADDR@l

# Fill SRAM with writes of 32GPRs
sram_loop:
    e_stmw  r0,0(r5)            # Write all 32 registers to SRAM
    e_addi  r5,r5,128           # Increment the RAM pointer to next 128bytes
    e_bdnz  sram_loop           # Loop for all of SRAM

sram_end:

#******************************
# Optimize Flash
#******************************
# Code is copied to RAM first, then executed, to avoid executing code from flash
# while wait states are changing.

	se_b copy_to_ram

# settings for 180MHz
#
# BIUCR = 0x00014a15
# M8PFE = 0b0           (Core 0 Nexus master pre-fetch disabled)
# M0PFE = 0b1           (Core 0 master pre-fetch enabled)
# APC and RWSC = 0b010  (2 additional hold cycles)
# WWSC = 0b01           (1 wait)
# DPFEN = 0b0           (data pre-fetch disabled)
# ARB = 0b0             (fixed priority arbitration)
# IPFEN = 0b1           (instruction pre-fetch enabled)
# PRI = 0b0             (core 0 higher priority)
# PFLIM = 0b1x          (prefetch on miss or hit)
# BFEN = 0b1            (read line buffer enabled)
#
# BIUCR3 = 0x00020015
# M9PFE = 0b0           (Core 1 Nexus master pre-fetch disabled)
# M6PFE = 0b0           (FlexRay master pre-fetch disabled)
# M5PFE = 0b0           (eDMA_B master pre-fetch disabled)
# M4PFE = 0b0           (eDMA_A master pre-fetch disabled)
# M1PFE = 0b1           (Core 1 master pre-fetch enabled)
# DPFEN = 0b0           (data pre-fetch disabled)
# IPFEN = 0b1           (instruction pre-fetch enabled)
# PFLIM = 0b1x          (prefetch on miss or hit)
# BFEN = 0b1            (read line buffer enabled)
#
flash_opt:
    e_lis   r3,0x0001
    e_or2i  r3,0x4a15
    e_lis   r4,0xC3F8
    e_or2i  r4,0x801C
    e_stw   r3,0(r4)        # BIUCR
    e_lis   r3,0x0002
    e_or2i  r3,0x0015
    e_lis   r4,0xC3F8
    e_or2i  r4,0x8028
    e_stw   r3,0(r4)        # BIUCR3
    se_isync
    msync
    se_blr
  
copy_to_ram:
    e_lis    r3,flash_opt@h
    e_or2i   r3,flash_opt@l
    e_lis    r4,copy_to_ram@h
    e_or2i   r4,copy_to_ram@l
    subf     r4,r3,r4
    se_mtctr r4
    e_lis    r5,0x4000
    se_mtlr  r5
copy:
    e_lbz   r6,0(r3)
    e_stb   r6,0(r5)
    e_addi  r3,r3,1
    e_addi  r5,r5,1
    e_bdnz  copy
    se_isync
    msync
    se_blrl

flashopt_end:

#*****************************
# enable BTB
#*****************************
    e_li    r3, 0x0201
    mtspr   1013, r3
    se_isync
	
#******************************
# lock the stack into cache and set stack pointer (core 1)
#******************************
	mfpir	r5					# Check core id
	se_cmpi	r5,0
	se_beq	stack_cache_0

stack_cache_1:
    e_lis     r3,__STACK_SIZE_1@h
    e_or2i    r3,__STACK_SIZE_1@l
    se_srwi   r3,5              # Shift the contents of R5 right by 5 bits (size/32)
    se_mtctr  r3
    e_lis     r3,__SP_END_1@h
    e_or2i    r3,__SP_END_1@l

lock_cache_loop_1:
    dcbz      r0,r3             # Establish address in cache for 32 bytes and zero
    dcbtls    0,r0,r3           # Lock the address into the cache
    se_addi   r3,32             # Increment to start of next cache line (+32 bytes)
    e_bdnz    lock_cache_loop_1 # Decrement the counter (CTR), branch if nonzero
    e_lis     r1,(__SP_INIT_1-0x10)@h
    e_or2i    r1,(__SP_INIT_1-0x10)@l

##--------------- Set up stack and run time environment Core 1 ---------
    e_lis   r1, __SP_INIT_1@h   # Initialize stack pointer r1 to
    e_or2i  r1, __SP_INIT_1@l   # value in linker command file.

    e_lis   r13, _SDA_BASE_@h   # Initialize r13 to sdata base
    e_or2i  r13, _SDA_BASE_@l   # (provided by linker).

    e_lis   r2, _SDA2_BASE_@h   # Initialize r2 to sdata2 base
    e_or2i  r2, _SDA2_BASE_@l   # (provided by linker).

    e_stwu  r0,-64(r1)			# Terminate stack.
    
    e_b     main

#******************************
# lock the stack into cache and set stack pointer (core 0)
#******************************
stack_cache_0:
    e_lis     r3,__STACK_SIZE_0@h
    e_or2i    r3,__STACK_SIZE_0@l
    se_srwi   r3,5              # Shift the contents of R5 right by 5 bits (size/32)
    se_mtctr  r3
    e_lis     r3,__SP_END_0@h
    e_or2i    r3,__SP_END_0@l

lock_cache_loop_0:
    dcbz      r0,r3             # Establish address in cache for 32 bytes and zero
    dcbtls    0,r0,r3           # Lock the address into the cache
    se_addi   r3,32             # Increment to start of next cache line (+32 bytes)
    e_bdnz    lock_cache_loop_0 # Decrement the counter (CTR), branch if nonzero
    e_lis     r1,(__SP_INIT_0-0x10)@h
    e_or2i    r1,(__SP_INIT_0-0x10)@l

#******************************
# call ghs init code (_start in crt0.s) This
# call to the GHS code insures heap etc. are
# configured and intialized correctly.
#******************************
	e_b _start

# hang if here 
loop_forever:
	se_b loop_forever





  
