/******************************************************************************
  FILE        : PWM_Cfg.c
  PURPOSE     : PWM configuration and PWM services provided as Duty cycle update.
  MACHINE     : Freescale 68HC12 (Target)
  LANGUAGE    : ANSI-C
  HISTORY     : Dec 2010   b30269 First version created
******************************************************************************/
#include "pwm_cfg.h"


#define CLK_A        0
#define CLK_SA       1

#define PWM_POL_HI   1
#define PWM_POL_LOW  0

#define PWM_DIR_IN   0
#define PWM_DIR_OUT  1

#define IO_LOW       0
#define IO_HI        1

#define PWM_PS_PTPRRL_USE  FALSE     /* PWM0-3 port in PORS4-7 */

#if (TRUE == PWM_PS_PTPRRL_USE)   
        
   #define PWM_Ch_IO_Dir(ch,dir)          PTPRRL |= (ch)
                                          if(dir)                  \
                                          {                        \
                                             DDRS   |= (ch << 4);  \
                                          }                        \
                                          else                     \
                                          {                        \
                                             DDRS   &= ~(ch << 4); \
                                          }                        \
      
   
#else   /* PWM in PORT P */
   #define PWM_Ch_IO_Dir(ch,dir)          if(dir)                  \
                                          {                        \
                                             DDRP   |= (ch);       \
                                          }                        \
                                          else                     \
                                          {                        \
                                             DDRP   &= ~(ch);      \
                                          }
   
#endif

#define Select_PWMn_Clk(ch,clk_src)       if(clk_src)              \
                                          {                        \
                                             PWMCLK |=(ch);        \
                                          }                        \
                                          else                     \
                                          {                        \
                                             PWMCLK &=(~ch);       \
                                          }                        \


#define PWM_CHn_OFF(ch)                   PWME &= (~ch)


Bool PWM_Off = TRUE;
 
/******************************************************************************
Function Name  : PWM_Ch_Enable
Engineer       : b30269	
Date           : 07/12/2010
Parameters     : PWM channel number
Returns        : NONE
Notes          : 
******************************************************************************/
void PWM_Ch_Enable(char channel)
{
  PWME |= channel; 
}

/******************************************************************************
Function Name  : Disable_PWM_CH
Engineer       : b30269	
Date           : 07/12/2010
Parameters     : PWM channel number
Returns        : NONE
Notes          : 
*****************************************************************************/

void Disable_PWM_CH(char channel)
{
  PWME &= ~channel; 
}

/******************************************************************************
Function Name  : PWM_Ch_Pol
Engineer       : b30269	
Date           : 07/12/2010
Parameters     : PWM channel number and initial polarity 
Returns        : NONE
Notes          : This function configures PWM CHx polarity.
*****************************************************************************/
void PWM_Ch_Pol(char channel, char pol)
{
  if(pol)
  {
     PWMPOL |= channel;  
  }
  else
  {
     PWMPOL &= ~channel;  
  }
}

/******************************************************************************
Function Name  : PWM_Ch_Shut_Dwn
Engineer       : b30269	
Date           : 07/12/2010
Parameters     : PWM Channel and IO state 
Returns        : NONE
Notes          : This Functions turns off PWM CHx
*****************************************************************************/
void PWM_Ch_Shut_Dwn(unsigned char ch)
{
   PWM_CHn_OFF(ch);
}

/******************************************************************************
Function Name  : PWM_Cfg
Engineer       : b30269	
Date           : 07/12/2010
Parameters     : NONE
Returns        : NONE
Notes          : This function configures PWM Module as follows:
                 1.- configures PIN direction to be used as output
                 2.- Selects the polarity to be used for CH1
                 3.- Selects CH1 clk source
                 4.- Configures PWM CH1 CLK speed
                 5.- Sets CH1 initial duty cycle
                 6.- Selects Align mode to be LEFT_ALIGN
                 7.- Disbles PWM module for LPM
                 8.- Enables PWM CH1.

                 
******************************************************************************/
void PWM_Cfg(void)
{
   PWM_Ch_IO_Dir(PWM_CH1, PWM_DIR_OUT);
   PWM_Ch_Pol(PWM_CH1, PWM_IO_LOW);
   Select_PWMn_Clk(PWM_CH1,CLK_SA);   
   PWM_Scale_SA_Clk(PWM_4KHz_SA_SCL_4MHz);  
   PWM_Set_Init_Duty(PWM_OUT_MIN);   
   PWM_Align_Mode(PWM_LEFT_ALIGN);  
   PWM_LPM_Mode(PWM_LPM_DISABLE);
   PWM_Ch_Enable(PWM_CH1);
}


