/**********************************************************
*
*	FILE				: motor_drv.c
*
*
*	PURPOSE			:Motor Controller functions based on 
*						  MC10B8CV1 Motor Controller Block
*						  Description
**********************************************************/

#include <hidef.h>					// common defines and macros
#include <mc9s12xhy256.h>				// derivative information
#include "motor_drv.h"				// Motor Driver header file
#include "cpu.h"						// CPU header file

word steps_to_move;
word num_steps;
word cur_ptr_pos;						// current pointer position var
word new_ptr_pos;

/**********************************************************
*	MC_init
*
*	setup motor controller (MC) module for HY64 demo applet
**********************************************************/
void MC_init(void){
	
	MCCTL0_MCPRE = DIV8;				// load pre-scalar value
	MCCTL0_FAST = 0;					// 11-bit resolution
	MCCTL1_RECIRC = 0;				// recirc in high-side transistors
//	MCCTL1_MCTOIE = 1;				// enable timer overflow interrupt
	
	MCCC0_MCOM = DUALFULLBRIDGE;	// dual full h-bridge configuration
	MCCC1_MCOM = DUALFULLBRIDGE;	// dual full h-bridge configuration
	
	MCDC0 = 0x554;						// 2/3 duty cycle, S = 0
	MCDC1 = 0x554;						// 2/3 duty cycle, S = 0
	MCCTL0_MCTOIF = 1;				// clear overflow flag
}

/**********************************************************
*	MC_zero_pointer
*
*	move motor pointer CCW to zero position.  Should be called 
* 	at application start 
**********************************************************/
void MC_zero_pointer(void){

	MC_step_dir(CW);					// setup to move CCW
	MC_step(10);							// 
	MC_step_dir(CCW);
	MC_step(300);							// move pointer to zero position
}

/**********************************************************
*	MC_step_dir
*
*	configures motor controller step direction 
*
*	INPUT:	dir - 0, CW: 1 - CCW
**********************************************************/
void MC_step_dir(byte dir){

	if(dir){									// CCW direction
		MCCC0_MCAM = LEFTALIGN;			// left aligned PWM output
		MCCC1_MCAM = RIGHTALIGN;		// right aligned PWM output
	} else{									// CW direction
		MCCC0_MCAM = RIGHTALIGN;		// right aligned PWM output
		MCCC1_MCAM = LEFTALIGN;			// left aligned PWM output
	}
}

/**********************************************************
*	MC_step
*
*	moves motor pointer by # of steps passed to function.  Pointer
*	movement direction must be setup prior to calling this function.  	
*	
*  total # of steps is limited to prevent damage to motor
*
*	Input:	#steps to move pointer	
**********************************************************/
void MC_step(word step)
{
	word n;
	if(step > MAX_STEP_CNT){
		num_steps = MAX_STEP_CNT;	// limit # of steps to max count
	} else num_steps = step;		// get step input
	
	MCPER = MC_PERIOD;				// start motor controller counter
	for(n=0; n<num_steps; n++){
		while(!MCCTL0_MCTOIF);		// wait for MC timer to overflow
		MCDC0 ^= 0x8000;				// toggle S-bit for motor 0, Ch 0
		MCDC1 ^= 0x8000;				// toggle S-bit for motor 0, Ch 1
		MCCTL0_MCTOIF =1;				// clear flag
	}
	MCPER = MC_PER_OFF;				// disable MC
}
