/******************************************************************************		
*                                                                                   	
*       Copyright (C) 2009 Freescale Semiconductor, Inc.                            	
*       All Rights Reserved								              				
*														              					
* Filename:       adc_flowtron.c               											
*														              					
* Description:	  16-bit ADC Driver file for MC9S08GW64	        	    	  	       						
*	
* Author:         Tanya Malik																					
* Notes:        																		
******************************************************************************/		

/******************************************************************************/
/* Include Files                                                              */
/*****************************************************************************/
#include "MC9S08GW64.h"
#include "adc_flowtron.h"    

/******************************************************************************/
/* Global Variables                                                              */
/*****************************************************************************/

unsigned int ADC_Data_A;
unsigned int ADC_Data_B;
unsigned int count_A;
unsigned int count_B;

/******************************************************************************/
/* FUNCTION          : ADC_Init                                               */
/******************************************************************************/
/* Abstract          : This function initializes adc interface                */
/*					           by configuring the internal registers                  */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.ADC_Index - ADC_0 /ADC_1                             */
/*                     2.ADC_Clk_Select -  ADICLK_BUS/ADICLK_BUS_2            */
/*                                         /ADICLK_ALTCLK/ADICLK_ADACK        */
/*                     3.ADC_Clk_Div - ADIV_1/ADIV_2/ADIV_3/ADIV_4            */
/*                     4.ADC_Mode - ADC_MODE_8/ADC_MODE_10                    */
/*                                  /ADC_MODE_12/ADC_MODE_16                  */
/*                     5.Trigger_Select - ADTRG_HW/ADTRG_SW                   */
/*                     6.Continuous_Conversion_Select - ADCO_SINGLE           */
/*                                                      /ADCO_CONTINUOUS      */
/*                     7.Volt_Ref_Select - ADC_REFSEL_EXT/ADC_REFSEL_ALT      */
/*                                         /ADC_REFSEL_BG/ADC_REFSEL_RES      */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/
void ADC_Init(unsigned char ADC_Index,unsigned char ADC_Clk_Select,
              unsigned char ADC_Clk_Div,unsigned char ADC_Mode,
              unsigned char Trigger_Select,unsigned char Continuous_Conversion_Select,
              unsigned char Volt_Ref_Select ) 
{
  ADC_L *adc_l_ptr;
  ADC_U *adc_u_ptr;
 
  if(ADC_Index == ADC_0)                 //if ADC0 is selected 
    {
      SCGC1_ADC0 = 1;                    //enables clock for ADC0
      adc_l_ptr = (ADC_L *)ADC0_L;
      adc_u_ptr = (ADC_U *)ADC0_U;       //points to ADC0
    }
  else                                    //if ADC1 is selected
   {
      SCGC1_ADC1 = 1;                     //enables clock for ADC1
      adc_l_ptr = (ADC_L *)ADC1_L;
      adc_u_ptr = (ADC_U *)ADC1_U;        //points to ADC1
   }

  adc_l_ptr->ADCCFG1.Bits.ADICLK = ADC_Clk_Select;   //sets the clk for ADC module
  adc_l_ptr->ADCCFG1.Bits.ADIV   = ADC_Clk_Div;      //sets the clk divider 
  adc_l_ptr->ADCCFG1.Bits.MODE   = ADC_Mode;         //sets the mode of conversion

  
  adc_u_ptr->ADCSC2.Bits.ADTRG   = Trigger_Select;   //sets the trigger to be used for ADC conversion
  adc_u_ptr->ADCSC2.Bits.REFSEL  = Volt_Ref_Select;  //sets the reference voltage
  adc_u_ptr->ADCSC3.Bits.ADCO    = Continuous_Conversion_Select; //selects between the continuous and single conversion
   
}

/******************************************************************************/
/* FUNCTION          : ADC_Channel_Config                                     */
/******************************************************************************/
/* Abstract          : This function selects the Channel and configures       */
/*                       the registers                                        */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.ADC_Index - ADC_0 /ADC_1                             */
/*                     2.Channel_Index - ADC_CHANNEL_A/ADC_CHANNEL_B          */
/*                     3.Interrupt_Enable - ADC_AIEN_OFF/ADC_AIEN_ON          */
/*                     4.Diff_Mode_Sel - ADC_SINGLE_MODE/ADC_DIFF_MODE        */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/

void ADC_Channel_Config(unsigned char ADC_Index ,
                        unsigned char Channel_Index,
                        unsigned char Interrupt_Enable,
                        unsigned char Diff_Mode_Sel) 
{
  ADC_L *adc_l_ptr;
  ADC_U *adc_u_ptr;
 
  if(ADC_Index == ADC_0)                 //if ADC0 is selected 
    {
      adc_l_ptr = (ADC_L *)ADC0_L;
      adc_u_ptr = (ADC_U *)ADC0_U;       //points to ADC0
    }
  else                                   //if ADC1 is selected
    {
      adc_l_ptr = (ADC_L *)ADC1_L;
      adc_u_ptr = (ADC_U *)ADC1_U;       //points to ADC1
    } 
 
 if(Channel_Index == ADC_CHANNEL_A)      //if channel A is selected
   {
      adc_l_ptr->ADCSC1A.Bits.AIENA = Interrupt_Enable;      //selects if the converison complete interrupt enabled for channel A
      adc_l_ptr->ADCSC1A.Bits.DIFFA = Diff_Mode_Sel;         //selects between the single mode or diff mode for channel A
   }
  else 
   {
      adc_l_ptr->ADCSC1B.Bits.AIENB = Interrupt_Enable;      //selects if the converison complete interrupt enabled for channel B
      adc_l_ptr->ADCSC1B.Bits.DIFFB = Diff_Mode_Sel;         //selects between the single mode or diff mode for channel B
   }
}

/******************************************************************************/
/* FUNCTION          : ADC_Compare_Enable                                     */
/******************************************************************************/
/* Abstract          : This function enables the compare function and sets    */
/*                     the various attributes                                 */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.ADC_Index - ADC_0 /ADC_1                             */
/*                     2.Comapre_GreaterThan_Enable - ADC_COMPARE_LESS        */
/*                                                    /ADC_COMPARE_GREATER    */
/*                     3.Compare_Range_Enable - ADC_COMPARE_RANGE_ENABLED     */
/*                                               /ADC_COMAPRE_RANGE_DISABLED  */
/* Return Parameter  : None                                                   */
/******************************************************************************/
   
void ADC_Compare_Enable(unsigned char ADC_Index,
                        unsigned char Compare_GreaterThan_Enable,
                        unsigned char Compare_Range_Enable,
                        unsigned int Compare_Value1,unsigned int Compare_Value2) 
{
  ADC_L *adc_l_ptr;
  ADC_U *adc_u_ptr;
 
  if(ADC_Index == ADC_0)                 //if ADC0 is selected 
    {
      adc_l_ptr = (ADC_L *)ADC0_L;
      adc_u_ptr = (ADC_U *)ADC0_U;       //points to ADC0 
    }
  else 
   {                                     //if ADC1 is selected
      adc_l_ptr = (ADC_L *)ADC1_L;
      adc_u_ptr = (ADC_U *)ADC1_U;       //points to ADC1
   }
    
  adc_u_ptr->ADCSC2.Bits.ACFE = 1;       //enables the compare function of the specified ADC
  adc_u_ptr->ADCSC2.Bits.ACFGT =Compare_GreaterThan_Enable;  //selects between compare greater than or less than the value in compare register 
  adc_u_ptr->ADCSC2.Bits.ACREN =Compare_Range_Enable;        //selects if the compare within the range is enabled or disabled
  adc_u_ptr->ADCCV1 = Compare_Value1;                        //sets the compare value to compare the conversion result with
  
  if(Compare_Range_Enable == ADC_COMPARE_RANGE_ENABLED) 
    {
      adc_u_ptr->ADCCV2 = Compare_Value2;  //sets the 2nd compare value if compare_range is enabled 
    }
}

/******************************************************************************/
/* FUNCTION          : ADC_Compare_Disable                                    */
/******************************************************************************/
/* Abstract          : This function disables the compare function of the     */
/*                     specified ADC                                          */
/*                                                                            */
/* Input Parameters  : 1.ADC_Index - ADC_0 /ADC_1                             */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/

void ADC_Compare_Disable(unsigned char ADC_Index) 
{
  ADC_L *adc_l_ptr;
  ADC_U *adc_u_ptr;
 
  if(ADC_Index == ADC_0)                 //if ADC0 is selected 
    {
      adc_l_ptr = (ADC_L *)ADC0_L;
      adc_u_ptr = (ADC_U *)ADC0_U;       //points to ADC0
    }
  else                                   //if ADC1 is selected
   {
      adc_l_ptr = (ADC_L *)ADC1_L;
      adc_u_ptr = (ADC_U *)ADC1_U;      //points to ADC1
   }
   
  adc_u_ptr->ADCSC2.Bits.ACFE = 0;      //disables the compare function in the specified ADC
}

/******************************************************************************/
/* FUNCTION          : ADC_Avg_Enable                                         */
/******************************************************************************/
/* Abstract          : This function enables the hardware averaging functions */
/*                     and sets the no of samples required                    */
/*                                                                            */
/*                                                                            */
/* Input Parameters  : 1.ADC_Index - ADC_0 /ADC_1                             */
/*                     2.Avg_Sample_Select- ADC_AVG_SAMPLE_4/ADC_AVG_SAMPLE_8 */
/*                                        /ADC_AVG_SAMPLE_16/ADC_AVG_SAMPLE_32*/
/* Return Parameter  : None                                                   */
/******************************************************************************/

void ADC_Avg_Enable(unsigned char ADC_Index,
                             unsigned char Avg_Sample_Select) 
{
  ADC_L *adc_l_ptr;
  ADC_U *adc_u_ptr;
 
  if(ADC_Index == ADC_0)                 //if ADC0 is selected 
    {
      adc_l_ptr = (ADC_L *)ADC0_L;
      adc_u_ptr = (ADC_U *)ADC0_U;       //points to ADC0
    }
  else                                   //if ADC1 is selected
   {
      adc_l_ptr = (ADC_L *)ADC1_L;
      adc_u_ptr = (ADC_U *)ADC1_U;       //points to ADC1
   }
   
  adc_u_ptr->ADCSC3.Bits.AVGE =1;        //enables the harware averaging for the specified ADC
  adc_u_ptr->ADCSC3.Bits.AVGS = Avg_Sample_Select;     //selects the No of Samples required
}

/******************************************************************************/
/* FUNCTION          : ADC_Avg_Disable                                        */
/******************************************************************************/
/* Abstract          : This function disables the H/W averaging function of   */
/*                     the specified ADC                                      */
/*                                                                            */
/* Input Parameters  : 1.ADC_Index - ADC_0 /ADC_1                             */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/ 
void ADC_Avg_Disable(unsigned char ADC_Index)
                              
{
  ADC_L *adc_l_ptr;
  ADC_U *adc_u_ptr;
 
  if(ADC_Index == ADC_0)                 //if ADC0 is selected 
    {
      adc_l_ptr = (ADC_L *)ADC0_L;
      adc_u_ptr = (ADC_U *)ADC0_U;       //points to ADC0
    }
  else 
   {
      adc_l_ptr = (ADC_L *)ADC1_L;       //if ADC1 is selected 
      adc_u_ptr = (ADC_U *)ADC1_U;       //points to ADC1
   }
   
  adc_u_ptr->ADCSC3.Bits.AVGE =0;        //disables the H/W averaging of the specified ADC
}

/******************************************************************************/
/* FUNCTION          : ADC_Diff_Channel_Select                                */
/******************************************************************************/
/* Abstract          : This function selects the differential pair for ADC    */
/*                     ADC conversion                                         */
/*                                                                            */
/* Input Parameters  : 1.ADC_Index - ADC_0 /ADC_1                             */
/*                     2.Channel_Index - ADC_CHANNEL_A/ADC_CHANNEL_B          */
/*                                                                            */
/* Return Parameter  : 0:fail 1:pass                                          */
/******************************************************************************/ 
unsigned char ADC_Diff_Channel_Select(unsigned char ADC_Index,unsigned char Channel_Index) 
{
 if(ADC_Index == ADC_0) 
  { 
    if(Channel_Index == ADC_CHANNEL_A)
        ADC0SC1A_ADCHA = ADC_0;          //DADP0/DADM0 selected on channel A 
    else
      {
        if(ADC0SC2_ADTRG == ADTRG_HW)
          ADC0SC1B_ADCHB = ADC_0;       //DADP0/DADM0 selected on channel B
        else 
          return 0;              //channel B not available in software trigger mode
      }
  }
 else
  {
   if(Channel_Index == ADC_CHANNEL_A)
        ADC1SC1A_ADCHA = ADC_1;          //DADP1/DADM1 selected on channel A
    else 
      {
        if(ADC1SC2_ADTRG == ADTRG_HW)
          ADC1SC1B_ADCHB = ADC_1;        //DADP1/DADM1 selected on channel B
        else
          return 0;          //channel B not available in software trigger mode
      }
  }
 return 1;      //channel successfully selected
}

/******************************************************************************/
/* FUNCTION          : ADC_Single_Channel_Select                              */
/******************************************************************************/
/* Abstract          : This function selects the single ended channel for ADC */
/*                     ADC conversion                                         */
/*                                                                            */
/* Input Parameters  : 1.Channel_Index: ADC_CHANNEL_A/ADC_CHANNEL_B           */
/*                     2.Channel : ADC_CHANNEL_AD0/ADC_CHANNEL_AD1............*/
/*                                 ADC_CHANNEL_AD15                           */
/*                                                                            */
/* Return Parameter  :                                                        */
/******************************************************************************/ 

unsigned char ADC_Single_Channel_Select(unsigned char Channel_Index,
                        unsigned char Channel) 
{
  //unsigned int i;
  switch(Channel) 
  {
    case ADC_CHANNEL_AD0:
        Channel = 1;
        goto a0;
        
    case ADC_CHANNEL_AD2:
        PTAPF4_A0 = 0x4;       //pin muxing 
        goto a0;
        
    case ADC_CHANNEL_AD4:
        PTFPF4_F0 = 0x2;       //pin muxing
        goto a0;
        
    case ADC_CHANNEL_AD5:
        PTFPF4_F1 = 0x2;       //pin muxing
        goto a0;
        
    case ADC_CHANNEL_AD8:
        PTGPF2_G2 = 0x2;       //pin muxing
        Channel = 12; 
        goto a0;
        
    case ADC_CHANNEL_AD9:
        PTGPF2_G3 = 0x2;       //pin muxing
        Channel = 13; 
        goto a0;
        
    case ADC_CHANNEL_AD12:     
        PTGPF1_G0 = 0x3;       //pin muxing
        Channel = 20;
        goto a0;
        
    case ADC_CHANNEL_AD13:
        PTGPF4_G1 = 0x3;        //pin muxing
        Channel = 21;
        goto a0;
        
    
    case ADC_CHANNEL_AD1:
        Channel = 0;
        goto a1;
        
    case ADC_CHANNEL_AD3:
        PTAPF1_A1 = 0x4;         //pin muxing
        goto a1;  
        
    case ADC_CHANNEL_AD6:
        PTGPF1_G0 = 0x2;         //pin muxing
        Channel = 8;
        goto a1;
        
    case ADC_CHANNEL_AD7:
        PTGPF1_G1 = 0x2;          //pin muxing
        Channel = 9;
        goto a1;
        
    case ADC_CHANNEL_AD10:
        PTGPF3_G4 = 0x3;          //pin muxing
        Channel = 17;
        goto a1;
        
    case ADC_CHANNEL_AD11:
        PTGPF3_G5 = 0x3;          //pin muxing
        Channel = 16;
        goto a1;
        
    case ADC_CHANNEL_AD14:
        PTHPF1_H0 = 0x3;          //pin muxing
        Channel = 23;
        goto a1;
        
    case ADC_CHANNEL_AD15:
        PTHPF1_H1 = 0x3;           //pin muxing
        Channel = 22;
        goto a1;
  }
  a0:
    if(Channel_Index == ADC_CHANNEL_A)
        ADC0SC1A_ADCHA = Channel;
    else 
    {
      if(ADC0SC2_ADTRG == ADTRG_HW) 
          ADC0SC1B_ADCHB = Channel;
      else 
          return 0;     //channel B is not available in software trigger
    }
    goto end;
    
  a1:
  if(Channel_Index == ADC_CHANNEL_A)
        ADC1SC1A_ADCHA = Channel;
    else 
    { 
      if(ADC1SC2_ADTRG == ADTRG_HW)
         ADC1SC1B_ADCHB = Channel;
      else
       return 0;   //channel B not available in software trigger
    }
 end:
 {
  return 1; //channel set successfully
 }
    
}



/******************************************************************************/
/* FUNCTION          : ADC_Cal                                                */
/******************************************************************************/
/* Abstract          : This function Callibrates the ADC and stores the       */
/*                     callibrated values in respective registers             */
/*                                                                            */
/* Input Parameters  : 1.ADC_Index - ADC_0 /ADC_1                             */
/*                                                                            */
/* Return Parameter  : 0:Callibration Fails /1: Callibration successful       */
/******************************************************************************/ 

unsigned char ADC_Cal(unsigned char ADC_Index)
{ 
 volatile unsigned int cal_var;
 unsigned char flag = 0;
 volatile unsigned char dummy;
 
 ADC_L *adc_l_ptr;
 ADC_U *adc_u_ptr;
 
  if(ADC_Index == ADC_0)               //if ADC0 is selected
 {
  adc_l_ptr = (ADC_L *)ADC0_L;
  adc_u_ptr = (ADC_U *)ADC0_U;         //points to ADC0
 } 
 else                                  //if ADC1 is selected
 {
 
  adc_l_ptr = (ADC_L *)ADC1_L;
  adc_u_ptr = (ADC_U *)ADC1_U;         //points to ADC1
 } 

  if(adc_u_ptr->ADCSC2.Bits.ADTRG) 
  {
   flag =1;
  }
  adc_u_ptr->ADCSC2.Bits.ADTRG = 0;     // Enable Software Conversion Trigger for Calibration Process
  adc_u_ptr->ADCSC3.Bits.CAL = 1;       // Start Callibration


 while (adc_l_ptr->ADCSC1A.Bits.COCOA == 0)
  	{
  	// Wait for conversion to complete
  	}             
 dummy =adc_l_ptr->ADCRA;     //clears the flag
  if (adc_u_ptr->ADCSC3.Bits.CALF == 1)     // Check for Calibration fail error 
    {
    adc_u_ptr->ADCSC3.Bits.CALF = ADC_CALF_FAIL;  // Clear Cal Fail flag
    return 0;  			 												// User will need to either re-attempt Calibration or generate fail indicator.
    }

  // Calculate plus-side calibration  
  cal_var = 0x00;

  cal_var =  (adc_u_ptr->ADCCLP.CLP0.Bits.CLP0); 
  cal_var += (adc_u_ptr->ADCCLP.CLP1.Bits.CLP1);
  cal_var += (adc_u_ptr->ADCCLP.CLP2.Bits.CLP2);
  cal_var += (adc_u_ptr->ADCCLP.CLP3.Bits.CLP3);
  cal_var += (adc_u_ptr->ADCCLP.CLP4.Bits.CLP4);
  cal_var += (adc_u_ptr->ADCCLP.CLPS.Bits.CLPS);

  cal_var = cal_var/2;
  
  cal_var += 0x8000; // Set MSB

  adc_u_ptr->ADCPG= cal_var;          //contains the gain error correction for the plus-side
                                      //input in differential mode 
                                      //or the overall conversion in single-ended mode

  /* Calculate minus-side calibration  */
  cal_var = 0x00;

  cal_var =  (adc_u_ptr->ADCCLM.CLM0.Bits.CLM0);
  cal_var += (adc_u_ptr->ADCCLM.CLM1.Bits.CLM1);
  cal_var += (adc_u_ptr->ADCCLM.CLM2.Bits.CLM2);
  cal_var += (adc_u_ptr->ADCCLM.CLM3.Bits.CLM3);
  cal_var += (adc_u_ptr->ADCCLM.CLM4.Bits.CLM4);
  cal_var += (adc_u_ptr->ADCCLM.CLMS.Bits.CLMS);

  cal_var = cal_var/2;

  cal_var += 0x8000; // Set MSB

  adc_u_ptr->ADCMG = cal_var;         //contains the gain error correction for the
                                      //minus-side input in differential mode

  /* Clear CAL bit */
  adc_u_ptr->ADCSC3.Bits.CAL = 0;
  if(flag) 
    {  
      adc_u_ptr->ADCSC2.Bits.ADTRG = 1;   //enables the H/W trigger if it was initially enabled in ADC_Init
    }
  return 1;  //callibration successful
}

/******************************************************************************/
/* FUNCTION          : Read_ADC0_A                                            */
/******************************************************************************/
/* Abstract          : This function reads the ADC0 channel A Data register   */
/*                                                                            */
/* Input Parameters  : Data - the data read is stored in the address          */
/*                            passed to this pointer                          */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/ 
void Read_ADC0_A(unsigned int* Data) 
{
/*unsigned int temp;

 while(!(ADC0SC1A_COCOA));    //Waiting for the conversion to be completed
 temp = ADC0RA;
 
 if(ADC0SC1A_DIFFA)
  {
    
    if((temp & 0x8000)) 
    {
     temp = ~temp;
     temp = temp+1;
    }
  }
    *Data = temp; */ 
    
 while(!(ADC0SC1A_COCOA));    //Waiting for the conversion to be completed
*Data =ADC0RA;
}

/******************************************************************************/
/* FUNCTION          : Read_ADC1_A                                            */
/******************************************************************************/
/* Abstract          : This function reads the ADC1 Channel A Data register   */
/*                                                                            */
/* Input Parameters  : Data - the data read is stored in the address          */
/*                            passed to this pointer                          */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/ 

void Read_ADC1_A(unsigned int* Data) 
{
 while(!(ADC1SC1A_COCOA));    //Waiting for the conversion to be completed
 *Data = ADC1RA;
}

/******************************************************************************/
/* FUNCTION          : Read_ADC0_B                                            */
/******************************************************************************/
/* Abstract          : This function reads the ADC0 Channel B Data register   */
/*                                                                            */
/* Input Parameters  : Data - the data read is stored in the address          */
/*                            passed to this pointer                          */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/ 

void Read_ADC0_B(unsigned int* Data)
{
 while(!(ADC0SC1B_COCOB));    //Waiting for the conversion to be completed
 *Data = ADC0RB;
}

/******************************************************************************/
/* FUNCTION          : Read_ADC1_B                                            */
/******************************************************************************/
/* Abstract          : This function reads the ADC1 Channel B Data register   */
/*                                                                            */
/* Input Parameters  : Data - the data read is stored in the address          */
/*                            passed to this pointer                          */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/ 

void Read_ADC1_B(unsigned int* Data)
{
 while(!(ADC1SC1B_COCOB));    //Waiting for the conversion to be completed
 
 *Data = ADC1RB;
}

/******************************************************************************/
/* FUNCTION          : Read_ADC_AnalogValue                                   */
/******************************************************************************/
/* Abstract          : This function reads the ADC converted value and        */
/*                     converts it to a analog value depending upon           */
/*                     VREFH & VREFL                                          */
/*                                                                            */
/* Input Parameters  : ADC_Index : ADC_0/ADC_1                                */
/*                     Channel_Index: ADC_CHANNEL_A/ADC_CHANNNEL_B            */
/*                                                                            */
/* Return Parameter  : analog value                                           */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

float Read_ADC_AnalogValue( unsigned char ADC_Index,unsigned char Channel_Index) 
{
  ADC_L *adc_l_ptr;
  unsigned int DataReg;
  unsigned char flag = 0;
  signed int temp;
 
  if(ADC_Index == ADC_0)                 //if ADC0 is selected 
    {
       adc_l_ptr = (ADC_L *)ADC0_L;
    }
  else                                    //if ADC1 is selected
    {
       adc_l_ptr = (ADC_L *)ADC1_L;
    }
   
 if(Channel_Index == ADC_CHANNEL_A) 
    {
       while(!(adc_l_ptr->ADCSC1A.Bits.COCOA));
       DataReg = adc_l_ptr->ADCRA;
       if(adc_l_ptr->ADCSC1A.Bits.DIFFA)
       flag = 1;
    }
  else
    {   
      while(!(adc_l_ptr->ADCSC1B.Bits.COCOB));
       DataReg = adc_l_ptr->ADCRB;
       if(adc_l_ptr->ADCSC1B.Bits.DIFFB)
       flag = 1;
    }
    
  switch(adc_l_ptr->ADCCFG1.Bits.MODE)
  {
    case ADC_MODE_8:
      temp = (signed int)DataReg;
      return ((temp/255.0) * (VREFH-VREFL));               //8bit result 
      
    case ADC_MODE_10:
      temp = (signed int)DataReg;
      return ((temp/1023.0) * (VREFH-VREFL));               //10bit result
      
    case ADC_MODE_12:
      temp = (signed int)DataReg;
      return ((temp/4095.0) * (VREFH-VREFL));               //12bit result
      
    case ADC_MODE_16:
      if(flag) 
        {
          temp = (signed int)DataReg;
          return ((temp/32767.0) * (VREFH-VREFL));          //15 bit result
        } 
      else 
        {
           return ((DataReg/65535.0)*(VREFH-VREFL));         //16 bit result
        }    
  }   
   
   
   
}
/*Interrupt subroutines*/
void interrupt VectorNumber_Vadc1 ADC1_ServiceInterrupt(void)
{ 
  if(ADC1SC1A_COCOA) 
  {
    ADC_Data_A = ADC1RA;
  
  }
  else {
    
  ADC_Data_B = ADC1RB;
  
  }
    //must read ADCR or write ADCSC1 to clear interrupt request
}

void interrupt VectorNumber_Vadc0 ADC0_ServiceInterrupt(void) 
{ 
  
  if(ADC0SC1A_COCOA) 
    { 
      ADC_Data_A = ADC0RA;
       
    }
  else 
    {
      ADC_Data_B = ADC0RB;
  
    }
   //must read ADCR or write ADCSC1 to clear interrupt request
}