/*******************************************************************************
*
*	Freescale Semiconductor Inc.
*	(c) Copyright 2010 Freescale Semiconductor Inc.
*	ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
*	@file sci_app_05.c
*	
*	@version 0.0.4.0
*	
*	@lastmodusr B16958
*	
*	@lastmoddate Jul-1-2010
*	
********************************************************************************
*
*	@brief eDMA emulating multiple SCI transmits and employs scatter gather option
*
*	This example demonstates how the eDMA could be used to emulate SCI transmiter
*	of buffer with fixed length and scatter gather option, using two DMA channels.
*	Configuration:
*		- DMA channel 0 writes to GPIO PCR_18 (PB2)
*		- DMA channel 0 activated using periodic triggers
*		- DMA channel 1 takes one byte from buffers and link to the channel 0
*		- DMA channel 1 activated explicitely by SW to transmit burst of bytes
*		- DMA channel 1 scatters to USER TCD, which disables the timer and hence the transmission
*		- Connection: Standard wiring on EVB: PB2 and PB3 -> SCI transciever
*	
*	This example is the same as sci_app_02, uses the same number of channels, but
*	also scatter-gather feature to solve buffering and disabling in one channel.
*	
*	@project AN4147
*	
*	@author B16958
*	
*	@ingroup sci_tx
*
******************************************************************************/

/******************************************************************************
* Includes
******************************************************************************/
#include "sci_app_05.h"
#include "MPC560xB.h"

/******************************************************************************
* Defines
******************************************************************************/
/* define location of user tcd's */
#define USER_TCD_0      (*(volatile struct tcd_t *)      (0x40007F00UL + 0 ))
#define USER_TCD_1      (*(volatile struct tcd_t *)      (0x40007F00UL + 32))


/******************************************************************************
* Internal variable Definitions
******************************************************************************/
static volatile unsigned char *pTxData=((unsigned char *)&SIU.PGPDO[3].R)+2; /*!< Pointer 
* to the data prepared for transmission, i.e. serialization. This is a reference
* to GPIO parallel access registers */
static  unsigned char stringBuffer[125]="Hello World! -please use period at the end of a sentence.";
/*!< String which is transmitted only ones over serial line. Note that 
* the transmission end is marked by '.' character */
static  unsigned long int stopTimer=0; /*!< Number to be moved to PIT control register
* to stop PIT and hence the SCI transmission */

/******************************************************************************
* Internal Function Declarations
******************************************************************************/


static void SciInit05(void);

static void SciRun05(void);


/******************************************************************************
* External Function Definitions
******************************************************************************/
extern void SciApp05(void)
{
	SciInit05();
	
	SciRun05();
}

/******************************************************************************
* Internal Function Definitions
******************************************************************************/

/***************************************************************************//*!
*
* @brief Initializes the SCI application to transmit one byte continuouslly,
*			whilst this byte is modified by other channel.
*
*	Initialization of the following peripherals:
*		- SIU: PCR registers for UART Tx signals
*		- PIT_0: To periodically trigger DMA at 9600Hz (one bit per timer overflow)
*		- DMAMUX: Routes PIT_0 flag to activate channel 0 DMA
*		- eDMA0: Setup one byte data movements
*			- from GPIO's PG15-PH9 (10 iterations of minor loop)
*			- to GPIO PB2 (SCI Tx)
*			- link to the channel 1 after major loop.
*		- eDMA1: Setup one byte transfer from data buffer to PH0-PH7 (data to be sent)
*			- this channel uses scatter gather operation to split the task into
*				buffering the data and disabling the channel.
*	
*	Note, that User TCD's are located in user RAM memory at locations reserved by linker 
*	command file (ram_user_tcd); they could be located also in flash.
*	User TCD's need to be alligned to the 32 bytes boundary.
*
*	@ingroup sci_tx
*
******************************************************************************/
static void SciInit05(void)
{
	/* Initialize PCR registers */
	 SIU.PCR[18].B.PA = 0;          /* Pin asigned as GPIO for DMA to perform SCI Tx */
	 SIU.PCR[18].B.OBE = 1;         /* Output buffer enable */
 	/* setup the line to logical high -- to get the first start bit right */
 	SIU.GPDO[18].R = 1;
	
	/* Initialize PIT 0 timer for periodic triggers of DMA channel 0 */
 	PIT.PITMCR.B.MDIS = 0;	/* enable the module */
  	PIT.CH[0].LDVAL.R=833;	/* setup 9600@8MHZ overflow rate */
  	PIT.CH[0].TFLG.B.TIF=1; /* clear the flag */

	/* Initialize the DMA channel 0 */
	DMAMUX.CHCONFIG[0].R =0x00; /* disable the channel activation source */
	/* Setup the TCD for channel 0 --> single SCI Tx */
	EDMA.TCD[0].SADDR = (vuint32_t)( &(SIU.GPDO[120].R)); /* Load data from byte-wise register PG15(=start bit), PH0-PH8(data), PH9(=stopBit) */
	EDMA.TCD[0].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[0].SOFF = -1; /* After transfer, decrement 1 */
	EDMA.TCD[0].SLAST = 10; /* After major loop, back to the beginning (10 iterations) */
	EDMA.TCD[0].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[0].DADDR = (vuint32_t) &(SIU.GPDO[18].R); /* writing to PCR 18 (GPIO-Tx) */
	EDMA.TCD[0].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[0].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[0].DLAST_SGA = 0; /* After major loop no change to destination addr */
	EDMA.TCD[0].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[0].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[0].BITER = 10; /* 10 iterations of major loop */
	EDMA.TCD[0].CITER = 10; /* Initialize current iteraction count */
	EDMA.TCD[0].D_REQ = 0; /* Disable channel when major loop is done (sending one byte only) */
	EDMA.TCD[0].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[0].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[0].CITERE_LINK = 0; /* no link after minor loop */
	EDMA.TCD[0].BITERE_LINK = 0;
	EDMA.TCD[0].MAJORE_LINK = 1; /* Linking the next channel to process buffering after major loop */
	EDMA.TCD[0].BITERLINKCH = 0;	
	EDMA.TCD[0].CITERLINKCH = 0;	
	EDMA.TCD[0].MAJORLINKCH = 1;	/* Link to channel 1 */
	EDMA.TCD[0].E_SG = 0;
	EDMA.TCD[0].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[0].START = 0; /* Initialize status flags */
	EDMA.TCD[0].DONE = 0;
	EDMA.TCD[0].ACTIVE = 0;
	/* Setup the TCD for channel 1 --> buffer for SCI Tx */
	EDMA.TCD[1].SADDR = (vuint32_t)(stringBuffer+1); /* Load data from the buffer (start with 2nd element) */
	EDMA.TCD[1].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[1].SOFF = 1; /* After transfer, decrement 1 */
	EDMA.TCD[1].SLAST = -12; /* After major loop, back to the beginning (-12 iterations) */
	EDMA.TCD[1].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[1].DADDR = (vuint32_t)pTxData; /* writing to PCR 18 (GPIO-Tx) */
	EDMA.TCD[1].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[1].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[1].DLAST_SGA = (vuint32_t)&(USER_TCD_0.SADDR); /* After major loop no change to destination addr */
	EDMA.TCD[1].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[1].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[1].BITER = 12; /* 12 iterations of major loop */
	EDMA.TCD[1].CITER = 11; /* Initialize current iteraction count, (start with 2nd data!) */
	EDMA.TCD[1].D_REQ = 0; /* Disable channel when major loop is done (sending the buffer once) */
	EDMA.TCD[1].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[1].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[1].CITERE_LINK = 0; /* Linking disabled */
	EDMA.TCD[1].BITERE_LINK = 0;
	EDMA.TCD[1].BITERLINKCH = 0;
	EDMA.TCD[1].CITERLINKCH = 0;
	EDMA.TCD[1].MAJORE_LINK = 0;
	EDMA.TCD[1].E_SG = 1;
	EDMA.TCD[1].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[1].START = 0; /* Initialize status flags */
	EDMA.TCD[1].DONE = 0;
	EDMA.TCD[1].ACTIVE = 0;

	/**********************
	*	Note: this user TCD is only for demonstration purpose located inside TCD memory.
	*	In reality, it could be stored anywhere within user data memory (commonly in Flash)
	*/
	/* Setup the USER_TCD to stop PIT0 after that */
	USER_TCD_0.SADDR = (vuint32_t)(&stopTimer); /* memory structure to enable/disable PIT0 */
	USER_TCD_0.SSIZE = 2; /* Read 2**2 = 4 byte per transfer */
	USER_TCD_0.SOFF = 0; /* After transfer no increment */
	USER_TCD_0.SLAST = 0; /* After major loop no change */
	USER_TCD_0.SMOD = 0; /* Source modulo feature not used */
	USER_TCD_0.DADDR = (vuint32_t)(&(PIT.CH[0].TCTRL.R)); /* PIT_0 - enable */
	USER_TCD_0.DSIZE =  2; /* Write 2**2 = 4 byte per transfer */
	USER_TCD_0.DOFF = 0; /* Do not increment destination addr */
	USER_TCD_0.DLAST_SGA = (vuint32_t)&(USER_TCD_1.SADDR); /* After major no change to destination addr */
	USER_TCD_0.DMOD = 0; /* Destination modulo feature not used */
	USER_TCD_0.NBYTES = 4; /* Transfer 4 byte per minor loop */
	USER_TCD_0.BITER = 1; /* 1 iterations of major loop */
	USER_TCD_0.CITER = 1; /* Initialize current iteraction count, (start with 2nd data!) */
	USER_TCD_0.D_REQ = 1; /* Disable channel when major loop is done (sending the buffer once) */
	USER_TCD_0.INT_HALF = 0; /* no interrupt in half */
	USER_TCD_0.INT_MAJ = 0; /* no interrupt in major loop completion */
	USER_TCD_0.CITERE_LINK = 0; /* Linking disabled */
	USER_TCD_0.BITERE_LINK = 0;
	USER_TCD_0.BITERLINKCH = 0;
	USER_TCD_0.CITERLINKCH = 0;
	USER_TCD_0.MAJORE_LINK = 0; 
	USER_TCD_0.MAJORLINKCH = 0; 
	USER_TCD_0.E_SG = 1;
	USER_TCD_0.BWC = 0; /* Default bandwidth control- no stalls */
	USER_TCD_0.START = 0; /* Initialize status flags */
	USER_TCD_0.DONE = 0;
	USER_TCD_0.ACTIVE = 0;
	/* Setup the USER_TCD to reinitialize back (so we can start another transmission afterwards) */	
	USER_TCD_1.SADDR = (vuint32_t)(stringBuffer+1); /* Load data from the buffer (start with 2nd element) */
	USER_TCD_1.SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	USER_TCD_1.SOFF = 1; /* After transfer, decrement 1 */
	USER_TCD_1.SLAST = -12; /* After major loop, back to the beginning (-12 iterations) */
	USER_TCD_1.SMOD = 0; /* Source modulo feature not used */
	USER_TCD_1.DADDR = (vuint32_t)pTxData; /* writing to PCR 18 (GPIO-Tx) */
	USER_TCD_1.DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	USER_TCD_1.DOFF = 0; /* Do not increment destination addr */
	USER_TCD_1.DLAST_SGA = (vuint32_t)&(USER_TCD_0.SADDR); /* After major loop no change to destination addr */
	USER_TCD_1.DMOD = 0; /* Destination modulo feature not used */
	USER_TCD_1.NBYTES = 1; /* Transfer 1 byte per minor loop */
	USER_TCD_1.BITER = 12; /* 12 iterations of major loop */
	USER_TCD_1.CITER = 11; /* Initialize current iteraction count, (start with 2nd data!) */
	USER_TCD_1.D_REQ = 0; /* Disable channel when major loop is done (sending the buffer once) */
	USER_TCD_1.INT_HALF = 0; /* no interrupt in half */
	USER_TCD_1.INT_MAJ = 0; /* no interrupt in major loop completion */
	USER_TCD_1.CITERE_LINK = 0; /* Linking disabled */
	USER_TCD_1.BITERE_LINK = 0;
	USER_TCD_1.BITERLINKCH = 0;
	USER_TCD_1.CITERLINKCH = 0;
	USER_TCD_1.MAJORE_LINK = 0;
	USER_TCD_1.E_SG = 1;
	USER_TCD_1.BWC = 0; /* Default bandwidth control- no stalls */
	USER_TCD_1.START = 0; /* Initialize status flags */
	USER_TCD_1.DONE = 0;
	USER_TCD_1.ACTIVE = 0;

	/* route DMA source to always enabled channel and enable periodic triggers */
	DMAMUX.CHCONFIG[0].R =0xC0 | 0x1F; /* only PIT0 -- always enabled channel */

}

/***************************************************************************//*!
*
* @brief	Transmits the message "Hello World!" to the serial line. After sending
* 			these 12 characters, the transmission is stopped.
*
*	Data transmission is started enabling the DMA channel 0, which transmits one byte.
*			After the transmission, links to the channel 1, which pops other byte from
*			transmitting buffer. After fixed number of itterations of channel 1 (major loop),
*			scatter-gather operation is performed and the user TCD is loaded to channel 1, 
*			which disables the timer, i.e. the transmission. Then, the same way another TCD
*			is loaded to the channel 1 (the original one), to setup the system for another
*			data transmission.
*
*	@ingroup sci_tx
*
******************************************************************************/
static void SciRun05(void)
{
  
	/* SETUP the first data to be sent */
	*pTxData = stringBuffer[0];	/* 8 data bits: PH0 - PH8 */
	SIU.GPDO[111].R = 1; /* STOP bit:  PH9 */
	SIU.GPDO[120].R = 0; /* START bit: PG15 */
 	
 	/* setup the line to logical high -- to get the first start bit right */
 	SIU.GPDO[18].R = 1;

  	PIT.CH[0].TFLG.B.TIF=1;	 /* clear the timer flag */
 	PIT.CH[0].TCTRL.B.TEN=1; /* and now start the timer */

	EDMA.SERQR.R = 0; 		/* start DMA channel 0 */

  for (;;) 
  {
  }
   

}
