/*******************************************************************************
*
*	Freescale Semiconductor Inc.
*	(c) Copyright 2010 Freescale Semiconductor Inc.
*	ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
*	@file sci_app_03.c
*	
*	@version 0.0.3.0
*	
*	@lastmodusr B16958
*	
*	@lastmoddate Jul-1-2010
*	
********************************************************************************
*
*	@brief eDMA emulating simple SCI reciever
*
*	This example demonstates how the eDMA could be used to recieve SCI.
*	Configuration:
*		- DMA channel 0 reads from GPIO PCR_19 (PB3)
*		- DMA channel 0 activated using periodic triggers
*		- DMA channel 1 activates PIT to recieve data
*		- DMA channel 1 activated using eMIOS0_CH0 -- input capture
*		- Connection: Standard wiring on EVB: PB2 and PB3=PA0 -> SCI transciever
*			- need a jumper between PB3 and PA0.
*			
*	@project AN4147
*	
*	@author B16958
*	
*	@ingroup sci_rx
*
******************************************************************************/

/******************************************************************************
* Includes
******************************************************************************/
#include "sci_app_03.h"
#include "MPC560xB.h"


/******************************************************************************
* Internal variable Definitions
******************************************************************************/
static  volatile unsigned char *pRxData=((unsigned char *)&SIU.PGPDO[3].R)+2;
/*!< Pointer to the data which were received using byte by byte DMA movements (deserialization). 
* This is a reference to GPIO parallel access registers */

static  volatile unsigned long int startTimer=(unsigned long int)1; /*< Transfering this
* data to appropriate PIT register will enable the timer and thus the SCi reception may
* start */


/******************************************************************************
* Internal Function Declarations
******************************************************************************/


static void SciInit03(void);

static void SciRun03(void);


/******************************************************************************
* External Function Definitions
******************************************************************************/
extern void SciApp03(void)
{
	SciInit03();
	
	SciRun03();
}

/******************************************************************************
* Internal Function Definitions
******************************************************************************/

/***************************************************************************//*!
*
* @brief Initializes the SCI application to recieve one byte.
*
*	Initialization of the following peripherals:
*		- SIU: PCR registers for DMA UART Rx signals, eMIOS0, and LINFEX UART Tx
*		- PIT_0: To periodically trigger DMA at 9600Hz (one bit per timer overflow)
*		- EMIOS_0: Channel0 -> input capture (as 5607B doesn't route EINT SIU to DMA)
*		- DMAMUX: Routes PIT_0 flag to activate channel 0 DMA, and EMIOS_0,ch0 to activate channel 1 DMA
*		- eDMA0: Setup one byte data movements from DMA UART Rx to PH0-PH7 (data bits)
*		- eDMA1: To start the timer (and data aquisition) after trigger (start bit)
*
*	@ingroup sci_rx
*
******************************************************************************/
static void SciInit03(void)
{
	/* Initialize PCR registers */
	 SIU.PCR[18].B.PA = 1;          /* Pin asigned as LINFLEX0 - UART Tx  */
	 SIU.PCR[18].B.OBE = 1;         /* Output buffer enable */
	 SIU.PCR[19].B.PA = 0;          /* Pin asigned as GPIO for DMA to perform SCI Tx */
	 SIU.PCR[19].B.IBE = 1;         /* Input buffer enable */
	 SIU.PCR[0].B.PA = 1;          /* the same pin --SCI Tx connect also to eMIOS channel */
	 SIU.PCR[0].B.IBE = 1;         /* Input buffer enable */

	/* Initialize LINFlex transmitter (UART, 9.600kb, 8data, 1stop, no parity) */
	LINFLEX_0.LINCR1.B.SLEEP = 0; /* Resets sleep mode */
	LINFLEX_0.LINCR1.B.INIT = 1;   /* Sets initialization mode */
	LINFLEX_0.UARTCR.B.UART = 1; /* Sets UART mode */
	LINFLEX_0.UARTCR.B.WL = 1;   /* 8 data bits */
	LINFLEX_0.UARTCR.B.PCE = 0;    /* Parity controll disabled */
	LINFLEX_0.UARTCR.B.RXEN = 0; /* Disable Receiver */
	LINFLEX_0.UARTCR.B.TXEN = 1; /* Enable Transmitter */
	LINFLEX_0.UARTCR.B.TDFL = 0; /* Sets 1Byte data lenght for Transmitter */
	LINFLEX_0.UARTCR.B.RDFL = 0; /* Sets 1Byte data lenght for Receiver */
	LINFLEX_0.LINIBRR.B.DIV_M = 0x34; /* Clock divisor's mantisa */
	LINFLEX_0.LINFBRR.B.DIV_F = 0; /* Clock divisor's fraction */	
	LINFLEX_0.LINCR1.B.INIT = 0;   /* Resets initialization mode */

	/* Initialize PIT 0 timer for periodic triggers of DMA channel 0 */
 	PIT.PITMCR.B.MDIS = 0;	/* enable the module */
  	PIT.CH[0].LDVAL.R=833;	/* setup 9600@8MHZ overflow rate */
  	PIT.CH[0].TFLG.B.TIF=1; /* clear the flag */

	/* Initialize the eMIOS0_ch0 for single action input capture */
	EMIOS_0.MCR.B.MDIS=0;
	EMIOS_0.MCR.B.GTBE=1; /* global time base out enable*/
	EMIOS_0.MCR.B.GPREN=1; /* enable prescaler */
	EMIOS_0.MCR.B.GPRE=0; /* presclaer 1:1 */
	EMIOS_0.OUDIS.R=0; /* enable all outputs */
	EMIOS_0.UCDIS.R=0; /* enable all channels */
	EMIOS_0.CH[0].A.R=833;
	EMIOS_0.CH[0].CCR.B.UCPRE=0; /* 1:1 */
	EMIOS_0.CH[0].CCR.B.UCPEN=1; /* enable prescaler */
	EMIOS_0.CH[0].CCR.B.DMA=1;	 /* DMA, not interrupt */
	EMIOS_0.CH[0].CCR.B.FCK=1; /* use main clock */
	EMIOS_0.CH[0].CCR.B.FEN=1; /* enable Flag -> DMA */
	EMIOS_0.CH[0].CCR.B.BSL=3; /* all channels internal counter bus */
	EMIOS_0.CH[0].CCR.B.EDSEL=1; /* saoc: output flip-flop toggled */
	EMIOS_0.CH[0].CCR.B.EDPOL=0; /* falling edge */
	EMIOS_0.CH[0].CCR.B.MODE=2; /* Single Action Input Capture */	


	/* Initialize the DMA channel 0 */
	DMAMUX.CHCONFIG[0].R =0x00; /* disable the channel activation source */
	DMAMUX.CHCONFIG[1].R =0x00; /* disable the channel activation source */
	/* Setup the TCD for channel 0 --> single SCI Tx */
	EDMA.TCD[0].SADDR = (vuint32_t)( &(SIU.GPDI[19].R)); /* reading data from PCR 19 (GPIO-Rx) */
	EDMA.TCD[0].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[0].SOFF = 0; /* After transfer, no change */
	EDMA.TCD[0].SLAST = 0; /* After major loop, no change */
	EDMA.TCD[0].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[0].DADDR = (vuint32_t) &(SIU.GPDO[119].R); /* Store data to byte-wise register PH0-PH8(data) */
	EDMA.TCD[0].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[0].DOFF = -1; /* Do decrement destination addr */
	EDMA.TCD[0].DLAST_SGA = 10; /* After major loop +10 bytes to destination addr */
	EDMA.TCD[0].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[0].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[0].BITER = 10; /* 10 iterations of major loop */
	EDMA.TCD[0].CITER = 10; /* Initialize current iteraction count */
	EDMA.TCD[0].D_REQ = 1; /* Keep the channel enabled when major loop is done (sending one byte only) */
	EDMA.TCD[0].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[0].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[0].CITERE_LINK = 0; /* no link after minor loop */
	EDMA.TCD[0].BITERE_LINK = 0;
	EDMA.TCD[0].MAJORE_LINK = 0; /* link disabled */
	EDMA.TCD[0].BITERLINKCH = 0;	
	EDMA.TCD[0].CITERLINKCH = 0;	
	EDMA.TCD[0].MAJORLINKCH = 0;	/* no linkage */
	EDMA.TCD[0].E_SG = 0;
	EDMA.TCD[0].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[0].START = 0; /* Initialize status flags */
	EDMA.TCD[0].DONE = 0;
	EDMA.TCD[0].ACTIVE = 0;
	/* Setup the TCD for channel 1 --> start the SCI Rx - enable the timer */
	EDMA.TCD[1].SADDR = (vuint32_t)(&startTimer); /* Load data from the buffer (start with 2nd element) */
	EDMA.TCD[1].SSIZE = 2; /* Read 2**2 = 4 byte per transfer */
	EDMA.TCD[1].SOFF = 0; /* After transfer no change */
	EDMA.TCD[1].SLAST = 0; /* After major loop no change */
	EDMA.TCD[1].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[1].DADDR = (vuint32_t)(&(PIT.CH[0].TCTRL.R)); /* writing PIT_0 TCTRL to enable (disable) */
	EDMA.TCD[1].DSIZE =  2; /* Write 2**2 = 4 byte per transfer */
	EDMA.TCD[1].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[1].DLAST_SGA = 0; /* After major loop no change to destination addr */
	EDMA.TCD[1].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[1].NBYTES = 4; /* Transfer 4 byte per minor loop */
	EDMA.TCD[1].BITER = 1; /* 12 iterations of major loop */
	EDMA.TCD[1].CITER = 1; /* Initialize current iteraction count, (start with 2nd data!) */
	EDMA.TCD[1].D_REQ = 0; /* Disable channel when major loop is done (sending the buffer once) */
	EDMA.TCD[1].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[1].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[1].CITERE_LINK = 0; /* Linking disabled */
	EDMA.TCD[1].BITERE_LINK = 0;
	EDMA.TCD[1].BITERLINKCH = 0;
	EDMA.TCD[1].CITERLINKCH = 0;
	EDMA.TCD[1].MAJORE_LINK = 0;
	EDMA.TCD[1].E_SG = 0;
	EDMA.TCD[1].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[1].START = 0; /* Initialize status flags */
	EDMA.TCD[1].DONE = 0;
	EDMA.TCD[1].ACTIVE = 0;


	/* route DMA source to always enabled channel and enable periodic triggers */
	DMAMUX.CHCONFIG[0].R =0xC0 | 0x1F; /* only PIT0 -- always enabled channel */
	DMAMUX.CHCONFIG[1].R =0x80 | 0x0D; /* eMIOS0, channel 0 */

}

/***************************************************************************//*!
*
* @brief	Recieves one byte from serial line.
*
*	Data transmission is started by eMIOS channel input capture event - falling edge
*	of the START bit. This triggers DMA channel0, which enables PIT0 and that is used
*	to pace data reception on GPIO pin PB3 (DMA UART Rx).
*	Data recieved are located at PH0-PH8.
*
*	@ingroup sci_rx
*
******************************************************************************/
static void SciRun03(void)
{
  unsigned char recievedData;
  

	EDMA.SERQR.R = 1; 		/* enable DMA channel 1 */
	EDMA.SERQR.R = 0; 		/* enable DMA channel 0 */

  for (;;) 
  {
  	while (EDMA.ERQRL.B.ERQ00==1) {}; /* Wait until data are recieved via DMA */ 
  	recievedData = *pRxData;


	/* now reset the timer */
  	PIT.CH[0].TCTRL.B.TEN=0; /* stop the timer */
  	PIT.CH[0].TFLG.B.TIF=1;	 /* clear the flag */
	/* reset DMA reciever */
	DMAMUX.CHCONFIG[0].R =0x0; /* disable the channel */
	EDMA.SERQR.R = 0; 		/* start DMA channel 0 */
	DMAMUX.CHCONFIG[0].R =0xC0 | 0x1F; /* route back to the PIT_0 */

  	/* send data via normal UART */
  	LINFLEX_0.BDRL.B.DATA0 = recievedData;
  	while ((LINFLEX_0.UARTSR.B.DTF)!=1) {}; /* Wait until a new data is recieved */ 
    LINFLEX_0.UARTSR.B.DTF = 1; /* Reset DTF flag */
  }
   

}
