/***********************************************************************************************\
* Freescale MMA865xQ Driver
*
* Filename: terminal.c
*
*
* (c) Copyright 2011, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
\***********************************************************************************************/

#include "system.h"

/***********************************************************************************************\
* Private macros
\***********************************************************************************************/

/***********************************************************************************************\
* Private type definitions
\***********************************************************************************************/


/***********************************************************************************************\
* Private prototypes
\***********************************************************************************************/

void Check800Hz (void);
void CheckHPF (byte);
byte ProcessHexInput (byte *in, byte *out);
void CopyXYZ (byte *ptr);
void PrintXYZdec8 (void);
void PrintXYZdec10 (void);
void PrintXYZdec12 (void);
void PrintXYZdec14 (void);
void PrintXYZdec (void);
void PrintXYZfrac (void);
void ClearDataFlash (void);
void PrintBuffering (void);
void PrintFIFO (void);
void ReadDataFlashXYZ (void);
void WriteFlashBuffer1 (void);
void WriteFlashBuffer2 (void);

/***********************************************************************************************\
* Private memory declarations
\***********************************************************************************************/

#pragma DATA_SEG __SHORT_SEG _DATA_ZEROPAGE

BIT_FIELD StreamMode;                       // stream mode control flags

extern BIT_FIELD RegisterFlag;
extern byte value[];                        // working value result scratchpad

static byte temp;                           // temporary byte variable
static byte actv;                           // temporary location to store active state
static byte index;                          // input character string index

static tword x_value;                       // X accelerometer value
static tword y_value;                       // Y accelerometer value
static tword z_value;                       // Z accelerometer value

extern byte SlaveAddressIIC;                // accelerometer slave I2C address
static byte deviceID;                       // contents from who_am_i register

extern byte functional_block;               // accelerometer function
                                        
extern byte address_in[3];                  // Data Flash input address pointer
extern byte address_out[3];                 // Data Flash output address pointer

#pragma DATA_SEG DEFAULT

extern tfifo_sample fifo_data[FIFO_BUFFER_SIZE];   // FIFO sample buffer

byte *const Modevalues[] =
{
  "Normal   ", "Low Noise", "High Res ", "Low Power"
};

byte *const ODRvalues [] = 
{
  "800", "400", "200", "100", "50", "12.5", "6.25", "1.56"
};

byte const HPFlookup [4][8] =
{
  { 0, 0, 1, 2, 3, 3, 3, 3 }, // Normal mode
  { 0, 0, 1, 2, 3, 5, 5, 5 }, // Low Noise mode
  { 0, 0, 0, 0, 0, 0, 0, 0 }, // High Res mode
  { 0, 1, 2, 3, 4, 6, 6, 6 }  // low Power mode
};

byte *const HPvalues [] = 
{
  "16", "8", "4", "2", "1", "0.5", "0.25", "0.125", "0.063", "0.031"
};

byte *const GRange [] =
{
  "Standby", "2g", "4g", "8g", "Rsvd"
};


/***********************************************************************************************\
* Public functions
\***********************************************************************************************/

/*********************************************************\
**  Terminal Strings
\*********************************************************/

const byte string_PromptX []      = {"\r\n????????> "};
const byte string_Prompt1 []      = {"\r\nMMA8652Q> "};
const byte string_Prompt2 []      = {"\r\nMMA8653Q> "};
const byte string_What []         = {" <-- what?"};

const byte string_Streaming []    = {" - Streaming XYZ data"};
const byte string_Counts []       = {" as signed counts\r\n"};
const byte string_Gs []           = {" as signed g's\r\n"};
const byte string_Interrupts []   = {" via INT"};
const byte string_FIFO []         = {" via FIFO"};

const byte string_a_N []          = {" a=N Normal data\r\n"};
const byte string_a_NH []         = {" a=N Normal data; a=H HPF data\r\n"};
const byte string_aa_N []         = {" CN=counts Normal, GN=gs Normal\r\n"};
const byte string_aa_NH []        = {" CN=counts Normal, CH=counts HPF, GN=g's Normal, GH=g's HPF\r\n"};

const byte string_warn_banner []     = {"*******************************************************************\r\n"};
const byte string_warn_slow_flash [] = {"* WARNING - The flash memory may not be reliable with 800 Hz data *\r\n"};
const byte string_warn_hpf []        = {"*  WARNING - HPF requires samples to settle after an activation   *\r\n"};
const byte string_prompt_spacer []   = {"            "};

const byte *string_Prompt;
const byte *string_a;
const byte *string_aa;

/*********************************************************\
**  Initialize Terminal Interface
\*********************************************************/
void TerminalInit (void)
{
  SCISendString("\r\n\n**  Freescale Semiconductor  **");
  SCISendString  ("\r\n**  MMA865xQ Demo            **");
  SCISendString  ("\r\n**      using the MC9S08QE8  **");
  SCISendString  ("\r\n**                           **");
  SCISendString  ("\r\n**  "__DATE__"    "__TIME__"  **\r\n\n");
  
  /*
  **  Read Flash.ID : SST25VF032B
  */
  SPI_SS_SELECT;
  SPI_ChrShift(0x9F);
  SPI_ChrShift(0x00);
  SPI_ChrShift(0x00);
  SPI_ChrShift(0x00);
  value[0] = SPI_ChrShiftR(0x00);
  value[1] = SPI_ChrShiftR(0x00);
  value[2] = SPI_ChrShiftR(0x00);
  SPI_SS_DESELECT;
  SCISendString("Data Flash JEDEC ID : Manf=");
  SCI_ByteOut(value[0]);
  SCISendString(" Type=");
  SCI_ByteOut(value[1]);
  SCISendString(" Capacity=");
  SCI_ByteOut(value[2]);
  if (value[0]==0xBF && value[1]==0x25 && value[2]==0x4A)
    SCISendString(" : SST25VF032B");
  else
    SCISendString(" : unknown");
  SCI_putCRLF();
  
  /*
  **  Prepare Data Flash.
  */
//  SCISendString  ("Erasing Data Flash\r");
//  while (SCItxActive);
  DATAFLASH_Unprotect();
  address_in[0] = 0;
  address_in[1] = 0;
  address_in[2] = 0;
  address_out[0] = 0;
  address_out[1] = 0;
  address_out[2] = 0;
  // Flash - Write Status Register - Clears Write Enable Latch
  SPI_SS_SELECT;
  SPI_ChrShift(0x01);
  SPI_ChrShift(0x00);
  SPI_SS_DESELECT;
//  ClearDataFlash();

  functional_block = FBID_MAX;
  StreamMode.Byte = 0;
}

/*********************************************************\
**  Set Terminal Prompt
\*********************************************************/
void TerminalSetPrompt (byte devID)
{
  deviceID = devID;
  switch(deviceID) {
    case MMA8652Q: string_Prompt = string_Prompt1; break;
    case MMA8653Q: string_Prompt = string_Prompt2; break;
    default:       string_Prompt = string_PromptX; break;
  }
  if (deviceID == MMA8653Q) {
    string_a  = string_a_N;
    string_aa = string_aa_N;
  } else {
    string_a  = string_a_NH;
    string_aa = string_aa_NH;
  }
}

/*********************************************************\
**  Process Terminal Interface
\*********************************************************/

void ProcessTerminal (void)
{
  /*
  **  Output command prompt if required.
  */
  if (PROMPT_WAIT == FALSE)
  {
    SCISendString(string_Prompt);
    PROMPT_WAIT = TRUE;
  }
  /*
  **  Get input from terminal.
  */
  if (SCI_INPUT_READY == TRUE)
  {
    INPUT_ERROR = FALSE;
    /*
    **  Use command line input if not streaming data
    */
    if ((XYZ_STREAM == FALSE) && (INT_STREAM == FALSE))
    {
      PROMPT_WAIT = FALSE;
      /*
      **  Get first input character - only uppercase
      */
      switch (BufferRx[0])
      {
        /***************************************************************************************/
        case '?':
          /*
          **  Help : list valid commands
          */
          SCISendString("List of MMA865x commands:\r\n");
          SCISendString("Mn       : Mode 1=Standby; 2=2g; 4=4g; 8=8g\r\n");
          SCISendString("On       : Oversampling 0=Normal; 1=LNLP; 2=HighRes; 3=LP\r\n");
          SCISendString("RO n     : ODR Hz 0=800; 1=400; 2=200; 3=100; 4=50; 5=12.5; 6=6.25; 7=1.56\r\n");
          SCISendString("RR xx    : Register xx Read\r\n");
          SCISendString("RW xx=nn : Register xx Write value nn\r\n");
          if (deviceID != MMA8653Q) {
            SCISendString("RH n     : High Pass Filter  0 - 3, 4=off\r\n");
          }
          SCISendString("RF       : Report ODR speed, HP Filter freq and Mode\r\n");
          SCISendString("C a      : XYZ data as signed counts:");
          SCISendString(string_a);
          SCISendString("G a      : XYZ data as signed g's:");
          SCISendString(string_a);
          SCISendString("S aa     : Stream XYZ:\r\n");
          SCISendString("         :  aa:");
          SCISendString(string_aa);
          SCISendString("I aa n   : Stream XYZ via Interrups\r\n");
          SCISendString("         :  aa:");
          SCISendString(string_aa);
          SCISendString("         :  n:  1=INT1; 2=INT2\r\n");
          if (deviceID==MMA8652Q) {
            SCISendString("F aa ww  : Stream XYZ via FIFO\r\n");
            SCISendString("         :  aa:");
            SCISendString(string_aa);
            SCISendString("         :  ww: Watermark= 1 to 31\r\n");
          }
          break;

        /***************************************************************************************/
        case 'M':
          /*
          **  Control sensor mode of operation
          */
          switch (BufferRx[1])
          {
            /////////////////////////////////////////////////////////////////////////////////////
            case '0':
              /*
              **  M0  : Enter Shutdown
              */
              SCISendString(" - Sensor Shutdown");
              SENSOR_SHUTDOWN;
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case '1':
              /*
              **  M1  : Enter Standby
              */
              SCISendString(" - Sensor Standby");
              SENSOR_ACTIVE;
              (void)MMA865x_Standby();
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case '2':
              /*
              **  M2  : Enter Active 2g
              */
              SCISendString(" - Sensor Active 2g");
              full_scale = FULL_SCALE_2G;
              SENSOR_ACTIVE;
              (void)MMA865x_Standby();
              temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
              temp = (temp & (~FS_MASK)) | full_scale;
              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              MMA865x_Active();
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case '4':
              /*
              **  M4  : Enter Active 4g
              */
              SCISendString(" - Sensor Active 4g");
              full_scale = FULL_SCALE_4G;
              SENSOR_ACTIVE;
              (void)MMA865x_Standby();
              temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
              temp = (temp & (~FS_MASK)) | full_scale;
              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              MMA865x_Active();
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case '8':
              /*
              **  M8  : Enter Active 8g
              */
              SCISendString(" - Sensor Active 8g");
              full_scale = FULL_SCALE_8G;
              SENSOR_ACTIVE;
              (void)MMA865x_Standby();
              temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
              temp = (temp & (~FS_MASK)) | full_scale;
              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              MMA865x_Active();
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            default:
              INPUT_ERROR = TRUE;
              break;
          }
          break;

        /***************************************************************************************/
        case 'O':
          /*
          **  Control sensor oversampling mode
          */
          actv = MMA865x_Standby();
          temp = IIC_RegRead(SlaveAddressIIC, CTRL_REG2);
          switch (BufferRx[1])
          {
            case '0': temp = (temp & ~MODS_MASK) | MOD_NORMAL;    break; // Normal
            case '1': temp = (temp & ~MODS_MASK) | MOD_LOW_NOISE; break; // LNLP
            case '2': temp = (temp & ~MODS_MASK) | MOD_HIGH_RES;  break; // HighRes
            case '3': temp = (temp & ~MODS_MASK) | MOD_LOW_POWER; break; // LP
            default:  INPUT_ERROR = TRUE; break;
          }
          IIC_RegWrite(SlaveAddressIIC, CTRL_REG2, temp);
          if (actv) MMA865x_Active();
          if (INPUT_ERROR != TRUE)
          {
            /*
            **  Echo back both ODR and HP register values
            */
            Print_ODR_HP();
          }
          break;

        /***************************************************************************************/
        case 'R':
          /*
          **  Sensor register access
          */
          switch (BufferRx[1])
          {
            /////////////////////////////////////////////////////////////////////////////////////
            case 'R':
              /*
              **  RR xx  : Register Read
              */
              index = 2;
              /*
              **  Skip past space characters
              */
              while (BufferRx[index] == ' ')
              {
                index++;
              }
              /*
              **  Process hexadecimal register address input
              */
              temp = ProcessHexInput (&BufferRx[index], &value[0]);
              if (temp != 0)
              {
                index += temp;
                /*
                **  Check for end of string null
                */
                if (BufferRx[index++] == 0)
                {
                  SCISendString("= ");
                  /*
                  **  Go read register and report result
                  */
                  temp = IIC_RegRead(SlaveAddressIIC, value[0]);
                  hex2ASCII(temp,&value[0]);
                  value[2] = 0;
                  SCISendString(&value[0]);
                }
                else
                {
                  INPUT_ERROR = TRUE;
                }
              }
              else
              {
                /*
                **  If no register identified, then read all registers
                */
                SCISendString("Read All Registers\r\n");
                for (value[5]=0; value[5]<0x32;)
                {
                  hex2ASCII(value[5],&value[0]);
                  value[2] = ' ';
                  value[3] = '=';
                  value[4] = 0;
                  SCISendString(&value[0]);

                  for (value[4]=4; value[4]!=0 && value[5]<0x32; value[4]--)
                  {
                    value[0] = ' ';
                    temp = IIC_RegRead(SlaveAddressIIC, value[5]++);
                    hex2ASCII(temp,&value[1]);
                    value[3] = 0;
                    SCISendString(&value[0]);
                  }
                  SCI_putCRLF();
                }
              }
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case 'W':
              /*
              **  RW xx = nn  : Register Write
              */
              index = 2;
              /*
              **  Skip past space characters
              */
              while (BufferRx[index] == ' ')
              {
                index++;
              }
              /*
              **  Process hexadecimal register address input
              */
              temp = ProcessHexInput (&BufferRx[index], &value[0]);
              if (temp != 0)
              {
                index += temp;
                /*
                **  Check for "="
                */
                while (BufferRx[index] == ' ')
                {
                  index++;
                }
                if (BufferRx[index++] == '=')
                {
                  while (BufferRx[index] == ' ')
                  {
                    index++;
                  }
                  /*
                  **  Process hexadecimal register data input
                  */
                  temp = ProcessHexInput (&BufferRx[index], &value[1]);
                  if (temp != 0)
                  {
                    index += temp;
                    /*
                    **  Check for end of string null
                    */
                    if (BufferRx[index++] == 0)
                    {
                      /*
                      **  Go write register
                      */
                      temp = MMA865x_Standby();
                      IIC_RegWrite(SlaveAddressIIC, value[0], value[1]);
                      if (temp)
                      {
                        MMA865x_Active();
                      }
                      /*
                      **  Go read register and verify
                      */
                      temp = IIC_RegRead(SlaveAddressIIC, value[0]);
                      if (temp == value[1])
                      {
                        SCISendString(" Success");
                        /*
                        **  Check and flag if ODR was set >= 400 Hz
                        */
                        if (value[0] == CTRL_REG1)
                        {
                          if ((temp & DR_MASK) <= DATA_RATE_400HZ)
                          {
                            USE_FLASH = TRUE;
                          }
                          else
                          {
                            USE_FLASH = FALSE;
                          }
                        }
                      }
                      else
                      {
                        SCISendString(" Fail");
                      }
                    }
                    else
                    {
                      INPUT_ERROR = TRUE;
                    }
                  }
                  else
                  {
                    INPUT_ERROR = TRUE;
                  }
                }
                else
                {
                  INPUT_ERROR = TRUE;
                }
              }
              else
              {
                INPUT_ERROR = TRUE;
              }
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case 'O':
              /*
              **  RO n  : Change Output Data Rate Register
              */
              index = 2;
              /*
              **  Skip past space characters
              */
              while (BufferRx[index] == ' ')
              {
                index++;
              }
              /*
              **  Accept numerical entries from '0' to '7'
              */
              value[0] = BufferRx[index] - '0';
              if (value[0] < 8)
              {
                value[0] <<= 3;
                temp = MMA865x_Standby();
                IIC_RegWrite(SlaveAddressIIC, CTRL_REG1,
                             (IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & (~DR_MASK)) | value[0]);
                if (temp)
                {
                   MMA865x_Active();
                }
                /*
                **  Echo back both ODR and HP register values
                */
                Print_ODR_HP();
              }
              else
              {
                INPUT_ERROR = TRUE;
              }
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case 'H':
              /*
              **  RH n  : Change High Pass Filter Register
              */
              index = 2;
              /*
              **  Skip past space characters
              */
              while (BufferRx[index] == ' ')
              {
                index++;
              }
              /*
              **  Accept numerical entries from '0' to '3'
              */
              value[0] = BufferRx[index] - '0';
              if (deviceID != MMA8653Q && value[0] < 4)
              {
                temp = MMA865x_Standby();
                IIC_RegWrite(SlaveAddressIIC, HP_FILTER_CUTOFF_REG, 
                    (IIC_RegRead(SlaveAddressIIC, HP_FILTER_CUTOFF_REG) & (~SEL_MASK)) | value[0]);
                if (temp)
                {
                   MMA865x_Active();
                }
                /*
                **  Echo back both ODR and HP register values
                */
                Print_ODR_HP();
              }
              else
              {
                INPUT_ERROR = TRUE;
              }
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            case 'F':
              /*
              **  RF  : Report ODR and HP Frequencies
              */
              Print_ODR_HP();
              break;

            /////////////////////////////////////////////////////////////////////////////////////
            default:            
              INPUT_ERROR = TRUE;
              break;
          }
          break;

        /***************************************************************************************/
        case 'C': // Read and display current XYZ data as signed counts
        case 'G': // Read and display current XYZ data as signed g's
          index = 1;
          /*
          **  Skip past space characters
          */
          while (BufferRx[index] == ' ')
          {
            index++;
          }
          /*
          **  Select High pass or normal output data
          */
          temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
          value[0] = temp;
          switch (BufferRx[index])
          {
            case 'N': temp &= ~HPF_OUT_MASK; break; // N = Normal XYZ data
            case 'H':  // H = HPF XYZ data
              if (deviceID != MMA8653Q) // HPF not supported on MMA8653Q
                temp |=  HPF_OUT_MASK;
              else
                INPUT_ERROR = TRUE;
              break;
            case  0: break;                         //   = repeat same type as last reading
            default:  INPUT_ERROR = TRUE; break;
          }
          
          if (INPUT_ERROR != TRUE)
          {
            // Check is the device is in standby mode
            actv = IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK;
          
            /*
            **  Need to put the device into Standby to change the HPF mode
            */
            if (((temp ^ value[0]) & HPF_OUT_MASK))
            {
              /*
              **  Change HPF filter mode
              */
              (void)MMA865x_Standby();
              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              actv = 0;
            }
            
            CheckHPF(1); // Warn if restarting the HPF filter
            
            /*
            **  Activate the device in case it was in Standby or the HPF was updated
            */
            if (!actv)
              MMA865x_Active();
            
            /*
            **  Wait for next sample to be acquired
            */
            while (!(IIC_RegRead(SlaveAddressIIC, STATUS_00_REG) & ZYXDR_MASK))
              ;
            /*
            **  Print out the data in the requested format
            */
            IIC_RegReadN(SlaveAddressIIC, OUT_X_MSB_REG, 6, &value[0]);
            CopyXYZ(&value[0]);
            if (BufferRx[0] == 'G')
              PrintXYZfrac();
            else
              PrintXYZdec();
          }
          break;

        /***************************************************************************************/
        case 'F': //  Stream XYZ data via FIFO interrupts
          if (deviceID != MMA8652Q) // FIFO is only supported for MMA8652Q
            INPUT_ERROR = TRUE;
        case 'S': //  Stream XYZ data
        case 'I': //  Stream XYZ data via interrupts
          StreamMode.Byte = 0;
          /*
          **  Skip past space characters
          */
          index = 1;
          while (BufferRx[index] == ' ')
            index++;

          /*
          **  Select output data format: counts or g's
          */
          switch (BufferRx[index++])
          {
            case 'C': StreamMode.Byte = STREAM_C_MASK; break; // C : Stream as signed counts
            case 'G': StreamMode.Byte = STREAM_G_MASK; break; // G : Stream as signed g's
            default: INPUT_ERROR = TRUE; break;
          }

          /*
          **  Select High pass or normal output data
          */
          if (!INPUT_ERROR)
          {
            temp = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
            value[0] = temp;
            switch (BufferRx[index++])
            {
              case 'N': temp &= ~HPF_OUT_MASK; break; // N = Normal XYZ data
              case 'H':  // H = HPF XYZ data
                if (deviceID != MMA8653Q) // HPF not supported on MMA8653Q
                  temp |=  HPF_OUT_MASK;
                else
                  INPUT_ERROR = TRUE;
                break;
              case  0: break;                         //   = repeat same type as last reading
              default:  INPUT_ERROR = TRUE; break;
            }
          }
          /*
          **  Skip past space characters
          */
          while (BufferRx[index] == ' ')
            index++;
            
          /*
          **  Select Interrupt number for Interrupt based streaming
          */
          if (!INPUT_ERROR && BufferRx[0]=='I')
          {
            /*
            **  Set INT number
            */
            switch (BufferRx[index]) {
              case '1' : value[1] = 0xFD; break; // Direct interrupts to INT1
              case '2' : value[1] = 0x00; break; // Direct interrupts to INT2
              default  : INPUT_ERROR = TRUE; break; // Invalid INT number
            }
          }
                    
          /*
          **  Get the desired FIFO watermark. (defaulting to the current wataermark setting)
          */
          if (!INPUT_ERROR && BufferRx[0]=='F')
          {
            value[1] = IIC_RegRead(SlaveAddressIIC, F_SETUP_REG) & F_WMRK_MASK;
            if (isnum (BufferRx[index]) == TRUE)
            {
              value[1] = BufferRx[index++] - '0';
              if (isnum (BufferRx[index]) == TRUE)
              {
                value[1] *= 10;
                value[1] += (BufferRx[index++] - '0');
              }
            }
            // Check that the watermark value is valid
            if (value[1] > 31)
              INPUT_ERROR = TRUE;
          }
          
          /*
          **  Activate streaming mode if there were no input errors
          */
          if (!INPUT_ERROR)
          {
            // Check is the device is in active or standby mode
            actv = IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK;
            
            /*
            **  Need to reactivate the device if:
            **    - the HPF mode changed
            **    - Interrupt of FIFO mode (because both need to set CTRL registers)
            **    - the FIFO watermark value changed
            */
            if (((temp ^ value[0]) & HPF_OUT_MASK) || (BufferRx[0]!='S'))
            {
              /*
              **  Enter Standby mode
              */
              (void)MMA865x_Standby();
              /*
              **  Change HPF filter mode
              */
              IIC_RegWrite(SlaveAddressIIC, XYZ_DATA_CFG_REG, temp);
              actv = 0;
            }
            
            /*
            **  Display streaming configuration
            */
            SCISendString(string_Streaming);
            if (BufferRx[0]=='I') // using Interrupts
            {  
              SCISendString(string_Interrupts);
              SCI_CharOut(BufferRx[index]);
            } else if (BufferRx[0]=='F') // using the FIFO
              SCISendString(string_FIFO);
            if (STREAM_C == TRUE)
              SCISendString(string_Counts);
            else
              SCISendString(string_Gs);
            
            CheckHPF(0);  // Warn if restarting the HPF filter
            //Check800Hz(); // Warn that 800 Hz data may not be stored correctly in flash
            
            /*
            **  Setup streaming operation
            */
            PROMPT_WAIT = TRUE;
            if (BufferRx[0]=='S') // Streaming XYZ data
            {
              functional_block = FBID_XYZ_SAMPLE;
              XYZ_STREAM  = TRUE;
              POLL_ACTIVE = TRUE;
            }
            else // FIFO and Streaming modes require interrupts
            {
              IIC_RegWrite(SlaveAddressIIC, CTRL_REG3, PP_OD_MASK);
              
              if (BufferRx[0]=='I') // Stream data via Interrupts
              {  
                functional_block = FBID_XYZ_SAMPLE;
                /*
                **  Activate sensor interrupts
                **  - enable Data Ready interrupt
                **  - route Data Ready to selected INT
                */
                IIC_RegWrite(SlaveAddressIIC, CTRL_REG4, INT_EN_DRDY_MASK);
                IIC_RegWrite(SlaveAddressIIC, CTRL_REG5, value[1]);
              }
              else // Stream data via FIFO
              {  
                functional_block = FBID_FIFO;
                /*
                **  Set the FIFO mode to circular
                **  this does not need to be in standby mode unless the watermark changes
                */
                IIC_RegWrite(SlaveAddressIIC, F_SETUP_REG, F_MODE_CIRCULAR | value[1]);
            
                /*
                **  Activate sensor interrupts
                **  - enable FIFO interrupt
                **  - route FIFO interrupt to INT2
                */
                IIC_RegWrite(SlaveAddressIIC, CTRL_REG4, INT_EN_FIFO_MASK);
                IIC_RegWrite(SlaveAddressIIC, CTRL_REG5, 0);

                /*
                **  Reset FIFO group ID
                */
                value[5] = 0;
              }
              /*
              **  Configure the interrupt pins for falling edge and open drain output
              */
              INT_STREAM = TRUE;
              INT_BOTH_FALLING;              
              // Clear and Enable MC9S08 Interrupts
              InterruptsActive ();
            }
            /*
            **  Activate the device in case it was in Standby or the HPF/Watermark changed
            */
            if (!actv)
              MMA865x_Active();
            
          }
          break;


        /***************************************************************************************/
        default:
          /*
          **  Undefined inputs are ignored.
          */
          INPUT_ERROR = TRUE;
          break;
      }
      if (INPUT_ERROR == TRUE)
      {
        SCISendString(string_What);
      }
      SCI_INPUT_READY = FALSE;
      temp = 0;
    }
    /*
    **  Data streaming is stopped by any character input.
    */
    else
    {
      /*
      **  Turn off data streaming
      */
      INT_PINS_DISABLED;
      POLL_ACTIVE = FALSE;
      XYZ_STREAM = FALSE;
      INT_STREAM = FALSE;
      SCI_INPUT_READY = FALSE;
      PROMPT_WAIT = FALSE;

      // Disable the FIFO in case it was using the FIFO
      value[2] = IIC_RegRead(SlaveAddressIIC, F_SETUP_REG) & F_WMRK_MASK;
      IIC_RegWrite(SlaveAddressIIC, F_SETUP_REG, value[2]);
      
      /*
      **  If samples were being buffered in FLASH, output them now.
      */
      if (USE_FLASH == TRUE)
      {
        SCI_putCRLF();
        address_out[0] = 0;
        address_out[1] = 0;
        address_out[2] = 0;
        /*
        **  Read samples stored in Data Flash
        */
        while (USE_FLASH == TRUE)
        {
          ReadDataFlashXYZ();
          /*
          **  Output formats are:
          **    - Stream XYZ data as signed counts
          **    - Stream XYZ data as signed g's
          */
          if (STREAM_C == 1)
            PrintXYZdec();
          else
            PrintXYZfrac();
          SCI_putCRLF();
          /*
          **  Adjust Data Flash address pointer
          */
          if (address_out[2] > 245)
          {
            address_out[2] = 0;
            address_out[1]++;
          }
          else
          {
            address_out[2] += 6;
          }
          /*
          **  Check if we're done.
          */
          if (address_out[0] >= address_in[0])
          {
            if (address_out[1] >= address_in[1])
            {
              if (address_out[2] >= address_in[2])
              {
                USE_FLASH = FALSE;
              }
            }
          }
        }
        /*
        **  Erase the Data Flash
        */
        USE_FLASH = TRUE;
        SCISendString  ("\r\nErasing Data Flash ... Please wait ");
        while (SCItxActive);
        ClearDataFlash();
      }
      functional_block = FBID_MAX;
    }
  }
}


/*********************************************************\
**  Terminal Output
\*********************************************************/
void OutputTerminal (byte BlockID, byte *ptr)
{
  switch (BlockID)
  {
    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_XYZ_SAMPLE:
      /*
      **  XYZ Sample Registers (0x01 - 0x06)
      */
      CopyXYZ(ptr);
      /*
      **  If the ODR >= 400Hz, buffer the samples in the Data Flash.
      */
      if (USE_FLASH == TRUE)
      {
        /*
        **  Provide a visual indication that we're buffering and write to Flash
        */
        WriteFlashBuffer1(); // First half of write to flash
        PrintBuffering();
        WriteFlashBuffer2(); // Second half of write to flash
      }
      else
      {
        /*
        **  Output formats are:
        **    - Stream XYZ data as signed counts
        **    - Stream XYZ data as signed g's
        */
        if (STREAM_C == 1)
          PrintXYZdec();
        else
          PrintXYZfrac();
        SCI_putCRLF();
      }
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_LANDSCAPE_PORTRAIT:
      /*
      **  Landscape/Portrait Orientation Status Register (0x18)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_FREEFALL_MOTION_1:
      /*
      **  Freefall/Motion 1 Source Register (0x24)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_TRANSIENT:
      /*
      **  Transient Source Register (0x2C)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_PULSE:
      /*
      **  Pulse Source Register (0x30)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_SYSTEM_MODE:
      /*
      **  System Mode Register (0x14)
      */
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    case FBID_FIFO:
      /*
      **  FIFO
      **
      **  If the ODR >= 400Hz, buffer the samples in the Data Flash.
      */
      if (USE_FLASH == TRUE)
      {
        /*
        **  Provide a visual indication that we're buffering
        */
        PrintBuffering();        
        /*
        **  Save the samples in the FIFO buffer into the Data Flash
        */
        value[0] = RegisterFlag.Byte & F_CNT_MASK;
        for (value[1]=0; value[1]!=value[0]; value[1]++)
        {
          /*
          **  Save sample to Data Flash
          */
          CopyXYZ(&fifo_data[value[1]].Sample.XYZ.x_msb);
          WriteFlashBuffer1(); // First half of write to flash
          WriteFlashBuffer2(); // Second half of write to flash
        }
      }
      else
      {
        PrintFIFO();
      }
      break;

    /////////////////////////////////////////////////////////////////////////////////////////////////
    default:
      break;
  }
}


/***********************************************************************************************\
* Private functions
\***********************************************************************************************/

/*********************************************************\
* Print a warning if trying to store data in Flash at 800 Hz
\*********************************************************/
void Check800Hz (void)
{
  /*
  **  Streaming mode at 800 Hz is unreliable because the flash is too slow
  */
  if ((IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & DR_MASK) == DATA_RATE_800HZ) {
    SCI_putCRLF();
    SCISendString(string_warn_banner);
    SCISendString(string_warn_slow_flash);
    SCISendString(string_warn_banner);
    SCI_putCRLF();
  }
}

/*********************************************************\
* Print a warning if the HPF is initializing
\*********************************************************/
void CheckHPF (byte spacer)
{
  /*
  **  HPF starts near normal XYZ data values after an activate() and then
  **  takes a number of samples to settle to the correct value
  */
  if (((IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK) == 0) &&
      ((IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG) & HPF_OUT_MASK)) )       
  {
    SCI_putCRLF();
    SCISendString(string_warn_banner);
    SCISendString(string_warn_hpf);
    SCISendString(string_warn_banner);
    SCI_putCRLF();
    if (spacer)
      SCISendString(string_prompt_spacer);  
  }
}

/*********************************************************\
* Print accelerometer's OverSample, ODR, HP and Mode
\*********************************************************/
void Print_ODR_HP (void)
{
  SCISendString("OverSample = ");
  value[0] = IIC_RegRead(SlaveAddressIIC, CTRL_REG2) & MODS_MASK;
  SCISendString(Modevalues[value[0]]);
  SCISendString("   ");
  
  SCISendString("ODR = ");
  value[1] = IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & DR_MASK;
  /*
  **  Use onboard flash to store XYZ data if ODR >= 400 Hz
  */
  if (value[1] <= DATA_RATE_400HZ)
  {
    USE_FLASH = TRUE;
  }
  else
  {
    USE_FLASH = FALSE;
  }
  value[1] >>= 3;
  SCISendString(ODRvalues[value[1]]);
  SCISendString("Hz   ");
  
  SCISendString("HP = ");
  if (deviceID == MMA8653Q) // HPF is not supportted in MMA8653Q
    SCISendString("--");
  else {
    value[2] = HPFlookup[value[0]][value[1]];
    value[2] += IIC_RegRead(SlaveAddressIIC, HP_FILTER_CUTOFF_REG) & SEL_MASK;
    SCISendString(HPvalues[value[2]]);
  }
  SCISendString("Hz   ");
  
  SCISendString("Mode = ");
  value[3] = IIC_RegRead(SlaveAddressIIC, CTRL_REG1) & ACTIVE_MASK;
  if (value[3])
  {
    value[3] = IIC_RegRead(SlaveAddressIIC, XYZ_DATA_CFG_REG);
    value[3] &= FS_MASK;
    value[3]++;
  }
  SCISendString(GRange[value[3]]);
}


/*********************************************************\
*
\*********************************************************/
void WriteFlashBuffer1 (void)
{
  /*
  **  Enable SO as a busy indicator
  */
  SPI_SS_SELECT;
  SPI_ChrShift(0x70);
  SPI_SS_DESELECT;
  /*
  **  Start multibyte write to Data Flash
  */
  DATAFLASH_WriteEnableLatch();
  SPI_SS_SELECT;
  SPI_ChrShift(0xAD);
  SPI_ChrShift(address_in[0]);
  SPI_ChrShift(address_in[1]);
  SPI_ChrShift(address_in[2]);
  SPI_ChrShift(x_value.Byte.hi);
  SPI_ChrShift(x_value.Byte.lo);
  SPI_SS_DESELECT;
  /*
  **  Wait for completion of the first write
  */
  SPI_SS_SELECT;
  while (SPI_SO==0);
  SPI_SS_DESELECT;
  /*
  **  Send next 2 bytes
  */
  SPI_SS_SELECT;
  SPI_ChrShift(0xAD);
  SPI_ChrShift(y_value.Byte.hi);
  SPI_ChrShift(y_value.Byte.lo);
  SPI_SS_DESELECT;
}


/*********************************************************\
*
\*********************************************************/
void WriteFlashBuffer2 (void)
{
  /*
  **  Wait for completion of the first write
  */
  SPI_SS_SELECT;
  while (SPI_SO==0); // Wait for the Flash to complete
  SPI_SS_DESELECT;
  /*
  **  Send last 2 bytes
  */
  SPI_SS_SELECT;
  SPI_ChrShift(0xAD);
  SPI_ChrShift(z_value.Byte.hi);
  SPI_ChrShift(z_value.Byte.lo);
  SPI_SS_DESELECT;
  /*
  **  Adjust Data Flash address pointer
  */
  if (address_in[2] > 245)
  {
    address_in[2] = 0;
    address_in[1]++;
  }
  else
  {
    address_in[2] += 6;
  }
  /*
  **  Wait for completion of the first write
  */
  SPI_SS_SELECT;
  while (SPI_SO==0); // Wait for the Flash to complete
  SPI_SS_DESELECT;
  /*
  **  Send WRDI to end the previous Write
  */
  SPI_SS_SELECT;
  SPI_ChrShift(0x04);
  SPI_SS_DESELECT;
  /*
  **  Disable SO as a busy indicator
  */
  SPI_SS_SELECT;
  SPI_ChrShift(0x80);
  SPI_SS_DESELECT;
}


/*********************************************************\
*
\*********************************************************/
void PrintBuffering (void)
{
  if (temp > 50)
  {
    SCI_putCRLF();
    temp = 0;
    address_out[2] += 50;
    if (address_out[2] < 50)
    {
      address_out[1]++;
      if (address_out[1] == 0)
      {
        address_out[0]++;
      }
    }
  }
  if (temp++ == 0)
  {
    hex2ASCII(address_out[0],&value[0]);
    hex2ASCII(address_out[1],&value[2]);
    value[4] = 0;
    SCISendString(&value[0]);
    hex2ASCII(address_out[2],&value[0]);
    value[2] = 0;
    SCISendString(&value[0]);
  }
  SCI_CharOut('*');
}


/*********************************************************\
*
\*********************************************************/
void PrintFIFO (void)
{
  /*
  **  Identify if this is either a FIFO Overflow or Watermark event
  */
  if (RegisterFlag.F_OVF_BIT == 1)
  {
    SCISendString  ("FIFO Overflow ");
  }
  if (RegisterFlag.F_WMRK_FLAG_BIT == 1)
  {
    SCISendString  ("FIFO Watermark ");
  }
  /*
  **  Display FIFO count
  */
  value[0] = RegisterFlag.Byte & F_CNT_MASK;
  SCISendString("Samples= ");
  SCI_Dec_Out(value[0],0);
  /*
  **  Identify FIFO group
  */
  SCISendString (" group= ");
  hex2ASCII(value[5],&value[1]);
  value[3] = 0;
  value[5]++;
  SCISendString(&value[1]);
  SCI_putCRLF();
  /*
  **  Output results
  */
  for (value[1]=0; value[1]!=value[0]; value[1]++)
  {
    CopyXYZ(&fifo_data[value[1]].Sample.XYZ.x_msb);
    /*
    **  Output formats are:
    **    - Stream XYZ data as signed counts
    **    - Stream XYZ data as signed g's
    */
    if (STREAM_C == 1)
    {
      PrintXYZdec();
    }
    else
    {
      PrintXYZfrac();
    }
    SCI_putCRLF();
  }
}


/*********************************************************\
*
\*********************************************************/
void ReadDataFlashXYZ (void)
{
  SPI_SS_SELECT;
  SPI_ChrShift(0x0B);
  SPI_ChrShift(address_out[0]);
  SPI_ChrShift(address_out[1]);
  SPI_ChrShift(address_out[2]);
  SPI_ChrShift(0x00);
  x_value.Byte.hi = SPI_ChrShiftR(0x00);
  x_value.Byte.lo = SPI_ChrShiftR(0x00);
  y_value.Byte.hi = SPI_ChrShiftR(0x00);
  y_value.Byte.lo = SPI_ChrShiftR(0x00);
  z_value.Byte.hi = SPI_ChrShiftR(0x00);
  z_value.Byte.lo = SPI_ChrShiftR(0x00);
  SPI_SS_DESELECT;
}


/*********************************************************\
*
\*********************************************************/
byte ProcessHexInput (byte *in, byte *out)
{
  byte data;

  data = *in++;
  if (ishex(data) == TRUE)
  {
    data = tohex(data);
  }
  else
  {
    return (0);
  }
  if (ishex(*in) == TRUE)
  {
    data <<= 4;
    data += tohex(*in);
    *out = data;
    return (2);
  }
  else if ((*in == ' ') || (*in == 0))
  {
    *out = data;
    return (1);
  }
  return (0);
}


/*********************************************************\
*
\*********************************************************/
void CopyXYZ (byte *ptr)
{
  x_value.Byte.hi = *ptr++;
  x_value.Byte.lo = *ptr++;
  y_value.Byte.hi = *ptr++;
  y_value.Byte.lo = *ptr++;
  z_value.Byte.hi = *ptr++;
  z_value.Byte.lo = *ptr;       
}


/*********************************************************\
*
\*********************************************************/
void PrintXYZdec8 (void)
{
  SCISendString("X= ");
  SCI_s8dec_Out(x_value);
  SCISendString("  Y= ");
  SCI_s8dec_Out(y_value);
  SCISendString("  Z= ");
  SCI_s8dec_Out(z_value);
}


/*********************************************************\
*
\*********************************************************/
void PrintXYZdec10 (void)
{
  SCISendString("X= ");
  SCI_s10dec_Out(x_value);
  SCISendString("  Y= ");
  SCI_s10dec_Out(y_value);
  SCISendString("  Z= ");
  SCI_s10dec_Out(z_value);
}


/*********************************************************\
*
\*********************************************************/
void PrintXYZdec12 (void)
{
  SCISendString("X= ");
  SCI_s12dec_Out(x_value);
  SCISendString("  Y= ");
  SCI_s12dec_Out(y_value);
  SCISendString("  Z= ");
  SCI_s12dec_Out(z_value);
}


/*********************************************************\
*
\*********************************************************/
void PrintXYZdec14 (void)
{
  SCISendString("X= ");
  SCI_s14dec_Out(x_value);
  SCISendString("; Y=");
  SCI_s14dec_Out(y_value);
  SCISendString("; Z=");
  SCI_s14dec_Out(z_value);
}


/*********************************************************\
*
\*********************************************************/
void PrintXYZdec (void)
{
  switch (deviceID) {
    case MMA8652Q: PrintXYZdec12(); break;
    case MMA8653Q: PrintXYZdec10(); break;
    default:       PrintXYZdec10(); break;
  }
}


/*********************************************************\
*
\*********************************************************/
void PrintXYZfrac (void)
{
  SCISendString("X= ");
  SCI_s14frac_Out(x_value);
  SCISendString("  Y= ");
  SCI_s14frac_Out(y_value);
  SCISendString("  Z= ");
  SCI_s14frac_Out(z_value);
}


/*********************************************************\
*
\*********************************************************/
void ClearDataFlash (void)
{
  address_in[0] = 0;
  address_in[1] = 0;
  address_in[2] = 0;
  address_out[0] = 0;
  address_out[1] = 0;
  address_out[2] = 0;
  DATAFLASH_Erase();
}

