/******************************************************************************		
*                                                                                   	
*       Copyright (C) 2009 Freescale Semiconductor, Inc.                            	
*       All Rights Reserved								              				
*														              					
* Filename:       ADC16.c               											
*														              					
* Description:	  16-bit ADC Driver file for MCF51EM256	        	    	  	       						
*																						
* Notes:        																		
******************************************************************************/		

/* include files */
#include "MCF51EM256.H" /* include peripheral declarations */
#include "ADC16.h"    

tADC *ADCPointer; /* Global Variable indicating which ADC is in use */

/******************************************************************************
Function Name	:	ADC_Init
Engineer      :	r54940
Date          :	11/09/09

Parameters		:	
Returns			  :	NONE
Notes         : Initialises ADC (includes Calibration)
******************************************************************************/
void ADC_Init(tADC *adc_ptr, unsigned char CONFIG1, unsigned char CONFIG2, unsigned short COMPAREA, unsigned short COMPAREB, unsigned char STATUS2, unsigned char STATUS3, unsigned char STATUS1A, unsigned char STATUS1B, unsigned char CHA,unsigned char CHB)
{
  ADC_Config(adc_ptr, CONFIG1, CONFIG2, COMPAREA, COMPAREB, STATUS2, STATUS3, STATUS1A, STATUS1B);
  ADC_Cal (adc_ptr);
  ADCPointer->ADCSC1A.Bits.ADCHA = CHA;
  ADCPointer->ADCSC1B.Bits.ADCHB = CHB;
  
}

/******************************************************************************
Function Name	:	ADC_Config
Engineer      :	r54940
Date          :	11/09/09

Parameters		:	
Returns			  :	NONE
Notes         :	Configures ADC
******************************************************************************/
void ADC_Config(tADC *adc_ptr, unsigned char CONFIG1, unsigned char CONFIG2, unsigned short COMPAREA, unsigned short COMPAREB, unsigned char STATUS2, unsigned char STATUS3, unsigned char STATUS1A, unsigned char STATUS1B)
{
  adc_ptr->ADCCFG1.Byte = CONFIG1;
  adc_ptr->ADCCFG2.Byte = CONFIG2;
  adc_ptr->ADCCVA.Word = COMPAREA;
  adc_ptr->ADCCVB.Word = COMPAREB;
  adc_ptr->ADCSC2.Byte = STATUS2;
  adc_ptr->ADCSC3.Byte = STATUS3;
  adc_ptr->ADCSC1A.Byte = STATUS1A;       /* Leave ADCH = 0x1F */
  adc_ptr->ADCSC1B.Byte = STATUS1B;       /* Leave ADCH = 0x1F */
}
 
/******************************************************************************
Function Name	:	ADC_Cal
Engineer      :	r54940
Date          : 11/09/09

Parameters		:	
Returns			  :	NONE
Notes         :	Calibrates the ADC. Required after reset and before a conversion is initiated
******************************************************************************/
void ADC_Cal(tADC *adc_ptr)
{

  volatile unsigned short cal_var;

  adc_ptr->ADCSC3.Bits.AVGE = 1;            /* Enable Averaging for Calibration Process */
  adc_ptr->ADCSC2.Bits.ADTRG = 0;           /* Enable Software Conversion Trigger for Calibration Process */
  adc_ptr->ADCSC3.Bits.CAL = 1;             /* Start CAL */

  while (ADCPointer->ADCSC1A.Bits.COCOA == 0)
  	{
  	/* Wait for conversion to complete */
  	}	

  if (adc_ptr->ADCSC3.Bits.CALF == 1)     /* Check for Calibration fail error */
    {
    adc_ptr->ADCSC3.Bits.CALF = CALF_FAIL;
    //while(1);
    }

  /* Calculate plus-side calibration as per 21.4.7 */
  cal_var = 0x00;

  cal_var += (adc_ptr->ADCCLP.CLP0.Bits.CLP0) + (adc_ptr->ADCCLP.CLP1.Bits.CLP1) + (adc_ptr->ADCCLP.CLP2.Bits.CLP2) + 
  (adc_ptr->ADCCLP.CLP3.Bits.CLP3) + (adc_ptr->ADCCLP.CLP4.Bits.CLP4) + (adc_ptr->ADCCLP.CLPS.Bits.CLPS);

  cal_var = cal_var/2;
  
  cal_var += 0x8000; // Set MSB

  adc_ptr->ADCPG.Word = cal_var;

  /* Calculate minus-side calibration as per 21.4.7 */
  cal_var = 0x00;

  cal_var += (adc_ptr->ADCCLM.CLM0.Bits.CLM0) + (adc_ptr->ADCCLM.CLM1.Bits.CLM1) + (adc_ptr->ADCCLM.CLM2.Bits.CLM2) + 
  (adc_ptr->ADCCLM.CLM3.Bits.CLM3) + (adc_ptr->ADCCLM.CLM4.Bits.CLM4) + (adc_ptr->ADCCLM.CLMS.Bits.CLMS);

  cal_var = cal_var/2;

  cal_var += 0x8000; // Set MSB

  adc_ptr->ADCMG.Word = cal_var;

  /* Clear CAL bit */
  adc_ptr->ADCSC3.Bits.CAL = 0;
  adc_ptr->ADCSC3.Bits.AVGE = 0;            /* Disable Averaging for main program */
  adc_ptr->ADCSC2.Bits.ADTRG = 1;           /* Enable Hardware Conversion Trigger for main program */
  

}
