/***************************************************************************************************** 
   u8SCI_init 
______________________________________________________________________________________________________
input parameters:   port          char  (1, 2, 3)
                    baudrate      long  (0-115200)
                    data length   char  (8=0, 9!=0)
                    parity        char  (none=0, even=1, odd=2)
                    TX mode       char  (disabled=0, not inverted=1, inverted = 2)
                    RX mode       char  (disabled=0, not inverted=1, inverted = 2)
       
output parameters:  char  (OK=0)
                          (invalid port=1)
                          (baudrate error=2)
                          (parity error=3)
                          (TXmode error=4)
                          (RXmode error=5)
                                              
***check it!!***    Busclock must be defined as SYSTEM_CLOCK                         
                    Interrupt events must be handled by the user              
*****************************************************************************************************/

#include "mcf5xxx.h"
#include "MCF51EM256.h"
#include "m51em256demo.h"
#include "infrared.h"

UINT8 u8SCI_init(UINT8 port, UINT32 baudrate, UINT8 length, UINT8 parity, UINT8 TXmode, UINT8 RXmode) 
  {
  
    if (0<RXmode>2) return 5;
    if (0<TXmode>2) return 4;
    if (0<parity>2) return 3;
    if (1<baudrate>115200) return 2;
    if (u8Set_baudrate(port, baudrate)) return 1;
    
    
   switch (port) 
   {
    case 1: 
    {
      //setting data length
      if (!length) SCI1C1_M = 0;
      else SCI1C1_M = 1;    
    
      //setting parity
      switch (parity) 
      {
        case 0: 
        {
          SCI1C1_PE = 0;  //parity disabled  
        }break;
        
        case 1: 
        {
          SCI1C1_PE = 1; //parity enabled
          SCI1C1_PT = 0; //even parity  
        }break;
        
        case 2: 
        {
          SCI1C1_PE = 1;  //parity enabled
          SCI1C1_PT = 1;  //odd parity    
        }break;
      }
    
      //transmission data mode
      switch (TXmode) 
      {
        case 0: 
        {
          SCI1C2_TE = 0;      //tx disabled
        }break;
        
        case 1: 
        {
          SCI1C2_TE = 1;      //tx enabled
          SCI1C3_TXINV = 0;   //tx not inverted
        }break;
        
        case 2: 
        {
          SCI1C2_TE = 1;      //tx enabled
          SCI1C3_TXINV = 1;   //tx inverted
        }break;
      }
      
      //reception data mode
      switch (RXmode) 
      {
        case 0: 
        {
          SCI1C2_RE = 0;      //rx disabled
        }break;
        
        case 1: 
        {
          SCI1C2_RE = 1;      //rx enabled
          SCI1S2_RXINV = 0;   //rx not inverted
        }break;
        
        case 2: 
        {
          SCI1C2_RE = 1;      //rx enabled
          SCI1S2_RXINV = 1;   //rx inverted
        }break;
      }
      
      //starting SCI1 clock
      SCGC1_SCI1 = 1;
    
    }break;

    
    case 2: 
    {
      //setting data length
      if (!length) SCI2C1_M = 0;
      else SCI2C1_M = 1;    
    
      //setting parity
      switch (parity) 
      {
        case 0: 
        {
          SCI2C1_PE = 0;  //parity disabled  
        }break;
        
        case 1: 
        {
          SCI2C1_PE = 1; //parity enabled
          SCI2C1_PT = 0; //even parity  
        }break;
        
        case 2: 
        {
          SCI2C1_PE = 1;  //parity enabled
          SCI2C1_PT = 1;  //odd parity    
        }break;
      }
    
      //transmission data mode
      switch (TXmode) 
      {
        case 0: 
        {
          SCI2C2_TE = 0;      //tx disabled
        }break;
        
        case 1: 
        {
          SCI2C2_TE = 1;      //tx enabled
          SCI2C3_TXINV = 0;   //tx not inverted
        }break;
        
        case 2: 
        {
          SCI2C2_TE = 1;      //tx enabled
          SCI2C3_TXINV = 1;   //tx inverted
        }break;
      }
      
      //reception data mode
      switch (RXmode) 
      {
        case 0: 
        {
          SCI2C2_RE = 0;      //rx disabled
        }break;
        
        case 1: 
        {
          SCI2C2_RE = 1;      //rx enabled
          SCI2S2_RXINV = 0;   //rx not inverted
        }break;
        
        case 2: 
        {
          SCI2C2_RE = 1;      //rx enabled
          SCI2S2_RXINV = 1;   //rx inverted
        }break;
      }
      
      //starting SCI2 clock
      SCGC1_SCI2 = 1;
    
    }break;
    
    
    case 3: 
    {
      //setting data length
      if (!length) SCI3C1_M = 0;
      else SCI3C1_M = 1;    
    
      //setting parity
      switch (parity) 
      {
        case 0: 
        {
          SCI3C1_PE = 0;  //parity disabled  
        }break;
        
        case 1: 
        {
          SCI3C1_PE = 1; //parity enabled
          SCI3C1_PT = 0; //even parity  
        }break;
        
        case 2: 
        {
          SCI3C1_PE = 1;  //parity enabled
          SCI3C1_PT = 1;  //odd parity    
        }break;
      }
    
      //transmission data mode
      switch (TXmode) 
      {
        case 0: 
        {
          SCI3C2_TE = 0;      //tx disabled
        }break;
        
        case 1: 
        {
          SCI3C2_TE = 1;      //tx enabled
          SCI3C3_TXINV = 0;   //tx not inverted
        }break;
        
        case 2: 
        {
          SCI3C2_TE = 1;      //tx enabled
          SCI3C3_TXINV = 1;   //tx inverted
        }break;
      }
      
      //reception data mode
      switch (RXmode) 
      {
        case 0: 
        {
          SCI3C2_RE = 0;      //rx disabled
        }break;
        
        case 1: 
        {
          SCI3C2_RE = 1;      //rx enabled
          SCI3S2_RXINV = 0;   //rx not inverted
        }break;
        
        case 2: 
        {
          SCI3C2_RE = 1;      //rx enabled
          SCI3S2_RXINV = 1;   //rx inverted
        }break;
      }
                    
      /*starting SCI3 clock*/
      SCGC1_SCI3 = 1;
        
    }break;    
   }
   return 0;                  //selection ok
  }

/*--------------------------------------------------------------------------------------------------*/ 

UINT8 u8Set_baudrate(UINT8 port, UINT32 baudrate) 
{
  //preprocessing to set baudrate
   UINT16 baud_divisor;
   baud_divisor = (UINT16)((BUS_CLOCK)/(baudrate * 16));
  
   if (1<port>3) return 1;                    //port error
   
   if (port==1) SCI1BD = baud_divisor;
   else if (port==2) SCI2BD = baud_divisor;
   else if (port==3) SCI3BD = baud_divisor;
   return 0;                                  //selection ok
}

/*--------------------------------------------------------------------------------------------------*/


/***************************************************************************************************** 
   u8IR_TX_init 
______________________________________________________________________________________________________
input parameters:   port            char  (1, 2, 3)
                    tx pin option   char (default=0, altarnative!=0)
                    *=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
                    =  SCI1: default pin.- PTB3 / alternative pin.- none  =
                    *  SCI2: default pin.- PTB1 / alternative pin.- PTE6  *
                    =  SCI3: default pin.- PTC1 / alternative pin.- PTF3  =
                    *=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
                    drive strength  char (no=0, yes=1, toggle=2)
                    mod source      char (none=0, tpmch0=1, tpmch1=2, mtim2=3, mtim3=4)
                    tx int enable   char (disabled=0, enabled!=0)
                    ****global interruptions must be enabled!****

                    ***** modulation frequency (timer config) is not included in this driver. 
                          User must define it *****

output parameters:  char  (OK=0)
                          (invalid port=1)
                          (invalid modulation source=2)
                          (invalid selection=3) 
*****************************************************************************************************/


UINT8 u8IR_TX_init(UINT8 port, UINT8 tx_pin_option, UINT8 strength, UINT8 modulation, UINT8 tx_ie)
  {
    //static UINT8 A;
    
    if (1<port>3) return 1;
    //A=u8modulation_set(port, modulation); 
    if (u8modulation_set(port, modulation)) return 2;
    if (u8drive_strength_set(port, tx_pin_option, strength)) return 3;
      
    switch (port)
      {
        case 1:
        {
          //pin function configuration
          if(!tx_pin_option) PTBPF2_B3 = 3; //PTB3 as TX1 (default option)
          else return 3;                    //Error because SCI1 pin options are only in PTB
          
          //enabling transmission interrupt
          if (tx_ie) SCI1C2_TIE = 1;
          else SCI1C2_TIE = 0;
               
        }break;
        
        case 2:
        {
          //pin function configuration
          if(!tx_pin_option) PTBPF2_B1 = 3; //PTB1 as TX2 (default option)
          else PTEPF1_E6 = 3;               //PTE6 as TX2 (alternative option)
                               
          //enabling transmission interrupt
          if (tx_ie) SCI2C2_TIE = 1;
          else SCI2C2_TIE = 0;
          
        }break;
        
        case 3:
        {
          //pin function configuration
          if(!tx_pin_option) PTCPF2_C1 = 3; //PTC1 as TX3 (default option)
          else                              //PTF3 as TX3 (alternative option)
            {                               //and NOT using the LCD
              LCDSUPPLY_VSUPPLY = 0b11;
              LCDRVC_RVEN = 0; //regulator voltage disabled
              LCDC1_FCDEN = 1;
              
              PTFPF2_F3 = 3;               
            }
          
          //enabling transmission interrupt
          if (tx_ie) SCI3C2_TIE = 1;
          else SCI3C2_TIE = 0;
            
        }break;
        
      }
      return 0;
  }

/*--------------------------------------------------------------------------------------------------*/

UINT8 u8drive_strength_set(UINT8 port, UINT8 tx_pin_option, UINT8 strength)
{
   if (1<port>3) return 1;                    //port error
   if (0<strength>2) return 3;                //selection error   
          
   switch (port)
   {
    case 1:
    {
      switch (strength)
      {
        case 0:
        {
           if(!tx_pin_option) PTBDS_PTBDS3 = 0;               
           //Drive Strength on PTB3 disabled
           else return 3;                  
           //SCI TX1 has only one pin option
        }break;
        
        case 1:
        {
           if(!tx_pin_option) PTBDS_PTBDS3 = 1;               
           //Drive Strength on PTB3 enabled
           else return 3;                  
           //SCI TX1 has only one pin option
        }break;
        
        case 2:
        {
           if(!tx_pin_option) PTBDS_PTBDS3 = ~PTBDS_PTBDS3;               
           //Drive Strength on PTB3 toggled
           else return 3;                  
           //SCI TX1 has only one pin option
        }break;
      }
    }break;
    
    case 2:
    {
      switch (strength)
      {
        case 0:
        {
           if(!tx_pin_option) PTBDS_PTBDS1 = 0;               
           //Drive Strength on PTB1 disabled
           else return 3;                  
           //SCI TX2 alternate pin (PTE6) does not support Drive Strength
        }break;
        
        case 1:
        {
           if(!tx_pin_option) PTBDS_PTBDS1 = 1;               
           //Drive Strength on PTB1 enabled
           else return 3;                  
           //SCI TX2 alternate pin (PTE6) does not support Drive Strength
        }break;
        
        case 2:
        {
           if(!tx_pin_option) PTBDS_PTBDS1 = ~PTBDS_PTBDS1;               
           //Drive Strength on PTB1 toggled
           else return 3;                  
           //SCI TX2 alternate pin (PTE6) does not support Drive Strength
        }break;
      }
    }break;
    
    case 3:
    {
      switch (strength)
      {
        case 0:
        {
           if(!tx_pin_option) PTCDS_PTCDS1 = 0;               
           //Drive Strength on PTC1 disabled
           else PTFDS_PTFDS3 = 0;               
           //Drive Strength on PTF3 disabled
        }break;
        
        case 1:
        {
           if(!tx_pin_option) PTCDS_PTCDS1 = 1;               
           //Drive Strength on PTC1 enabled
           else PTFDS_PTFDS3 = 1;               
           //Drive Strength on PTF3 enabled
        }break;
        
        case 2:
        {
           if(!tx_pin_option) PTCDS_PTCDS1 = ~PTCDS_PTCDS1;               
           //Drive Strength on PTC1 toggled
           else PTFDS_PTFDS3 = ~PTCDS_PTCDS1;               
           //Drive Strength on PTF3 toggled
        }break;
      }
    }break;
   
   }
   return 0;
   //selection ok
}

/*--------------------------------------------------------------------------------------------------*/

UINT8 u8modulation_set(UINT8 port, UINT8 modulation)
{
   if (1<port>3) return 1;                    //port error 
   if (0<modulation>4) return 2;              //invalid modulation source
   
   switch (port)
   {
    case 1:
    {
      switch (modulation)
      {
        case 0:
        {
          SIMIPS2_MODTX1 = 0;    //modulation disabled
        }break;
            
        case 1:
        {
          SIMIPS2_MTBASE1 = 0;   //SCI1 TX modulated by TPM ch0
          SIMIPS2_MODTX1 = 1;
        }break;
            
        case 2:
        {
          SIMIPS2_MTBASE1 = 1;   //SCI1 TX modulated by TPM ch1
          SIMIPS2_MODTX1 = 1;
        }break;
            
        case 3:
        {
          SIMIPS2_MTBASE1 = 2;   //SCI1 TX modulated by MTIM2
          SIMIPS2_MODTX1 = 1;
        }break;
            
        case 4:
        {
          SIMIPS2_MTBASE1 = 3;   //SCI1 TX modulated by MTIM3
          SIMIPS2_MODTX1 = 1;
        }break;
      }
    }break;
    
    case 2:
    {
      switch (modulation)
      {
        case 0:
        {
          SIMIPS2_MODTX2 = 0;    //modulation disabled
        }break;
            
        case 1:
        {
          SIMIPS2_MTBASE2 = 0;   //SCI2 TX modulated by TPM ch0
          SIMIPS2_MODTX2 = 1;
        }break;
            
        case 2:
        {
          SIMIPS2_MTBASE2 = 1;   //SCI2 TX modulated by TPM ch1
          SIMIPS2_MODTX2 = 1;
        }break;
            
        case 3:
        {
          SIMIPS2_MTBASE2 = 2;   //SCI2 TX modulated by MTIM2
          SIMIPS2_MODTX2 = 1;
        }break;
            
        case 4:
        {
          SIMIPS2_MTBASE2 = 3;   //SCI2 TX modulated by MTIM3
          SIMIPS2_MODTX2 = 1;
        }break;
      }
    }break;
    
    case 3:
    {
      if (modulation) return 3;  //modulation not supported by SCI3
    }break;
    
   }
   return 0;
}

/*--------------------------------------------------------------------------------------------------*/


/***************************************************************************************************** 
   u8IR_RX_init 
______________________________________________________________________________________________________
input parameters:   port            char (1, 2, 3) ***only port 1 and 2 can be connected to ACMP*** 
                    rx pin option   char (default=0, altarnative=!0)
                    *=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=**
                    =  SCI1: default pin.- PTB2 / alternative pin.- none =
                    *  SCI2: default pin.- PTB0 / alternative pin.- none *
                    =  SCI3: default pin.- PTC0 / alternative pin.- PTF2 =
                    *=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=**
                    cmp level       char (not used=0, used=1-31)
                    rx int enable   char (disabled=0, enabled!=0) 
                    ****global interruptions must be enabled!****
                    
                    *****Comparator level supports internal reference only in this driver*****

output parameters:  char  (OK=0)
                          (invalid port=1)
                          (invalid comparator level=2)
                          (invalid selection=3) 
*****************************************************************************************************/  

UINT8 u8IR_RX_init(UINT8 port, UINT8 rx_pin_option, UINT8 cmp_level, UINT8 rx_ie)
  {
    if (0<cmp_level>31) return 2;
    
    switch(port)
      {
        case 1:
        {
          //pin function configuration
          if(rx_pin_option) return 3;         //Error because SCI1 pin options are only in PTB
          else PTBPF2_B2 = 3;                 //PTB2 as RX1 (default option)
                    
          //comparator configuration
          if(!cmp_level) SIMIPS2_RX1IN = 0;   //input directly from RX pin
          else
          {
            
            /*Comparator 1 input*/  
            PTBPF2_B2 = 2; /* CMP1P0 */
            
            /*Comparator 1 output*/   
            PTEPF2_E0 = 2; /* CMP1O */
            /*This pin is only in 100-pin package*/    

            /***PRACMP1 init***/  
            
            PRACMP1CS = 0;            /*ACMP disabled*/
            /*PRACMP1CS_ACIEN = 1;*/  /* interrupt */
            PRACMP1CS_ACOPE = 1;      /* output on ACMPxO */
            PRACMP1C0_ACPSEL = 0;     /* external reference zero */ /* CMP1P1 */
            PRACMP1C0_ACNSEL = 7;     /* internal programmable reference*/ 
            PRACMP1C1_PRGINS = 1;     /* using external Vdd reference (probably 3.3V) */
            PRACMP1C2 = 0x3F;         /* external analog channel 0 is allowed!!! */
            PRACMP1C1_PRGEN = 1;      /* start programmable */
            PRACMP1CS_ACEN = 1;       /*ACMP enabled*/

            /***PRACMP end***/
            
            SIMIPS2_RX1IN = 1;                //input from analog comparator 1 output
            PRACMP1C1_PRGOS = cmp_level;      //comparator reference level
          }
         
          //enabling reception interrupt
          if (rx_ie) SCI1C2_RIE = 1;
          else SCI1C2_RIE = 0;
          
        }break; 
    
        case 2:
        {
          //pin function configuration
          if(!rx_pin_option) PTBPF2_B0 = 3; //PTB0 as RX2 (default option)
          else return 3;                    //Error because SCI2 pin options are only in PTB
          
          //comparator configuration
          if(!cmp_level) SIMIPS2_RX2IN = 0;   //input directly from RX pin
          else
          {
            /*Comparator 2 inputs*/  
            PTBPF2_B0 = 2; /* CMP2P0 */
            /*Comparator 2 output on pin is not included*/  
            
            /***PRACMP2 init***/  
  
            PRACMP2CS = 0;            /*ACMP disabled*/
            /*PRACMP2CS_ACIEN = 1;*/  /* interrupt */
            PRACMP2CS_ACOPE = 1;      /* output on ACMPxO */
            PRACMP2C0_ACPSEL = 0;     /* external reference zero */ /* CMP2P1 */
            PRACMP2C0_ACNSEL = 7;     /* internal programmable reference*/ 
            PRACMP2C1_PRGINS = 1;     /* using external Vdd reference (probably 3.3V) */
            PRACMP2C2 = 0x3F;         /* external analog channel 0 is allowed!!! */
            PRACMP2C1_PRGEN = 1;      /* start programmable */
            PRACMP2CS_ACEN = 1;       /*ACMP enabled*/

            /***PRACMP end***/
            
            SIMIPS2_RX2IN = 1;                //input from analog comparator output
            PRACMP2C1_PRGOS = cmp_level;      //comparator reference level
          }
          
          //enabling reception interrupt
          if (rx_ie) SCI2C2_RIE = 1;
          else SCI2C2_RIE = 0;

        }break;
    
        case 3:
        {
          if(cmp_level) return 3;             //RX through comparator not supported by SCI3
          
          //pin function configuration
          if(!rx_pin_option) PTCPF2_C0 = 3; //PTC0 as RX3 (default option)
          else                              //PTF2 as RX3 (alternative option)
            {                               //and NOT using the LCD
              LCDSUPPLY_VSUPPLY = 0b11;
              LCDRVC_RVEN = 0; //regulator voltage disabled
              LCDC1_FCDEN = 1;
              
              PTFPF2_F2 = 3;               
            }
            
          //enabling reception interrupt
          if (rx_ie) SCI3C2_RIE = 1;
          else SCI3C2_RIE = 0;
          
        }break;
      }
         return 0;
      }
