/******************************************************************************		
*                                                                                   	
*       Copyright (C) 2008 Freescale Semiconductor, Inc.                            	
*       All Rights Reserved								              				
*														              					
* Filename:       ADC16.h               											
*														              					
* Description:	  16-bit ADC Header files for MCF51EM256	        	    	   	       						
*																						
* Notes:          Based on 28 June 2008 reference manual
*     																		
******************************************************************************/		

/******************************************************************************/

#include "m51em256evb.h"

#ifndef ADC16_H        /*prevent duplicated includes*/
#define ADC16_H

/* 
   required for ColdFire -> Stores data to reduce data size instead of improving 
                            execution performance.
   pop -> see end of this file
*/
/* #pragma pack(push,1)  */
#pragma options align=packed

/************************* Project Include Files *****************************/

/************************* Variables *****************************************/

typedef union 
{
  UINT8 Byte;
  struct 
    {
    UINT8 ADCHA       :5;        /* Input Channel Select */
    UINT8 DIFFA       :1;        /* Differential Mode Enable */
    UINT8 AIENA       :1;        /* Interrupt Enable */
    UINT8 COCOA       :1;        /* Conversion Complete Flag */
    } Bits;
}ADC_SC1ARegType;


typedef union 
{
  UINT8 Byte;
  struct 
    {
    UINT8 ADCHB       :5;        /* Input Channel Select */
    UINT8 DIFFB       :1;        /* Differential Mode Enable */
    UINT8 AIENB       :1;        /* Interrupt Enable */
    UINT8 COCOB       :1;        /* Conversion Complete Flag */
    } Bits;
}ADC_SC1BRegType;

typedef union 
{
  UINT8 Byte;
  struct 
    {
    UINT8 ADICLK      :2;        /* Input Clock Select */
    UINT8 MODE        :2;        /* Conversion Mode Select */
    UINT8 ADLSMP      :1;        /* Sample Time Configuration */
    UINT8 ADIV        :2;        /* Clock Divide Select */
    UINT8 ADLPC       :1;        /* Low Power Configuration */
    } Bits;
}ADC_CFG1RegType;


typedef union 
{
  UINT8 Byte;
  struct 
    {
    UINT8 ADLSTS      :2;        /* Long Sample Time Select */                    
    UINT8 ADHSC       :1;        /* High Speed Configuration */
    UINT8 ADACKEN     :1;        /* Asynchronous Clock Output Enable */
    UINT8             :4;
    } Bits;
}ADC_CFG2RegType;

typedef union 
{
  UINT16 Word;
  struct 
    {
    UINT16 D           :16;        /* Data */
    } Bits;
}ADC_RARegType;

typedef union 
{
  UINT16 Word;
  struct 
    {
    UINT16 D           :16;        /* Data */
    } Bits;
}ADC_RBRegType;

typedef union 
{
  UINT16 Word;
  struct 
    {
    UINT16 CV          :16;        /* Compare Value */
    } Bits;
}ADC_CVARegType;

typedef union 
{
  UINT16 Word;
  struct 
    {
    UINT16 CV          :16;        /* Compare Value */
    } Bits;
}ADC_CVBRegType;

typedef union 
{
  UINT8 Byte;
  struct 
    {
    UINT8 REFSEL      :2;        /* Voltage Reference Selection */
    UINT8             :1;
    UINT8 ACREN       :1;        /* Compare Function Range Enable */
    UINT8 ACFGT       :1;        /* Compare Function Greater Than Enable */
    UINT8 ACFE        :1;        /* Compare Function Enable */
    UINT8 ADTRG       :1;        /* Conversion Trigger Select */        
    UINT8 ADACT       :1;        /* Conversion Active */
    } Bits;
}ADC_SC2RegType;

typedef union 
{
  UINT8 Byte;
  struct 
    {
    UINT8 AVGS        :2;        /* Hardware Average Select */
    UINT8 AVGE        :1;        /* Hardware Average Enable */
    UINT8 ADCO        :1;        /* Continous Conversion Enable */        
    UINT8             :2;
    UINT8 CALF        :1;        /* Calibration Failed Flag */
    UINT8 CAL         :1;        /* Calibration */
    } Bits;
}ADC_SC3RegType;

typedef union 
{
  UINT16 Word;
  struct 
    {
    UINT16 OFS         :16;        /* Offset Value */
    } Bits;
}ADC_OFSRegType;

typedef union 
{
  UINT16 Word;
  struct 
    {
    UINT16 PG          :16;        /* Plus Gain Value */
    } Bits;
}ADC_PGRegType;

typedef union 
{
  UINT16 Word;
  struct 
    {
    UINT16 MG          :16;        /* Minus Gain Value */
    } Bits;
}ADC_MGRegType;

typedef struct 
{
  union 
  {
    UINT8 Byte;
    struct 
      {
      UINT8 CLPD        :6;        /* Plus-side General Calibration */       
      UINT8             :2; 
      } Bits;
  }CLPD;
  union 
  {
    UINT8 Byte;
    struct 
      {
      UINT8 CLPS        :6;        /* Plus-side General Calibration */        
      UINT8             :2;
      } Bits;
  }CLPS;
  union 
  {
    UINT16 Word;
    struct 
      {
      UINT16 CLP4        :10;        /* Plus-side General Calibration */        
      UINT16             :6;
      } Bits;
  }CLP4;
  union 
  {
    UINT16 Word;
    struct 
      {
      UINT16 CLP3        :9;        /* Plus-side General Calibration */        
      UINT16             :7;
      } Bits;
  }CLP3;
  union 
  {
    unsigned char Byte;
    struct 
      {
      unsigned char CLP2        :8;        /* Plus-side General Calibration */
      } Bits;
  }CLP2;
  union 
  {
    unsigned char Byte;
    struct 
      {
      unsigned char CLP1        :7;        /* Plus-side General Calibration */        
      unsigned char             :1;
      } Bits;
  }CLP1;
  union 
  {
    unsigned char Byte;
    struct 
      {
      unsigned char CLP0        :6;        /* Plus-side General Calibration */        
      unsigned char             :2;
      } Bits;
  }CLP0;
}ADC_CLPRegType;

typedef struct 
{
  union 
  {
    unsigned char Byte;
    struct 
      {
      unsigned char CLMD        :6;        /* Minus-side General Calibration */        
      unsigned char             :2;
      } Bits;
  }CLMD;
  union 
  {
    unsigned char Byte;
    struct 
      {
      unsigned char CLMS        :6;        /* Minus-side General Calibration */        
      unsigned char             :2;
      } Bits;
  }CLMS;
  union 
  {
    UINT16 Word;
    struct 
      {
      UINT16 CLM4        :10;      /* Minus-side General Calibration */        
      UINT16             :6;
      } Bits;
  }CLM4;
  union 
  {
    UINT16 Word;
    struct 
      {
      UINT16 CLM3        :9;       /* Minus-side General Calibration */        
      UINT16             :7;
      } Bits;
  }CLM3;
  union 
  {
    unsigned char Byte;
    struct 
      {
      unsigned char CLM2         :8;       /* Minus-side General Calibration */
      } Bits;
  }CLM2;
  union 
  {
    unsigned char Byte;
    struct 
      {
      unsigned char CLM1         :7;       /* Minus-side General Calibration */        
      unsigned char              :1;
      } Bits;
  }CLM1;
  union 
  {
    unsigned char Byte;
    struct 
      {
      unsigned char CLM0         :6;       /* Minus-side General Calibration */        
      unsigned char              :2;
      } Bits;
  }CLM0;
}ADC_CLMRegType;


typedef struct				/*ADC datastructure	*/
  {
  ADC_SC1ARegType	ADCSC1A;	/*ADC status and control register 1A */
  ADC_SC1BRegType	ADCSC1B;	/*ADC status and control register 1B */
  ADC_CFG1RegType ADCCFG1;  /*ADC config 1 register */
  ADC_CFG2RegType ADCCFG2;  /*ADC config 2 register */
  ADC_RARegType   ADCRA;    /*ADC result A register */
  ADC_RBRegType   ADCRB;    /*ADC result B register */
  ADC_CVARegType  ADCCVA;   /*ADC compare value A register */
  ADC_CVBRegType  ADCCVB;   /*ADC compare value B register */
  ADC_SC2RegType	ADCSC2;	  /*ADC status and control register 2 */
  ADC_SC3RegType	ADCSC3;	  /*ADC status and control register 3 */
  ADC_OFSRegType  ADCOFS;   /*ADC offset register */
  ADC_PGRegType   ADCPG;    /*ADC plus-input gain */
  ADC_MGRegType   ADCMG;    /*ADC minus-input gain */
  ADC_CLPRegType  ADCCLP;   /*ADC plus-side general calibration registers */
  ADC_CLMRegType  ADCCLM;   /*ADC minus-side general calibration registers */
 }tADC;

/* #pragma pack(pop) */
#pragma options align=reset


extern tADC *ADCPointer;  /* Global Variable indicating which ADC is in use */

  
/************************* #Defines ******************************************/

/* ADCSC1 */
#define COCO_COMPLETE     0x80
#define COCO_NOT          0x00

#define AIEN_ON           0x40
#define AIEN_OFF          0x00

#define DIFF_SINGLE       0x00
#define DIFF_DIFFERENTIAL 0x20

/* ADCCFG1 */
#define ADLPC_NORMAL      0x00
#define ADLPC_LOW         0x80

#define ADIV_1            0x00
#define ADIV_2            0x20
#define ADIV_4            0x40
#define ADIV_8            0x60

#define ADLSMP_SHORT      0x00
#define ADLSMP_LONG       0x10

#define MODE_8            0x00
#define MODE_12           0x04
#define MODE_10           0x08
#define MODE_16           0x0C

#define ADICLK_BUS        0x00
#define ADICLK_BUS_2      0x01
#define ADICLK_ALTCLK     0x02
#define ADICLK_ADACK      0x03

/* ADCCFG2 */
#define ADACKEN_DISABLED  0x00
#define ADACKEN_ENABLED   0x08

#define ADHSC_NORMAL      0x00
#define ADHSC_HISPEED     0x04

#define ADLSTS_20          0x00
#define ADLSTS_12          0x01
#define ADLSTS_6           0x02
#define ADLSTS_2           0x03

/* ADCSC2 */
#define ADACT_INACTIVE     0x00
#define ADACT_ACTIVE       0x80

#define ADTRG_SW           0x00
#define ADTRG_HW           0x40

#define ACFE_DISABLED      0x00
#define ACFE_ENABLED       0x20

#define ACFGT_LESS         0x00
#define ACFGT_GREATER      0x10

#define ACREN_DISABLED     0x00
#define ACREN_ENABLED      0x08

#define REFSEL_EXT         0x00
#define REFSEL_ALT         0x01
#define REFSEL_BG          0x02
#define REFSEL_RES         0x03     /* reserved but defaults to Vref */

/* ADCSC3 */
#define CAL_BEGIN          0x80
#define CAL_OFF            0x00

#define CALF_NORMAL        0x00
#define CALF_FAIL          0x40

#define ADCO_SINGLE        0x00
#define ADCO_CONTINUOUS    0x08

#define AVGE_DISABLED      0x00
#define AVGE_ENABLED       0x04

#define AVGS_4             0x00
#define AVGS_8             0x01
#define AVGS_16            0x02
#define AVGS_32            0x03

/************************* #Defines ******************************************/
#define ADC1 0xFFFF8400
#define ADC2 0xFFFF8440
#define ADC3 0xFFFF8480
#define ADC4 0xFFFF84C0

/************************* Macros ********************************************/

/************************* Prototypes ****************************************/
void vfnADC_Init(tADC *adc_ptr, unsigned char CONFIG1, unsigned char CONFIG2, UINT16 COMPAREA, UINT16 COMPAREB, unsigned char STATUS2, unsigned char STATUS3, unsigned char STATUS1A, unsigned char STATUS1B);
void ADC_Cal(tADC *adc_ptr);
void ADC_Config(tADC *adc_ptr, unsigned char CONFIG1, unsigned char CONFIG2, UINT16 COMPAREA, UINT16 COMPAREB, unsigned char STATUS2, unsigned char STATUS3, unsigned char STATUS1A, unsigned char STATUS1B);


#endif /* ADC16_H */