#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "metering_algorithms.h"
#include "DFT_coef.h"
#include "tpm.h"

//Metering algorithms internal variables
extern word i;             // Internal loops index
#if DFT_INTERMEDIATE == 0
static long Sum_D;     // Intermediate result (32-bits)
#else
static long long Sum_D;     // Intermediate result (64-bits)
#endif      

static long long Sum;       // Intermediate result for the RMS calculation (64)
static dword Acc = 0, Acc_temp; // Square Root internal variables
extern word Number_of_interactions[N_SAMPLES];     //Debug purposes
//extern dword time;

//--------------------------------------------
//   RMS
//
//   Input parameter: pointer to 16-bit signed.
//   Output parameter: unsigned 16-bits.
//--------------------------------------------
/* RMS calculation*/   
word  RMS_calc(short *input){
     
  Sum = 0;                                 // clear accumulator variable
  for(i=0;i<N_SAMPLES;i+=RMS_DEC){         // execute interaction 
#if ASM_SUM == 1                           // Assembly implementation of 64-bit sum
      ADD64bits(&Sum,((*input)*(*input))); // call Assembly function
      input+=RMS_DEC;                      // update pointer
#else                                      // if not using assembly
    Sum = Sum + (input[i])*(input[i]);     // multiply and accumulate
#endif                                     //
  }                                        //
  Sum = (Sum>>FAC_SAMPLE_FOR_RMS);         // divide final sum by number of samples
  return(SquareRoot((dword)Sum));          // output square root of Sum

}
//--------------------------------------------
//   Power and Energy Calculations
//
//   Input parameter: -Voltage pointer to 16-bit signed.
//                    -Current pointer to 16-bit signed.
//                    -Output Vector pointer.
//                    -Output Time vector pointer.
//   Output parameter: unsigned 16-bits.
//--------------------------------------------   
void Power_Calc(short *V, short *I, Power_vec *Out, Power_time *tm){

 long A;                                    //local var used as ACC
 short *tempV = V, *tempI = I;              //local pointers
                                            //
  //Active Energy Calculation               //
  StartTPM();                               //Start timer to measure execution time 
  Sum = 0;                                  //clear accumulator variable
  for(i=0;i<N_SAMPLES;i++){                 //interaction for the 256 samples
    Sum = Sum + (*tempV++)*(*tempI++);      //multiply and accumulate
  }                                         //
  Out->Act_Eng = Sum;                       //output result
  tm->Act_Eng = Stop_Read_TPM();            //Stop timer to measure execution time
                                            //
  //Active Power Calculation                //
  StartTPM();                               //Start timer to measure execution time
  Out->Act_Pwr = (long)(Sum>>FAC_N_SAMPLES);//Active Power Calculation
  tm->Act_Pwr = Stop_Read_TPM();            //Stop timer to measure execution time
                                            //
  //Voltage RMS Calculation                 //
  StartTPM();                               //Start timer to measure execution time
  Out->Vrms = RMS_calc(V);                  //Voltage RMS calculation
  tm->Vrms = Stop_Read_TPM();               //Stop timer to measure execution time
                                            //
  //Current RMS Calculation                 //
  StartTPM();                               //Start timer to measure execution time
  Out->Irms = RMS_calc(I);                  //Current RMS calculation
  tm->Irms = Stop_Read_TPM();               //Stop timer to measure execution time
                                            //
  //Aparent Power Calculation               //
  StartTPM();;                              //Start timer to measure execution time
  Out->Apr_Pwr = (long)((Out->Vrms) * (Out->Irms));//Aparent Power Calculation
  if((Out->Apr_Pwr) < (Out->Act_Pwr))       //Check if Aparent power is smaller then active
    Out->Apr_Pwr = Out->Act_Pwr;            //Correct output
  tm->Apr_Pwr = Stop_Read_TPM();            //Stop timer to measure execution time
                                            //
  //Power Factor Calculation                //
  StartTPM();                               //Start timer to measure execution time
  if((Out->Apr_Pwr)<=(Out->Act_Pwr))        //If aparent power equals active
    Out->Pwr_fct = 65535;                   //Power Factor equals one
  else                                      //
    Out->Pwr_fct = (word) ((((long long)(Out->Act_Pwr))<<16)/(Out->Apr_Pwr));//calc P. Fct
  tm->Pwr_fct = Stop_Read_TPM();            //Stop timer to measure execution time
                                            //
  //Reactive Energy Calculation             //                                            
  StartTPM();                               //Start timer to measure execution time
  A = (((Out->Apr_Pwr)>>16)*((Out->Apr_Pwr)>>16)) - (((Out->Act_Pwr)>>16)*((Out->Act_Pwr)>>16));
    Out->React_Pwr = (SquareRoot((dword)A))<<16; 
  tm->React_Pwr = Stop_Read_TPM();          //Stop timer to measure execution time
}                

//--------------------------------------------
//   Square root
//
//   Input parameter: unsigned 32-bits 
//   Output parameter: unsigned 16-bits
//--------------------------------------------
/*Square root*/
word  SquareRoot(dword A){

  word j;                          // local variable 
  Acc = ASM_FF1(A);                // initial seed
  for(j=0;j<MAX_INT;j++){          // execute maximum of MAX_INT interactions
    Acc_temp = (A/Acc + Acc)/2;    // Calculate interaction
    if((abs(Acc - Acc_temp) < 0x0001)||(Acc_temp == 0)) // Test is precision is good enough     
      break;                       // break if a good precision was reached
    Acc = Acc_temp;                // Save previous interaction
  }
  //Number_of_interactions[i] = (unsigned short) (j+1); // For debugging purposes
  return((word)Acc_temp);          // Return parameter 
}

//--------------------------------------------
//   Guess the initial root square seed
//
//   Input parameter: unsigned 32-bits 
//   Output parameter: unsigned 16-bits
//--------------------------------------------
/*Root Square Initial Seed*/
dword ASM_FF1(dword A){
  asm{                            //
  ff1.l d0                        // find the fist one
  }                               //
  A= (32-A)>>1;                   // first one index diveded by 2
  A = (unsigned long)1<<A;        // multiplies by two
  return(A);                      // Return parameter
}

//--------------------------------------------
//   DFT
//
//   Input parameter: signed 16-bits pointer 
//   Output parameter: Complex Structure
//--------------------------------------------
Complex DFT(short *input){

  Complex Res;                                   // Complex structure to store result
  short *P1, *P2;                                // pointers
  P1= input;                                     // input pointer initialization (Real)
  P2 = &Cos_coef_k_1[0];                         // coeficients pointer init (Real)
                                                 //
  Sum_D = 0;                                     // clear accumulator variable
  for(i=0;i<N_SAMPLES;i+=DEC){                   // calculate real component
    Sum_D = Sum_D + (((*P1)*(*P2))>>DFT_SCALING);// square and accumulate the N_SAMPLES_DFT samples 
    P1+=DEC;                                     // increment pointers
    P2+=DEC;                                     // increment pointers
  }                                              // 
  Res.Real = (long)((Sum_D)>>(FAC_SAMPLE_FOR_DFT + COEF_MAX - 1 - DFT_SCALING));
                                                 //
  P1= input;                                     // input pointer initialization (Imag)
  P2 = &Sin_coef_k_1[0];                         // coeficients pointer init (Imag)
                                                 //
  Sum_D = 0;                                     // clear accumulator variable
  for(i=0;i<N_SAMPLES;i+=DEC){                   // calculate Img component
     Sum_D = Sum_D + (((*P1)*(*P2))>>DFT_SCALING);// square and accumulate the N=256 buffer samples
     P1+=DEC;                                    // increment pointers
     P2+=DEC;                                    // increment pointers
  }                                              //
  Res.Img =  (long)((Sum_D)>>(FAC_SAMPLE_FOR_DFT + COEF_MAX - 1 - DFT_SCALING));
                                                 //
  return(Res);                                   // output the result

}
//--------------------------------------------
//   THD
//--------------------------------------------
word THD_calc(word Total_RMS, word Fund_RMS){

  if(Total_RMS >= Fund_RMS)
    return( (((dword)(SquareRoot((Total_RMS*Total_RMS) - (Fund_RMS*Fund_RMS))))<<16) / (Fund_RMS));
  else
    return(0);


}
//--------------------------------------------
//   RMS DFT
//--------------------------------------------
word RMS_DFT_calc(Complex A){

  return(SquareRoot((((dword)A.Real*A.Real + (dword)A.Img*A.Img))>>1));

}

//--------------------------------------------
//   Active Power
//--------------------------------------------
dword Act_Pwr_DFT_calc(Complex V, Complex I){

  return(((dword)V.Real*I.Real + (dword)V.Img*I.Img));

}

//--------------------------------------------
//   Reactive Power
//--------------------------------------------
dword Reac_Pwr_DFT_calc(Complex V, Complex I){

  return(((dword)V.Img*I.Real - (dword)V.Real*I.Img));

}

asm long MAC_BUFF(short *B1, short *B2, short N){

/*    move.l  #0xFFFFFFFF,MASK
    move.l	#0x00000000,MACSR		// load MACSR for Sat|signed|Integer|round
    clr.l   D1
    move.l  D1,ACC				      // clear the acc
    move.l  (A0)+,D1
    move.l  (A1)+,D2

Back:    
    macl.w	D1.l,D2.l,(A0)+,D3   			// acc + x(n-2)*B2
    macl.w D1.u,D2.u,(A1)+,D4
    macl.w	D3.l,D4.l,(A0)+,D1   			// acc + x(n-2)*B2
    macl.w D3.u,D4.u,(A1)+,D2

    subq.l #1,D0
    bne.b Back

    move.l ACC, D0       */

    rts
}
//--------------------------------------------
//   64-bit accumulate
//
//   Input parameter: signed 64-bits pointer
//   Output parameter: signed 32-bits 
//--------------------------------------------
/*64-bit accumulate*/
asm void ADD64bits(long long* ACC, long A){

         ADD.L  D0,4(A0)                    // Add 32-bit to 64-bit accumulator
         BCC    OUT                         // Check overflow
         ADDQ.L #01,(A0)                    // if overflow increase MS 32-bits of ACC
    OUT:                                    //
         RTS                                // Return
}

