/************************************************************************************
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale.
*
************************************************************************************/
#ifndef _ENGINE_H_
#define _ENGINE_H_

/************************************************************************************
*************************************************************************************
* Public macros
*************************************************************************************
************************************************************************************/
#define ENG_BUFFER_SIZE    0x200  //read/write maximum buffer size

/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/
typedef enum {
  engReadReq_c      = 0x01,
  engReadResp_c     = 0x02,
  engWriteReq_c     = 0x03,
  engCommitReq_c    = 0x04,
  engEraseReq_c     = 0x05,
  engCmdCnf_c       = 0xF0
} ENGCommandId_t;

typedef enum {
  gEngValidReq_c = 0x00,
  gEngInvalidReq_c,
  gEngSuccessOp_c,
  gEngWriteError_c,
  gEngReadError_c,
  gEngCRCError_c,
  gEngCommError_c,
  gEngExecError_c,
  gEngNoConfirm_c
} ENGCmdStatus_t;

typedef enum
{
  engSecured_c    = 0xC3,
  engUnsecured_c  = 0x3C
} ENGSecureOption_t;

//pack all structures
#pragma pack(1)
typedef struct ENGReadReq_tag
{
  uint32_t        startAddress;  //start address in FLASH for the read operation
  uint16_t        length;        //number of bytes to read
} ENGReadReq_t;

typedef struct ENGReadResp_tag
{
  uint8_t         cmdStatus;     //Should be ENGCmdStatus_t(enum),but that is int in VC++
  uint16_t        length;        //valid data bytes in the buffer
  uint8_t         buffer[ENG_BUFFER_SIZE];
} ENGReadResp_t;

typedef struct ENGWriteReq_tag
{
  uint32_t        startAddress;  //start address for the write operation
  uint16_t        length;        //number of bytes to be written from the command buffer
  uint8_t         buffer[ENG_BUFFER_SIZE];
} ENGWriteReq_t;

typedef struct ENGCommitReq_tag
{
  uint32_t        binLength;     //length of the binary commited to FLASH
  uint8_t         secured;       //secured option
} ENGCommitReq_t;

typedef struct ENGEraseReq_tag
{
  uint32_t        address;       //the bits from [AMS-A12] are used to determine the sector to be erased. For all F's, all FLASH will be erased.
}ENGEraseReq_t;

typedef struct ENGCmdCnf_tag
{
  uint8_t  cmdStatus;     //Should be ENGCmdStatus_t(enum),but that is int in VC++
} ENGCmdCnf_t;

typedef union ENGFields_tag
{
  ENGReadReq_t         readReq;
  ENGReadResp_t        readResp;
  ENGWriteReq_t        writeReq;
  ENGCommitReq_t       commitReq;
  ENGEraseReq_t        eraseReq;
  ENGCmdCnf_t          cmdCnf;
} ENGFields_t;

typedef struct ENGCommand_tag
{
  uint8_t         commandId;  //Should be ENGCommandId_t(enum),but that is int in VC++
  ENGFields_t     fields;
} ENGCommand_t;

//return to default packing
#pragma pack()


/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

/************************************************************************************
*  Send the command to erase a sector or all sectors of the MC1322x. Note that the 
* last sector of the internal FLASH of the MC1322x hold production data and will 
* not be erased.
*
*  Input parameters:
*  - hComm: comm port opened for the communication with the MC1322x.
*  - address: address in the FLASH sector to be erased. The LSB's of the address will
*      be ignored, only the bits specifying the sector are relevant. If the address 
*      is 0xFFFFFFFF, all the FLASH is erased.
*  - confirmTimeout: time, in miliseconds, to wait for a confirm from the SSL running 
*        on the MC1322x.
*  Return:
*  - ENGCmdStatus_t: status of the erase operation.
************************************************************************************/
ENGCmdStatus_t ENG_Erase(HANDLE hComm, uint32_t address, uint32_t confirmTimeout);

/************************************************************************************
*  Send the command to write data into the internal FLASH of the MC1322x. All of the
* written bytes in FLASH are read back and verified by the SSL before returning the
* confirm for the write operation.
*
*  Input parameters:
*  - hComm: comm port opened for the communication with the MC1322x.
*  - address: address in the FLASH to start writing from. It can not be in the last 
*      sector, as the last sector of the internal FLASH holds production data.
*  - pBuffer: pointer to the buffer containing the data to be written to the FLASH.
*  - length: nuber of bytes to be written from the buffer. Needs to be smaller than
*      the maximum payload of a write command (ENG_BUFFER_SIZE).
*  - confirmTimeout: time, in miliseconds, to wait for a confirm from the SSL running 
*      on the MC1322x.
*  Return:
*  - ENGCmdStatus_t: status of the write operation.
************************************************************************************/
ENGCmdStatus_t ENG_Write(HANDLE hComm, uint32_t address, uint8_t* pBuffer, uint16_t length, uint32_t confirmTimeout);

/************************************************************************************
*  Send the command to read data from the internal FLASH of the MC1322x.
*
*  Input parameters:
*  - hComm: comm port opened for the communication with the MC1322x.
*  - address: address in the FLASH to start reading from. It can not be in the last
*      sector, as the last sector of the internal FLASH holds production data.
*  - pBuffer: pointer to the buffer ready to receive the data read from the FLASH.
*  - length: nuber of bytes to be read from the FLASH. Needs to be smaller than
*      the maximum payload of a read command (ENG_BUFFER_SIZE).
*  - confirmTimeout: time, in miliseconds, to wait for a confirm from the SSL running 
*      on the MC1322x.
*  Return:
*  - ENGCmdStatus_t: status of the read operation.
************************************************************************************/
ENGCmdStatus_t ENG_Read(HANDLE hComm, uint32_t address, uint8_t* pBuffer, uint16_t length, uint32_t readTimeout);

/************************************************************************************
*  Commit the image written the internal FLASH of the MC1322x by writing the FLASH 
* header with the image descriptor.
*
*  Input parameters:
*  - hComm: comm port opened for the communication with the MC1322x.
*  - length: the length of the image from the FLASH. This is the number of bytes that
*      the bootstrap will load from FLASH to RAM, starting from the first location
*      after the FLASH header.
*  - secureOption: secure or unsecure load of the image by the bootloader from FLASH
*      to RAM. Secure means that the ETM module shall not be enabled, thus
*      preventing the image to be copied.
*  - commitTimeout: time, in miliseconds, to wait for a confirm from the SSL running 
*        on the MC1322x.
*  Return:
*  - ENGCmdStatus_t: status of the commit operation.
************************************************************************************/
ENGCmdStatus_t ENG_Commit(HANDLE hComm, uint32_t length, ENGSecureOption_t secureOption, uint32_t commitTimeout);

#endif _ENGINE_H_
