/************************************************************************************
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale.
*
************************************************************************************/
#include "stdafx.h"
#include "UART.h"
#include "Engine.h"

/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/
#define UART_BAUD_RATE  115200          //Baud rate used to communicate with the MC1322x
#define BIN_FILE_BUFFER_SIZE  4*1024    //Buffer allocated for reading the SSL binary image
#define FLASH_IMAGE_HEADER    8         //FLASH image header, containing secure option (4 bytes) and binary length (4 bytes)
#define BIN_MAX_SIZE    (96*1024 - FLASH_IMAGE_HEADER) //RAM size of MC1322x minus the FALSH header size
#define ERASE_ALL_TO    5000            //Timeout for erasing all of the MC1322x FLASH, in ms
#define WRITE_DATA_TO   250             //Timeout for writing a data buffer
#define READ_DATA_TO    250             //Timeout for reading a data buffer

#if (BIN_FILE_BUFFER_SIZE < ENG_BUFFER_SIZE)
  #error *** BIN_FILE_BUFFER_SIZE has to be bigger than ENG_BUFFER_SIZE.
#endif


#define VERIFY_CLOSE_HANDLE(handle)      if(handle != INVALID_HANDLE_VALUE)   \
                                         {                                    \
                                          CloseHandle(handle);                \
                                         }
/************************************************************************************
*************************************************************************************
* Private type declarations
*************************************************************************************
************************************************************************************/
typedef struct inputParams_tag
{
  HANDLE hCom;      //Handle to the opened COM for MC1322x communication
  HANDLE hBinFile;  //Handle for the binary file to be downloaded into the MC1322x FLASH
  HANDLE hSSLBin;   //Handle for the binary file containing the SSL image
  BOOL bSecured;    //Secure/unsecure option
} inputParams_t;

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
char binFileBuffer[BIN_FILE_BUFFER_SIZE];

/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/
void  PrintHelpText();
int   ParseArguments(int argc, _TCHAR* argv[], inputParams_t* pInputParams);
int   Connect(HANDLE hCom);
int   DownloadSSL(HANDLE hCom, HANDLE hSSLBin);
int   DownloadBin(inputParams_t* pInputParams);
int   DumpFLASH(HANDLE hCom, char* fileName, uint32_t startAddress, uint32_t length);
void  CleanUp(inputParams_t* pInputParams);

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
int _tmain(int argc, _TCHAR* argv[])
{
  inputParams_t inputParams;
  
  //Init handles
  inputParams.hBinFile = INVALID_HANDLE_VALUE;
  inputParams.hCom = INVALID_HANDLE_VALUE;
  inputParams.hSSLBin = INVALID_HANDLE_VALUE;

  //Parse and process the input arguments
  if(ParseArguments(argc, argv, &inputParams) != 0)
  {
    CleanUp(&inputParams);
    return -1;
  }

  //Update the user on the progress then try to connect to the MC1322x
  printf("\nOpened COM port. Connecting to the MC1322x.");
  if(Connect(inputParams.hCom) != 0)
  {
    CleanUp(&inputParams);
    return -1;
  }
  
  //Connected to MC1322x. Move on to downloading the second stage load.
  printf("\nConnected to the MC1322x. Sending the SSL binary.");
  if((DownloadSSL(inputParams.hCom, inputParams.hSSLBin)!= 0) != 0)
  {
    CleanUp(&inputParams);
    return -1;
  }
  
  //SSL binary downloaded. Move on to downloading the flash image.
  printf("\nDownloading the %s binary image to FLASH.", argv[3]);
  if(DownloadBin(&inputParams)!= 0)
  {
    printf("\nBinary image download failure.");
  }
  else
  {
    printf("\nAll done.");
  }
  
  //Exit the application
  CleanUp(&inputParams);
  printf("\n");
	return 0;
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*  Will print a help text regarding the use of the application.
*
*  Input parameters:
*  - None
*  Return:
*  - Nothing
************************************************************************************/
void PrintHelpText()
{
    printf("\n\n");
    printf("*********************************************************************\n\n");
    printf("  The application must be called with three parameters:\n");
    printf("    - the COM number on which the MC1322x is connected\n");
    printf("    - secure or un-secure write of the binary to flash (S/U)\n");
    printf("    - the name of the binary to be loaded on MC1322x\n");
    printf("\n  The call to write c:\\MyApps\\myMC13224App.bin binary, secured,\n");
    printf("  to a MC13224 board connected on COM10 would look like: \n");
    printf("    c:\\ConsoleLoader 10 S c:\\MyApps\\myMC13224App.bin\n");
    printf("\n  The call to write c:\\MyApps\\myMC13224App.bin binary, un-secure,\n");
    printf("  to a MC13224 board connected on COM10 would look like: \n");
    printf("    c:\\ConsoleLoader 10 U c:\\MyApps\\myMC13224App.bin\n");
    printf("\n  Ensure that the second stage loader binary (ssl.bin) is in \n");
    printf("  the same folder as the ConsoleLoader.exe and that MC1322x\n"); 
    printf("  has the internal FLASH erased. Reset the MC1322x just before\n");
    printf("  executing the application.\n\n");
    printf("*********************************************************************\n\n");
}

/************************************************************************************
*  Parses and validates the input parameters. Tries to open the needed files and COM 
* port.
*
*  Input parameters:
*  - argc: number of arguments to main.
*  - argv: list of pointers to main arguments.
*  - pInputParams: return structure holding the needed opened handles and secure option
*  Return:
*  -  0: success. Input parameters validated, all needed haldles opened.
*  - -1: failure. Input parameters could not be validated, or at least one handle 
*        could not be opened.
************************************************************************************/
int ParseArguments(int argc, _TCHAR* argv[], inputParams_t* pInputParams)
{
  int comNumber = 0;
  char comName[16];

  // Test if we have received 3 and only 3 arguments (plus one being the executable path and name)
  if(argc != 4)
  {
    printf("\nError: Invalid number of arguments.\n");
    PrintHelpText();
    return -1;
  }

  //Get the secured/unsecured attribute
  switch(argv[2][0])
  {
  case 's':
  case 'S':
    pInputParams->bSecured = TRUE;
    break;
  case 'u':
  case 'U':
    pInputParams->bSecured = FALSE;
    break;
  default:
    printf("\nError: invalid secure option.\n");
    PrintHelpText();
    return -1;
    break;
  }
  
  //Get the COM number
  comNumber = atoi(argv[1]);
  if(comNumber == 0)
  {
    printf("\nError: Invalid COM number.\n");
    PrintHelpText();
    return -1;
  }
  else
  {
    //Convert the com number to a suitable string
    if(comNumber > 9)
    {
      sprintf(comName, "%s%d", "\\\\.\\COM", comNumber);
    }
    else
    {
      sprintf(comName, "%s%d", "\\COM", comNumber);
    }
    //Try to open the COM port
    pInputParams->hCom = UART_OpenCom(comName, UART_BAUD_RATE);
    if(pInputParams->hCom == INVALID_HANDLE_VALUE)
    {
      printf("\nError: Can't open the specified COM port.\n");
      PrintHelpText();
      return -1;
    }
  }

  //Try to open the binary file to be written into flash
  pInputParams->hBinFile = CreateFile(argv[3], GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if( pInputParams->hBinFile == INVALID_HANDLE_VALUE)
  {
    printf("\nError: Can't open the specified binary file: %s\n", argv[3]);
      PrintHelpText();
      return -1;
  }

  //Try to open the embedded second stage loader
  pInputParams->hSSLBin = CreateFile("ssl.bin", GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if( pInputParams->hSSLBin == INVALID_HANDLE_VALUE)
  {
      printf("\nError: Can't open the ssl.bin file.\n");
      PrintHelpText();
      return -1;
  }
  
  return 0;
}

/************************************************************************************
*  Tries to connect to the MC1322x out of reset: send 0x00 for baudrate detection,
* then wait for the "CONNECT" string from the MC1322x. The process is repeated a
* number of times, waiting for the user to reset the MC1322x.
*
*  Input parameters:
*  - hCom: comm port opened for the communication with the MC1322x.
*  Return:
*  -  0: success. Connection was established.
*  - -1: failure. Connection could not be established.
************************************************************************************/
int Connect(HANDLE hCom)
{
  char dummy;
  DWORD bytesRead;
	DWORD bytesWritten;
  char AutoBaudData[1];
  char connectString[8] = "NC";
  int tries = 100;

  printf("\nPress the reset switch on the board.\n");

  AutoBaudData[0] = 0;	
  while(tries--)
  {
    //Pull the RTS low
    do
    {
    ReadFile(hCom, &dummy, 1, &bytesRead, NULL);
    }while(bytesRead);
    
    //Send autobaud data
	  WriteFile(hCom, AutoBaudData, 1, &bytesWritten, NULL);
    
    Sleep(25);
    //Expect CONNECT response
    ReadFile(hCom, connectString, 7, &bytesRead, NULL);
    connectString[7] = 0;
	  if(strcmp(connectString, "CONNECT") == 0)
	  {
      //Connection established - we have received the CONNECT string 
      return 0;
    }
    printf(".");
  }
  
  //We can not connect. Display an error message and return.
  printf("\nError: Can not connect to MC1322x.\n");
  PrintHelpText();
  return -1;
}

/************************************************************************************
*  Tries to download the SSL image to the MC1322x. At this point, the bootstrap 
* should be running on the MC1322x, waiting for a file length and then the binary 
* itself to download to RAM. After the binary is received, the bootstrap will run the
* executable. The SSL application shall send READY to the host to signal it is 
* running on the MC1322x.
*
*  Input parameters:
*  - hCom: comm port opened for the communication with the MC1322x.
*  - hSSLBin: handle opened for the SSL binary.
*  Return:
*  -  0: success. Connection with the SSL established.
*  - -1: failure. SSL could not be downloaded or the connection with the SSL could 
*        not be established.
************************************************************************************/
int DownloadSSL(HANDLE hCom, HANDLE hSSLBin)
{
  DWORD fileSize = 0;
  DWORD bytesOps = 0;
  char readyString[6] = "R";

  //Get the size of the file in bytes
  fileSize =  GetFileSize(hSSLBin, NULL);
  if(INVALID_FILE_SIZE == fileSize)
  {
    printf("\nError: Can not get the filesize of SSL.bin.\n");
    PrintHelpText();
    return -1;
  }
  //Check the filesize against the statically allocated read buffer size
  if(fileSize > BIN_FILE_BUFFER_SIZE)
  {
    printf("\nError: SSL.bin is too big.\n");
    PrintHelpText();
    return -1;
  }

  //Read the bin file to a buffer
  ReadFile(hSSLBin, binFileBuffer, fileSize, &bytesOps, NULL);
  if(bytesOps != fileSize)
  {
    printf("\nError: Could not read SSL.bin.\n");
    PrintHelpText();
    return -1;
  }

  //We now have all that we need. Download the SSL to MC1322x.
  //Send the filesize
  WriteFile(hCom, (char*)(&fileSize), 4, &bytesOps, NULL);
  //Send the binary
  WriteFile(hCom, &binFileBuffer, fileSize, &bytesOps, NULL);
  
  //Expect READY response
  ReadFile(hCom, readyString, 5, &bytesOps, NULL);
  readyString[5] = 0;
	if(strcmp(readyString, "READY") == 0)
	{
    //SSL downloaded 
    return 0;
  }
  else
  {
    printf("\nError: READY not received from SSL.\n");
    PrintHelpText();
    return -1;
  }
  return 0;
}

/************************************************************************************
*  Tries to download the user specified binary image to the MC1322x FLASH. At this 
* point, the SSL should be running on the MC1322x, waiting for commands from the
* host to execute on the MC1322x (see Engine.h for available commands). The file 
* will be partitioned into data packets of the maximum supported size by the SSL and
* send to the SSL to be written into FLASH. After the file is transmitted, the binary
* is commited by writing the MC1322x FLASH header indicating the presence in the FLASH
* of a valid executable.
*  
*  Input parameters:
*  - pInputParams: structure containing opened handles to the bin file and comm port
*    with the MC1322x, as well as the secure option specified by the user.
*  Return:
*  -  0: success. Connection was established.
*  - -1: failure. Connection could not be established.
************************************************************************************/
int DownloadBin(inputParams_t* pInputParams)
{
  DWORD fileSize = 0;
  DWORD bytesOps = 0;
  ENGCmdStatus_t status;
  uint32_t currentWriteAddress;
  int binReminder;
  int toRead;

  //Get the size of the file in bytes
  fileSize =  GetFileSize(pInputParams->hBinFile, NULL);
  if(INVALID_FILE_SIZE == fileSize)
  {
    printf("\nError: Can not get the filesize of the binary FLASH image for writing in FLASH.\n");
    PrintHelpText();
    return -1;
  }
  //Check the filesize against the max size supported by MC1322x
  if(fileSize > BIN_MAX_SIZE)
  {
    printf("\nError: the binary FLASH image is too big.\n");
    PrintHelpText();
    return -1;
  }
  
  //All seems OK. Start the write process.
  //Erase all the FLASH on MC1322x
  status = ENG_Erase(pInputParams->hCom, 0xFFFFFFFF, ERASE_ALL_TO);
  if(status != gEngSuccessOp_c)
  {
    printf("\nError trying to erase the FLASH: %d", status);
    PrintHelpText();
    return -1;
  }
  //Initialize the write address with the first byte after the FLASH header
  currentWriteAddress = FLASH_IMAGE_HEADER;
  //Start the read and write loop. Print progress information.
  printf("\n");
  binReminder = fileSize;
  while(binReminder != 0)
  {
    //Determine how much can we read
    if(binReminder > ENG_BUFFER_SIZE)
    {
      toRead = ENG_BUFFER_SIZE;
    }
    else
    {
      toRead = binReminder;
    }
    //Update the remaining data to read
    binReminder-= toRead;
    //Read the data
    ReadFile(pInputParams->hBinFile, binFileBuffer, toRead, &bytesOps, NULL);
    if(bytesOps == toRead)
    {
      //Read operation finished sucessfully, write the data to FLASH
      status = ENG_Write(pInputParams->hCom, currentWriteAddress, (uint8_t*)binFileBuffer, toRead, WRITE_DATA_TO);
      if(status != gEngSuccessOp_c)
      {
        printf("\nError trying to write a data chunck to the FLASH: %d", status);
        PrintHelpText();
        return -1;
      }
      //Update the write address for the next write
      currentWriteAddress+= toRead;
      printf(".");
    }
    else
    {
      //Error reading the data. Inform the user and exit the function.
      printf("\nError: Could not read the binary.\n");
      PrintHelpText();
      return -1;
    }
  }

  //All data written, commit the image by writing the FLASH header
  if(pInputParams->bSecured)
  {
    status = ENG_Commit(pInputParams->hCom, fileSize, engSecured_c, WRITE_DATA_TO);
  }
  else
  {
    status = ENG_Commit(pInputParams->hCom, fileSize, engUnsecured_c, WRITE_DATA_TO);
  }
  if(status != gEngSuccessOp_c)
  {
    printf("\nError executing commit: %d", status);
    PrintHelpText();
    return -1;
  }
  
  //All done, return success
  return 0;
}

/************************************************************************************
*  Tries to dump the FLASH content of the MC1322x to the user specified file. The SSL
* should be running on the MC1322x, waiting for commands from the host to execute on
* the MC1322x (see Engine.h for available commands).
*  
*  Input parameters:
*  - hCom: comm port opened for the communication with the MC1322x.
*  - fileName: file name to save the FLASH content into. The function will try to open
*      this file.
*  - startAddress: start address in FLASH to the dump from.
*  - length: number of bytes to dump.
*
*  Return:
*  -  0: success.
*  - -1: failure.
************************************************************************************/
int DumpFLASH(HANDLE hCom, char* fileName, uint32_t startAddress, uint32_t length)
{
  HANDLE hFile;
  DWORD bytesOps = 0;
  ENGCmdStatus_t status;
  uint32_t currentReadAddress;
  int binReminder;
  int toRead;

  //Try to open the binary file to be written into flash
  hFile = CreateFile(fileName, GENERIC_WRITE, FILE_SHARE_READ, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
  if( hFile == INVALID_HANDLE_VALUE)
  {
      printf("\nError: Can't create the specified binary file.\n");
      return -1;
  }

  //Initialize the write address with the first byte after the FLASH header
  currentReadAddress = startAddress;
  binReminder = length;
  while(binReminder != 0)
  {
    //Determine how much can we read
    if(binReminder > ENG_BUFFER_SIZE)
    {
      toRead = ENG_BUFFER_SIZE;
    }
    else
    {
      toRead = binReminder;
    }
    //Update the remaining data to read
    binReminder-= toRead;
    status = ENG_Read(hCom, currentReadAddress, (uint8_t*)binFileBuffer, toRead, READ_DATA_TO);
    if(status != gEngSuccessOp_c)
    {
      printf("\nError trying to read a data chunck from the FLASH: %d", status);
      CloseHandle(hFile);
      return -1;
    }
    WriteFile(hFile, binFileBuffer, toRead, &bytesOps, NULL);
    if(bytesOps != toRead)
    {
      printf("\nError trying to write a data chunck to the file");
      CloseHandle(hFile);
      return -1;
    }
    //Update the read address
    currentReadAddress+= toRead;
  }

  CloseHandle(hFile);
  return 0;
}

/************************************************************************************
*  Clean-up before exiting. All handles are checked and closed if found open.
*  
*  Input parameters:
*  - pInputParams: structure containing opened handles to the bin file and comm port
*    with the MC1322x.
*
*  Return:
*  - Nothing.
************************************************************************************/
void CleanUp(inputParams_t* pInputParams)
{
  //Clean-up: all allocated handles must be freed.
  VERIFY_CLOSE_HANDLE(pInputParams->hCom);
  VERIFY_CLOSE_HANDLE(pInputParams->hBinFile);
  VERIFY_CLOSE_HANDLE(pInputParams->hSSLBin);
}


