/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2008 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file      mcu.c
*
* @author    rc574c
*
* @version   1.0.0.0
*
* @date      Dec-11-2008
*
* @brief     MCU init related function.
*
******************************************************************************/

/******************************************************************************
* Include
******************************************************************************/

#include "jdp.h"
#include "mcu.h"
#include "typedefs.h"

/***************************************************************************//*!
*
* @brief   Init 64MHz MCU bus clock, when using 8MHz crystal and enable all
* 		   peripherals.
*
* @return  N/A
*
******************************************************************************/

void Mcu_Per_Enb_Osc_Init(void)
{
	// Enable all peripheral clocks, division ration 1
	CGM.SC_DC[0].B.DE = 1;
	CGM.SC_DC[0].B.DIV = 0;
	CGM.SC_DC[1].B.DE = 1;
	CGM.SC_DC[1].B.DIV = 0;
	CGM.SC_DC[2].B.DE = 1;
	CGM.SC_DC[2].B.DIV = 0;
    // Setting RUN Configuration Register ME_RUN_PC[0]
    // Peripheral ON in every mode
    ME.RUNPC[0].R = 0x000000FE;

    // Re-enter in DRUN mode to update
    // Mode & Key
    ME.MCTL.R = 0x30005AF0;
    // Mode & Key inverted
    ME.MCTL.R = 0x3000A50F;

    // Configure CLKOUT on PA0 (for debugging)
    // Enable Output Clock
    CGM.OC_EN.B.EN = 1;
    // Divide Output Clock by 8
    CGM.OCDS_SC.B.SELDIV = 3;
    // PLL is Output Clock (Xtal/RC/PLL - 0/1/2)
    CGM.OCDS_SC.B.SELCTL = 2;

    // Switch on external osc
    ME.DRUN.B.FXOSC0ON = 1;
    // RE enter the drun mode, to update the configuration
    // Mode & Key
    ME.MCTL.R = 0x30005AF0;
    // Mode & Key inverted
    ME.MCTL.R = 0x3000A50F;
    // Wait for external osc to stabilize
    while(ME.GS.B.S_FXOSC == 0);
    // Wait for mode entry to complete
    while(ME.GS.B.S_MTRANS == 1);
    // Check DRUN mode has been entered
    while(ME.GS.B.S_CURRENTMODE != 3);

    // Max bus 64MHz, VCO 256-512MHz
    // Fpll = XTAL * NDIV / IDF / ODF = 64MHz
    // Fvco = XTAL * NDIV / IDF = 512MHz
    // Divide by 1
    CGM.FMPLL_CR.B.IDF = 0x0000;
    // Divide by 8
    CGM.FMPLL_CR.B.ODF = 0x2;
    // Loop divide by 64
    CGM.FMPLL_CR.B.NDIV = 64;

	// Enable pll
    ME.DRUN.B.FMPLLON = 1;
    // System clock is IRC
    ME.DRUN.B.SYSCLK = 0x0;

    // RE enter the drun mode, to update the configuration
    // Mode & Key
    ME.MCTL.R = 0x30005AF0;
    // Mode & Key inverted
    ME.MCTL.R = 0x3000A50F;
    // Wait for mode entry to complete
    while(ME.GS.B.S_MTRANS == 1);
    // Check DRUN mode has been entered
    while(ME.GS.B.S_CURRENTMODE != 0x3);

    // Wait for PLL to lock - will not lock before DRUN re-entry
    while(CGM.FMPLL_CR.B.S_LOCK == 0);

	// System clock is PLL
    ME.DRUN.B.SYSCLK = 0x4;
    // Mode & Key
    ME.MCTL.R = 0x30005AF0;
    // Mode & Key inverted
    ME.MCTL.R = 0x3000A50F;
    // Wait for mode entry to complete
    while(ME.GS.B.S_MTRANS == 1);
    // Check DRUN mode has been entered
    while(ME.GS.B.S_CURRENTMODE != 0x3);

    /* ME_GS Poll Global status register to get current system clock
	fail if system clock is not pll
		0000 16MHz internal RC oscillator
		0001 divided 16MHz internal RC oscillator
		0010 4MHz crystal oscillator
		0011 divided 4MHz crystal oscillator
		0100 system PLL
		1111 system clock is disabled */
    // Fail is stuck here
    while(ME.GS.B.S_SYSCLK != 4){};
}

/***************************************************************************//*!
*
* @brief   Disable Watchdog.
*
* @return  N/A
*
******************************************************************************/

void Wd_Disable(void)
{
	// Key
	SWT.SR.R = 0x0000c520;
	// Key
	SWT.SR.R = 0x0000d928;
	// Disable WEN
	SWT.CR.R = 0x8000010A;
}

/***************************************************************************//*!
*
* @brief   Initialize interrupts
*
* @return  N/A
*
******************************************************************************/

void MCU_Int_Init(void)
{
	vuint8_t i;

	// IVPR = address base used with IVOR's
	asm ("e_lis r5, __IV_ADDR@h");
  	asm ("e_or2i r5, __IV_ADDR@l");
  	asm ("mtIVPR r5 ");

    // Initialize INTC
    INTC.MCR.B.HVEN = 1;
    // Offset 4B
    INTC.MCR.B.VTES = 0;

    // Interrupt priority 4 for each vector
    for (i = 0; i < 148; i++)
    {
    	INTC.PSR[i].R = 4;
    }

    // Ensure INTC's current priority is 0
    INTC.CPR.B.PRI = 0;
}

/***************************************************************************//*!
*
* @brief   Generate 4ms delay.
*
* @return  N/A
*
******************************************************************************/

void Delay_4ms(void)
{
	uint32_t i;

	for(i=0; i<DELAY_4mS; i++ )
	{
		asm (" nop " );
	}
}
