/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2008 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file      emios.c
*
* @author    rc574c
*
* @version   1.0.0.0
*
* @date      Dec-11-2008
*
* @brief     EMIOS 0 routines.
*
******************************************************************************/

/******************************************************************************
* Include
******************************************************************************/

#include "jdp.h"
#include "emios.h"

/******************************************************************************
* Definition
******************************************************************************/

// Channels parameters array
chPar ch[];

/***************************************************************************//*!
*
* @brief   EMIOS 0 OPWMT channels configuration.
*
* @return  N/A
*
******************************************************************************/

void Mcu_Emios_0_Channel_OPWMT_Init(vuint8_t i)
{
	vuint32_t chDutyCycle, chStartDelay, chTriggerAdc, counterPeriod;
	vuint16_t chDutyCycleStep, chStartDelayStep, chTriggerAdcStep;


	// Select channel PWM period: 100Hz, 200Hz
	switch(ch[i].perSel)
	{
		case 100:
		{
			// Change counter clock to 100Hz (Counter Bus A)
			EMIOS_0.CH[i].CCR.B.BSL = 0;

			// Load recalculation constants
			chDutyCycleStep = START_DELAY_TIME_STEP_100;
			chStartDelayStep = DUTY_CYCLE_STEP_100;
			chTriggerAdcStep = TRIGGER_ADC_STEP_100;
			counterPeriod = COUNTER_PERIOD_100;
			break;
		}
		case 200:
		{
			// Change counter clock to 200Hz (Counter Bus D)
			EMIOS_0.CH[i].CCR.B.BSL = 1;

			// Load recalculation constants
			chDutyCycleStep = START_DELAY_TIME_STEP_200;
			chStartDelayStep = DUTY_CYCLE_STEP_200;
			chTriggerAdcStep = TRIGGER_ADC_STEP_200;
			counterPeriod = COUNTER_PERIOD_200;
			break;
		}
	}


    // Channel Start delay time
    // Recalculate PWM leading edge Start delay time
	chStartDelay = (vuint32_t)(ch[i].startDelayTime * chStartDelayStep);
	// Load PWM leading edge start delay time to A1
	EMIOS_0.CH[i].CADR.R = chStartDelay;


    // Channel PWM Duty Cycle
	// Recalculate Duty cycle
	chDutyCycle = (vuint32_t)(ch[i].dutyCycle * chDutyCycleStep);
    // Load duty cycle to channel B2 register (transfered to B1 on counter A1 match)
	if (chDutyCycle == counterPeriod)
	{
		// PWM Duty cycle 100%
		EMIOS_0.CH[i].CBDR.R = chDutyCycle;
	}
	else if ((chDutyCycle + chStartDelay) > counterPeriod)
	{
		// PWM trailing edge after counter roll over
		EMIOS_0.CH[i].CBDR.R = chDutyCycle + chStartDelay - counterPeriod;
	}
	else
	{
		// PWM trailing edge before counter roll over
		EMIOS_0.CH[i].CBDR.R = chDutyCycle + chStartDelay;
	}


    // Channel ADC trigger point
	// Recalculate Trigger ADC
	chTriggerAdc = (vuint32_t)(ch[i].triggerAdc * chTriggerAdcStep);
	// Load channel trigger ADC to A2 register
	if ((chTriggerAdc + chStartDelay) > counterPeriod)
	{
		// Channel ADC trigger occurs after counter roll over
		EMIOS_0.CH[i].ALTCADR.R = chTriggerAdc + chStartDelay - counterPeriod;
	}
	else
	{
		// Channel ADC trigger occurs before counter roll over
		EMIOS_0.CH[i].ALTCADR.R = chTriggerAdc + chStartDelay;
	}
}

/***************************************************************************//*!
*
* @brief   EMIOS 0 channels initial parameters storage in channel data filed
*
* @return  N/A
*
******************************************************************************/

void Mcu_Emios_0_Channel_Param_Load(void)
{
	// Channel PWM1
	// Period 100Hz
	ch[PWM1].perSel = 100;
	// Leading edge at begin of counting cycle
	ch[PWM1].startDelayTime = 0;
	// Duty cycle 50%
	ch[PWM1].dutyCycle = 100;
	// ADC trigger point 25%
	ch[PWM1].triggerAdc = 50;

	// Channel PWM2
	// Period 100Hz
	ch[PWM2].perSel = 100;
	// Leading edge at 5% from begin of counting cycle
	ch[PWM2].startDelayTime = 10;
	// Duty cycle 50%
	ch[PWM2].dutyCycle = 100;
	// ADC trigger point 25%
	ch[PWM2].triggerAdc = 50;

	// Channel PWM3
	// Period 100Hz
	ch[PWM3].perSel = 100;
	// Leading edge at 10% from begin of counting cycle
	ch[PWM3].startDelayTime = 20;
	// Duty cycle 95%
	ch[PWM3].dutyCycle = 190;
	// ADC trigger point 25%
	ch[PWM3].triggerAdc = 50;
}

/***************************************************************************//*!
*
* @brief   EMIOS 0 timebase init
*
* @return  N/A
*
******************************************************************************/

void Mcu_Emios_0_Channel_Timebase_Init (vuint8_t ch, vuint16_t ctr)
{
	// Channel ch configuration - MC mode
	// Enable prescaler (divide ration 4), global clock, MC mode
	EMIOS_0.CH[ch].CCR.R = 0x0E000310;
	// Config A1
	EMIOS_0.CH[ch].CADR.R = ctr - 1;
}

/***************************************************************************//*!
*
* @brief   EMIOS 0 module initialization.
*
* @return  N/A
*
******************************************************************************/

void Mcu_Emios_0_Init(void)
{
	// Module Configuration register
	// Global prescaler enable, divide ratio 4
	EMIOS_0.MCR.R = 0x04000300;

	// Module Output Update Disable register
	// Enable transfers (A2 -> A1, B2 -> B1) for all 28 channel registers
	EMIOS_0.OUDR.R = 0x00000000;

    // Enable channels: 0, 1, 3, 5, 23 (0 -> enable)
	EMIOS_0.UCDIS.R = 0xFFFFFFFF;
	EMIOS_0.UCDIS.B.CHDIS0 = 0;
	EMIOS_0.UCDIS.B.CHDIS1 = 0;
	EMIOS_0.UCDIS.B.CHDIS3 = 0;
	EMIOS_0.UCDIS.B.CHDIS5 = 0;
	EMIOS_0.UCDIS.B.CHDIS23 = 0;

	// Channel 23 configuration - MC mode - counter bus A 100Hz
	Mcu_Emios_0_Channel_Timebase_Init(CTRA, COUNTER_PERIOD_100);
	// Channel 0 configuration - MC mode - counter bus B 200Hz
	Mcu_Emios_0_Channel_Timebase_Init(CTRB, COUNTER_PERIOD_200);

	// PWM1 channel configuration - OPWMT mode
	// Enable prescaler (divide ration 1), counter bus A clock, OPWMT mode
	EMIOS_0.CH[PWM1].CCR.R = 0x02000026;
	// A match comparator A sets output, while match B clears it.
	EMIOS_0.CH[PWM1].CCR.B.EDPOL = 1;
	// Flag generate DMA request (CTU trigger)
	EMIOS_0.CH[PWM1].CCR.B.DMA   = 1;
	// Enable the flag generate DMA request (CTU trigger)
	EMIOS_0.CH[PWM1].CCR.B.FEN   = 1;

	// PWM2 channel configuration - OPWMT mode
	// Enable prescaler (divide ration 1), counter bus A clock, OPWMT mode
	EMIOS_0.CH[PWM2].CCR.R = 0x02000026;
	// A match comparator A sets output, while match B clears it.
	EMIOS_0.CH[PWM2].CCR.B.EDPOL = 1;
	// Flag generate DMA request (CTU trigger)
	EMIOS_0.CH[PWM2].CCR.B.DMA   = 1;
	// Enable the flag generate DMA request (CTU trigger)
	EMIOS_0.CH[PWM2].CCR.B.FEN   = 1;

	// PWM3 channel configuration - OPWMT mode
	// Enable prescaler (divide ration 1), counter bus A clock, OPWMT mode
	EMIOS_0.CH[PWM3].CCR.R = 0x02000026;
	// A match comparator A sets output, while match B clears it.
	EMIOS_0.CH[PWM3].CCR.B.EDPOL = 1;
	// Flag generate DMA request (CTU trigger)
	EMIOS_0.CH[PWM3].CCR.B.DMA   = 1;
	// Enable the flag generate DMA request (CTU trigger)
	EMIOS_0.CH[PWM3].CCR.B.FEN   = 1;

	// Load PWM channels initial data into related channels structures
	Mcu_Emios_0_Channel_Param_Load();

	// Configure eMIOS channels PWM1, PWM2, PWM3
	Mcu_Emios_0_Channel_OPWMT_Init(PWM1);
	Mcu_Emios_0_Channel_OPWMT_Init(PWM2);
	Mcu_Emios_0_Channel_OPWMT_Init(PWM3);

	// Global time base enable
	EMIOS_0.MCR.B.GTBE = 1;
}
