/******************************************************************************
*                                                  COPYRIGHT (c) Freescale 2009
* FILE NAME: stepper_driver.c     REVISION 1.0
*                                                                           
* PURPOSE: Contains the Stepper motor driver external APIs
*                                                                           
*******************************************************************************
*******************************************************************************  
**  THIS CODE IS ONLY INTENDED AS AN EXAMPLE FOR THE CODEWARRIOR COMPILER    **  
**      AND THE S08 EVB AND HAS ONLY BEEN GIVEN A MIMIMUM LEVEL OF TEST.     **  
**  IT IS PROVIDED 'AS SEEN' WITH NO GUARANTEES AND NO PROMISE OF SUPPORT.   **  
*******************************************************************************  

*******************************************************************************
Freescale reserves the right	to make	changes	without	further	notice to any
product	herein to improve reliability, function	or design. Freescale	does not
assume any liability arising out of	the	application	or use of any product,
circuit, or software described	herein;	neither	does it	convey any license
under its patent rights	nor	the	rights of others. Freescale products are	not
designed, intended,	or authorized for use as components	in systems  intended for
surgical implant into the body, or	other applications intended to support life,
or	for	any	other application in which the failure of the  Freescale product
could create a	situation where	personal injury	or death may occur. Should
Buyer	purchase or	use	Freescale products for any such unintended or
unauthorized	application, Buyer shall idemnify and hold Freescale	and	its
officers, employees, subsidiaries, affiliates, and distributors	harmless
against	all	claims costs, damages, and expenses, and reasonable	attorney fees 
arising	out	of,	directly or	indirectly,	any	claim of personal injury or	death
associated with	such unintended	or unauthorized use,	even if	such claim alleges
that Freescale was negligent	regarding the design or	manufacture	of the part.
Freescale and the Freescale logo*	are	registered trademarks of Freescale Ltd.
*******************************************************************************
*******************************************************************************
*
*
* DESCRIPTION:  file for Stepper Motor Driver
*
* NOTE:
*
* AUTHOR: Saurabh Jhamb        LOCATION: IDC      LAST EDIT DATE: 09JAN09
*
******************************************************************************/
#include "stepper_driver.h"
#include "ref_design.h"


/****************************************************************************
 * Function Name: InitStepper
 *
 * Agruments:
 * NONE
 *
 * Return Type: VOID
 *
 * Description:
 *    It initializes the working mode of Stepper as PIN1-TPM1CH0, PIN2-PTI0
 * , PIN3-PTI1, PIN4-TPM1CH1. It initializes all the required TPM and GPIO
 *  registers, but does not enables the TPM clock.
 *
 * 
 ****************************************************************************/
void InitStepper()
{
  current_position = 0;
#if MODE_2TPM_2GPIO
/* This is the mode in which 2 of the 4 pins of stepper motor are operated with TPM signals
   and the remaining 2 are operated with GPIOs
   PIN 1 - TPM1CH0    PTF1
   PIN 2 - PTA4 GPIO
   PIN 3 - PTA5 GPIO
   PIN 4 - TPM1CH1    PTH4
*/
  SCGC1_TPM1=0x1;
  //PINPS2_TPM10 = 1;
  PINPS2_TPM11 = 1;

  TPM1CNT = 0x01;       //clears the TPM counter
  TPM1MOD = 862;       //specify the modulus value  //1/4 ms

  //configure TPM channels to operate in Edge-aligned mode.
  TPM1C0SC_MS0x = 0x2;
  TPM1C1SC_MS1x = 0x2;

  //configure TPM channels to follow Low True pulses
  TPM1C0SC_ELS0x = 0x1;
  TPM1C1SC_ELS1x = 0x1;

  TPM1SC_TOIE = 1;
  
  //GPIO Init
  PTADD_PTADD4 = 0x1;
  PTADD_PTADD5 = 0x1;
  PTHDD_PTHDD4 = 0x1;
  PTFDD_PTFDD1 = 0x1;
  PTAPE = 0xFF;

  //GPIO for CS of current driver IC
  PTIDD_PTIDD3 = 0x1;
  PTID_PTID3 = CURRENT_DRIVER_IC_OFF;
#endif
}

/****************************************************************************
 * Function Name: DriveStepper_to_zero
 *
 * Agruments:
 * NONE
 *
 * Return Type: VOID
 *
 * Description:
 *    It follows a specified process to get stepper to initial ZERO position
 * This function might be different for different stepper motors. It
 * basically follows a predefined movement to avoid jitter and noisy movement
 * to initial ZERO position.
 *
 * 
 ****************************************************************************/
void DriveStepper_to_zero()
{
  current_position = 0;

  Move10DegreesClkwise();       //move 10 degrees clkwise with initial speed
  StepperSpeedUpAntiClkwise();  //move 8 degree anticlkwise with increasing
                                //speed, then move 300 degrees with full speed
}

/****************************************************************************
 * Function Name: DriveStepper_to_Max
 *
 * Agruments:
 * NONE
 *
 * Return Type: VOID
 *
 * Description:
 *    It follows a specified process to get stepper to MAX position
 * This function might be different for different stepper motors. It
 * basically follows a predefined movement to avoid jitter and noisy movement
 * to MAX position.
 *
 * 
 ****************************************************************************/
void DriveStepper_to_Max()
{
  StepperSpeedUpClkwise();  //move 8 degree anticlkwise with increasing
                                //speed, then move 300 degrees with full speed
  current_position = MAX_RPM;
}

/****************************************************************************
 * Function Name: MotorCurrentDriverCE_Toggle
 *
 * Agruments:
 * 1.currentdriver: It holds the enable/disable value for CE of the driver IC
 *
 * Return Type: VOID
 *
 * Description:
 *    It toggles the ChipSelect signal(Active low) for the current driver IC
 * , which is required to operate the stepper motor.
 *
 * 
 ****************************************************************************/
void MotorCurrentDriverCE_Toggle(unsigned char currentdriver)
{
  //Enable here the current driver IC to start driving the Stepper motor
  PTID_PTID3 = currentdriver;
}

/****************************************************************************
 * Function Name: move_motor_microstep
 *
 * Agruments:
 * 1.num_of_steps: It holds the number of steps to be moved in stepper motor
 * 2.direction: It holds the direction in which the specified steps are moved
 *
 * Return Type: VOID
 *
 * Description:
 *    It is the function providing movement of stepper motor in microsteps
 * that is 1/12 degree in each micro step
 *
 * 
 ****************************************************************************/
void move_motor_microstep(unsigned int num_of_steps, unsigned char direction)
{
  microstep_active = 1;
  num_steps_taken = 0;

  //Start the already configured TPM and GPIOs
  //PTAD_PTAD4 = 0x1;
  //PTAD_PTAD5 = 0x1;
  PTAD_PTAD4 = 0x0;
  PTAD_PTAD5 = 0x0;
  TPM1MOD = 862;    //modify the waveform period accordingly
  TPM1CNT = 0x01;
  TPM1C0SC_ELS0x = 0x1;
  TPM1C1SC_ELS1x = 0x1;
  TPM1SC_CLKSA = 1;
  
  wait = 1;
  
  while((!num_of_steps) || (num_steps_taken < num_of_steps))
  {
    while(wait);//cleared by interrupt
    motor_micro_update_pwm(current_position + num_steps_taken, direction);

    num_steps_taken++;
  }
  //switch off the TPM and GPIO waveform to 0.
  TPM1C0SC_ELS0x = 0x0;
  TPM1C1SC_ELS1x = 0x0;
  TPM1SC_CLKSA = 0x0;
  PTHD_PTHD4 = 0;
  PTFD_PTFD1 = 0;
  PTAD_PTAD4 = 0x0;
  PTAD_PTAD5 = 0x0;

  microstep_active = 0;
  //if(STEPPER_MOVE_ANTICLKWISE == direction)
  //{
    //current_position = (current_position > num_steps_taken)? (current_position - num_steps_taken):0;
    //decrement the current position
  //}
  //else
  {
    current_position+= num_steps_taken;     //increment the current position
  }
}

/****************************************************************************
 * Function Name: move_motor_partialstep
 *
 * Agruments:
 * 1.num_of_steps: It holds the number of steps to be moved in stepper motor
 * 2.direction: It holds the direction in which the specified steps are moved
 *
 * Return Type: VOID
 *
 * Description:
 *    It is the function providing movement of stepper motor in partialsteps
 * that is 1/3 degree in each partial step
 *
 * 
 ****************************************************************************/
void move_motor_partialstep(unsigned int num_of_steps, unsigned char direction)
{  
  //In case motor is not aligned to partial step boundary, 1 partial step
  //will just align the motor to partial step boundary in the direction requested
  if(current_position%4)
  {
    if(STEPPER_MOVE_CLKWISE == direction)
    {
      //move motor to partialstep boundary clkwise
      move_motor_microstep(4-(current_position%4), direction);
    }
    else
    {
      //move motor to partialstep boundary clkwise
      move_motor_microstep(current_position%4, direction);
    }
    return;
  }
  
  //PTAD_PTAD4 = 0x1;
  //PTAD_PTAD5 = 0x1;
  PTAD_PTAD4 = 0x0;
  PTAD_PTAD5 = 0x0;
  TPM1C0V = 3448;
  TPM1C1V = 3448;
  TPM1MOD = 3448;//1ms period
  TPM1C0SC_ELS0x = 0x2;
  TPM1C1SC_ELS1x = 0x2;
  TPM1SC_CLKSA = 1;
  num_steps_taken = 0;

  while((!num_of_steps) || (num_steps_taken < num_of_steps))
  {
    motor_partial_update_pwm(current_position + (num_steps_taken*4), direction);

    while(wait);//cleared by interrupt
    
    num_steps_taken++;
  }
  TPM1C0SC_ELS0x = 0x0;
  TPM1C1SC_ELS1x = 0x0;
  TPM1SC_CLKSA = 0x0;
  PTHD_PTHD4 = 0;
  PTFD_PTFD1 = 0;
  PTAD_PTAD4 = 0x0;
  PTAD_PTAD5 = 0x0;

  num_steps_taken *= 4;
  
  if(STEPPER_MOVE_ANTICLKWISE == direction)
  {
    current_position = (current_position > num_steps_taken)? (current_position - num_steps_taken):0;
    //decrement the current position
  }
  else
  {
    current_position+= num_steps_taken;     //increment the current position
  }
}

/****************************************************************************
 * Function Name: move_motor_fullstep
 *
 * Agruments:
 * 1.num_of_steps: It holds the number of steps to be moved in stepper motor
 * 2.direction: It holds the direction in which the specified steps are moved
 *
 * Return Type: VOID
 *
 * Description:
 *    It is the function providing movement of stepper motor in fullsteps
 * that is 1 degree in each full step
 *
 * 
 ****************************************************************************/
void move_motor_fullstep(unsigned int num_of_steps, unsigned char direction)
{
  //In case motor is not aligned to full step boundary, 1 full step
  //will just align the motor to full step boundary in the direction requested
  if((current_position/4)%3)
  {
    if(current_position%4)
    {
      //if motor is also not aligned to partial step boundary
      if(STEPPER_MOVE_CLKWISE == direction)
      {
        move_motor_microstep(4-(current_position%4), direction);
        move_motor_partialstep((24-(current_position%24))/4, direction);
      }
      else
      {
        move_motor_microstep(current_position%4, direction);
        move_motor_partialstep((current_position%24)/4, direction);
      }
    }
    else
    {
    //if motor is also aligned to partial step boundary
      if(STEPPER_MOVE_CLKWISE == direction)
      {
        move_motor_partialstep((24-(current_position%24))/4, direction);
      }
      else
      {
        move_motor_partialstep((current_position%24)/4, direction);
      }
    }
    return;
  }
  //initialize GPIOs and TPM waveform period
  PTAD_PTAD4 = 0x1;
  PTAD_PTAD5 = 0x1;
  TPM1C0V = 3448;
  TPM1C1V = 3448;
  TPM1MOD = 3448;
  TPM1C0SC_ELS0x = 0x1;
  TPM1C1SC_ELS1x = 0x1;
  TPM1SC_CLKSA = 1;

  num_steps_taken = 0;

  while((!num_of_steps) || (num_steps_taken < (3*num_of_steps)))
  {
    motor_partial_update_pwm(current_position + num_steps_taken, direction);

    while(wait);//cleared by interrupt
    
    num_steps_taken+=4;
  }
  //switch off the TPM and GPIO functionality to stop motor
  TPM1C0SC_ELS0x = 0x0;
  TPM1C1SC_ELS1x = 0x0;
  TPM1SC_CLKSA = 0x0;
  PTHD_PTHD4 = 0;
  PTFD_PTFD1 = 0;
  PTAD_PTAD4 = 0x0;
  PTAD_PTAD5 = 0x0;

  if(STEPPER_MOVE_ANTICLKWISE == direction)
  {
    current_position = (current_position > num_steps_taken)? (current_position - num_steps_taken):0;
    //decrement the current position
  }
  else
  {
    current_position+= num_steps_taken;     //increment the current position
  }
}

/*
 * TPM1 block overflow interrupt service routine
 */
void interrupt VectorNumber_Vtpm1ovf  TPM1_OVF_ISR()
{
  byte dummy;

  //clear the interrupt flag
  dummy = TPM1SC;
  TPM1SC_TOF = 0;
  
  if(!microstep_active)
  {
    //update GPIO only in case of microstepping
    PTAD_PTAD4 = pin2val;
    PTAD_PTAD5 = pin3val;
  }
  if(wait)
  {
    wait = 0;   //clear the flag used for synchronization
  }
}