#include "lcd.h"
#include "lcd_config.h"
#include "ref_design.h"
#include "mc9s08lg32.h"

void LCDClearAll()
{
  LCDClearAllSegments();
  LCDGlowSegment(FREESCALE, 0);
}

void LCDActivateSkeleton()
{
  LCDClearAll();

  LCDGlowSegment(BATTERY, 0);
  LCDGlowSegment(FUELBORDER, 0);
  LCDGlowSegment(THERMOMETER, 0);
}

void LCDChangeFuelLevel(unsigned char level)
{
  LCDGlowSegment(FUELBORDER, 0);
  LCDClearSegment(FUELLEVEL);
  LCDGlowSegment(FUELLEVEL, level);
}

void LCDChangeBattLevel(unsigned char level)
{
  LCDGlowSegment(BATTERY, 0); //battery sign cover
  LCDClearSegment(BATTLEVEL);
  LCDGlowSegment(BATTLEVEL, level);
}

void LCDChangeOilIndication(unsigned char oil_indication_on)
{
  if(oil_indication_on)
  {
    LCDGlowSegment(OIL, 0);
  }
  else
  {
    LCDClearSegment(OIL);
  }
}

void LCDColdHotIndication(unsigned char condition)
{
  switch(condition)
  {
    case ENGINE_TEMP_HOT:
      LCDGlowSegment(HOT, 0);
      break;
    case ENGINE_TEMP_NEUTRAL:
      LCDGlowSegment(THERMOMETER, 0);
      break;
    case ENGINE_TEMP_COLD:
      LCDGlowSegment(COLD, 0);
  }
}

void LCDHVACControls(unsigned char condition)
{
  switch(condition)
  {
    case LCD_HVAC_FAN_CONTROL:
      LCDGlowSegment(HVACFAN, 0);
      break;
    case LCD_HVAC_AIRFLOW_CONTROL:
      LCDGlowSegment(HVACAIRFLOW, 0);
      break;
    case LCD_HVAC_AIRFLOWSINGLE_CONTROL:
      LCDGlowSegment(HVACAIRFLOWSINGLE, 0);
      break;
    case LCD_HVAC_AIRFLOWDUAL_CONTROL:
      LCDGlowSegment(HVACAIRFLOWDUAL, 0);
      break;
    case LCD_HVAC_AIRFLOWWINDOW_CONTROL:
      LCDGlowSegment(HVACAIRFLOWWINDOW, 0);
  }
}

void LCDServiceIndication(unsigned char servicedue)
{
  if(servicedue)
  {
    LCDGlowSegment(SERVICE, 0);
  }
  else
  {
    LCDClearSegment(SERVICE);
  }
}

void LCDUpdateSpeed(unsigned int speed)
{
  unsigned char character;
  unsigned char displaydigit = 3;

  if(speed > LCD_MAX_SPEED)
  {
    speed = LCD_MAX_SPEED;
  }

  LCDClearDigit(1);
  LCDClearDigit(2);
  LCDClearDigit(3);
  LCDGlowSegment(KMPH, 0);

  do
  {
    character = speed%10;
    character += '0';
    LCDPutChar(character, displaydigit);
    displaydigit--;
    speed /= 10;
  }while(speed);
}

void LCDUpdateDistance(unsigned long int odo)
{
  unsigned char character;
  unsigned char displaydigit = 9;

  if(odo > LCD_MAX_DISTANCE)
    return;

  LCDClearDigit(4);
  LCDClearDigit(5);
  LCDClearDigit(6);
  LCDClearDigit(7);
  LCDClearDigit(8);
  LCDClearDigit(9);

  do
  {
    character = odo%10;
    character += '0';
    LCDPutChar(character, displaydigit);
    displaydigit--;
    odo /= 10;
    if(displaydigit < 4)
      break;
  }while(odo);
}

void LCDUpdateTemperature(unsigned long int temp)
{
  unsigned char character;
  unsigned char displaydigit = 7;

  if(temp > LCD_MAX_TEMPERATURE)
    return;

  LCDClearDigit(4);
  LCDClearDigit(5);
  LCDClearDigit(6);
  LCDClearDigit(7);

  do
  {
    character = temp%10;
    character += '0';
    LCDPutChar(character, displaydigit);
    displaydigit--;
    temp /= 10;
    if(displaydigit < 4)
      break;
  }while(temp);
}

void LCDUpdateTrips(unsigned char tripID, double tripDistance)
{
  unsigned char character, displaydigit = 9;
  unsigned long int odo = (unsigned long)tripDistance;

  LCDClearDigit(4);
  LCDClearDigit(5);
  LCDClearDigit(6);
  LCDClearDigit(7);
  LCDClearDigit(8);
  LCDClearDigit(9);
  LCDClearSegment(TRIPA);
  LCDClearSegment(TRIPB);
  LCDGlowSegment(KM_ODO, 0);
  LCDGlowSegment(TRIP, 0);
  if(LCD_TRIPA_ID == tripID)
  {
    LCDGlowSegment(TRIPA, 0);
  }
  else
  {
    LCDGlowSegment(TRIPB, 0);
  }
  character = ((unsigned int)(tripDistance*10))%10;
  character += '0';
  LCDPutChar(character, displaydigit--);

  LCDGlowSegment(DP, 0);

  do
  {
    character = odo%10;
    character += '0';
    LCDPutChar(character, displaydigit);
    displaydigit--;
    odo /= 10;
    if(displaydigit < 4)
      break;
  }while(odo);
}

void LCD_UpdateClock(unsigned char hours, unsigned char minutes, unsigned char IsPM)
{
  unsigned char character, displaydigit = 9;

  LCDClearDigit(4);
  LCDClearDigit(5);
  LCDClearDigit(6);
  LCDClearDigit(7);
  LCDClearDigit(8);
  LCDClearDigit(9);
  LCDGlowSegment(CLOCK, 0);
  LCDClearSegment(PM);
  LCDClearSegment(AM);

  if(IsPM)
  {
    LCDGlowSegment(PM, 0);
  }
  else
  {
    LCDGlowSegment(AM, 0);
  }
  
  if(curr_state.seconds%2)
  {
    LCDGlowSegment(COL, 0);
  }
  else
  {
    LCDClearSegment(COL);
  }

  do
  {
    character = minutes%10;
    character += '0';
    LCDPutChar(character, displaydigit);
    displaydigit--;
    minutes /= 10;
  }while(7 != displaydigit);
  
  do
  {
    character = hours%10;
    character += '0';
    LCDPutChar(character, displaydigit);
    displaydigit--;
    hours /= 10;
  }while(5 != displaydigit);
}

void LCD_UpdateCurrentDisplay()
{
  switch(curr_state.lcddisplaymode)
  {
    case LCD_DISPLAY_TRIPA:
      LCDGlowSegment(TRIP, 0);
      LCDGlowSegment(TRIPA, 0);
      LCDUpdateTrips(LCD_TRIPA_ID, curr_state.tripAdistance);
      break;
    case LCD_DISPLAY_TRIPB:
      LCDGlowSegment(TRIP, 0);
      LCDGlowSegment(TRIPB, 0);
      LCDUpdateTrips(LCD_TRIPB_ID, curr_state.tripBdistance);
      break;
    case LCD_DISPLAY_KLEFT:
      LCDGlowSegment(KMLEFT, 0);
      LCDGlowSegment(KM_ODO, 0);
      LCDUpdateDistance(curr_state.kmleft);
      break;
    case LCD_DISPLAY_MILE:
      LCDGlowSegment(MILEAGE, 0);
      LCDGlowSegment(KMPL, 0);
      LCDUpdateDistance(curr_state.mileage);
      break;
    case LCD_DISPLAY_ODOM:
      LCDGlowSegment(ODO, 0);
      LCDGlowSegment(KM_ODO, 0);
      LCDUpdateDistance(curr_state.distance);
      break;
    case LCD_DISPLAY_CLCK:
      LCDGlowSegment(CLOCK, 0);
      LCDGlowSegment(COL, 0);
      LCD_UpdateClock(curr_state.hours, curr_state.minutes, curr_state.IsPM);
      break;
    case LCD_DISPLAY_TEMP:
      LCDGlowSegment(TEMPERATURE, 0);
      LCDGlowSegment(DEGREE, 0);
      LCDPutChar('C', 8);
      LCDUpdateTemperature(curr_state.displaytemp);
  }
}

void LCD_ActivateMode(unsigned char mode)
{
  LCDClearSegment(TRIP);
  LCDClearSegment(TRIPA);
  LCDClearSegment(TRIPB);
  LCDClearSegment(KMLEFT);
  LCDClearSegment(MILEAGE);
  LCDClearSegment(ODO);
  LCDClearSegment(CLOCK);
  LCDClearSegment(TEMPERATURE);
  LCDClearSegment(AM);
  LCDClearSegment(PM);
  LCDClearSegment(KMPL);
  LCDClearSegment(KM_ODO);
  LCDClearSegment(DP);
  LCDClearSegment(COL);
  LCDClearSegment(DEGREE);
  LCDClearDigit(4);
  LCDClearDigit(5);
  LCDClearDigit(6);
  LCDClearDigit(7);
  LCDClearDigit(8);
  LCDClearDigit(9);

  switch(mode)
  {
    case LCD_DISPLAY_TRIPA:
      LCDGlowSegment(TRIP, 0);
      LCDGlowSegment(TRIPA, 0);
      LCDUpdateTrips(LCD_TRIPA_ID, curr_state.tripAdistance);
      break;
    case LCD_DISPLAY_TRIPB:
      LCDGlowSegment(TRIP, 0);
      LCDGlowSegment(TRIPB, 0);
      LCDUpdateTrips(LCD_TRIPB_ID, curr_state.tripBdistance);
      break;
    case LCD_DISPLAY_KLEFT:
      LCDGlowSegment(KMLEFT, 0);
      LCDGlowSegment(KM_ODO, 0);
      LCDUpdateDistance(0);
      break;
    case LCD_DISPLAY_MILE:
      LCDGlowSegment(MILEAGE, 0);
      LCDGlowSegment(KMPL, 0);
      LCDUpdateDistance(0);
      break;
    case LCD_DISPLAY_ODOM:
      LCDGlowSegment(ODO, 0);
      LCDGlowSegment(KM_ODO, 0);
      LCDUpdateDistance(curr_state.distance);
      break;
    case LCD_DISPLAY_CLCK:
      LCDGlowSegment(CLOCK, 0);
      LCDGlowSegment(COL, 0);
      LCD_UpdateClock(curr_state.hours, curr_state.minutes, curr_state.IsPM);
      break;
    case LCD_DISPLAY_TEMP:
      LCDGlowSegment(TEMPERATURE, 0);
      LCDGlowSegment(DEGREE, 0);
      LCDPutChar('C', 8);
      LCDUpdateTemperature(curr_state.displaytemp);
  }
}

void LCDBlinkStateChange(unsigned char state)
{
  LCDBCTL_BRATE = 0x2;
  LCDBCTL_BLINK = state;
}