/**************************************************************************/
/* FILE NAME: main.c                       COPYRIGHT (c) Freescale 2008   */
/*                                                All Rights Reserved     */
/* DESCRIPTION:                                                           */
/* Main file for Engine Position Freemaster demo.                         */
/*========================================================================*/
/* REV      AUTHOR        DATE        DESCRIPTION OF CHANGE               */
/* ---   -----------    ----------    ---------------------               */
/* 1.0   G. Emerson     22/Dec/08      Initial version.                   */
/**************************************************************************/

#include "mpc5500_usrccdcfg.h"
#include "mpc5500_xcptn.h" 
#include "freemaster.h"
#include "freemaster/freemaster.h"
#include "etpu_toothgen.h"         /* eTPU ToothGen API */
#include "etpu_app_eng_pos.h"         /* eTPU EMDCT API */
#include "eng_pos_etpu_gct.h"     /* eTPU engine initialization - generated by eTPU GCT */
#include "Freemaster_config.h"
#include "fs_gpio.h"
#include "etpu_util.h"
#include "mpc563m.h"
/******************************************************************************
* Global variables
******************************************************************************/
/* Application State Machine */
vuint8_t app_state;
vuint8_t app_fault;
vuint8_t app_switch;

/* Crank Error reset */
vuint8_t CrankErrorReset =0;

/* Tooth Generator */
vuint32_t tooth_duty_cycle = 0x800000; /* 50% */
vuint32_t motor_speed_start = 500;        
vuint32_t motor_speed = 500; 
vuint32_t accel_rate = 0x28f5c;        /* 1% */
vuint8_t  cam_start = 3 ;//+ 36;
vuint8_t  cam_stop = 5 ;//+ 36;
vuint8_t  error_list = 0;
vuint8_t  active_error_list ;
vuint8_t  missing_tooth_number_1 = 5;
vuint8_t  missing_tooth_number_2 = 5;
/* these values give good nosie pulses at 5Mhz TCR1 - use 5000 @ RPM*/
vuint8_t  crank_noise_tooth_number = 9;
vuint32_t crank_noise_pulse_shift = 1000;  /* 250 us */
vuint32_t crank_noise_pulse_width = 280;  /*  50 us */
/* these values give good nosie pulses at 5Mhz TCR1 - use 5000 @ RPM*/
vuint8_t  cam_noise_tooth_number = 10 ;
vuint32_t cam_noise_pulse_shift = 250;  /* 50us */
vuint32_t cam_noise_pulse_width = 2500;  /* 500us */
vuint8_t Crank_Status = 0; 
vuint8_t Cam_Status = 0;
vuint8_t cam_error_status;  
vuint8_t crank_error_status;  
vuint8_t Eng_Pos_Status =0;
vuint32_t cam_window_open =39000 ;
vuint32_t cam_window_width =1000;
vuint32_t cam_edge_angle ; 
vuint32_t cam_new_window_open =39000;
vuint32_t cam_new_window_end = 40000;
vuint32_t cam_new_window_width =1000;
vuint32_t cam_engine_cycle_0_count;
vuint32_t crank_blank_time_ms = 1 ; /* 1ms */
vuint8_t crank_blank_tooth_count = 6; 
vuint32_t crank_tcr2_ticks_per_tooth = 100;
ufract24_t crank_gap_ratio = 0x9fffff; /* 0.625 */
ufract24_t crank_windowing_ratio_normal = 0x199999; /* 0.1 */
ufract24_t crank_windowing_ratio_across_gap = 0x199999; /* 0.1 */
ufract24_t crank_windowing_ratio_after_gap = 0x199999; /* 0.1 */
ufract24_t crank_windowing_ratio_timeout = 0x199999; /* 0.1 */
vuint32_t crank_first_tooth_timeout_us = 40000 ; /* 40,000 us */
vuint8_t crank_number_of_missing_teeth =1;
vuint8_t crank_number_of_physical_teeth =35;
vuint8_t  crank_tooth_number;
int32_t error =0;
vuint8_t num;
vuint32_t i, j;
/* eTPU and eTPU Load */
extern struct eTPU_struct *eTPU;
extern uint32_t fs_etpu_data_ram_start;
vuint32_t * toothgen_pba;
vuint32_t * toothgen_crank_pba;
vuint32_t * toothgen_cam_pba;
vuint32_t * crank_pba;
vuint32_t * cam_pba;
vuint32_t * QOM_pba;
vuint32_t * knock_pba;
vuint32_t * toothgen_cba;
vuint32_t * toothgen_crank_cba;
vuint32_t * toothgen_cam_cba;
vuint32_t * crank_cba;
vuint32_t * cam_cba;
vuint32_t * knock_cba;
vuint32_t * QOM_cba;
extern uint32_t etpu_a_tcr1_freq;
extern uint32_t etpu_b_tcr1_freq;
vuint16_t etpu_load;
extern uint32_t etpu_load_max;
extern uint32_t etpu_load_min;
extern uint32_t etpu_load_avg;
extern uint32_t misc_ref_time;
extern uint32_t misc_act_time;
vuint8_t crank_error_state = 0;
vint32_t delta;
vuint32_t crank_link_1 = 0x0A0E0707; /* 14 (0x0E) is CAM; 10 (0x0A) is KNOCK */
vuint32_t crank_link_2 = 0x07070707;
vuint32_t crank_link_3 = 0x07070707;
vuint32_t crank_link_4 = 0x07070707;
vuint32_t measured_speed;
extern uint32_t IntcIsrVectorTable[];
extern void IVOR4Handler (void);

#define KNOCK_NUMBER_OF_WINDOWS 8
vuint8_t knock_number_of_windows = KNOCK_NUMBER_OF_WINDOWS;
vuint8_t knock_number_of_physical_teeth =35 ;
vuint8_t knock_number_of_missing_teeth =1;
vuint32_t knock_tcr2_ticks_per_tooth = 100;
uint32_t my_knock_window_angle_table[KNOCK_NUMBER_OF_WINDOWS*2];


/******************************************************************************
* Constants and macros
******************************************************************************/
/* app_state options */
#define APP_STATE_OFF          0
#define APP_STATE_ENABLE       1
#define APP_STATE_ON           2
#define APP_STATE_DISABLE      3
#define APP_STATE_FAULT        4

#define NULL            ((void *)0)

/* app_fault options */
#define APP_FAULT_NO                    0
#define APP_FAULT_OTHER                 1
#define APP_FAULT_ETPU_MICROCODE_GE     2
#define APP_FAULT_ETPU_ILLEGAL_INSTR    3
#define APP_FAULT_ETPU_MISC_ERROR       4

/* app_switch options */
#define APP_SWITCH_OFF   0
#define APP_SWITCH_ON    1

#define CRANK_NOISE_OFF   0
#define CRANK_NOISE_ON    1

#define ETPU_OUTPUT_PAD_CONFIG FS_GPIO_PRIMARY_FUNCTION + \
                               FS_GPIO_MAXIMUM_SLEW_RATE + \
                               FS_GPIO_OUTPUT_DRAIN_DISABLE + \
                               FS_GPIO_READBACK_ENABLE + \
                               FS_GPIO_OUTPUT_BUFFER_ENABLE
                               
#define ETPU_INPUT_PAD_CONFIG  FS_GPIO_PRIMARY_FUNCTION + \
                               FS_GPIO_INPUT_BUFFER_ENABLE  
      
/* FreeMASTER TSA support */
FMSTR_TSA_TABLE_BEGIN(first_table)
    FMSTR_TSA_RW_VAR(knock_number_of_windows, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(cam_new_window_open, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(cam_new_window_width, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(cam_new_window_end, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(crank_tooth_number, FMSTR_TSA_UINT8)
    FMSTR_TSA_RO_VAR(app_state, FMSTR_TSA_UINT8)
    FMSTR_TSA_RO_VAR(app_fault, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(app_switch, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(CrankErrorReset, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(tooth_duty_cycle, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(motor_speed, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(accel_rate, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(cam_start, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(cam_stop, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(error_list, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(missing_tooth_number_1, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(missing_tooth_number_2, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(crank_noise_tooth_number, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(crank_noise_pulse_shift, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_noise_pulse_width, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(cam_noise_tooth_number, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(cam_noise_pulse_shift, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(cam_noise_pulse_width, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(eTPU, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(toothgen_pba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(toothgen_crank_pba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(toothgen_cam_pba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(crank_pba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(cam_pba, FMSTR_TSA_UINT32)    
    FMSTR_TSA_RO_VAR(knock_pba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(toothgen_cba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(toothgen_crank_cba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(toothgen_cam_cba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(crank_cba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(cam_cba, FMSTR_TSA_UINT32)    
    FMSTR_TSA_RO_VAR(knock_cba, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(etpu_a_tcr1_freq, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(etpu_b_tcr1_freq, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(Crank_Status, FMSTR_TSA_UINT8)
    FMSTR_TSA_RO_VAR(Cam_Status, FMSTR_TSA_UINT8)
    FMSTR_TSA_RO_VAR(crank_error_status, FMSTR_TSA_UINT8)
    FMSTR_TSA_RO_VAR(cam_error_status, FMSTR_TSA_UINT8)
    FMSTR_TSA_RO_VAR(Eng_Pos_Status, FMSTR_TSA_UINT8)
    FMSTR_TSA_RO_VAR(etpu_load, FMSTR_TSA_UINT16)
    FMSTR_TSA_RW_VAR(etpu_load_max, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(etpu_load_min, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(etpu_load_avg, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(misc_ref_time, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(misc_act_time, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(cam_window_open, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(cam_window_width, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(cam_edge_angle, FMSTR_TSA_UINT32)
    FMSTR_TSA_RO_VAR(cam_engine_cycle_0_count, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_blank_time_ms, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_blank_tooth_count, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(crank_tcr2_ticks_per_tooth, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_gap_ratio, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_windowing_ratio_normal, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_windowing_ratio_across_gap, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_windowing_ratio_after_gap, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_windowing_ratio_timeout, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_first_tooth_timeout_us, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(crank_number_of_missing_teeth, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(crank_number_of_physical_teeth, FMSTR_TSA_UINT8)
    FMSTR_TSA_RO_VAR(fs_etpu_data_ram_start, FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[0],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[1],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[2],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[3],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[4],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[5],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[6],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[7],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[8],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[9],  FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[10], FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[11], FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[12], FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[13], FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[14], FMSTR_TSA_UINT32)
    FMSTR_TSA_RW_VAR(my_knock_window_angle_table[15], FMSTR_TSA_UINT32)
    
    
FMSTR_TSA_TABLE_END()

FMSTR_TSA_TABLE_LIST_BEGIN()
    FMSTR_TSA_TABLE(first_table)
FMSTR_TSA_TABLE_LIST_END()
/* End of FreeMASTER TSA support */

void eSCI_init();
void eMIOS_init();
void INTC_init();

void main (void) { 

   eSCI_init();
   eMIOS_init();
   INTC_init();
   FMSTR_Init();
   initIrqVectors();

  my_knock_window_angle_table[0] = 3000;
  my_knock_window_angle_table[1] = 1500; 
  my_knock_window_angle_table[2] = 9000;
  my_knock_window_angle_table[3] = 3000;
  my_knock_window_angle_table[4] = 15000;
  my_knock_window_angle_table[5] = 3000;
  my_knock_window_angle_table[6] = 21000;
  my_knock_window_angle_table[7] = 3000;
  my_knock_window_angle_table[8] = 27000;
  my_knock_window_angle_table[9] = 3000;
  my_knock_window_angle_table[10] = 33000;
  my_knock_window_angle_table[11] = 3000;
  my_knock_window_angle_table[12] = 39000;
  my_knock_window_angle_table[13] = 3000;
  my_knock_window_angle_table[14] = 44000;
  my_knock_window_angle_table[15] = 3000;

  /* configure the eTPU channel pads */
  
  /* these for engine A - CAM & CRANK and an Output pin for KNOCK */
	fs_gpio_config( FS_GPIO_TCRCLKA, ETPU_INPUT_PAD_CONFIG); /* needed for parts like MPC5554 etc */
	fs_gpio_config( FS_GPIO_ETPUA0, ETPU_INPUT_PAD_CONFIG); /* added for Monaco */
	fs_gpio_config( FS_GPIO_ETPUA14, ETPU_INPUT_PAD_CONFIG);   
	fs_gpio_config( FS_GPIO_ETPUA10, ETPU_OUTPUT_PAD_CONFIG); 
	
	  
  
	fs_gpio_config( FS_GPIO_ETPUA1, ETPU_OUTPUT_PAD_CONFIG);   
	fs_gpio_config( FS_GPIO_ETPUA15, ETPU_OUTPUT_PAD_CONFIG);   
	
	/* ESCI_A pins */

                         	
	fs_gpio_config( FS_GPIO_TXDA, FS_GPIO_PRIMARY_FUNCTION + \
                               FS_GPIO_MINIMUM_SLEW_RATE + \
                               FS_GPIO_OUTPUT_DRAIN_DISABLE + \
                               FS_GPIO_OUTPUT_BUFFER_ENABLE + \
                               FS_GPIO_WEAK_PULL_UP);   
                               
	fs_gpio_config( FS_GPIO_RXDA, FS_GPIO_PRIMARY_FUNCTION + \
                               FS_GPIO_MINIMUM_SLEW_RATE + \
                               FS_GPIO_OUTPUT_DRAIN_DISABLE + \
                               FS_GPIO_INPUT_BUFFER_ENABLE + \
                               FS_GPIO_WEAK_PULL_UP);   


           

  /* assign PRAM non dynamically  - ease of debug etc */
  
  eTPU->CHAN[TOOTHGEN0_CRANK   ].CR.B.CPBA=0x20;
  eTPU->CHAN[APP_ENG_POS0_CAM	 ].CR.B.CPBA=0x40;
  eTPU->CHAN[APP_ENG_POS0_CRANK].CR.B.CPBA=0x60;
  eTPU->CHAN[KNOCK_WINDOW0_CHANNEL].CR.B.CPBA=0xA0;
                      
  /* init Application State Machine */ 
  app_state = APP_STATE_OFF;
  app_fault = APP_FAULT_NO;
  app_switch = APP_SWITCH_OFF;
  /* init eTPU load measurement reference time */
  misc_ref_time = 533;  /* applies during first ON state, later is measured */

  /* enable interrupts */
  asm ("wrteei 1");

  /* background loop */
  while (1)
  {
    cam_engine_cycle_0_count = fs_etpu_eng_pos_get_engine_cycle_0_tcr2_count();
    cam_edge_angle = fs_etpu_eng_pos_get_cam_edge_angle ( APP_ENG_POS0_CAM, APP_ENG_POS0_CRANK ); 
    delta = cam_edge_angle - cam_engine_cycle_0_count;

    /* The FreeMASTER poll call must be called in the main application loop
       to handle the communication interface and protocol.
       In LONG_INTR FreeMASTER interrupt mode, all the processing is done
       during the communication interrupt routine and the FMSTR_Poll() is
       compiled empty. */
    FMSTR_Poll();
  }    

} /* end of main() */


void eMIOS_init()
{
/*.
    EMIOS Configuration
--------------------------------------------
    Clock Frequency: 80.000 MHz , Clock Divider: 1, Global Prescaler Enable: Enable 
    Module: Run , Debug/Freeze Operation: Run 
    Global Time Base Enable: Enable , Counter Bus[A] Time Base: Unified Channel 23 , 
    Doze Mode Enable: Disable
    UC[0] value update: Enable , UC[1] value update: Enable , UC[2] value update: Enable , UC[3] value update: Enable 
    UC[4] value update: Enable , UC[5] value update: Enable , UC[6] value update: Enable , UC[7] value update: Enable 
    UC[8] value update: Enable , UC[9] value update: Enable , UC[10] value update: Enable , UC[11] value update: Enable 
    UC[12] value update: Enable , UC[13] value update: Enable , UC[14] value update: Enable , UC[15] value update: Enable 
    UC[16] value update: Enable , UC[17] value update: Enable , UC[18] value update: Enable , UC[19] value update: Enable 
    UC[20] value update: Enable , UC[21] value update: Enable , UC[22] value update: Enable , UC[23] value update: Enable
.*/
	EMIOS.MCR.R = 0x14000000;
/*.
    EMIOS Configuration
--------------------------------------------
    UC[0] : Clock Frequency: 80.000 MHz , Filter period: 15.625 ns , Mode: MC Up, int. clk 
            Bus selection: Int. Counter->UC all , Edge Selection: Clear , Edge Polarity: Set 
    UC[1] : Clock Frequency: 80.000 MHz , Filter period: 15.625 ns , Mode: MC Up, int. clk 
            Bus selection: Int. Counter->UC all , Edge Selection: Clear , Edge Polarity: Set 
.*/
	EMIOS.CH[0].CADR.R = 0x00001900;
	EMIOS.CH[8].CADR.R = 0x00138800;
	EMIOS.CH[0].CCR.R = 0x020206D0;
	EMIOS.CH[8].CCR.R = 0x020206D0;
}

void eSCI_init()
{
/*.
    ESCI_A Configuration
--------------------------------------------
    Enable module: Enable , Baud Rate: 115.942 kBd 
    Enable transmitter: Enable , Enable receiver: Enable 
    Enable TX DMA: Disable, Enable RX DMA: Disable
    Stop DMA TX on error: Enable 
    Enable parity: Disable, 
    Data format: 8 data bits , Break length: 10/11 bits , Enable Loop operation: Disable
    Wakeup condition: By idle , Idle count start: After start bit 
    RX idle int: Disable, RX full int: Enable 
    TX complete int: Disable, TX empty int: Enable , LIN bit error int: Disable
    Enable LIN mode: Disable, LIN parity generation: Disable
    LIN debug mode: Disable, LIN double stop flags: Disable
    LIN CRC polynomial: 0xC599, LIN wakeup delimiter length: 4 bits 
    LIN CRC error int: Disable, LIN checksum error int: Disable, LIN frame complete int: Disable
    LIN overflow int: Disable, LIN physical bus error int: Disable, LIN RX ready int: Disable
    LIN slave timeout error int: Disable, LIN TX ready int: Disable, LIN wakeup int: Disable
.*/
	ESCI_A.CR1.R =  0x002b00AC;        
	ESCI_A.CR2.R =  0x2000;        
         
}
void INTC_init()
{
  INTC.MCR.B.HVEN = 0;	/* Initialize INTC for software vector mode */
  INTC.MCR.B.VTES = 0;	/* Use the default vector table entry offsets of 4 bytes */
  INTC.IACKR.R = (uint32_t) &IntcIsrVectorTable[0]; /* Set INTC ISR vector table base addr. */	INTC.CPR.R = 0;

	INTC.PSR[51].R = 3;	
	INTC.PSR[59].R = 1;	
	INTC.PSR[67].R = 4;	
	INTC.PSR[146].R = 2;
}
/************************************************************************/
/* FUNCTION     : initIrqVectors                                        */
/* PURPOSE      : This function intializes the IVPR and IVOR4 registers */
/*                  for exceptions.                                     */
/* INPUT NOTES  : None                                                  */
/* RETURN NOTES : None                                                  */
/* WARNING      : None                                                  */
/************************************************************************/

void initIrqVectors(void) {
  asm ("lis r5, __IV_ADDR@h");   /* IVPR = address base used with IVOR's */
  asm ("mtIVPR r5 ");
  asm ("lis r5, IVOR4Handler@h"); /* IVOR4 = address of handler */ 							
  asm ("ori r5, r5, IVOR4Handler@l");	
  asm ("mtIVOR4 r5");					
}

/******************************************************************************
* Interrupt Handlers
******************************************************************************/

/******************************************************************************
* Function name : EMIOS_UC8_Isr
* Description   : 10ms periodic interrupt from eMIOS used to handle Application
*                 State Machine.
*
* Parameters    : None
*
* Returns       : None
*
******************************************************************************/
void EMIOS_UC8_Isr(uint32_t vec)
{
  switch (app_state)
  {
  case APP_STATE_OFF:
    /* state machine */
    if (app_switch == APP_SWITCH_ON)
    {
      app_state = APP_STATE_ENABLE;
    }
    break; 
  case APP_STATE_ENABLE:
    /* Reset parameters */
    motor_speed = motor_speed_start; 

    ETPU.MCR.B.GTBE=0; /* turn off to enable AM legally */
    EMIOS.MCR.B.GTBE=0;
    
    /* Initialize eTPU */
    if (my_system_etpu_init() != 0)
    {
      /* state machine */
      app_state = APP_STATE_FAULT;
      app_fault = APP_FAULT_OTHER;
    }
    else
    {


   
   
    /* Get function's parameter and channel base addresses - for FreeMASTER */
      toothgen_cba       = fs_etpu_data_ram(TOOTHGEN0_CRANK);
      toothgen_crank_cba = &eTPU->CHAN[TOOTHGEN0_CRANK].CR.R;
      toothgen_cam_cba   = &eTPU->CHAN[TOOTHGEN0_CAM].CR.R;
      crank_cba = &eTPU->CHAN[APP_ENG_POS0_CRANK].CR.R;
      cam_cba   = &eTPU->CHAN[APP_ENG_POS0_CAM].CR.R;
      knock_cba   = &eTPU->CHAN[KNOCK_WINDOW0_CHANNEL].CR.R;
      
      cam_pba            = fs_etpu_data_ram(APP_ENG_POS0_CAM);
      toothgen_pba       =  fs_etpu_data_ram(TOOTHGEN0_CRANK);    
      toothgen_crank_pba = fs_etpu_data_ram(TOOTHGEN0_CAM);
      toothgen_cam_pba   = fs_etpu_data_ram(TOOTHGEN0_CAM);
      crank_pba          = fs_etpu_data_ram(APP_ENG_POS0_CRANK);
      knock_pba          = fs_etpu_data_ram(KNOCK_WINDOW0_CHANNEL);
                      
      /* Start eTPU */
      my_system_etpu_start();
      /* start eTPU load measurement */
      fs_etpu_load_start_measure();
      /* state machine */
      app_state = APP_STATE_ON;
      /* Get status of the channels */
      Crank_Status = fs_etpu_eng_pos_get_crank_status ();
      Cam_Status = fs_etpu_eng_pos_get_cam_status ( APP_ENG_POS0_CAM );
      Eng_Pos_Status = fs_etpu_eng_pos_get_engine_position_status ( APP_ENG_POS0_CAM );
      cam_error_status = fs_etpu_eng_pos_get_cam_error_status ( APP_ENG_POS0_CAM );
      crank_error_status = fs_etpu_eng_pos_get_crank_error_status ( APP_ENG_POS0_CRANK );
      num =0;

    }
    break; 
  case APP_STATE_ON:

    /* Update ToothGen */
    num++;
    fs_etpu_toothgen_adj(TOOTHGEN0_CRANK, accel_rate, motor_speed, etpu_a_tcr1_freq);
                                 

    fs_etpu_toothgen_set_error_mimics( TOOTHGEN0_CRANK,
                                       error_list,
                                       missing_tooth_number_1,
                                       missing_tooth_number_2,
                                       crank_noise_tooth_number,
                                       crank_noise_pulse_shift,
                                       crank_noise_pulse_width,
                                       cam_noise_tooth_number,
                                       cam_noise_pulse_shift,
                                       cam_noise_pulse_width );

                                       
    for (j=0; j<(knock_number_of_windows); j++)
		{
    error = fs_etpu_knock_window_update ( KNOCK_WINDOW0_CHANNEL,
                                          APP_ENG_POS0_CAM, 
                                          j, 
                                          my_knock_window_angle_table[j*2], 
                                          my_knock_window_angle_table[j*2+1] );
    }
    
    /* Get status of the channels */
    Crank_Status = fs_etpu_eng_pos_get_crank_status  ();
    Cam_Status = fs_etpu_eng_pos_get_cam_status  ();
    Eng_Pos_Status = fs_etpu_eng_pos_get_engine_position_status ();
    cam_engine_cycle_0_count = fs_etpu_eng_pos_get_engine_cycle_0_tcr2_count ();
    cam_edge_angle = fs_etpu_eng_pos_get_cam_edge_angle (); 

    crank_tooth_number =  fs_etpu_eng_pos_get_tooth_number  ();
    cam_new_window_width = cam_new_window_end - cam_new_window_open; 

    error = fs_etpu_eng_update_cam_window (cam_new_window_open, 
                                            cam_new_window_width
                                          );
                                          
    error = fs_etpu_eng_pos_set_wr_normal( crank_windowing_ratio_normal);
    cam_error_status = fs_etpu_eng_pos_get_cam_error_status();
    crank_error_status = fs_etpu_eng_pos_get_crank_error_status();
    
    active_error_list = fs_etpu_get_chan_local_8(APP_ENG_POS0_CRANK, FS_ETPU_TOOTHGEN_ERRORSWITCH_OFFSET);
    
    measured_speed = fs_etpu_eng_pos_get_engine_speed(etpu_a_tcr1_freq);
    


                             					                             
    if (CrankErrorReset == 1)
    {
       error= fs_etpu_eng_pos_clear_crank_error_status();
       CrankErrorReset = 0;
    }
    
    /* state machine */
    if (app_switch == APP_SWITCH_OFF)
    {
      app_state = APP_STATE_DISABLE;
    }

      
    break; 
  case APP_STATE_DISABLE:
    /* Disable all active channels */
    
    fs_etpu_disable(APP_ENG_POS0_CRANK);
    fs_etpu_disable(APP_ENG_POS0_CAM);    
    fs_etpu_disable(TOOTHGEN0_CRANK);
    fs_etpu_disable(TOOTHGEN0_CAM);
    fs_etpu_disable(KNOCK_WINDOW0_CHANNEL);


    /* start eTPU load measurement - measure reference time when etpu_load = 0 */
    fs_etpu_load_start_measure();
    /* state machine */
    app_state = APP_STATE_OFF;
    
    break; 
  case APP_STATE_FAULT:
    /* state machine */
    if (app_switch == APP_SWITCH_OFF)
    {
      app_state = APP_STATE_OFF;
    }
    break; 
  }

  /* Clear UC1 flag */
  	EMIOS.CH[8].CSR.R |= 0x1;
;
  
}

/******************************************************************************
* Function name : ESCI_A_Isr
* Description   : Interrupt from eSCI A handled by FreeMASTER tool.
*
* Parameters    : None
*
* Returns       : None
*
******************************************************************************/
void ESCI_A_Isr(uint32_t vec)
{
  FMSTR_Isr(vec);
}

/******************************************************************************
* Function name : EMIOS_UC0_Isr
* Description   : 50us periodic interrupt from eMIOS used to call FreeMASTER
*                 Recorder.
*
* Parameters    : None
*
* Returns       : None
*
******************************************************************************/
void EMIOS_UC0_Isr(uint32_t vec)
{
  /* Call FreeMASTER Recorder */
  FMSTR_Recorder();
  /* Clear UC0 flag */
  	EMIOS.CH[0].CSR.R |= 0x1;
}

/******************************************************************************
* Function name : ETPU_globalexception_Isr
* Description   : eTPU Global Exception handler distinguishes the type of
*                 global exception. The SCM MISC Error exception is used to
*                 measure the eTPU load instead of standard fault handling.
*
* Parameters    : None
*
* Returns       : None
*
******************************************************************************/
void ETPU_globalexception_Isr(uint32_t vec)
{
  /* If Microcode Global Exception is asserted */
  if (eTPU->MCR.B.MGE1 == 1)
  {
    app_state = APP_STATE_FAULT;
    app_fault = APP_FAULT_ETPU_MICROCODE_GE;
  }
  /* If Illegal Instruction Flag is asserted */
  else if(eTPU->MCR.B.ILF1 == 1)
  {
    app_state = APP_STATE_FAULT;
    app_fault = APP_FAULT_ETPU_ILLEGAL_INSTR;
  }
  /* If SCM MISC Flag is asserted */
  else if(eTPU->MCR.B.SCMMISF == 1)
  {
    /* SCM MISC is used to measure eTPU load */
    if (app_state == APP_STATE_OFF)
    {
      fs_etpu_load_get_ref_time();
    }
    else
    {
      etpu_load = fs_etpu_load_evaluate();
    }
    fs_etpu_load_start_measure();
    /*
    app_state = APP_STATE_FAULT;
    app_fault = APP_FAULT_ETPU_MISC_ERROR;
    */
  }
  /* Clear eTPU global interrupt flag */
  eTPU->MCR.B.GEC = 1;
}

/******************************************************************************
 *
 * Copyright:
 *  Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 *****************************************************************************/



