/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2006 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file    etpu_load.c
*
* @author  r54529
*
* @version 0.0.0.1
*
* @date    Sep-07-2006
*
* @brief   Enables to measure eTPU load in run-time.
*
*******************************************************************************
*
* The eTPU feature SCM MISC (Share Code Memory Multiple Input Signature 
* Calculator) is used to measure the time when the eTPU engine is busy, and 
* calculate the eTPU load ratio.
* 
* The Multiple Input Signature Calculator (MISC) comprises special hardware that
* sequentially reads all SCM positions and calculates the signature.
* The MISC hardware is active only when the eTPU engine is not servicing. 
* Hence, the total MISC calculation time corresponds to the time the eTPU is 
* free of load, plus the time when the eTPU is busy.
* 
* The MISC calculation time need to be measured, at first, when the eTPU is
* completly free of load. The result, called misc_ref_time, includes only
* the time necessary to calculate the signature. The misc_ref_time is 
* stored and later used to calculate the etpu_load ratio.
* Then, the MISC calculation time can be measured in run-time, when the eTPU is 
* loaded. The result, called misc_act_time, includes both the time the eTPU is
* free of load and the time the eTPU is busy.
* 
* The eTPU load ratio is calculated using the formula:
* 
*                    misc_ref_time
*    etpu_load = 1 - -------------
*                    misc_act_time
* 
* The result is slightly less then the actual load beacause the MISC works
* during 2 of 3 TST (Time Slot Transition) cycles, which precede every service
* thread. The TST cycles should be included into the time when eTPU is busy, but
* 2 of them are not. The difference is usually up to 10% of the actual load.
*  
* On single eTPU engine devices, e.g. MPC5553, the etpu_load ration can be up to
* 50%. Only on two engine devices, e.g. MPC5554, and when both engines are
* active, the load could be over 50%.
*
* The misc_act_time and misc_ref_time are measured so, that the signature 
* compare value (MISCCMP register) is set a wrong value, causing the MISC 
* to raise global eTPU exception when done. The calculation start time is read 
* when the MISC is enabled, and the calculation stop time is read 
* on the interrupt.
* 
* The misc_ref_time is roughly 25us. Exactly, the time depends on SCM size
* (S) and processor clock frequency (f):
* 
*                    S[bytes]
*    misc_ref_time = ---------
*                    4 * f[Hz]
*
******************************************************************************/
#include "etpu_load.h"

/******************************************************************************
* Global variables
******************************************************************************/
uint16_t etpu_load_min = 0xFFFF;
uint16_t etpu_load_max = 0x0000;
uint32_t etpu_load_avg = 0x80000000;
uint32_t misc_ref_time;
uint32_t misc_act_time;

/******************************************************************************
* Constants and macros
******************************************************************************/

/******************************************************************************
* Local types
******************************************************************************/

/******************************************************************************
* Local function prototypes
******************************************************************************/

/******************************************************************************
* Local variables
******************************************************************************/
static uint32_t misc_start_time;

/******************************************************************************
* Local functions
******************************************************************************/

/******************************************************************************
* Global functions
******************************************************************************/

/**************************************************************************//*!
*
* FUNCTION: fs_etpu_load_start_measure
* 
* @return void 
*
* @brief  Starts the eTPU load measurement.
*
*******************************************************************************
*
* This function sets a wrong value to the MISC Compare Register and restarts
* the MISC operation.
* Thanks to the wrong MISC compare value, an eTPU global exception interrupt
* is rised when one cycle of the MISC calculation is finised. A function
* fs_etpu_load_get_ref_time or fs_etpu_load_evaluate should be called
* on this event.
*
******************************************************************************/
void fs_etpu_load_start_measure(void)
{
  /* Disable MISC */
  eTPU->MCR.B.SCMMISEN = 0;
  /* Set a wrong value to MICS Compare Reg. */
  eTPU->MISCCMPR.R += 1;
  /* Enable MISC */
  eTPU->MCR.B.SCMMISEN = 1;
  /* Store start TCR1 time */
  misc_start_time = eTPU->TB1R_A.R;
}

/***************************************************************************//*!
*
* FUNCTION: fs_etpu_load_get_ref_time
* 
* @return misc_ref_time
*
* @brief  Measures the misc_ref_time
*
********************************************************************************
*
* This function should be called on the eTPU global exception interrupt, if
* the MISC Flag (SCMMISF) is asserted, and when the eTPU engine is ensured
* not to service any events. The function gets the current time and 
* using the MISC_StartTime calculates the misc_ref_time.
*
*******************************************************************************/
uint32_t fs_etpu_load_get_ref_time(void)
{
  uint32_t stop_time;

  /* Get stop TCR1 time */
  stop_time = eTPU->TB1R_A.R;
  /* Get reference time */
  misc_ref_time = 0x00FFFFFF & (stop_time - misc_start_time);
  /* Correct the wrong value in MICS Compare Reg. */
  eTPU->MISCCMPR.R -= 1;
  /* Return misc_ref_time */
  return(misc_ref_time);
}

/**************************************************************************//*!
*
* FUNCTION: fs_etpu_load_evaluate
* 
* @return etpu_load ratio as 16-bit unsigned fractional value. 
*         0x0000 corresponds to 0%, and 0xFFFF to 100%
*
* @brief  Evaluates the eTPU load.
*
*******************************************************************************
*
* This function should be called on the eTPU global exception interrupt, if
* the MISC Flag (SCMMISF) is asserted. The function gets the current time, 
* using the MISC_StartTime calculates the misc_act_time first, and then using
* the misc_ref_time calculates and returns the eTPU load within the last
* MISC cycle.
*
******************************************************************************/
uint16_t fs_etpu_load_evaluate(void)
{
  uint32_t stop_time;
  uint32_t etpu_load;

  /* Get stop TCR1 time */
  stop_time = eTPU->TB1R_A.R;
  /* Get actual calculation time */
  misc_act_time = 0x00FFFFFF & (stop_time - misc_start_time);
  /* Correct the wrong value in MISC Compare Reg. */
  eTPU->MISCCMPR.R -= 1;
  /* Calculate load ratio: 1 - ref_time/act_time */
  etpu_load = (uint16_t)(0x10000 - (misc_ref_time<<16)/misc_act_time);
  /* Update etpu_load_min and etpu_load_max */
  if (etpu_load > etpu_load_max)
  {
    etpu_load_max = etpu_load;
  }
  else if (etpu_load < etpu_load_min)
  {
    etpu_load_min = etpu_load;
  }
  /* Update etpu_load_avg */
  etpu_load_avg = etpu_load_avg - (etpu_load_avg >> 12) + ((uint32_t)etpu_load << 4); 
  /* Return etpu_load */
  return(etpu_load);
  
}

/*********************************************************************
 *
 * Copyright:
 *	Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ********************************************************************/
