/**
 * Copyright (c) 2006, Freescale Semiconductor
 * Freescale Confidential Proprietary
 *
 * File name   : main.c
 *
 * Author      : Paolo Alcantara
 * Department  : RTAC Americas
 *
 * Description : UART-DMA api
 */
 
#include "mcf5222x_reg.h"
#include "uart-dma.h"
#include "driver_reg.h"

/* Functions */

/*
 * DMAIRQinit: enable DMA interrupts
 *
 * Parameters: none
 *
 * Return : none.
 */
void DMAIRQinit()
{
	/* setting processor in supervisor mode */
	asm(move.w #0x2000,SR);
    /* Set IMRH to enable interrupt request from ADC*/
	MCF_INTC0_IMRH &= ~(MCF_INTC_IMRH_MASK63);
	/* enable all interrupt sources */
	MCF_INTC0_IMRL &= ~(MCF_INTC_IMRL_MASK9 	/* DMA channel 0 */
					 | MCF_INTC_IMRL_MASK10 	/* DMA channel 1 */
					 | MCF_INTC_IMRL_MASK11 	/* DMA channel 2 */
					 | MCF_INTC_IMRL_MASK12		/* DMA channel 3 */
					 | MCF_INTC_IMRL_MASKALL);	/* all Interrupts */
	/* Levels of interruptions */
	/* Level 6, Priority x */
	MCF_INTC0_ICR9  = MCF_INTC_ICR_IP(7) + MCF_INTC_ICR_IL(4);
	MCF_INTC0_ICR10 = MCF_INTC_ICR_IP(6) + MCF_INTC_ICR_IL(4);
	MCF_INTC0_ICR11 = MCF_INTC_ICR_IP(5) + MCF_INTC_ICR_IL(4);
	MCF_INTC0_ICR12 = MCF_INTC_ICR_IP(4) + MCF_INTC_ICR_IL(4);
	return;
}

/*
 * UARTDMAinit: configure UART to transfer data transfered by DMA
 *			Can be used just any UART port
 *
 * Parameters: port: UART channel to configure
 *
 * Return : none.
 */
void UARTDMAinit(hcc_u8 port)
{
	register hcc_u16 ubgs;
	
	/* Set Port UART to initialize */ 
	
	/* UART0 pin assignment */
	if (port == 0)
	{
		MCF_GPIO_PUAPAR |=  (0
					//| MCF_GPIO_PUAPAR_CTS0_CTS0 /* CTS signal */
					//| MCF_GPIO_PUAPAR_RTS0_RTS0 /* RTS signal */
					| MCF_GPIO_PUAPAR_RXD0_RXD0 /* RXS signal */
					| MCF_GPIO_PUAPAR_TXD0_TXD0 /* TXS signal */
					);		
	}
	/* UART1 pin assignment */
	else if (port == 1)
	{
		MCF_GPIO_PUBPAR |=  (0
					//| MCF_GPIO_PUBPAR_CTS1_CTS1 /* CTS signal */
					//| MCF_GPIO_PUBPAR_RTS1_RTS1 /* RTS signal */
					| MCF_GPIO_PUBPAR_RXD1_RXD1 /* RXS signal */
					| MCF_GPIO_PUBPAR_TXD1_TXD1 /* TXS signal */
					);
	}
	/* UART2 pin assignment */
	else if (port == 2)
	{
		MCF_GPIO_PUCPAR |=  (0
					//| MCF_GPIO_PUCPAR_CTS2_CTS2 /* CTS signal */
					//| MCF_GPIO_PUCPAR_RTS2_RTS2 /* RTS signal */
					| MCF_GPIO_PUCPAR_RXD2_RXD2 /* RXS signal */ 
					| MCF_GPIO_PUCPAR_TXD2_TXD2 /* TXS signal */
					);		
	}
	
	/* Reset the selected UART module */
	MCF_UART_UCR(port) = MCF_UART_UCR_RESET_TX; /* Reset transmitter command */
	MCF_UART_UCR(port) = MCF_UART_UCR_RESET_RX; /* Reset receiver command */
	MCF_UART_UCR(port) = MCF_UART_UCR_RESET_MR; /* Reset mode command */

	/* No parity, Receive interrupt, 8-bits per character */
	MCF_UART_UMR(port) = (0		
		| MCF_UART_UMR_PM_NONE					/* no parity */
		| MCF_UART_UMR_BC_8 					/* 8 bits per character */
		);

	/* Local loopback, 1 stop bit */
	MCF_UART_UMR(port) = (0
		| MCF_UART_UMR_CM_NORMAL				/* normal channel mode */
		| MCF_UART_UMR_SB_STOP_BITS_1			/* Stop bit 1.000 */
		);

	/* Set Rx and Tx baud by timer */
	MCF_UART_UCSR(port) = (0
		| MCF_UART_UCSR_RCS_SYS_CLK				/* Prescale RX internal clk */
		| MCF_UART_UCSR_TCS_SYS_CLK				/* Prescale TX internal clk */
		);
	
	/* Calculate baud settings */
	ubgs = (hcc_u16)((SYS_CLOCK*1000000)/(SERIAL_BAUD * 32));

	/* Write baud settings */
	MCF_UART_UBG1(port) = (hcc_u8)((ubgs & 0xFF00) >> 8);
	MCF_UART_UBG2(port) = (hcc_u8)(ubgs & 0x00FF);

	/* Enable receiver and transmitter */
	MCF_UART_UCR(port) = (0
		| MCF_UART_UCR_TX_ENABLED				/* enable UART transsmiter */
		| MCF_UART_UCR_RX_ENABLED);				/* enable UART receiver */
	
    /* Set respective PACR to allow read/write from selected UART */
    MCF_SCM_PACR_UART = MCF_SCM_PACR_UART_USERMODE(port);
    
    /* SRAM backdoor for dual port access (9th bit) */
	MCF_SCM_RAMBAR = 0x20000200;
	
	return;
}

/*
 * UARTprint: send a character array to UART port
 *
 * Parameters: text: array to send
 * 			   size: array's length
 *			   port: UART channel to send array
 *
 * Return : none.
 */
void UARTprint(char *text, hcc_u8 size, hcc_u8 port)
{
	hcc_u8 i;
	
	for(i = 0; i < size; i++)
	{
		/* put character in tx buffer */
		MCF_UART_UTB(port) = text[i];
		/* wait until it can receive another character */
		while (!(MCF_UART_USR(port)&MCF_UART_USR_TXRDY))
		; /* Empty body */
	}
	
	return;
}

/*
 * DMAstart: configure DMA to transfer from UART from/to SRAM
 *			Can use any DMA channel
 *
 * Parameters: dma_state: DMA descriptor
 *
 * Return : none.
 */
void DMAstart(dma_state_t *dma_state)
{
	/* clear transfer complete */
	MCF_DMA_DSR(dma_state->dmaChannel) = MCF_DMA_DSR_DONE;
	/* set DMA source */
	MCF_DMA_SAR(dma_state->dmaChannel) = dma_state->source;
	/* set DMA destination */
	MCF_DMA_DAR(dma_state->dmaChannel) = dma_state->destination;
	/* set DMA count */
	MCF_DMA_BCR(dma_state->dmaChannel) = dma_state->byteCount;
		
	return;
}

/*
 * DMARXinit: start transfer between UART and SRAM
 *
 * Parameters: dmaChannel: DMA channel to start
 *
 * Return : none.
 */
void DMARXinit(hcc_u8 dmaChannel)
{
	MCF_DMA_DSR(dmaChannel) = MCF_DMA_DSR_DONE;	/* clear transfer complete */

	/* set DMA configuration for UART to SRAM */
	MCF_DMA_DCR(dmaChannel) = (0
		//| MCF_DMA_DCR_INT						/* enable interrupts */
   		| MCF_DMA_DCR_EEXT						/* enable external request */
   		| MCF_DMA_DCR_CS 						/* single RW per request */
		| MCF_DMA_DCR_SSIZE(1)					/* size of source */
		| MCF_DMA_DCR_DSIZE(1)					/* size of destination */
		| MCF_DMA_DCR_DINC  					/* DMA destination increment */
	    );
	
	/* Assign DMA request to UART Receive */
	MCF_SCM_DMAREQC |= MCF_SCM_DMAREQC_DMAC_RX(dmaChannel,UARTCHANNEL);
			
	return;
}

/*
 * DMATXinit: configure transfer between SRAM to UART
 *
 * Parameters: dmaChannel: DMA channel to start
 *
 * Return : none.
 */
void DMATXinit(hcc_u8 dmaChannel)
{
	MCF_DMA_DSR(dmaChannel) = MCF_DMA_DSR_DONE;	/* clear transfer complete */

	/* set DMA configuration for SRAM to UART */
	MCF_DMA_DCR(dmaChannel) = (0
   		| MCF_DMA_DCR_D_REQ						/* EEXT bit is cleared */
   		| MCF_DMA_DCR_CS 						/* single RW per request */
		| MCF_DMA_DCR_SSIZE(1)					/* size of source */
		| MCF_DMA_DCR_DSIZE(1)					/* size of destination */
		| MCF_DMA_DCR_SINC  					/* DMA source increment */
	    );
	
	/* Assign DMA request to UART Transmit */
	MCF_SCM_DMAREQC |= MCF_SCM_DMAREQC_DMAC_TX(dmaChannel,UARTCHANNEL);
	
	return;
}

/*
 * DMATXstart: start transfer between SRAM to UART
 *
 * Parameters: none
 *
 * Return : none.
 */
void DMATXstart()
{
	/* allow transmission */
	MCF_DMA_DCR(DMATRANSMIT) |= MCF_DMA_DCR_EEXT;
	
	return;
}

__declspec(interrupt:0x2000) void DMA0ISR()
{	
	/* clear transfer complete */
	MCF_DMA_DSR(0) = MCF_DMA_DSR_DONE;	

	/* fill with code */

	return;
}

__declspec(interrupt:0x2000) void DMA1ISR()
{	
	/* clear transfer complete */
	MCF_DMA_DSR(1) = MCF_DMA_DSR_DONE;	

	/* fill with code */

	return;
}

__declspec(interrupt:0x2000) void DMA2ISR()
{	
	/* clear transfer complete */
	MCF_DMA_DSR(2) = MCF_DMA_DSR_DONE;	

	/* fill with code */

	return;
}

__declspec(interrupt:0x2000) void DMA3ISR()
{	
	/* clear transfer complete */
	MCF_DMA_DSR(3) = MCF_DMA_DSR_DONE;	

	/* fill with code */

	return;
}