/*
 * File:		I2C.c
 * Purpose:		I2C transfer functions and interrupt handler
 *
 * Notes:		
 *  
 */
 
#include "hcc_types.h"
#include "mcf5222x_reg.h"
#include "driver_reg.h"
#include "i2c.h"

/* Globals to be shared throughout the project */
I2C_BUFFER i2c_tx_buffer;
I2C_BUFFER i2c_rx_buffer;

/* Globals for this file only */
hcc_u8 master_mode;
hcc_u8 master_tx_done;

//#define DEBUG_I2C

/* I2C memory functions */

#define SLAVE_ADDRESS	0xA0

void I2CwriteCharacter(hcc_u8 address, hcc_u8 character)
{
	hcc_u8 temp[2];
	temp[0] = address;
	temp[1] = character;
	
	I2CsetTxBuffer(&temp[0], sizeof(temp));
	I2Caction(I2C_TX,SLAVE_ADDRESS);
	/* delay for IIC memory */
	cpu_pause(5000);
	return;
}

void I2CreceiveString(hcc_u8 address, hcc_u16 length)
{
  	I2CsetStartAddress(address);
 	I2CsetRxBuffer(length);
    I2Caction(I2C_TXRX,SLAVE_ADDRESS);
    	
	return;
}

/*
 * Pause for the specified number of micro-seconds.
 * Uses DTIM3 as a timer
 */
void
cpu_pause(hcc_u32 usecs)
{
    /* Enable the DMA Timer 3 */
    MCF_DTIM3_DTRR = (usecs - 1);
    MCF_DTIM3_DTER = MCF_DTIM_DTER_REF;
    MCF_DTIM3_DTMR = 0
        | MCF_DTIM_DTMR_PS(80)
        | MCF_DTIM_DTMR_ORRI
        | MCF_DTIM_DTMR_FRR
        | MCF_DTIM_DTMR_CLK_DIV1
        | MCF_DTIM_DTMR_RST;

    while ((MCF_DTIM3_DTER & MCF_DTIM_DTER_REF) == 0) 
    ; /* waiting */
    
    /* Disable the timer */
    MCF_DTIM3_DTMR = 0;
}

/* I2C memory functions end*/


void I2Cinit(hcc_u8 slave_address)
{
	/* set interrupt's level */
	asm(move.w #0x2000,SR);
	
	/* Initialize PASP0 and PASP1 to I2C functions */
	MCF_GPIO_PASPAR = 0
					| MCF_GPIO_PASPAR_PASPAR0(1)	/* SCL function*/
					| MCF_GPIO_PASPAR_PASPAR1(1)	/* SDA function*/
					;

    /* Enable I2C interrupt */
    MCF_INTC_IMRL(0) &= ~(0 
    			   | MCF_INTC_IMRL_MASK17 
    			   | MCF_INTC_IMRL_MASKALL
    			   );
    /* Set to Level 1, Priority 1 */
    MCF_INTC_ICR17(0) = 0
    			   | MCF_INTC_ICR_IP(1)
    			   | MCF_INTC_ICR_IL(1)
    			   ;
    			   
    /***** Initialize I2C Module *****/
    /* Set transmission frequency = ~384kHz */
    MCF_I2C_I2FDR = MCF_I2C_I2FDR_IC(0x0A);
    /* Set module's I2C address */
    MCF_I2C_I2AR = slave_address;
    /* Enable I2C interrupts and module */
    MCF_I2C_I2CR = 0
    			 | MCF_I2C_I2CR_IIEN
    			 | MCF_I2C_I2CR_IEN
    			 ;
	return;
}

/******************************************************************************/
/*	Purpose:  	General function for performing I2C master transfers.  Capable
 *				of performing Master TX, Master RX, and Master TX/RX w/ Repeated
 *				Start.
 *				
 *	Arguments:	mode - Valid modes include I2C_TX, I2C_RX, and I2C_TXRX 
 *					(all modes defined in i2c.h)
 *				slave_address - The slave address of the I2C module that needs
 *					to be dealt with.  
 */
void
I2Caction(hcc_u8 mode, hcc_u8 slave_address)
{    
    /* Interrrupt-Driven */
   
    master_mode = mode;
    master_tx_done = FALSE;

    /* Reset index for TX and RX buffers */
    i2c_tx_buffer.tx_index = 0;
    i2c_rx_buffer.rx_index = 0;
    
    /* Transmit followed by receive using RSTA */
    if (mode == I2C_TXRX)
    {
		/* Make sure bus is idle */
		while (MCF_I2C_I2SR & MCF_I2C_I2SR_IBB)
		;
		/* Put module in master TX mode (generates START) */
		MCF_I2C_I2CR |= (MCF_I2C_I2CR_MSTA | MCF_I2C_I2CR_MTX);
		/* Put target address into IBDR */
		MCF_I2C_I2DR = (hcc_u8)( 0 | slave_address | I2C_TX);
		/* Wait for I2SR[IBB] (bus busy) to be set */
		while (!(MCF_I2C_I2SR & MCF_I2C_I2SR_IBB))
		;

		/* Wait for TX to finish before starting RX */
		while (!master_tx_done)
		;
	
		master_mode = I2C_RX;
		/* Set IBCR.RSTA and put in master RX mode */
		MCF_I2C_I2CR |= (0 | MCF_I2C_I2CR_RSTA);
		/* Put target address into IBDR and set the R/!W bit */
		MCF_I2C_I2DR = (hcc_u8)(0 | slave_address | I2C_RX);
	
		/* Wait for bus to become free before continuing */
		while (MCF_I2C_I2SR & MCF_I2C_I2SR_IBB)
		;
	
		/* Restore module to it's idle (but active) state */
		MCF_I2C_I2CR = 0xC0;
	
		return;
  	}
    /* Single TX or RX */
    else if ( (mode == I2C_TX) | (mode == I2C_RX) )
    {
		/* Make sure bus is idle */
		while (MCF_I2C_I2SR & MCF_I2C_I2SR_IBB)
		;
		/* Put module in master TX mode (generates START) */
		MCF_I2C_I2CR |= (0 | MCF_I2C_I2CR_MSTA | MCF_I2C_I2CR_MTX);
		/* Put target address into IBDR */
		MCF_I2C_I2DR = (hcc_u8)( 0 | slave_address | mode);
		/* Wait for I2SR[IBB] (bus busy) to be set */
		while (!(MCF_I2C_I2SR & MCF_I2C_I2SR_IBB))
		;
	
		/* Wait for bus to become free before continuing */
		while (MCF_I2C_I2SR & MCF_I2C_I2SR_IBB)
		;

		/* Restore module to it's idle (but active) state */
		MCF_I2C_I2CR = 0xC0;
	
		return;
   	}

	return;
}

void I2CsetStartAddress(hcc_u8 address)
{
    /* Initialize TX buffer structure */
    i2c_tx_buffer.tx_index = 0;
    i2c_tx_buffer.rx_index = 0;
    i2c_tx_buffer.data_present = TRUE;
    i2c_tx_buffer.length = 1;	
	i2c_tx_buffer.buf[0] = address;
	return;
}

void I2CsetTxBuffer(hcc_u8 *data, hcc_u16 length)
{
	hcc_u16 i;
	
    /* Initialize TX buffer structure */
    i2c_tx_buffer.tx_index = 0;
    i2c_tx_buffer.rx_index = 0;
    i2c_tx_buffer.data_present = TRUE;
    i2c_tx_buffer.length = length;
    for (i = 0; i < length; i++)
		i2c_tx_buffer.buf[i] = data[i];	
	return;	
}

void I2CsetRxBuffer(hcc_u16 length)
{
	/* Initialize RX buffer structure */
    i2c_rx_buffer.tx_index = 0;
    i2c_rx_buffer.rx_index = 0;
    i2c_rx_buffer.data_present = FALSE;
    i2c_rx_buffer.length = length;
    return;
}

void printArray(hcc_u8 *data, hcc_u16 length)
{
	hcc_u16 i;

	for(i = 0; i < length; i++)
	{
		printf("%c",data[i]);
	}
	
	printf("\n\r\n\r");
	return;
}

/******************************************************************************/
/*	Purpose:  	General I2C handler, created using the flowchart example 
 *				included in the I2C chapter of the UM.  
 */
__declspec(interrupt:0)
void i2c_handler(void)
{
    /* Temp variable for dummy reads */
    hcc_u8 dummy_read;		

    /* Clear the I2C Interrupt Flag. */
    MCF_I2C_I2SR &= ~MCF_I2C_I2SR_IIF;
    
    /* Check if this device is in Master or Slave Mode. */
    if (MCF_I2C_I2CR & MCF_I2C_I2CR_MSTA)
    {
		/* Master Mode - Check if this device is in Transmit or Receive Mode. */
		if (MCF_I2C_I2CR & MCF_I2C_I2CR_MTX)
		{
	   		/* Master Transmit Mode - Check if last byte was tranmitted. */
	    	if ((i2c_tx_buffer.length == 0) && (master_mode != I2C_RX))
	    	{
				/* Last byte was transmitted - 
		   		   Generate Stop signal by changing to Slave Mode. */

				/* If TXRX mode (Repeated Start), signal end of TX */
				if (master_mode == I2C_TXRX)
				{
					master_tx_done = TRUE;	
				}
				/* Issue STOP */
				else
				{
					MCF_I2C_I2CR &= ~MCF_I2C_I2CR_MSTA;
					i2c_tx_buffer.data_present = FALSE;
				}
		
                #ifdef DEBUG_I2C
		     		printf("Master TX mode: Last Byte\n\r");
                #endif
	    	}
	    	else
	    	{
				/* More bytes to be transmitted - Check if ACK received. */
				if (MCF_I2C_I2SR & MCF_I2C_I2SR_RXAK)
				{
		    		/* ACK not received - Generate STOP */
		    		MCF_I2C_I2CR &= ~MCF_I2C_I2CR_MSTA;
		    
                    #ifdef DEBUG_I2C
		         		printf("Master TX mode: NAK\n\r");
                    #endif
				}
				else
				{
		    		/* Check if end of address cycle */
		    		if (master_mode == I2C_RX)
		    		{
						MCF_I2C_I2CR &= ~MCF_I2C_I2CR_MTX;
						dummy_read = MCF_I2C_I2DR;
			
                        #ifdef DEBUG_I2C
			     			printf("Master TX mode: End of RX address cycle, switch to RX mode.\n\r");
                        #endif
		    		}
		    		/* ACK received, send data */
		    		else
		    		{
						/* Not end of address cycle - Write next byte to MBDR */
						MCF_I2C_I2DR = i2c_tx_buffer.buf[i2c_tx_buffer.tx_index++];
						i2c_tx_buffer.length--;
						#ifdef DEBUG_I2C
			     			printf("Byte sent\t");
                        #endif
		    		}
				}
	    	} 
		}
		else
		{
	    	/* Master Receive Mode - Check if this is last byte to be read. */	    
	    	if (i2c_rx_buffer.length == 1)
	    	{
				/* Last byte to be read - 
				   Generate Stop signal by changing to Slave Mode. */
				MCF_I2C_I2CR &= ~MCF_I2C_I2CR_MSTA;

				#ifdef DEBUG_I2C
		     		printf("Master RX mode: All data received, send STOP.\n\r");
                #endif
	    	}
	    	else
	    	{
				/* Not last byte to be read - Check if second to last byte. */
				if (i2c_rx_buffer.length == 2)
				{
		    		/* Second to last byte to be read - Set Transmit Acknowledge Enable
		    		   bit so no ACK is sent after the next byte is received, which
		    		   indicates "end of data" to the slave. */
		    		MCF_I2C_I2CR |= MCF_I2C_I2CR_TXAK;
		    
		    		#ifdef DEBUG_I2C
		         		printf("Master RX mode: Second-to-last byte received, set TXAK.\n\r");
                    #endif
				}
	    	}

	    	/* Store received data in RX buffer */
	    	i2c_rx_buffer.buf[i2c_rx_buffer.rx_index++] = MCF_I2C_I2DR;
	    	i2c_rx_buffer.length--;
		}
    }
    else
    {
		/* Slave Mode - Check if Arbitration Lost. */
		if (MCF_I2C_I2SR & MCF_I2C_I2SR_IAL)
		{
	    	#ifdef DEBUG_I2C
	         	printf("Arbitration Lost.\n\r");
            #endif

	    	/* Clear IAL bit */
	    	MCF_I2C_I2SR &= ~MCF_I2C_I2SR_IAL;
	    
	    	/* Arbitration Lost - Check if this device is being addressed as slave.
	    	   (If not, nothing more needs to be done.) */
	    	if (MCF_I2C_I2SR & MCF_I2C_I2SR_IAAS)
	    	{
				/* Addressed as slave - 
				   Check if master was reading from slave or writing to slave. */
				if (MCF_I2C_I2SR & MCF_I2C_I2SR_SRW)
				{
		    		/* Set tx_index to 0 */
		    		i2c_tx_buffer.tx_index = 0;
		    
		    		/* Master was reading from slave - Set Transmit Mode. */
		    		MCF_I2C_I2CR |= MCF_I2C_I2CR_MTX;
		    
		    		/* Write data to MBDR. */
		    		MCF_I2C_I2DR = i2c_tx_buffer.buf[i2c_tx_buffer.tx_index++];

		    		#ifdef DEBUG_I2C
		         		printf("Arbitration Lost: Addressed as slave - TX mode.\n\r");
                    #endif
				}
				else
				{
		    		/* Set rx_index to 0 */
		    		i2c_rx_buffer.rx_index = 0;
		    
		    		/* Master was writing to slave - Set Receive Mode. */
		    		MCF_I2C_I2CR &= ~MCF_I2C_I2CR_MTX;
		    
		    		/* Dummy read from MBDR, to clear the ICF bit. */
		    		dummy_read = MCF_I2C_I2DR;
		    
		    		#ifdef DEBUG_I2C
		         		printf("Arbitration Lost: Addressed as slave - RX mode.\n\r");
                    #endif
				}
	    	}	    
		}
		else
		{
	    	/* Arbitration Not Lost - Check if data byte is this devices's Slave Address byte. */
	    	if (MCF_I2C_I2SR & MCF_I2C_I2SR_IAAS)
	    	{
				/* Data byte is Slave Address byte - Check Slave Read/Write bit. */
				if (MCF_I2C_I2SR & MCF_I2C_I2SR_SRW)
				{
		    		/* Set tx_index to 0 */
		    		i2c_tx_buffer.tx_index = 0;
		    
		    		/* Master was reading from slave - Set Transmit Mode. */
		    		MCF_I2C_I2CR |= MCF_I2C_I2CR_MTX;
		    
		    		/* Write data to MBDR. */
		    		MCF_I2C_I2DR = i2c_tx_buffer.buf[i2c_tx_buffer.tx_index++];
				}
				else
				{
		    		/* Master has specified Slave Receive Mode.
		    		   Set Receive Mode.  (Writing to MBCR clears IAAS.) */
		    
		    		/* Set rx_index to 0 */
		    		i2c_rx_buffer.rx_index = 0;
		    
		    		MCF_I2C_I2CR &= ~MCF_I2C_I2CR_MTX;
		    
		    		/* Read address data from MBDR and store it. */
		    		dummy_read = MCF_I2C_I2DR;

		    		#ifdef DEBUG_I2C
		         		printf("Slave RX: Receive address.\n\r");
                    #endif
				}
	    	}
	    	else
	    	{
				/* Data byte received is not Slave Address byte - 
		   		   Check if this device is in Transmit or Receive Mode. */
				if (MCF_I2C_I2CR & MCF_I2C_I2CR_MTX)
				{
		    		/* Last byte received? */
		    		if (MCF_I2C_I2SR & MCF_I2C_I2SR_RXAK)
		    		{
						MCF_I2C_I2CR &= ~MCF_I2C_I2CR_MTX;
						dummy_read = MCF_I2C_I2DR;

						#ifdef DEBUG_I2C
			     			printf("Slave TX: Last byte has been sent.\n\r");
                        #endif
		    		}
		    		else
		    		{
						/* Write data to MBDR. */
						MCF_I2C_I2DR = i2c_tx_buffer.buf[i2c_tx_buffer.tx_index++];
						i2c_tx_buffer.length--;
		    		}
				}
				else
				{
		    		/* Receive Mode - Read data from MBDR and store it. */
		    		i2c_rx_buffer.buf[i2c_rx_buffer.rx_index++] = MCF_I2C_I2DR;
		    		i2c_rx_buffer.length++;
		    		i2c_rx_buffer.data_present = TRUE;
				}
	    	}
		}
    }
}
