 
/*
 * File:		int_handlers.c
 * Purpose:		Interrupt handlers
 *
 */

#include "mwerks.h"
#include "mcf5282.h"
#include <stdio.h>


#define MCF5282_QADC_QASR1_CWPQ1	(0x3F00)
#define MCF5282_QADC_QASR1_CWPQ2	(0x003F)
#define EOQ	63  // end of queue code

int32 first_entry_flag = 1; // 1 = first entry to interrupt routine, 0 = another entry
void first_entry (void); // function prototype

static int32 previous_CWPQ1; // points to last CCW where respective result was taken in Q1
static int32 previous_CWPQ2; // points to last CCW where respective result was taken in Q2


/***********************************************************************************************************/

__interrupt__ 
void irq_handler (void)  /* general interrupt handler */
{
	printf("general interrupt handler\n");
}


/***********************************************************************************************************/

__interrupt__
void EPORT_IRQ7_button_pressing (void)  /* Edge Port - IRQ7 (black button on M5282EVB) */
{
	MCF5282_EPORT_EPFR = MCF5282_EPORT_EPFR_EPF7; // clearing flag from IRQ7
	printf("\n\r					START Q2\n\r");	
	MCF5282_QADC_QACR2 |= MCF5282_QADC_QACRx_SSE;	
}


/***********************************************************************************************************/
     
__interrupt__
void QADC_Q1_conversion_pause (void)  /* QADC - Queue 1 conversion pause */
{	
	MCF5282_QADC_QASR0 &= ~MCF5282_QADC_QASR0_PF1; // clear event condition	flag	
	
	if (first_entry_flag) // write init values if first pause interrupt occurs (Q1 or Q2)
	{
		first_entry ();
	}	
	
	printf("\n\rGate was closed during Q1 scan\n\r");
		
	MCF5282_QADC_QACR1 |= MCF5282_QADC_QACRx_SSE; 
	   // set for continuation of scanning (in case single-scan mode)
}


/***********************************************************************************************************/

__interrupt__
void QADC_Q1_conversion_complete (void)  /* QADC - Queue 1 conversion complete */
{
	int32 i;
	int16 channel;
	int32 CWPQ1; // Q1 command word pointer
		
	CWPQ1 = (MCF5282_QADC_QASR1 & MCF5282_QADC_QASR1_CWPQ1) >> 8; // read position of last executed CCW
	MCF5282_QADC_QASR0 &= ~MCF5282_QADC_QASR0_CF1; // clear event condition	flag
	
	if (first_entry_flag) // write init values if first pause interrupt occurs (Q1 or Q2)
	{
		first_entry ();
	}	

	for (i = 1 + previous_CWPQ1; i <= CWPQ1; i++)
	{
		channel = MCF5282_QADC_CCW(i) & 0x003F;  // get channel number from respective CCW line
		if (channel != EOQ) // exclude End-of-Queue code
		{
			printf("%d",channel);
			printf(":");
			printf("%6d",(int16) MCF5282_QADC_LJSRR(i)); // signed results
			printf("\n\r");
		}
	}
	previous_CWPQ1 = -1; // reset pointer before beginning of Q1
	printf("         C\n\n\r");
	
	asm (move.w #0x2000,SR); // allows other interrupt sources to interrupt this cpu pause (this ISR)
	cpu_pause(100000); // set interval beetween two scans by cpu pause
	MCF5282_QADC_QACR1 |= MCF5282_QADC_QACRx_SSE; 
		// set for continuation of scanning (in case single-scan mode)

}


/***********************************************************************************************************/

__interrupt__
void QADC_Q2_conversion_pause (void)  /* QADC - Queue 2 conversion pause */
{
	int32 i;
	int16 channel;
	int32 CWPQ2; // Q2 command word pointer
		
	CWPQ2 = MCF5282_QADC_QASR1 & MCF5282_QADC_QASR1_CWPQ2; // read position of last executed CCW	
	
	MCF5282_QADC_QASR0 &= ~MCF5282_QADC_QASR0_PF2; // clear event condition	flag
	
	if (first_entry_flag) // write init values if first pause interrupt occurs (Q1 or Q2)
	{
		first_entry ();
	}

	for (i = 1 + previous_CWPQ2; i <= CWPQ2; i++)
	{
		channel = MCF5282_QADC_CCW(i) & 0x003F;  // get channel number from respective CCW line
		if (channel != EOQ) // exclude End-of-Queue code
		{
			printf("					%d",channel);
			printf(":");
			printf("%6d",(int16) MCF5282_QADC_LJSRR(i)); // signed results
			printf("\n\r");
		}
	}
	previous_CWPQ2 = CWPQ2; //  next time it will determine position of first read result register
	printf("					         P\n\r"); 
}


/***********************************************************************************************************/

__interrupt__
void QADC_Q2_conversion_complete (void)  /* QADC - Queue 2 conversion complete */
{
	int32 i;
	int16 channel;
	int32 CWPQ2; // Q2 command word pointer	
	
	CWPQ2 = MCF5282_QADC_QASR1 & MCF5282_QADC_QASR1_CWPQ2; // read position of last executed CCW
	MCF5282_QADC_QASR0 &= ~MCF5282_QADC_QASR0_CF2; // clear event condition	flag
	
	if (first_entry_flag) // write init values if first pause interrupt occurs (Q1 or Q2)
	{
		first_entry ();
	}

	for (i = 1 + previous_CWPQ2; i <= CWPQ2; i++)
	{
		channel = MCF5282_QADC_CCW(i) & 0x003F;  // get channel number from respective CCW line
		if (channel != EOQ) // exclude End-of-Queue code
		{
			printf("					%d",channel);
			printf(":");
			printf("%6d",(int16) MCF5282_QADC_LJSRR(i)); // signed results
			printf("\n\r");
		}
	}
	previous_CWPQ2 = (MCF5282_QADC_QACR2 & MCF5282_QADC_QACRx_BQ(127)) - 1; // reset pointer before beginning of Q1
	printf("					         C\n\n\r");	
}


/***********************************************************************************************************/

void first_entry (void) /* shared function for all QADC interrupt handlers */
{
	int32 BQ2; // beginning of Q2
	previous_CWPQ1 = -1; // if pause interrupt occurs 1st time, set pointer before beginning of Q1
	BQ2 = MCF5282_QADC_QACR2 & MCF5282_QADC_QACRx_BQ(127); // beginning of Q2
	previous_CWPQ2 = BQ2 - 1; // if pause interrupt occurs 1st time, set pointer before beginning of Q2
	first_entry_flag = 0; // this function will no longer be called 
}

/***********************************************************************************************************/

void 
exception_handler (void *framep) /* Called by asm_exception_handler */
{
	/*
	 * This is the exception handler for all defined exceptions.  Most
	 * exceptions do nothing, but some of the more important ones are
	 * handled to some extent.
	 */

	switch (MCF5200_RD_SF_FORMAT(framep))
	{
		case 4:
		case 5:
		case 6:
		case 7:
			break;
		default:
			printf("\nIllegal stack type! -- PC = %#08X\n", \
				MCF5200_SF_PC(framep));
			break;
	}

	switch (MCF5200_RD_SF_VECTOR(framep))
	{
		case 2:
			printf("Access Error -- PC = %#08X\n", MCF5200_SF_PC(framep));
			switch (MCF5200_RD_SF_FS(framep))
			{
				case 4:
					printf("Error on instruction fetch\n");
					break;
				case 8:
					printf("Error on operand write\n");
					break;
				case 9:
					printf("Attempted write to write-protected space\n");
					break;
				case 12:
					printf("Error on operand read\n");
					break;
				default:
					printf("Reserved Fault Status Encoding\n");
					break;
			}
			break;
		case 3:
			printf("Address Error -- PC = %#08X\n", MCF5200_SF_PC(framep));
			switch (MCF5200_RD_SF_FS(framep))
			{
				case 4:
					printf("Error on instruction fetch\n");
					break;
				case 8:
					printf("Error on operand write\n");
					break;
				case 9:
					printf("Attempted write to write-protected space\n");
					break;
				case 12:
					printf("Error on operand read\n");
					break;
				default:
					printf("Reserved Fault Status Encoding\n");
					break;
			}
			break;
		case 4:
			printf("Illegal instruction -- PC = %#08X\n", MCF5200_SF_PC(framep));
			break;
		case 8:
			printf("Privilege violation -- PC = %#08X\n", MCF5200_SF_PC(framep));
			break;
		case 9:
			printf("Trace Exception -- PC = %#08X\n", MCF5200_SF_PC(framep));
			break;
		case 10:
			printf("Unimplemented A-Line Instruction -- PC = %#08X\n", \
				MCF5200_SF_PC(framep));
			break;
		case 11:
			printf("Unimplemented F-Line Instruction -- PC = %#08X\n", \
				MCF5200_SF_PC(framep));
			break;
		case 12:
			printf("Debug Interrupt -- PC = %#08X\n", MCF5200_SF_PC(framep));
			break;
		case 14:
			printf("Format Error -- PC = %#08X\n", MCF5200_SF_PC(framep));
			break;
		case 15:
			printf("Unitialized Interrupt -- PC = %#08X\n", \
				MCF5200_SF_PC(framep));
			break;
		case 24:
			printf("Spurious Interrupt -- PC = %#08X\n", \
				MCF5200_SF_PC(framep));
			break;
		case 25:
		case 26:
		case 27:
		case 28:
		case 29:
		case 30:
		case 31:
			printf("Autovector interrupt level %d\n",
				MCF5200_RD_SF_VECTOR(framep) - 24);
			break;
		case 32:
		case 33:
		case 34:
		case 35:
		case 36:
		case 37:
		case 38:
		case 39:
		case 40:
		case 41:
		case 42:
		case 43:
		case 44:
		case 45:
		case 46:
		case 47:
			printf("TRAP #%d\n", MCF5200_RD_SF_VECTOR(framep) - 32);
			break;
		case 5:
		case 6:
		case 7:
		case 13:
		case 16:
		case 17:
		case 18:
		case 19:
		case 20:
		case 21:
		case 22:
		case 23:
		case 48:
		case 49:
		case 50:
		case 51:
		case 52:
		case 53:
		case 54:
		case 55:
		case 56:
		case 57:
		case 58:
		case 59:
		case 60:
		case 61:
		case 62:
		case 63:
			printf("Reserved: #%d\n",
				MCF5200_RD_SF_VECTOR(framep));
			break;
		default:
			printf("User Defined Vector #%d\n",
				MCF5200_RD_SF_VECTOR(framep));
			break;
	}
}
/********************************************************************/
