/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2005 Freescale Semiconductor, Inc.
* (c) Copyright 2001-2004 Motorola, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file   main.c
*
* @brief  Embedded Control of Cloth Iron with MC9RS08KA2
*
* @author r79251
* 
* @version 1.1.0.0
* 
* @date Jan 2007
* 
*******************************************************************************
*
* This is the main C file of the application.
*
*******************************************************************************/


/****************************************************************************//*!
*
*  @mainpage
*
*  This application is concerned with full control of cloth iron by microcontroler
*  MC9RS08KA2, or else - by replacing of traditional bimetal control for embeded control
*  system. 
*
*  -# This application is devided to several mouduls according its function.
*     The modules are:
*     DRIVERS: 
*     ADC -  emulated ADC converter for a reading of desired temperature and a 
*            measuring of a sole plate temperature
*     SCI -  by software emulated SCI transceiver for needs of a debugging
*             both drivers are followed by configuration files (adc_cfg.h; sci_cfg.h)
*             where is possible to change some parameters of the modules
*             
*     MODULES:
*     main        - the main file where is a base of the whole software with 
*                   a definition of a Control Frame that determinates exact position
*                   of measuring, switching-on of triac etc.
*     controller  - there is algorithm of PI regulator
*     math        - there are basic mathematical operations with limitation that are
*                   needed for the regulator module
*      
*  -# usage of pines
*     PTA0/KBI0/ACMP+ reserve and SCI transmit PIN (output) for debug
*     PTA1/KBI1/ACMP- input for measuring 
*     PTA2/KBI2       input wave for zerocrossing
*     PTA3/BKGD       control of Triac      
*     PTA4/KBI4       Selection of Thermo sensor for measurement
*     PTA5/KBI5       selection of Potenciometer for measurement 
*  -# both measured values by the ADC are filtered by a IIR filter
*  -# in main.c files are also some parameters for modification of control 
*     system behavior. The parameters are inside marked section.
*  -# the defined duration of Control Frame (CF) is 32 periods which represents
*     a time 0.64 s. In every the CF are done these steps:
*     four measuring desired a actual temperature including the IIR filtration;
*     a calculating of a duty value;
*     in dependence on the duty the triac is switched-on.
*
*******************************************************************************
*
* @note 1. In code are left lines marked - "for debugging". The lines can help 
*          in debug process. The lines are mainly with a SCI_putc function. 
*          The function can be copied and edited for a sending some  
*          other information according actual needs
*          
* @note 2. All regulator parameters and other parameters in section
*         'Parameters for a modification of control system behavior'
*          are dependent on specific 'hardware' - on actual iron.
*          
*******************************************************************************/


#include <hidef.h>      /* for EnableInterrupts macro                      */
#include <stdlib.h>     /* include header file of a standard library       */
#include "derivative.h" /* include peripheral declarations                 */

#include "main.h"

#include "sci.h"        /* include header file of a SCI module             */
#include "adc.h"        /* include header file of a ADC module             */
#include "controller.h" /* include header file of a controller module      */
#include "math.h"       /* include base mathematical function declarations */

#pragma MESSAGE DISABLE C4301   /* 'Inline expansion done for function call    '*/
#pragma MESSAGE DISABLE C3604   /* 'Static <Internal Object> was not referenced'*/

                        
/* Function prototypes */
void MCU_init(void);               /* Device initialization function declaration */
void SwitchOn (void);              /* Triac switch function declaration          */

#pragma INLINE
BYTE IIR_Filter (UBYTE filterInput, int * filterState, UBYTE filterShift) {      /* IIR filtering  */
    *filterState += (filterInput - *filterState) >> filterShift;         
    return (BYTE)*filterState;
}


/* Variable definition */  
volatile APPSCSTR   _APPSC;        /* Application Status and Control Register */

UBYTE  desiredTemperature;         /* desired temperature from pot */
UBYTE  lastDesTemperature;         /* desired temperature before on cycle */
int    desiredTemperatureBuffer;   /* filtered desired temperature from pot */
UBYTE  actualTemperature;          /* filtered measured temperature from sensor */
int    actualTemperatureBuffer;    /* filtered desired temperature from pot */
BYTE   controlDifference;          /* Difference between desiredTemperature and actualTemperature - input to reg */
BYTE   outputReg;                  /* output from regulator */
UBYTE  startReg;                   /* target temperature - /CBAND/ when the regulator takes control */

UBYTE  duty;                       /* number of switched wave */
UBYTE  wave;                       /* counting of wave inside controlFrame */

#define TIMEOUT_10MS 313           /* 313 x 0.032 (the timer period) = 10.016 ms - offset for switch-on negative       	     */
                                   /* half vawe to switch whole periods, if the switch-on is not quite sure, the value 	     */
                                   /* is possible to increase. In case some bigger disturbance (pulse before the switch-on), */
                                   /* to decrease with checkout of a good switching.				 	                               */
     
/* Parameters for a modification of control system behavior */
  #define CBAND                3     /* control band 1 +/-50%; 2 +/-25%; 3 +/-11%; 4 +/-6%; 5 +/-3%*/
  #define HIGH_CONTROL_LIMIT   1     /* 1 - when desired value is exceeded the duty =0; 0 - the feature is disabled */
  #define CONTROL_FRAME       15     /* number of wave in regulation frame including zero - 32 waves; quicker - 16 waves */
  #define CONTROL_SCALE        3     /* 2 - for Control Frame 32 waves; 3 - for Control Frame 16 waves */


  REG_PIparamsType regParams = {     /* parameters for controler algorithm    */
  /* UBYTE proportionalGain */ 200,
  /* UBYTE integralGain     */ 10,
  /* int   integral_K1      */0x1000 /* feed the controler just at switching over from full power to control by controler */
  };                                 /*  (Ox1FFF is 25 % from Ox7FFF)   */
                                     /*  (Ox1000 is 12.5 % from Ox7FFF) */
                                     /*  (Ox0666 is  5 % from Ox7FFF)   */
                                     
  #define FILTER_SHIFT	    2        /*  a weighted coefficient of IIR filter */                                                                         
                                
/* end of 'Parameters for a modification of control system behavior' section  */
                                   


/***************************************************************************//*!
*
* @brief  Application Loop
*
* @remarks 
*
* 1) definition of Control Frame (32 periods)
* 2) switch-on triac depending on calculated duty
* 3) in last four periods inside Control Frame are both mearuring executed
* 4) when the desired temperature is changed the bChange flag is set 
* 5) a  controlDifference is calculated
* 6) the regulator algorithm gives result
* 7) according the controlDifference and regulator result is a duty determined
* 8) flag for limitation to max possible power is set for next run when the max power is reached
*
****************************************************************************/

     
#define GET_HIGH(x)  (BYTE)((x)>>8)

void main(void) {
                                              
  MCU_init();                                                                 /* call Device Initialization */
  duty = 0;                                                                   /* starting value of a duty   */
                                                                              
  for(;;) {

    
     __RESET_WATCHDOG();
    for (wave=0; wave <= CONTROL_FRAME ; wave++)                              /* setting of the Control Frame */
    {   
        WAIT();                                                               /* waitting for KBI interupt   */
        if (KBISC_KBF == 1)                                                   /* check KBI interupt          */
        {
            KBISC_KBACK = 1;                                                  /* Clear  KBI pending interrupt      */
            if (wave > (CONTROL_FRAME - 4))                                   /* both measuring in last four waves */
            {
                /* measuring and filtering of both measured values by IIR filter */
                desiredTemperature = IIR_Filter(ADC_Read(CHAN_POTENTIOMETER), &desiredTemperatureBuffer, FILTER_SHIFT);
                actualTemperature  = IIR_Filter(ADC_Read(CHAN_THERMISTOR), &actualTemperatureBuffer, FILTER_SHIFT); 
            }
            if ((wave < (duty+1)) && (duty > 0)) SwitchOn();                  /* switch-on triac according a computed duty */                                                                    

            if (lastDesTemperature  != desiredTemperature) APPSC_bChange = 1; /* setting of change_flag that indicates a changing of desired temperature */
            lastDesTemperature = desiredTemperature;                          /* storing of desired temperature for next run - for change_flag settings  */
     
            /* for a debug only - it sends actual temperature once every Control Frame. There is possible place a Transmission
               any variable(s) according needs...                
                
            if (wave == CONTROL_FRAME)
            {
                SCI_putc(desiredTemperature);
                SCI_putc(actualTemperature);
                SCI_putc(duty);                                
            }                                      */            
        }                                                 
    }                                                      
 
    __RESET_WATCHDOG();
    
    
          /* temperature limit for a switching from full power to control of regulator   */
          /* startReg = dT- dT/8 = 89% of desiredTemperature (example for CBAND=3 ~ 89%) */
    startReg = desiredTemperature - (desiredTemperature >> CBAND);     
                                                                       
          /* if actual temp. is lower then startReg and the bChange is set, the power is full*/
    if ((actualTemperature < startReg) && (APPSC_bChange == 1)) duty = CONTROL_FRAME;  
    else 
    {     /* switch from full power to set integral_K1 as a start conditions when */ 
          /* the desired value is near - CBAND is reached */ 
                                  
        APPSC_bChange = 0;           
        
          /* calculation of difference between temperatures */
        controlDifference = uSUB8((UBYTE)desiredTemperature, (UBYTE)actualTemperature);  
  
    
          /* Call PI controller - computing of a duty */
        outputReg = GET_HIGH(regPIsat(controlDifference, &regParams, APPSC_bPmax));
        

          /* set saturation flag - already is reached full power */
        APPSC_bPmax = (outputReg == 0x1F ? 1 : 0);
        
          /* Scale controller output to heating frame (from +/- 128 to +/- 32 range)      */
        duty  = (BYTE)(outputReg > 0 ? outputReg >> CONTROL_SCALE : 0);
          /* duty = 0 when desired value is exceeded - this is possible turn-off by comment */ 
        if ((actualTemperature > desiredTemperature) && HIGH_CONTROL_LIMIT) duty = 0; 
    }

  } /* loop forever */
  /* please make sure that you never leave main */
}
                                                                                          

/***************************************************************************//*!
*
* @brief   Switch On one a whole wave
*
* @remarks the function switch-on triac by setting of OUTPUT, starts timer and holding the 
*  output set, waits < TIMEOUT_10MS * timer period> for negative half wave when the OUTPUT is
*  set to zero - the switch pulse is end off
****************************************************************************/

void SwitchOn (void) 
{   
    int countTOF;                                /* duration of half wave - number of TOF flag         */
    
    OUTPUT = 1;                                  /* switching-on of triac for pozitive half wave       */
    MTIMSC = MTIMSC_TRST_MASK|MTIMSC_TOIE_MASK;  /* Reset and enable overflow interupt from timer      */
    WAIT();                                      /* waiting for MTIM pending interrupt, output is held */
    MTIMSC_TOF = 0;                              /* Clear MTIM pending interrupt   */

    for (countTOF = TIMEOUT_10MS; countTOF > 0;) /* waiting for negative half wave */
    {                                            /* output is still held */
        WAIT();                                  /* waiting for further MTIM interrupt */
        if (MTIMSC_TOF == 1) {                   /* test of MTIM interrupt   */
            MTIMSC_TOF = 0;                      /* reset of MTIM interrupt  */
            countTOF--;                          /* decrementation of countTOF every timer period */
        }
    }
    OUTPUT = 0;                                  /* end off the switch puls              */
    if (MTIMSC_TOF == 1) MTIMSC_TOF = 0;         /* Clear  MTIM pending interrupt        */
    MTIMSC_TOIE = 0;                             /* disable overflow interupt from timer */
}


