/* epic.c - pic subroutines to handle interrupts */

/*  Copyright 2005-2006 Freescale Semiconductor */
/*
modification history
--------------------
01a,19jul05,ahk rewritten from edink
*/

/*
DESCRIPTION

INCLUDE Files: epic.h, tsec.h
*/

/* includes */

#include <stdio.h>
#include "epic.h"
#include "e500.h"

/* defines */
/* MSR */
#define GET_MSR(lval)   asm volatile ("mfmsr %0" : "=r" (lval))
#define SET_MSR(value)  asm volatile ("mtmsr %0" : :"=r" (value))
#define MSR_EE 0x00008000


/* typedefs */


/* globals */



/* forward declaration */

struct SrcVecTable SrcVecTable[MAXVEC] = /* Addr/Vector/ISR Addr cross-reference tbl */
    {
 	  /* 12 external internal sources */     
	  {  EPIC_EIVPR0_REG,  "Ext IRQ Source 0", 0x0},
	  {  EPIC_EIVPR1_REG,  "Ext IRQ Source 1", 0x0},
	  {  EPIC_EIVPR2_REG,  "Ext IRQ Source 2", 0x0},
	  {  EPIC_EIVPR3_REG,  "Ext IRQ Source 3", 0x0},
	  {  EPIC_EIVPR4_REG,  "Ext IRQ Source 4", 0x0},
	  {  EPIC_EIVPR5_REG,  "Ext IRQ Source 5", 0x0},
  	  {  EPIC_EIVPR6_REG,  "Ext IRQ Source 6", 0x0},
	  {  EPIC_EIVPR7_REG,  "Ext IRQ Source 7", 0x0},
	  {  EPIC_EIVPR8_REG,  "Ext IRQ Source 8", 0x0},
	  {  EPIC_EIVPR9_REG,  "Ext IRQ Source 9", 0x0},
	  {  EPIC_EIVPR10_REG,  "Ext IRQ Source 10", 0x0},
	  {  EPIC_EIVPR11_REG,  "Ext IRQ Source 11", 0x0},
	  
	  /* 4 message sources */
	  {  EPIC_MIVPR0_REG, "Message Source 0", 0x0},
	  {  EPIC_MIVPR1_REG, "Message Source 0", 0x0},
	  {  EPIC_MIVPR2_REG, "Message Source 0", 0x0},
	  {  EPIC_MIVPR3_REG, "Message Source 0", 0x0},
	   	
	  /* 4 inter-processor sources */
  	  { EPIC_IPIVPR0_REG,  "IPI Source 0", 0x0},
	  { EPIC_IPIVPR1_REG,  "IPI Source 1", 0x0},
	  { EPIC_IPIVPR2_REG,  "IPI Source 2", 0x0},
	  { EPIC_IPIVPR3_REG,  "IPI Source 3", 0x0},

	  /* 4 global timer sources */
 	  {  EPIC_GTVPR0_REG,  "Timer Source 0", 0x0},
	  {  EPIC_GTVPR1_REG,  "Timer Source 1", 0x0},
	  {  EPIC_GTVPR2_REG,  "Timer Source 2", 0x0},
	  {  EPIC_GTVPR3_REG,  "Timer Source 3", 0x0},	   

	  /* 32 internal sources and 23 out of them are used */	  
	  {  EPIC_IIVPR0_REG,  "Internal Source 0", 0x0},
  	  {  EPIC_IIVPR1_REG,  "Internal Source 1", 0x0},
  	  {  EPIC_IIVPR2_REG,  "Internal Source 2", 0x0},
  	  {  EPIC_IIVPR3_REG,  "Internal Source 3", 0x0},
	  {  EPIC_IIVPR4_REG,  "Internal Source 4", 0x0},
	  {  EPIC_IIVPR5_REG,  "Internal Source 5", 0x0},
	  {  EPIC_IIVPR6_REG,  "Internal Source 6", 0x0},
	  {  EPIC_IIVPR7_REG,  "Internal Source 7", 0x0},
	  {  EPIC_IIVPR8_REG,  "Internal Source 8", 0x0},
	  {  EPIC_IIVPR9_REG,  "Internal Source 9", 0x0},
	  
	  {  EPIC_IIVPR10_REG,  "Internal Source 10", 0x0},
  	  {  EPIC_IIVPR11_REG,  "Internal Source 11", 0x0},
  	  {  EPIC_IIVPR12_REG,  "Internal Source 12", 0x0},
  	  {  EPIC_IIVPR13_REG,  "Internal Source 13", 0x0}, /* TSEC1 transmit interrupt */
	  {  EPIC_IIVPR14_REG,  "Internal Source 14", 0x0}, /* TSEC1 receive interrupt */
	  {  EPIC_IIVPR15_REG,  "Internal Source 15", 0x0},
	  {  EPIC_IIVPR16_REG,  "Internal Source 16", 0x0},
	  {  EPIC_IIVPR17_REG,  "Internal Source 17", 0x0},
	  {  EPIC_IIVPR18_REG,  "Internal Source 18", 0x0}, /* TSEC1 receive/transmit error interrupt */
	  {  EPIC_IIVPR19_REG,  "Internal Source 19", 0x0},	  
  	  {  EPIC_IIVPR20_REG,  "Internal Source 20", 0x0},
  	  {  EPIC_IIVPR21_REG,  "Internal Source 21", 0x0},
  	  {  EPIC_IIVPR22_REG,  "Internal Source 22", 0x0},
  	  {  EPIC_IIVPR23_REG,  "Internal Source 23", 0x0},
	  {  EPIC_IIVPR24_REG,  "Internal Source 24", 0x0},
	  {  EPIC_IIVPR25_REG,  "Internal Source 25", 0x0},
	  {  EPIC_IIVPR26_REG,  "Internal Source 26", 0x0},
	  {  EPIC_IIVPR27_REG,  "Internal Source 27", 0x0},
	  {  EPIC_IIVPR28_REG,  "Internal Source 28", 0x0},
	  {  EPIC_IIVPR29_REG,  "Internal Source 29", 0x0},
	  {  EPIC_IIVPR30_REG,  "Internal Source 30", 0x0},
	  {  EPIC_IIVPR31_REG,  "Internal Source 31", 0x0},	  	  
    };



/****************************************************************************
*
*  epicInit - Initializes the EPIC unit by:
*
*  Setting the reset bit in the Global Configuration Register which will:
*  		-  Disables all interrupts
*  Clears all pending and in-service interrupts
*     	-  Sets epic timers to base count 
*     	-  Sets the value of the Processor Current Task Priority to the
*          highest priority (0xF)
*  Reset spurious vector to 0xFF
*	-  Default to pass-through mode
*  Sets the EPIC operation mode to Mixed Mode (vs. Pass 
*  	Through or 8259 compatible mode).
*
* RETURNS: None
*      
* ERRNO: None 
*
*/

void epicInit (void)
    {
    ULONG tmp;					/* to hold tmp register values */
    unsigned int counter;       /* to hold tmp register values */
    unsigned int noIRQs;        /* to hold # IRQs */
    unsigned int noCPUs;        /* to hold # CPUs */
    unsigned int verID;         /* to hold Version ID */

    /*  Disable External Interrupts */
    
    printf("EPIC: Disable External Interrupts\n\r");
    CoreExtIntDisable();

    /* 
    * Make sure the PIR[P0] bit is cleared. CPU
    * should never set P0 bit of PIR register. If this
    * bit is set then the system will be in a deadlock 
    * state. Therefore, check that this bit is cleared
    * to start with. We can clear this bit but not set 
    * it. The doc says that CPU should attempt to write
    * to it. Need to check. 
    */

    tmp = sysEUMBBARRead(EPIC_PROC_INIT_REG);
    sysEUMBBARWrite(EPIC_PROC_INIT_REG, tmp & 0xFFFFFFFE);

    /* 
    * Set the RST bit (bit # 30) GCR register to reset EPIC.
    * This bit is self-clearing 
    */

    tmp = sysEUMBBARRead(EPIC_GLOBAL_REG);
    tmp |= 0x80000000;                  
    sysEUMBBARWrite(EPIC_GLOBAL_REG, tmp); 
    printf("EPIC: Reset in progress...\n\r"); 


    /* Wait until reset sequence completes */
   
    while (sysEUMBBARRead(EPIC_GLOBAL_REG) & 0x80000000); 
    printf("EPIC: Reset completed...\n\r"); 

    /* Set PIC to mixed mode */

    sysEUMBBARWrite(EPIC_GLOBAL_REG, 0x20000000);
    printf("EPIC configured in mixed mode...\n\r");

    /* 
    * Get number of IRQs for false interrupt handling 
    * out of reset.
    */

    epicFeatures(&noIRQs, &noCPUs, &verID);

	#ifdef EPICDBG
    
    	printf("noIRQs: %d noCPUs: %x verID: %x\n\r", noIRQs, noCPUs, verID);
    	printf("EPIC: Lower Current Task Priority\n\r");

	#endif /* EPICDBG */

    /* step 2 */
    epicCurTaskPrioSet(0);

	#ifdef EPICDBG
    	
    	printf("EPIC: Enable: \n\r");
	
	#endif /* EPICDBG */
    
    /* step 4*/
    for (counter = 0; counter <= noIRQs; counter++)
        {
        epicIntEnable(counter);
	
	#ifdef EPICDBG
		if((counter % 10)==0)
		{
			printf("\n\r");
		}
		printf("%d ",counter);

	#endif /* EPICDBG */
        }
	
	#ifdef EPICDBG
    
    	printf("\n\r");
	
	#endif /* EPICDBG */

    for (counter = 0; counter <= noIRQs; counter++)
        {
        tmp = epicIntAck();
        epicEOI();
	
		#ifdef EPICDBG
    
        	printf("PASS: %d  IACK: %x and EOI\n\r", counter, tmp);
	
		#endif /* EPICDBG */
        }                    

	#ifdef EPICDBG
    
    	printf("EPIC: Disable: \n\r");
	
	#endif /* EPICDBG */
    
    for (counter = 0; counter <= noIRQs; counter++)
        {
        epicIntDisable(counter);
	
	#ifdef EPICDBG
		if((counter % 10)==0)
		{
			printf("\n\r");
		}	
        printf("%d ",counter);
	#endif /* EPICDBG */
        }             
	
	#ifdef EPICDBG
    	printf("\n\r");
	#endif /* EPICDBG */

    epicDefault();	/* set EPIC unit to default set up */
}

/****************************************************************************
*
*  epicDefault - sets up EPIC unit into a default mode for the 
*		 
*
*  - Lastly, it enables external interrupts. 
*
*
*  RETURNS:  None
*
***************************************************************************/
void epicDefault(void)
{
	int status;

	/* 
	* CPM (30) interrupt --> index 54 
	*/
	printf("EPIC: CPM Configure...\n\r");
	status = epicIntSourceConfig(54,1,0,10); /* vector, polarity (should be 1 all time), sense(x), priority */ 
	printf("Connect ISR...\n\r");
	epicISRConnect(54,cpmIsr);

	/* Enable interrupt vector 54 */
	printf("Enable\n\r");
	printf("\n\r");	
	epicIntEnable(54); 	
		
	/* Enable External Interrupts */
	printf("EPIC: Enable External Interrupts in MSR\n\r");
	printf("\n\r");
	CoreExtIntEnable();	
}
	

/****************************************************************************
*
*  epicIntEnable - Enable an interrupt source
*
*  This routine clears the mask bit of an external, an internal or
*  a Timer interrupt vector priority register to enable the interrupt.
*
*  RETURNS:  None
*
***************************************************************************/
void epicIntEnable
    (
    int intVec				/* Interrupt Vector Number */
    )
    {
    ULONG tmp;
    ULONG srAddr;

    srAddr = SrcVecTable[intVec].srcAddr;  /* Retrieve src Vec/Prio register */
    tmp = sysEUMBBARRead(srAddr);
    tmp &= 0x7fffffff;             /* Clear the mask bit */
    sysEUMBBARWrite(srAddr, tmp);
    return;
    }

/****************************************************************************
*
*  epicIntDisable - Disable an interrupt source
*
*  This routine sets the mask bit of an external, an internal or
*  a Timer interrupt vector priority register to disable the interrupt.
*
*  RETURNS: None 
*
****************************************************************************/

void epicIntDisable
    ( 
    int intVec 				/* Interrupt vector number */
    )
    {

    ULONG tmp, srAddr;

    srAddr = SrcVecTable[intVec].srcAddr;
    tmp = sysEUMBBARRead(srAddr);
    tmp |= 0x80000000;                      /* Set the mask bit */
    sysEUMBBARWrite(srAddr, tmp);
    return;
    }

/****************************************************************************
 * epicIntSourceConfig - Set properties of an interrupt source
 *
 * This function sets interrupt properites (Polarity, Sense, Interrupt
 * Prority, and Interrupt Vector) of an Interrupt Source.  The properties
 * can be set when the current source is not in-request or in-service,
 * which is determined by the Activity bit.  This routine return ERROR
 * if the the Activity bit is 1 (in-request or in-service).
 *
 * This function assumes that the Source Vector/Priority register (input)
 * is a valid address.
 * 
 * RETURNS:  OK or ERROR
 **************************************************************************/

int epicIntSourceConfig
    ( 
    int   Vect,                         /* interrupt source vector number */
    int   Polarity,                     /* interrupt source polarity */
    int   Sense,                        /* interrupt source Sense */
    int   Prio                          /* interrupt source priority */
    )

    {
    ULONG tmp, newVal;
    ULONG actBit, srAddr;

    srAddr = SrcVecTable[Vect].srcAddr;
    tmp = sysEUMBBARRead(srAddr);
    actBit = (tmp & 40000000) >> 30;    /* retrieve activity bit - bit 30 */
    if (actBit == 1)
        return ERROR;

    tmp &= 0xff30ff00;     /* Erase previously set P,S,Prio,Vector bits */
    newVal = (Polarity << 23) | (Sense << 22) | (Prio << 16) | Vect;
    sysEUMBBARWrite(srAddr, tmp | newVal );
    return (OK);
    }

/****************************************************************************
 * epicIntAck - acknowledge an interrupt
 *
 * This function reads the Interrupt acknowldge register and return
 * the vector number of the highest pending interrupt.
 *
 * RETURNS: Interrupt Vector number.
 */

unsigned int epicIntAck()
{
    return(sysEUMBBARRead( EPIC_PROC_INT_ACK_REG ));
}

/****************************************************************************
 * epicEOI - signal an end of interrupt
 *
 * This function writes 0x0 to the EOI register to signal end of interrupt.
 * It is usually called after an interrupt routine is served.
 *
 * RETURNS: None
 */

void extern epicEOI()
    {
    sysEUMBBARWrite(EPIC_PROC_EOI_REG, 0x0);
    }

/****************************************************************************
 *  epicCurTaskPrioSet - sets the priority of the Processor Current Task
 *
 *  This function should be called after epicInit() to lower the priority
 *  of the processor current task.
 *
 *  RETURNS:  OK or ERROR
 */

int epicCurTaskPrioSet
    (
    int prioNum                 /* New priority value */
    )
    {

    if ( (prioNum < 0) || (prioNum > 0xF))
        return ERROR;
    sysEUMBBARWrite(EPIC_PROC_CTASK_PRI_REG, prioNum);
    return OK;
    }


/************************************************************************
 * function: epicIntTaskGet
 *
 * description: Get value of processor current interrupt task priority register
 *
 * note:
 ***********************************************************************/
unsigned char epicIntTaskGet()
{
	/* get the interrupt task priority register */
    ULONG reg;
    unsigned char rec;

    reg = sysEUMBBARRead( EPIC_PROC_CTASK_PRI_REG );
    rec = ( reg & 0x0F );
    return rec;
}

/**************************************************************
 * function: epicISR
 * 
 * description: EPIC service routine called by the core exception
 *              at 0x500
 *
 * note: added body to routine. tpeters 7/30/99
 **************************************************************/
void epicISR()
{
	VOIDFUNCPTR intISRAddr;
	unsigned int vector;

	/* read the IACK register to get interrupt vector number
	   and look up ISR address in SrcVecTable using vector number.
	   we should check for a spurios vector, if it is we should print
	   something and return (no EOI). */
	vector = epicIntAck();

	/* Check if we got a spurious vector */
	/* Modify this later to look up actual spurious vector
	   value set by user in the SVR */
	if (vector == 0xff)
		{
#ifdef EPICDBG
		//printf("EPIC: In epicISR() for a Spurious Vector\n\r");
#endif /* EPICDBG */
		return;
		}

	intISRAddr = SrcVecTable[vector].isrAddr;

#ifdef EPICDBG
	//printf("EPIC: In epicISR() for vector#: %d\n\r", vector);
#endif /* EPICDBG */

	/* tpeters 03/29/00
	   set up dink transfer table so that DINK's dynamic functions
	   are available to the ISR. This is necessary if the user 
	   manually downloaded an ISR and connected to epic using
	   "dev epic ISRCnt" and wishes to do things like //printf */

	//epicSetupTransferTable();

	/* Now branch to service routine */
	intISRAddr(); 
			
#ifdef EPICDBG
	//printf("EPIC: Back from ISR\n");
#endif /* EPICDBG */

	/* returned from ISR now write to EOI register to end cycle */
	epicEOI();
}

/**************************************************************
 * function: epicISRConnect
 * 
 * description: EPIC routine called to register an ISR with EPIC
 *              for a particular interrupt vector.
 *
 * note: added routine. tpeters 7/30/99
 **************************************************************/
void epicISRConnect(int vector, VOIDFUNCPTR srcISRAddr)
{
	/* add ISR address to lookup table for given Vector  */
	SrcVecTable[vector].isrAddr = srcISRAddr;
	
#ifdef EPICDBG
	//printf("epicISRConnect(): Vector: %d -> ISRAddr: %x\n",\
						 vector, srcISRAddr);
#endif  /* EPICDBG */
}

/************************************************************
 * function: epicModeGet
 *
 * description: query EPIC mode, return 0 if pass through mode
 *                               return 1 if mixed mode
 *
 * note:
 *************************************************************/
unsigned int epicModeGet()
{
    ULONG val;

    val = sysEUMBBARRead( EPIC_GLOBAL_REG );
    return (( val & 0x20000000 ) >> 29);
}


/*********************************************
 * function: epicConfigGet
 *
 * description: Get the EPIC interrupt Configuration
 *              return 0 if not error, otherwise return 1
 *
 * note:
 ********************************************/
void epicConfigGet( unsigned int *clkRatio, unsigned int *serEnable)
{
    ULONG val;
	
    val = sysEUMBBARRead( EPIC_INT_CONF_REG );
    *clkRatio = ( val & 0x70000000 ) >> 28;
    *serEnable = ( val & 0x8000000 ) >> 27;
}


/*******************************************************************
 *  sysEUMBBARRead - Read a 32-bit EUMBBAR register
 *
 *  This routine reads the content of a register in the Embedded
 *  Utilities Memory Block, and swaps to big endian before returning
 *  the value.
 *
 *  RETURNS:  The content of the specified EUMBBAR register.
 */

ULONG sysEUMBBARRead
    (
    ULONG regNum
    )
    {
    ULONG temp;

    temp = *(ULONG *) (EUMBBAR + regNum);
#ifndef MPC8560ADS    
    return ( LONGSWAP(temp));
#else
    return (temp);
#endif
    }

/*******************************************************************
 *  sysEUMBBARWrite - Write a 32-bit EUMBBAR register
 *
 *  This routine swaps the value to little endian then writes it to
 *  a register in the Embedded Utilities Memory Block address space.
 *
 *  RETURNS: N/A
 */

void sysEUMBBARWrite
    (
    ULONG regNum,               /* EUMBBAR register address */
    ULONG regVal                /* Value to be written */
    )
    {
#ifndef MPC8560ADS
    *(ULONG *) (EUMBBAR + regNum) = LONGSWAP(regVal);
#else
    *(ULONG *) (EUMBBAR + regNum) = regVal;
#endif
    
    return ;
    }


/********************************************************
 * function: epicVendorId
 *
 * description: return the EPIC Vendor Identification
 *              register:
 * 
 *              siliccon version, device id, and vendor id
 *
 * note:
 ********************************************************/
void epicVendorId
   ( 
    unsigned int *step,
    unsigned int *devId,
    unsigned int *venId
   )
   {
    ULONG val;
    val = sysEUMBBARRead( EPIC_VENDOR_ID_REG );
    *step  = ( val & 0x00FF0000 ) >> 16;
    *devId = ( val & 0x0000FF00 ) >> 8;
    *venId = ( val & 0x000000FF );
    }


/****************************************************************************
*
*  epicFeatures - provides features of the PIC unit
*
*
* RETURNS: number of IRQ supported, number of CPU, 
*		   and the version of the  OpenEPIC
*      
* ERRNO: None 
*
*/

void epicFeatures
    ( 
    unsigned int *noIRQs, 
    unsigned int *noCPUs,
    unsigned int *verId
    )
    {
    ULONG val;

    val = sysEUMBBARRead( EPIC_FEATURES_REG );
    *noIRQs  = ( val & 0x07FF0000 ) >> 16;
    *noCPUs  = ( val & 0x00001F00 ) >> 8;
    *verId   = ( val & 0x000000FF );
}


/*********************************************************
 * function: epciTmFrequncySet
 *
 * description: Set the timer frequency reporting register
 ********************************************************/
void epicTmFrequencySet( unsigned int frq )
{
    sysEUMBBARWrite(EPIC_TM_FREQ_REG, frq);
}

/*******************************************************
 * function: epicTmFrequncyGet
 *
 * description: Get the current value of the Timer Frequency
 * Reporting register
 *
 ******************************************************/
unsigned int epicTmFrequencyGet()
{
    return( sysEUMBBARRead(EPIC_TM_FREQ_REG)) ;
}
			      

/****************************************************
 * function: epicTmBaseSet
 *
 * description: Set the #n global timer base count register
 *              return 0 if no error, otherwise return 1.
 *
 * note:
 ****************************************************/
unsigned int epicTmBaseSet
    ( 
    ULONG srcAddr,         /* Address of the Timer Base register */
    unsigned int cnt,	   /* Base count */
    unsigned int inhibit   /* 1 - count inhibit */
    )
{

    unsigned int val = 0x80000000;
    /* First inhibit counting the timer */
    sysEUMBBARWrite(srcAddr, val) ;

    /* set the new value */
    val = (cnt & 0x7fffffff) | ((inhibit & 0x1) << 31);
    sysEUMBBARWrite(srcAddr, val) ;
    return 0;
}

/***********************************************************************
 * function: epicTmBaseGet
 *
 * description: Get the current value of the global timer base count register
 *              including the CI bit. return 0 if no error, else return 1.
 *
 * note:
 ***********************************************************************/
unsigned int epicTmBaseGet( ULONG srcAddr, unsigned int *val )
{
    *val = sysEUMBBARRead( srcAddr );
    return 0;
}

/***********************************************************
 * function: epicTmCountGet
 *
 * description: Get the value of a given global timer
 *              current count register and toggle bit.
 *              return 0 if no error, otherwise return 1
 * note:
 **********************************************************/
unsigned int epicTmCountGet( ULONG srcAddr, unsigned int *val )
{
    *val = sysEUMBBARRead( srcAddr );
    return 0;
}



/***********************************************************
 * function: epicTmInhibit
 *
 * description: Stop counting of a given global timer
 *              return 0 if no error, otherwise return 1
 *
 * note:
 ***********************************************************/
unsigned int epicTmInhibit( ULONG srcAddr )
{
    ULONG val;

    val = sysEUMBBARRead( srcAddr ); 
    val |= 0x80000000;
    sysEUMBBARWrite( srcAddr, val ); 
    return 0;
}

/******************************************************************
 * function: epicTmEnable
 *
 * description: Enable counting of a given global timer
 *              return 0 if no error, otherwise return 1
 *
 * note:
 *****************************************************************/
unsigned int epicTmEnable( ULONG srcAddr )
{
    ULONG val;

    val = sysEUMBBARRead( srcAddr );
    val &= 0x7fffffff;
    sysEUMBBARWrite( srcAddr, val );
    return 0;
}


/******************************************************************
 * function: epicSourcePrint
 *
 * description: 
 *
 * note:
 *****************************************************************/
void epicSourcePrint(int Vect)
    {
    ULONG srcVal;

    srcVal = sysEUMBBARRead(SrcVecTable[Vect].srcAddr);
    //printf("%s\n", SrcVecTable[Vect].srcName);
    //printf("Address   = 0x%lx\n", SrcVecTable[Vect].srcAddr);
    //printf("Vector    = %d\n", (srcVal & 0x000000FF) );
    //printf("Mask      = %d\n", srcVal >> 31);
    //printf("Activitiy = %d\n", (srcVal & 0x40000000) >> 30);
    //printf("Polarity  = %d\n", (srcVal & 0x00800000) >> 23);
    //printf("Sense     = %d\n", (srcVal & 0x00400000) >> 22);
    //printf("Priority  = %d\n", (srcVal & 0x000F0000) >> 16);
    }


/*******************************************************
 * function: epicStatPrint
 *
 ******************************************************/
void epicStatPrint()
    {
    unsigned int tmp1, tmp2;
    unsigned int tmp;
    unsigned int noIRQs;

    /* get max number of IRQs */
    epicFeatures(&noIRQs, &tmp1, &tmp2);

    /* report the EPIC mode */
    tmp = epicModeGet();
    //PRINT("EPIC mode          %s\n", tmp ? "Mixed" : "Pass Though");

    /* report the epic interrupt configuration */
    epicConfigGet( &tmp1, &tmp2 );
    //PRINT("EPIC Operation     %s\n", tmp2 ? "Serial":"Direct");
    //if (tmp2 == 1)
        //PRINT("Clock Ratio        0x%x\n", tmp1);
    /* report the processor task priority */
    //PRINT("Current task Pri. 0x%x\n\n", epicIntTaskGet() );

    /* report the vector number, source address and description */
    //PRINT("Vector   Source Address   ISR Address          Source Interrupt\n");
    //for (tmp = 0; tmp <= noIRQs; tmp++)
      //  PRINT("  %2d         %lx          %8lx    %s\n" , tmp, 
		//			SrcVecTable[tmp].srcAddr,
		//			SrcVecTable[tmp].isrAddr,
          //                              SrcVecTable[tmp].srcName);
    //PRINT("\n");
    }


/****************************************************************************
*
*  CoreExtIntDisable - Disables the external interrupt
*
*
* RETURNS: None
*      
* ERRNO: None 
*
*/
void CoreExtIntDisable(void)
{
	unsigned long	msr_value;
	//unsigned long delay_loop = 0x100;
	
	/* Clear the EE bit of MSR */
	
	GET_MSR(msr_value);
	//asm("msync");
	asm("isync");
	//asm("mfmsr r3");
	//asm("lis r4, 0xffff");
	//asm("ori r4, r4, 0x7fff");
	//asm("and r3,r3,r4");
	//asm("mtmsr r3");
	
	msr_value &= ~MSR_EE;
	//while (delay_loop != 0x0) delay_loop--;
	//msr_value &= ~0x00008000;
	SET_MSR(msr_value);
	//asm("sync");
	asm("isync");	
}

/****************************************************************************
*
*  CoreExtIntEnable - Enables the external interrupt
*
*
* RETURNS: None
*      
* ERRNO: None 
*
*/
void CoreExtIntEnable (void)
{
	unsigned long	msr_value;
	
	/* Set the EE bit of MSR */
	
	GET_MSR(msr_value);
	msr_value |= MSR_EE;
	SET_MSR(msr_value);
	
}