//--------------------------------------------------------------------------                
// FILE: MCC_HDLC.c
//
//
// DESCRIPTION: 
//
//  Exercises MCC1 (Multi-channel controller #1) transmit/receive (TX / RX)
//  functions, using RX interrupts and internal loopback in the serial 
//  interface.
// 
//  This modified version of a previous PQ2 MCC program sends 8 Buffer Descriptors  
//  (BDs) of data per super channel (aka Multichannels or MCNs) over 128 MCC  
//  channels using HDLC Superchannel mode.
//  Each of a channel's 8 BDs contains a different data pattern. 
//
//  CONNECTIONS <<<IMPORTANT:  PLEASE READ>>>:
//  
//  The code sets up BRG5 and TMR1 to be the clock and sync respectively
//  for TDMA1. These signals must be routed to the appropriate inputs
//  in the CPM Expansion Connector as follows:
//     1) Connect D9 to D1 and D4.  This is BRG5out (PC23) to CLK1 (PC31, 
//        the clock input for the TDM) and TIN1 (PC28)
//     AND: 
//     2) Connect D2 and B26. This is TOUT1 (PC30) and L1RSYNC (PA6).
//
//
// NOTES:
//
//     1) Specifically Designed to run on MPC8560ADS board.
//
//
// REFERENCES: 
//
//      1) MPC8560 Users Manual
//      2) e500 Users manual
//
// HISTORY:
//
// DEC 1998   ggh     	Initial version
// MAY 1999   ggh     	Moved initialization of imm
// SEP 1999   ggh     	Final modifications for new web release,
//                     	incorporates alpha modifications and configs for
//                     	new startup file
// NOV 1999   jms     	Modified code for PILOT Rev boards.  The code now 
//				reads	BCSR2 for the board revision and then 
//				chooses the correct bit positionings for BCSR0
//				and BCSR1.
// JUL 2002   ddc 	Modified code to use 24 channels (channels 64-87) 
// AUG 2007   mj	Modified for 8560ADS and to run 128 channels in HDLC superchannel mode
//  
//-------------------------------------------------------------------------
#include <stdio.h>
#include <string.h>
#include <stdlib.h>        
#include "netcomm.h"       // global defines 
#include "mpc8560.h"       // Device definitions and declarations 
#include "mcc.h"           // Local header file 

//*********************
// Global Declarations 
//*********************
mpc_imm *pCCSRBAR;
BDRINGS *RxTxBD;        // buffer descriptors base pointer 
VUWORD IntCounter;      // counts times the interrupt handler is entered
VUBYTE NotDone;         // Termination of Rx flag 
VUWORD RxGood;          // Successful RX flag 
VUBYTE SI;              // Which SI is in use for this example 
VUBYTE TDM;             // Which TDM is in use for this example 
VUBYTE MCC;             // Which MCC is in use for this example 
UBYTE testmode;  	// determines whether tests runs once or continuously loops 

//----------------------------------------------------------
// Set of Data Buffers for transmit and receive             
// data. The Tx buffer pool will take up the first set of   
// buffers and Rx buffer pools will begin after that.       
//----------------------------------------------------------
volatile UBYTE *BufferPool;

//*********************
// Function Prototypes 
//*********************
void     main(void);
void     ClockingInit(void);
void     LoadTxBuffers(void);
void     InitBDs(UHWORD);
void     InitParallelPorts(void);
void     InterruptControlInit(void);
void     MCCGlobalInit(void);
void     MCCChanSpecInit(UHWORD);
void     MCCExChanSpecInit(UHWORD,UHWORD,UHWORD);
void     SuperchanTableEntryInit(UHWORD,UHWORD);
void     StopChannel(UHWORD);
void     StartChannel(UHWORD);
void     StartChannel(UHWORD);
void     InitTxRxParams(UHWORD);
void     SIinit(void);
UHWORD   BDRxError(UHWORD);
UHWORD   BDEmpty(UHWORD);
UHWORD   LastBD(UHWORD);
extern 	 void epicInit(void);

//--------------------------------------------------------------------------
//
// FUNCTION NAME:  main 
//
// DESCRIPTION:
//
//  Main function for MPC8560 MCC example code. 
//                 
// PARAMETERS:  None
//
// RETURNS: None
//
//-------------------------------------------------------------------------
void main(void)
{
	printf("\nMCC HDLC superchannel example software\n\n");

   	pCCSRBAR = (mpc_imm *)CCSRBAR_ADDR;

   	memset((UBYTE *)(CPM_DP1_ADDR),0,0x4000); // Clear contents of DPRAM area 1
   	memset((UBYTE *)(CPM_DP2_ADDR),0,0x4000); // Clear contents of DPRAM area 2

	//---------------------------------------------------------------
	// Initialize which SI, MCC, and TDM 
	//---------------------------------------------------------------
	SI = SI1;       // Note that if SI1 is in use, MCC1 must also be used 
	MCC = MCC1;     // and the range of channel numbers is restricted to  
	            	// 0-127.  Likewise, usage of SI2 requires usage of   
	            	// MCC2 and channels numbers must be within 128-255.  
	            	// If user changes which SI is in use, they are also  
	            	// responsible for ensuring that SIRAM and the        
	            	// parallel I/O ports are properly configured.         

	            	// If different channel numbers are used, the user is     
	            	// responsible for making channel-specific calls           
	            	// for the new channels, as well as modifying   
	            	// the SIRAM programming performed in SIinit().         

	TDM = TDMA;     // If the user wishes to change which TDM is being    
	            	// used, the user is also responsible for proper      
	            	// configuration of parallel I/O ports.            

	//-------------------------------
	// Miscellaneous initializations 
	//-------------------------------
	testmode = RUN_ONCE;  	// CONTINUOUS or RUN_ONCE
				// determines whether tests runs once or continuously loops   
	                                                    
	BufferPool = (UBYTE *)(BUFFERPOOLBASE);  // initialize pointer to buffer space

	IntCounter = 0;   // reset global counter for how many interrupts have been
	              	  // received, useful in continuous testing mode 

	               
	//----------------------------------------------------------------
	// Initialize the Interrupt Controller to properly enable or mask 
	// interrupts                                                    
	//----------------------------------------------------------------
	InterruptControlInit();
	epicInit();

	//---------------------------------------------------------
	// Establish base pointer for Tx and Rx buffer descriptors 
	//---------------------------------------------------------
	RxTxBD = (BDRINGS *)(BDRING_BASE);    

	//------------------------------------------------
	// Load the Tx buffer pool with the test patterns 
	//------------------------------------------------
	LoadTxBuffers();

	//----------------------------------------------
	// Program the Serial Interface including SIRAM 
	//----------------------------------------------
	SIinit();
    
	//------------------------------------------------------------
	// Set up baud rate generators for clock and sync for the TDM 
	//------------------------------------------------------------
	ClockingInit();

	//----------------------------------
	// Initialize the parallel I/O pins 
	//----------------------------------
	InitParallelPorts();

	// END OF "INITIALIZE ONCE" TYPE CALLS 

	//--------------------------------------------
	// Beginning of loop that will be repeatedly  
	// executed when in continuous mode           
	//--------------------------------------------
	while(1)
	{
		NotDone = TRUE;     // initialize as not done 
		RxGood = TRUE;      // initialize as good 

		//--------------------------
		// Initialize RX and TX BDs 
		//--------------------------
		InitBDs(CH0BDOFFSET);
		InitBDs(CH16BDOFFSET);
		InitBDs(CH32BDOFFSET);
		InitBDs(CH48BDOFFSET);    
		InitBDs(CH64BDOFFSET);
		InitBDs(CH80BDOFFSET);
		InitBDs(CH96BDOFFSET);
		InitBDs(CH112BDOFFSET);         

	        //------------------------------------
	        // Initialize MCC's Global parameters 
	        //------------------------------------
	        MCCGlobalInit();

	        //----------------------------------------------
	        // Initialize Extra Channel Specific parameters 
	        //----------------------------------------------
	        MCCExChanSpecInit(0,CH0BDOFFSET+64,CH0BDOFFSET);
	        MCCExChanSpecInit(16,CH16BDOFFSET+64,CH16BDOFFSET);
	        MCCExChanSpecInit(32,CH32BDOFFSET+64,CH32BDOFFSET);
	        MCCExChanSpecInit(48,CH48BDOFFSET+64,CH48BDOFFSET);
	        MCCExChanSpecInit(64,CH64BDOFFSET+64,CH64BDOFFSET);
	        MCCExChanSpecInit(80,CH80BDOFFSET+64,CH80BDOFFSET);
	        MCCExChanSpecInit(96,CH96BDOFFSET+64,CH96BDOFFSET);
	        MCCExChanSpecInit(112,CH112BDOFFSET+64,CH112BDOFFSET);        

	        //----------------------------------------
	        // Initialize Channel Specific parameters 
	        //----------------------------------------
	        MCCChanSpecInit(0);
	        MCCChanSpecInit(16);
	        MCCChanSpecInit(32);
	        MCCChanSpecInit(48);
	        MCCChanSpecInit(64);
	        MCCChanSpecInit(80);
	        MCCChanSpecInit(96);
	        MCCChanSpecInit(112);        

	        //-------------------------------------------
	        // Issue STOP commands for any superchannels 
	        //-------------------------------------------
	    	StopChannel(0);		
	        StopChannel(16);
	        StopChannel(32);	
	        StopChannel(48);
	    	StopChannel(64);		
	        StopChannel(80);
	        StopChannel(96);	
	        StopChannel(112);        

	        //-----------------------------------------------------------------
	        // Issue Init RX & TX Parameters Command for MCC.                 
	        // Be aware that issuing an init params command to an MCC channel  
	        // actually initializes the 32 consecutive channels starting with  
	        // the channel numbers specified below. In other words, in this    
	        // case it is not necessary to issue the command for channels 1   
	        // to 31 after issuing the command for channel 0.                    
	        //-----------------------------------------------------------------
	        InitTxRxParams(0);
	        InitTxRxParams(32);
	        InitTxRxParams(64);
	        InitTxRxParams(96);        

	        //------------
	        // Enable TDM 
	        //------------
	        pCCSRBAR->cpm.simcc[SI].sigmr = (VUBYTE)(1 << TDM);
		
	        //--------------------------
	        // Enable superchannels 
	        //--------------------------
	        StartChannel(0);	
	        StartChannel(16);
	        StartChannel(32);	
	 	StartChannel(48);	
	        StartChannel(64);	
	        StartChannel(80);
	        StartChannel(96);	
	 	StartChannel(112);
     
	        //------------------------------------------------------------------
	        // Wait in this "NotDone" loop until all frames have been sent and  
	        // received.                                                        
	        //                                                                  
	        // If there were any errors, they will be highlighted using printf()                              
	        //------------------------------------------------------------------

	        while (NotDone){};     // wait here until reception is done 

		// shut down superchannels and stop the TDM 
	        StopChannel(0);		
	        StopChannel(16);     
		StopChannel(32);	
	        StopChannel(48);  
	        StopChannel(64);		
	        StopChannel(80);     
		StopChannel(96);	
	        StopChannel(112);            

	        pCCSRBAR->cpm.simcc[SI].sigmr = 0x0;
		
        	if (testmode == RUN_ONCE) 
        	{
        		printf("\nTEST LOOP COMPLETED SUCCESSFULLY\n");
        		while(1){};  	// If this is a "one-shot" test, stop here          		
        	}
    	};  // end of while loop for continuous mode 
}  // End main 


//-------------------------------------------------------------------------
//
// FUNCTION NAME:  ClockingInit
//
//           
// DESCRIPTION:  Sets up the clock and sync that the TDM will use in this
//  example.
//                                            
// PARAMETERS: none
//
// RETURNS: none
//
//-------------------------------------------------------------------------
void ClockingInit(void)
{
	//-----------------------------------------
	// Connect MCC1's Tx and Rx clocks to BRG5 
	//-----------------------------------------
	pCCSRBAR->cpm.mux.cmxsi1cr = ALL_ZEROS; // this uses the default clock
	                               		// setting. (For TDMA1 this is
	                               		// CLK1 and CLK2 for RX and TX) 
	
	pCCSRBAR->cpm.timers.tmr1 = 0x1f0e; // timer used to supply TDM sync 
	pCCSRBAR->cpm.timers.trr1 = 0x00ff;
	pCCSRBAR->cpm.timers.tgcr1 = 0x09;
	pCCSRBAR->cpm.brgc5[0] = 0x0001000A; // BRG5 used to supply TDMA clocks (and TIN1)
}

//-------------------------------------------------------------------------
//
// FUNCTION NAME:  LoadTxBuffers
//
//           
// DESCRIPTION:
//
//  This example uses multiple superchannels, each of which will send 8
//  buffers containing the following patterns. Each buffer will be 256 
//  bytes in length.
//
//     Buffer 0: 0x55
//     Buffer 1: 0xAA
//     Buffer 2: 0x00
//     Buffer 3: 0xFF
//     Buffer 4: Increasing Walking Ones
//     Buffer 5: Decreasing Walking Ones
//     Buffer 6: Increment from 0
//     Buffer 7: Decrement from 255                                              
//
// PARAMETERS: none
//
// RETURNS: none				
//
//-------------------------------------------------------------------------
void  LoadTxBuffers(void)
{
	UHWORD   index, pattern;

	//---------------------------------------------------
	// Load buffers 0 through 3 with the following data  
	// patterns:                                         
	//                                                   
	// Buffer[0] = 0x55                                  
	// Buffer[1] = 0xAA                                  
	// Buffer[2] = 0x00                                  
	// Buffer[3] = 0xFF                                  
	//---------------------------------------------------
	for (index = 0; index < (BUFFER_SIZE+1); index++)

	{
		*(BufferPool + index) = 0x55;
		*(BufferPool + 320+ index) = 0xAA;     
		*(BufferPool + 2*320 + index) = 0x00;
		*(BufferPool + 3*320 + index) = 0xFF;      
	}

	//-----------------------------------------
	// Buffer[4]: Load increasing walking ones 
	//-----------------------------------------
	for (index = 0, pattern = 1; index < (BUFFER_SIZE); index++, pattern<<=1)
	{
		if (pattern == 0x0100)
	 		pattern = 0x01;
		*(BufferPool + 4*320 + index) = (UBYTE)pattern;     
	}

	//-----------------------------------------
	// Buffer[5]: Load decreasing walking ones 
	//-----------------------------------------
	for (index = 0, pattern = 0x80; index < (BUFFER_SIZE); index++, pattern>>=1)
	{
		if (pattern == 0x00)
	 		pattern = 0x80;

		*(BufferPool + 5*320 + index) = (UBYTE)pattern;
	}

	//--------------------------------------------
	// Buffer[6]: Load "Increment from 0" pattern 
	//--------------------------------------------
	for (index = 0; index < (BUFFER_SIZE); index++)
	{
		*(BufferPool + 6*320 + index) = (UBYTE)index;	  
	}

	//----------------------------------------------
	// Buffer[7]: Load "Decrement from 255" pattern 
	//----------------------------------------------
	for (index = 0; index < (BUFFER_SIZE); index++)
	{
		*(BufferPool + 7*320 + index) = (UBYTE)(255-index);
	}										                           
} // End of LoadTxBuffers 


//--------------------------------------------------------------------------
//
// FUNCTION NAME:  InitBDs
//
//
// DESCRIPTION:
//
//  Initializes BD rings for channels to point RX BDs to first half 
//  of buffer pool and TX BDs to second half of buffer pool. This function
//  also initializes the buffer descriptors control and data length fields.
//  It also ensures that transmit and receive functions are disabled before
//  buffer descriptors are initialized.
//
// PARAMETERS:
// 
//  offset -- used to access the BD area for a particular super channel
//
// RETURNS: None
//
//-------------------------------------------------------------------------
void InitBDs(UHWORD offset)
{
	UHWORD index;	

	//-------------------
	// Initialize RxBDs. 
	//-------------------
	for (index = 0; index < NUM_RXBDS; index++) 
	{
		if( index != (NUM_RXBDS-1) )  // If not the last RxBD for this channel 
		{
			RxTxBD->RxBD[index+offset].bd_cstatus = 0x9000; // Empty, Interrupt 
		}
		else  // if last RxBD for this channel 
		{
			RxTxBD->RxBD[index+offset].bd_cstatus = 0xB000; // Empty, Interrupt, Wrap                                	  		
		}

		//-------------------------
		// clear the buffer length 
		//-------------------------
      		RxTxBD->RxBD[index+offset].bd_length = 0;

		//----------------------------------------------------
		// set address to point to proper receive area for    
		// this BD in the bufferpool scheme this program uses 
		//----------------------------------------------------
		RxTxBD->RxBD[index+offset].bd_addr = (BufferPool + (8+offset+index)*1024);
   	} // end for loop initializing RxBDs 

	//-------------------
	// Initialize TxBDs. 
	//-------------------
   	for (index=0; index < NUM_TXBDS; index++) 
   	{
		if( index != (NUM_TXBDS-1) )  // If not the last TxBD for this channel 
		{
			//---------------
			// Set Ready bit 
			//---------------
			RxTxBD->TxBD[index+offset].bd_cstatus = 0x8000;
			
			//----------------------------------------------------
			// If this channel is HDLC, also set Last and TC bits 
			//----------------------------------------------------
			RxTxBD->TxBD[index+offset].bd_cstatus |= 0x0C00;              

      		}
      		else  // if last TxBD for this channel 
		{
			//----------------------
			// Set Ready, Wrap bits 
			//----------------------
			RxTxBD->TxBD[index+offset].bd_cstatus = 0xA000;

			//----------------------------------------------------
			// If this channel is HDLC, also set Last and TC bits 
			//----------------------------------------------------
            		RxTxBD->TxBD[index+offset].bd_cstatus |= 0x0C00;     
      		}

		//------------------------
		// load the buffer length 
		//------------------------
		RxTxBD->TxBD[index+offset].bd_length = (BUFFER_SIZE-4);  

		//----------------------------------------------------
		// set address to point to proper receive area for    
		// this BD in the bufferpool scheme this program uses 
		//----------------------------------------------------
		RxTxBD->TxBD[index+offset].bd_addr = (BufferPool + 320*index);
	}
} // end InitBDs 


//--------------------------------------------------------------------------
//
// FUNCTION NAME:  InitParallelPorts
//
//
// DESCRIPTION:
//
//   Sets up the parallel I/O pins for proper operation of this mode.
//
// SPECIAL NOTE:  The user is responsible for generating new settings if
//   they wish to change which SI, BRG or timer in use. Refer to the 
//   8560 reference manual for details.
//   
// EXTERNAL EFFECTS: Initializes relevant PIO registers.
//
// PARAMETERS: None
//
// RETURNS: None
//
//-------------------------------------------------------------------------
void InitParallelPorts(void)
{
	// Clear the Port Pin Assignment Registers 
	pCCSRBAR->cpm.iop[PORTA].ppar = 0x00000000;
	pCCSRBAR->cpm.iop[PORTB].ppar = 0x00000000;
	pCCSRBAR->cpm.iop[PORTC].ppar = 0x00000000;
	pCCSRBAR->cpm.iop[PORTD].ppar = 0x00000000;

	// Clear the Port Data Direction Registers 
	pCCSRBAR->cpm.iop[PORTA].pdir = 0x00000000;
	pCCSRBAR->cpm.iop[PORTB].pdir = 0x00000000;
	pCCSRBAR->cpm.iop[PORTC].pdir = 0x00000000;
	pCCSRBAR->cpm.iop[PORTD].pdir = 0x00000000;

	// Program the Port Special Options Registers 
	pCCSRBAR->cpm.iop[PORTA].psor = 0x03c00000;
	pCCSRBAR->cpm.iop[PORTB].psor = 0x00000000;
	pCCSRBAR->cpm.iop[PORTC].psor = 0x00020002;
	pCCSRBAR->cpm.iop[PORTD].psor = 0x00000000;

	// Program the Port Data Direction Registers 
	pCCSRBAR->cpm.iop[PORTA].pdir = 0x00400000;
	pCCSRBAR->cpm.iop[PORTB].pdir = 0x0000c000;
	pCCSRBAR->cpm.iop[PORTC].pdir = 0x00028502;
	pCCSRBAR->cpm.iop[PORTD].pdir = 0x00000000;

	// Program the Port Open-Drain Registers 
	pCCSRBAR->cpm.iop[PORTA].podr = 0x00000000;
	pCCSRBAR->cpm.iop[PORTB].podr = 0x00000000;
	pCCSRBAR->cpm.iop[PORTC].podr = 0x00000000;
	pCCSRBAR->cpm.iop[PORTD].podr = 0x00000000;

	// Program the Port Pin Assignment Registers 
	pCCSRBAR->cpm.iop[PORTA].ppar = 0x03c00000;
	pCCSRBAR->cpm.iop[PORTB].ppar = 0x0000c000;
	pCCSRBAR->cpm.iop[PORTC].ppar = 0x0002850B;  
	pCCSRBAR->cpm.iop[PORTD].ppar = 0x00000000;

}  // End of InitParallelPorts 


//------------------------------------------------------------------------
//
// FUNCTION NAME:  InterruptControlInit 
//                  
//
// DESCRIPTION:
//
//  Initializes interrupt controller for enabling/masking interrupts
//                 
// EXTERNAL EFFECT:
//
//  Interrupts enabled for MCC
//
// PARAMETERS: None
//
// RETURNS: None 
//
//-----------------------------------------------------------------------
void InterruptControlInit(void)
{
	//---------------------------------------------------------------------
	// Note that we will be using the default priority config. in the IC   
	//---------------------------------------------------------------------

	//-------------------------------------------------------
	// Enable MCC Interrupts to the Interrupt Controller          
	//-------------------------------------------------------
	if (MCC==MCC1) 
		pCCSRBAR->cpm.ic.simr_l = SIMR_L_MCC1;
	else 
		pCCSRBAR->cpm.ic.simr_l = SIMR_L_MCC2; 
	                          
} // End of InterruptControlInit 


//------------------------------------------------------------------------
//
// FUNCTION NAME:  MCCGlobalInit 
//                  
//
// DESCRIPTION:
//
//  MCC Initialization Routine.
//                 
// EXTERNAL EFFECT:
//
//  Parameter RAM and various registers on the 8560 including interrupt 
//  related, MCC-specific registers. 
//
// PARAMETERS: None
//
// RETURNS: None 
//
//-----------------------------------------------------------------------
void MCCGlobalInit(void)
{
	UWORD chan = 0;		// Used to set-up superchannel table
	UWORD superchan = 0;	// Used to set-up superchannel table

	// set up abbreviated pointer 
	volatile mcc_pram* MCCglobal;
	MCCglobal = (mcc_pram *)&(pCCSRBAR->cpm.dpram2.pram.mcc[MCC]); 

	//*************************************
	// Global Parameter RAM Initialization 
	//*************************************   
	MCCglobal->mccbase  = BDRING_BASE;  // base of BDs used by this MCC 
	MCCglobal->mccstate = ALL_ZEROS; 
	MCCglobal->mrblr    = MCC_MRBLR;    // max Rx buffer length 
	MCCglobal->grfthr   = NUM_RXBDS*NUMBER_OF_CHANNELS; // Interrupt on all buffers received 
	MCCglobal->grfcnt   = MCCglobal->grfthr; // grfthr decrementer 
	MCCglobal->rinttmp  = ALL_ZEROS;
	MCCglobal->_data[0] = ALL_ZEROS;
	MCCglobal->_data[1] = ALL_ZEROS;
	MCCglobal->tint.base = TX_INTCQ;            // base of Tx int. circular queue 
	MCCglobal->tint.ptr  = MCCglobal->tint.base; // temp ptr for tx interrupts 
	MCCglobal->tinttmp  = ALL_ZEROS;           // must be cleared by user     
	MCCglobal->sctpbase = SUPERCHAN_TABLE;     // base of superchannel table  
	MCCglobal->c_mask32 = CRC32;               // 32 bit crc constant 
	MCCglobal->xtrabase = MCCXTRABASE;         // base of chan-specific xtra params 
	MCCglobal->c_mask16 = CRC16;               // 16 bit crc constant 
	// Note that we are using only 1 one of the four possible Rx int. queues 
	MCCglobal->_rinttmp[0] = ALL_ZEROS;    // must be cleared by user 
	MCCglobal->_rinttmp[1] = ALL_ZEROS;    // must be cleared by user 
	MCCglobal->_rinttmp[2] = ALL_ZEROS;    // must be cleared by user 
	MCCglobal->_rinttmp[3] = ALL_ZEROS;    // must be cleared by user 
	MCCglobal->rint[0].base = RX_INTCQ;    // base of Rx int. circular queue #0 
	MCCglobal->rint[0].ptr = MCCglobal->rint[0].base;   // current RxINTCQ0 ptr 
	MCCglobal->rint[1].base = ALL_ZEROS;   // base of Rx int. circular queue #1 
	MCCglobal->rint[1].ptr = MCCglobal->rint[1].base;   // current RxINTCQ1 ptr 
	MCCglobal->rint[2].base = ALL_ZEROS;   // base of Rx int. circular queue #2 
	MCCglobal->rint[2].ptr = MCCglobal->rint[2].base;   // current RxINTCQ2 ptr 
	MCCglobal->rint[3].base = ALL_ZEROS;   // base of Rx int. circular queue #3 
	MCCglobal->rint[3].ptr = MCCglobal->rint[3].base;   // current RxINTCQ3 ptr 
    
	//------------------------------------
	// Superchannel Table Initialization  
	//------------------------------------
	// User is responsible for altering this table if SIRAM programming is modified     
	for(chan=0;chan<128;chan++)
	{
		SuperchanTableEntryInit(superchan,chan); 
		
		if(!((chan+1)%16))
		{
			superchan += 16;
		}
	}                         

	//----------------------------------
	// Initialize MCCF:                                                
	// Set channels to use correct TDM                                                  
	//----------------------------------
	// Note that for the purposes of this example, this line sets all 
	// channel groups to use the exact same TDM.  
	pCCSRBAR->cpm.simcc[MCC].mccf = (VUBYTE)(TDM | (TDM << 2) | (TDM << 4) | (TDM << 6));                                                              

	//****************************************************
	// Miscellaneous MCC interrupt-related intializations 
	//****************************************************
	memset((UBYTE *)TX_INTCQ, 0, 0x200); // clear interrupt queue areas 
	memset((UBYTE *)RX_INTCQ, 0, 0x200);

	*((UWORD *)TX_INTCQ + 0x7F) = 0x40000000; // set wrap bit at end of TX intcq 
	*((UWORD *)RX_INTCQ + 0x7F) = 0x40000000; // set wrap bit at end of RX intcq 

	//-------------------------
	// Enable RINT0 interrupts 
	//-------------------------
	pCCSRBAR->cpm.simcc[MCC].mccm = 0x4000;

	//-----------------------------------------
	// Clear MCCE Register by writing all 1's. 
	//-----------------------------------------
	pCCSRBAR->cpm.simcc[MCC].mcce = (VUHWORD)ALL_ONES;

} // End of MCCGlobalInit 


//------------------------------------------------------------------------
//
// FUNCTION NAME:  MCCChanSpecInit 
//                  
// DESCRIPTION:
//
//  Initializes the MCC channel-specific parameter ram for channels
//
// PARAMETERS: 
//
//  mcn -- multichannel number
//
// RETURNS: None 
//
//-----------------------------------------------------------------------
void MCCChanSpecInit(UHWORD mcn)
{
	volatile mcc_chan_param* MCCmch;
	MCCmch=  (mcc_chan_param*)(CPM_DP1_ADDR + (64 * mcn));
      
	//***********************************************
	// Channel-Specific Parameter RAM Initialization 
	//***********************************************
	MCCmch->zistate = 0x10000207;	// regular channel 
	MCCmch->zidata0 = ALL_ONES;
	MCCmch->zidata1 = ALL_ONES;
	MCCmch->_tbdflags = ALL_ZEROS;
	MCCmch->_tbdcnt  = ALL_ZEROS;
	MCCmch->_tbdptr = ALL_ZEROS;
	MCCmch->intmask = 0x0005;	// enable interrupts for RXF, RXB 
	MCCmch->chamr = 0xE080;     	// set MODE to HDLC, POL, CRC 
	MCCmch->mflr = MCC_MRBLR;	// HDLC max frame receive length 
	MCCmch->zdstate = 0x00FFFFE0;
	MCCmch->_tcrc = ALL_ZEROS;
	MCCmch->zddata0 = ALL_ONES;
	MCCmch->zddata1 = ALL_ONES;
    	MCCmch->_rbdflags = ALL_ZEROS;
    	MCCmch->_rbdcnt  = ALL_ZEROS;
    	MCCmch->_rbdptr = ALL_ZEROS;
    	MCCmch->_max_cnt = ALL_ZEROS;
    	MCCmch->_rcrc  = ALL_ZEROS;

	// initialize states for superchannelling (to be set in StartChannel) 
	MCCmch->tstate = ALL_ZEROS;	
	MCCmch->rstate = ALL_ZEROS;	

} // end MCCChanSpecInit() 


//------------------------------------------------------------------------
//
// FUNCTION NAME:  MCCExChanSpecInit 
//
// DESCRIPTION:
//
//  Initializes the extra MCC channel-specific parameter ram 
//
// PARAMETERS:
//
//  mcn -- multichannel number
//
//  mctbase -- offset to base of TxBDs for this mcn
//
//  mcrbase -- offset to base of RxBDs for this mcn 
//
// RETURNS: None 
//
//-----------------------------------------------------------------------
void MCCExChanSpecInit(UHWORD mcn, UHWORD mctbase, UHWORD mcrbase) 
{
	//establish pointer to multichannel extra params area
	volatile mcc_chxparam* MCCmchx;
	MCCmchx=(mcc_chxparam *)(CPM_DP1_ADDR + MCCXTRABASE + (8 * mcn));

	//-----------------------------------------------------------------
	// Let MCC know where base of TxBDs are for this super channel. 
	// Note that tbase is an offset from MCCBASE in the global params. 
	//-----------------------------------------------------------------
	MCCmchx->tbase = mctbase;
	MCCmchx->tbptr = MCCmchx->tbase;	// init "current TxBD ptr" 

	//-----------------------------------------------------------------
	// Let MCC know where base of RxBDs are for this super channel  
	// Note that rbase is an offset from MCCBASE in the global params. 
	//-----------------------------------------------------------------
	MCCmchx->rbase = mcrbase;
	MCCmchx->rbptr = MCCmchx->rbase;	// init "current RxBD ptr" 
}


//------------------------------------------------------------------------
//
// FUNCTION NAME:  SuperchanTableEntryInit 
//                  
// DESCRIPTION:
//
//  Adds an entry to MCC's superchannel table 
//
// PARAMETERS: 
//
//  mcn -- superchannel number
//  vcn -- MCC channel number
//
// RETURNS: None 
//
//-----------------------------------------------------------------------
void SuperchanTableEntryInit(UHWORD mcn, UHWORD vcn)
{
	VUHWORD* superchannel;

    	superchannel = (VUHWORD *) (CPM_DP1_ADDR + SUPERCHAN_TABLE + (2*vcn));
    	*superchannel = (UHWORD)(mcn << 6);
}


//------------------------------------------------------------------------
//
// FUNCTION NAME:  StopChannel 
//                  
// DESCRIPTION:
//
//  Issues STOP commands for the specified multichannel. Necessary for
//  superchannel initialization procedure. 
//
// PARAMETERS: 
//
//  mcn -- multichannel number
//
// RETURNS: None 
//
//-----------------------------------------------------------------------
void StopChannel(UHWORD mcn)
{
	VUWORD psbc;

	if (MCC == MCC1) 
		psbc = CPCR_MCC1_PSBC;
	else
		psbc = CPCR_MCC2_PSBC;

	//-------------------------------------------
	// Issue Stop TX Command for the given mcn  
	//-------------------------------------------
    	while ((pCCSRBAR->cpm.cp.cpcr & CPCR_FLG) != READY_TO_RX_CMD){}; 
    	pCCSRBAR->cpm.cp.cpcr = CPCR_STOP_TX |
                 psbc |
                 (mcn << 6) | 
                 CPCR_FLG;              // ISSUE COMMAND 

	//-------------------------------------------
	// Issue Stop RX Command for the given mcn
	//-------------------------------------------
	while ((pCCSRBAR->cpm.cp.cpcr & CPCR_FLG) != READY_TO_RX_CMD){};

    	pCCSRBAR->cpm.cp.cpcr = CPCR_MCC_STOP_RX |
                 psbc |
                 (mcn << 6) | 
                 CPCR_FLG;              // ISSUE COMMAND 
                
    	while ((pCCSRBAR->cpm.cp.cpcr & CPCR_FLG) != READY_TO_RX_CMD){};
}


//------------------------------------------------------------------------
//
// FUNCTION NAME:  StartChannel 
//                  
// DESCRIPTION:
//
//  Initializes RSTATE and TSTATE for given multichannel. Necessary for
//  superchannel initialization procedure. 
//
// PARAMETERS:
//
//  mcn -- multichannel number
//
// RETURNS: None 
//
//-----------------------------------------------------------------------
void StartChannel (UHWORD mcn)
{
	// establish pointer to channel-specific params 
	volatile mcc_chan_param* MCCmch;
	MCCmch = (mcc_chan_param*)(CPM_DP1_ADDR + (64 * mcn));

	if (BD_BUS == LOCAL)
	{
		MCCmch->tstate |= 0x01000000;	// set BDB 
		MCCmch->rstate |= 0x01000000;	// set BDB 
	}

	if (BUFFER_BUS == LOCAL)
	{
		MCCmch->tstate |= 0x02000000;	// set DTB 
		MCCmch->rstate |= 0x02000000;	// set DTB 
	} 

    	MCCmch->tstate |= 0x38800000;	// global, big endian
    	MCCmch->rstate |= 0x38800000;	// global, big endian
}


//------------------------------------------------------------------------
//
// FUNCTION NAME:  InitTxRxParams 
//                  
// DESCRIPTION:
//
//  Issues the INIT_TX_RX_PARAMS command to the CPCR for the given mcn.
//
// PARAMETERS:
//
//  mcn -- multichannel number
//
// RETURNS: None 
//
//-----------------------------------------------------------------------
void InitTxRxParams(UHWORD mcn)
{
	VUWORD psbc;

	if (MCC == MCC1) 
		psbc = CPCR_MCC1_PSBC;
    	else
		psbc = CPCR_MCC2_PSBC;

    	while ((pCCSRBAR->cpm.cp.cpcr & CPCR_FLG) != READY_TO_RX_CMD){}; 

    	pCCSRBAR->cpm.cp.cpcr = CPCR_INIT_TX_RX_PARAMS |
                 psbc |  
                 (mcn<<6)|
                 CPCR_FLG;              // ISSUE COMMAND 

    	while ((pCCSRBAR->cpm.cp.cpcr & CPCR_FLG) != READY_TO_RX_CMD){};
}


//------------------------------------------------------------------------
//
// FUNCTION NAME:  SIinit 
//                  
// DESCRIPTION:
//
//  Initializes the serial interface RAM entries and registers 
//
// PARAMETERS: None
//
// RETURNS: None 
//
//-----------------------------------------------------------------------
void SIinit(void)
{
	UHWORD entry = 0;
	UWORD chan = 0;
	
	//TX SIRAM
	// Channel 0
	pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT_FIRST | SIxRAM_CNT_1);
	pCCSRBAR->cpm.siram[SI].tx[entry] |= (0 << 5);	
	entry += 8;
	for(chan=1;chan<16;chan++)
	{
		pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT | SIxRAM_CNT_8);
		pCCSRBAR->cpm.siram[SI].tx[entry] |= (chan << 5);
		
		entry += 8;		
	}
	
	// Channel 16
	entry = 1;
	pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT_FIRST | SIxRAM_CNT_1);
	pCCSRBAR->cpm.siram[SI].tx[entry] |= (16 << 5);	
	entry += 8;
	for(chan=17;chan<32;chan++)
	{
		pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT | SIxRAM_CNT_8);
		pCCSRBAR->cpm.siram[SI].tx[entry] |= (chan << 5);
		
		entry += 8;		
	}
	
	// Channel 32
	entry = 2;
	pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT_FIRST | SIxRAM_CNT_1);
	pCCSRBAR->cpm.siram[SI].tx[entry] |= (32 << 5);	
	entry += 8;
	for(chan=33;chan<48;chan++)
	{
		pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT | SIxRAM_CNT_8);
		pCCSRBAR->cpm.siram[SI].tx[entry] |= (chan << 5);
		
		entry += 8;		
	}
	
	// Channel 48
	entry = 3;
	pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT_FIRST | SIxRAM_CNT_1);
	pCCSRBAR->cpm.siram[SI].tx[entry] |= (48 << 5);	
	entry += 8;
	for(chan=49;chan<64;chan++)
	{
		pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT | SIxRAM_CNT_8);
		pCCSRBAR->cpm.siram[SI].tx[entry] |= (chan << 5);
		
		entry += 8;		
	}			

	// Channel 64
	entry = 4;
	pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT_FIRST | SIxRAM_CNT_1);
	pCCSRBAR->cpm.siram[SI].tx[entry] |= (64 << 5);	
	entry += 8;
	for(chan=65;chan<80;chan++)
	{
		pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT | SIxRAM_CNT_8);
		pCCSRBAR->cpm.siram[SI].tx[entry] |= (chan << 5);
		
		entry += 8;		
	}

	// Channel 80
	entry = 5;
	pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT_FIRST | SIxRAM_CNT_1);
	pCCSRBAR->cpm.siram[SI].tx[entry] |= (80 << 5);	
	entry += 8;
	for(chan=81;chan<96;chan++)
	{
		pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT | SIxRAM_CNT_8);
		pCCSRBAR->cpm.siram[SI].tx[entry] |= (chan << 5);
		
		entry += 8;		
	}

	// Channel 96
	entry = 6;
	pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT_FIRST | SIxRAM_CNT_1);
	pCCSRBAR->cpm.siram[SI].tx[entry] |= (96 << 5);	
	entry += 8;
	for(chan=97;chan<112;chan++)
	{
		pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT | SIxRAM_CNT_8);
		pCCSRBAR->cpm.siram[SI].tx[entry] |= (chan << 5);
		
		entry += 8;		
	}
	
	// Channel 112
	entry = 7;
	pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT_FIRST | SIxRAM_CNT_1);
	pCCSRBAR->cpm.siram[SI].tx[entry] |= (112 << 5);	
	entry += 8;
	for(chan=113;chan<128;chan++)
	{
		pCCSRBAR->cpm.siram[SI].tx[entry] = (SIxRAM_MCC | SIxRAM_SUPER | SIxRAM_BYT | SIxRAM_CNT_8);
		pCCSRBAR->cpm.siram[SI].tx[entry] |= (chan << 5);
		
		entry += 8;		
	}
	pCCSRBAR->cpm.siram[SI].tx[127] |= 0x0001;	//LAST 	
	

                
	//RX SIRAM
	pCCSRBAR->cpm.siram[SI].rx[0] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[1] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[2] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[3] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[4] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[5] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[6] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[7] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[8] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[9] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[10] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[11] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[12] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[13] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[14] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[15] = 0x8E02;   // byte to mcn112 

	pCCSRBAR->cpm.siram[SI].rx[16] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[17] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[18] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[19] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[20] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[21] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[22] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[23] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[24] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[25] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[26] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[27] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[28] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[29] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[30] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[31] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[32] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[33] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[34] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[35] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[36] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[37] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[38] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[39] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[40] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[41] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[42] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[43] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[44] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[45] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[46] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[47] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[48] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[49] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[50] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[51] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[52] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[53] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[54] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[55] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[56] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[57] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[58] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[59] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[60] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[61] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[62] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[63] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[64] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[65] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[66] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[67] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[68] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[69] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[70] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[71] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[72] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[73] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[74] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[75] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[76] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[77] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[78] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[79] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[80] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[81] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[82] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[83] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[84] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[85] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[86] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[87] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[88] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[89] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[90] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[91] = 0x8602;   // byte to mcn48  
	pCCSRBAR->cpm.siram[SI].rx[92] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[93] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[94] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[95] = 0x8E02;   // byte to mcn112                                            

	pCCSRBAR->cpm.siram[SI].rx[96] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[97] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[98] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[99] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[100] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[101] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[102] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[103] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[104] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[105] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[106] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[107] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[108] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[109] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[110] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[111] = 0x8E02;   // byte to mcn112     

	pCCSRBAR->cpm.siram[SI].rx[112] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[113] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[114] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[115] = 0x8602;   // byte to mcn48  
	pCCSRBAR->cpm.siram[SI].rx[116] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[117] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[118] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[119] = 0x8E02;   // byte to mcn112            

	pCCSRBAR->cpm.siram[SI].rx[120] = 0x8002;   // byte to mcn0 
	pCCSRBAR->cpm.siram[SI].rx[121] = 0x8202;   // byte to mcn16 
	pCCSRBAR->cpm.siram[SI].rx[122] = 0x8402;   // byte to mcn32 
	pCCSRBAR->cpm.siram[SI].rx[123] = 0x8602;   // byte to mcn48 
	pCCSRBAR->cpm.siram[SI].rx[124] = 0x8802;   // byte to mcn64 
	pCCSRBAR->cpm.siram[SI].rx[125] = 0x8A02;   // byte to mcn80 
	pCCSRBAR->cpm.siram[SI].rx[126] = 0x8C02;   // byte to mcn96 
	pCCSRBAR->cpm.siram[SI].rx[127] = 0x8E03;   // byte to mcn112 - WRAP SET   

	// Enable TDM in loopback mode (CRT set) 
	pCCSRBAR->cpm.simcc[SI].sixmr[TDM] = 0x0840;
}


//--------------------------------------------------------------------------
//
// FUNCTION NAME: BDRxError
//
// DESCRIPTION:
//
//     Return TRUE if Buffer Descriptor Status bd_cstatus indicates Receive 
//     Error; Return FALSE otherwise, note Receive Errors are as follows:
//
//     0x80: DPLL Error (DE)
//     0x20: Length Violation (LG)
//     0x10: Non-Octet Aligned (NO)
//     0x8:  Rx Abort Sequence (AB)
//     0x4:  Rx CRC Error (CR)
//     0x2:  Overrun (OV)
//     0x1:  Carrier Detect Lost (CD)
//
// EXTERNAL EFFECTS: None
//
// PARAMETERS:  
//
//     bd_cstatus
//
// RETURNS: TRUE if there was an error and FALSE if there wasn't
//
//-------------------------------------------------------------------------
UHWORD BDRxError(UHWORD bd_cstatus)
{
   	if (bd_cstatus & BD_RX_ERROR)
		return TRUE;
   	else
      		return FALSE;
} // end BDRxError 


//--------------------------------------------------------------------------
//
// FUNCTION NAME: LastBD
//
// DESCRIPTION:   Return TRUE if Buffer Descriptor with status and 
//                control register bd_cstatus is last in frame; Return 
//                FALSE otherwise.
//
// EXTERNAL EFFECTS: None
//
// PARAMETERS:  
//
//     bd_cstatus - status word of this buffer descriptor
//
// RETURNS:
//
//-------------------------------------------------------------------------
UHWORD LastBD(UHWORD bd_cstatus)
{
	if (bd_cstatus & 0x0800)
      		return TRUE;
   	else
      		return FALSE;
} // end LastBD 


//--------------------------------------------------------------------------
//
// FUNCTION NAME: BDEmpty
//
// DESCRIPTION:
//
//     Return TRUE if Buffer Descriptor Status bd_cstatus is empty. Return 
//     FALSE otherwise.
//
// EXTERNAL EFFECTS: None
//
// PARAMETERS:  
//
//     bd_cstatus - status word of this buffer descriptor
//
// RETURNS: TRUE if empty and FALSE if it isn't
//
//-------------------------------------------------------------------------
UHWORD BDEmpty(UHWORD bd_cstatus)
{
   	if (bd_cstatus & 0x8000)
		return TRUE;
   	else
      		return FALSE;
} // end BDEmpty 


//-------------
// END PROGRAM 
//-------------



