/************************************************************************************************/
/*                                      Includes Section                                        */
/************************************************************************************************/
#include "LCD.h"

/************************************************************************************************/
/*                                     Constants Section                                        */
/************************************************************************************************/

const unsigned int gau16CharactersArray[] =
{
     (unsigned int)(SEG_A|SEG_B|SEG_C|SEG_D|SEG_E|SEG_F|SEG_J|SEG_N),      //0
     (unsigned int)(SEG_B|SEG_C|SEG_J),                                    //1
     (unsigned int)(SEG_A|SEG_B|SEG_K|SEG_G|SEG_E|SEG_D),                  //2
     (unsigned int)(SEG_A|SEG_B|SEG_K|SEG_C|SEG_D),                        //3
     (unsigned int)(SEG_F|SEG_G|SEG_K|SEG_B|SEG_C),                        //4
     (unsigned int)(SEG_A|SEG_F|SEG_G|SEG_K|SEG_C|SEG_D),                  //5
     (unsigned int)(SEG_A|SEG_F|SEG_G|SEG_K|SEG_C|SEG_D|SEG_E),            //6
     (unsigned int)(SEG_A|SEG_B|SEG_K|SEG_C),                              //7
     (unsigned int)(SEG_A|SEG_B|SEG_C|SEG_D|SEG_E|SEG_F|SEG_G|SEG_K),      //8
     (unsigned int)(SEG_A|SEG_F|SEG_G|SEG_K|SEG_B|SEG_C),                  //9
     (unsigned int)(0),   //:
     (unsigned int)(0),   //;
     (unsigned int)(0),   //<
     (unsigned int)(0),   //=
     (unsigned int)(0),   //>
     (unsigned int)(0),   //?
     (unsigned int)(0),   //@
     (unsigned int)(SEG_A|SEG_B|SEG_C|SEG_E|SEG_F|SEG_G|SEG_K),            //A
     (unsigned int)(SEG_A|SEG_B|SEG_C|SEG_D|SEG_E|SEG_F|SEG_K),            //B
     (unsigned int)(SEG_A|SEG_F|SEG_E|SEG_D),                              //C
     (unsigned int)(SEG_F|SEG_E|SEG_N|SEG_H),                              //D
     (unsigned int)(SEG_A|SEG_F|SEG_E|SEG_D|SEG_G),                        //E
     (unsigned int)(SEG_A|SEG_F|SEG_E|SEG_G),                              //F
     (unsigned int)(SEG_A|SEG_F|SEG_K|SEG_C|SEG_D|SEG_E),                  //G
     (unsigned int)(SEG_F|SEG_E|SEG_G|SEG_K|SEG_B|SEG_C),                  //H
     (unsigned int)(SEG_A|SEG_I|SEG_M|SEG_D),                              //I
     (unsigned int)(SEG_B|SEG_C|SEG_D|SEG_E),                              //J
     (unsigned int)(SEG_F|SEG_E|SEG_G|SEG_J|SEG_L),                        //K
     (unsigned int)(SEG_F|SEG_E|SEG_D),                                    //L
     (unsigned int)(SEG_F|SEG_E|SEG_H|SEG_J|SEG_B|SEG_C),                  //M
     (unsigned int)(SEG_F|SEG_E|SEG_H|SEG_L|SEG_B|SEG_C),                  //N
     (unsigned int)(SEG_A|SEG_B|SEG_C|SEG_D|SEG_E|SEG_F),                  //O
     (unsigned int)(SEG_F|SEG_E|SEG_A|SEG_B|SEG_K|SEG_G),                  //P
     (unsigned int)(SEG_A|SEG_B|SEG_C|SEG_D|SEG_E|SEG_F|SEG_L),            //Q
     (unsigned int)(SEG_F|SEG_E|SEG_A|SEG_B|SEG_K|SEG_G|SEG_L),            //R
     (unsigned int)(SEG_D|SEG_L|SEG_H|SEG_A),                              //S
     (unsigned int)(SEG_F|SEG_A|SEG_B|SEG_I|SEG_M),                        //T
     (unsigned int)(SEG_F|SEG_E|SEG_D|SEG_C|SEG_B),                        //U
     (unsigned int)(SEG_H|SEG_L|SEG_C|SEG_B),                              //V
     (unsigned int)(SEG_F|SEG_E|SEG_D|SEG_M|SEG_C|SEG_B),                  //W
     (unsigned int)(SEG_H|SEG_N|SEG_L|SEG_J),                              //X
     (unsigned int)(SEG_H|SEG_M|SEG_J),                                    //Y
     (unsigned int)(SEG_A|SEG_J|SEG_N|SEG_D)                               //Z
};

const unsigned char *gpau8PortArray[] =
{
     (unsigned char *)(&LCD_DIGIT1_PORT),
     (unsigned char *)(&LCD_DIGIT2_PORT),
     (unsigned char *)(&LCD_DIGIT3_PORT),
     (unsigned char *)(&LCD_DIGIT4_PORT),
     (unsigned char *)(&LCD_DIGIT5_PORT),
     (unsigned char *)(&LCD_DIGIT6_PORT),
     (unsigned char *)(&LCD_DIGIT7_PORT),
     (unsigned char *)(&LCD_DIGIT8_PORT)
};

const unsigned char gau8NibblePerDigit[] =
{
     LCD_DIGIT1_NIBBLE,
     LCD_DIGIT2_NIBBLE,
     LCD_DIGIT3_NIBBLE,
     LCD_DIGIT4_NIBBLE,
     LCD_DIGIT5_NIBBLE,
     LCD_DIGIT6_NIBBLE,
     LCD_DIGIT7_NIBBLE,
     LCD_DIGIT8_NIBBLE
};

/************************************************************************************************/
/*                                Function Prototipes Section                                   */
/************************************************************************************************/


/************************************************************************************************/
/*                                Global Variables Section                                      */
/************************************************************************************************/
_uSegments LCDBuffer[MAX_DIGITS];
unsigned char gu8LCDTimeElapsed = 0;
unsigned char gu8LCDPhase = 0;
unsigned char gu8Cycle = 0;
unsigned int g16ContrastValue = CONSTRAST_LOW;
unsigned char gu8ContrastToggle = 0;
unsigned char gu8InitialIndex = 0;

/************************************************************************************************/
/*                                      Functions Section                                       */
/************************************************************************************************/

void TIMERInit(void)
{

/********       Timer Configuration        ********/

     TIMER_REGISTER = 0x0F;                  /* Select busclk as clock source */
                                             /* Set prescaler = 128 */
                                             /* Timer Frequency = busclk/prescaler = 20MHz/128 */

/********       Timer Channel Configuration        ********/

     TIM_CHANNEL_REGISTER = 0x50;            /* Channel Interrupt Enabled */
                                             /* Config channel as Output Compare */
                                             /* The timer pin is not modified on Compare */

     RELOAD_TIMER_VALUE = WAVE_FORM_PERIOD;  /* Set Time Period for Channel 0 */
}

void vfnLCDInit(void)
{
     LCD_DIGIT1_PORT_INIT;
     LCD_DIGIT2_PORT_INIT;
     LCD_DIGIT3_PORT_INIT;
     LCD_DIGIT4_PORT_INIT;
     LCD_DIGIT5_PORT_INIT;
     LCD_DIGIT6_PORT_INIT;
     LCD_DIGIT7_PORT_INIT;
     LCD_DIGIT8_PORT_INIT;
     
     LCD_BKP0_INIT;
     LCD_BKP1_INIT;
     LCD_BKP2_INIT;
     LCD_BKP3_INIT;
     
     TIMERInit();
}

void vfnLCDPrintMessage(unsigned char *u8pTextSource,unsigned char u8TopDigit)
{
     unsigned char u8CopyIndex = 0;

     while(u8CopyIndex < MAX_DIGITS)
     {
          if (((gu8InitialIndex + u8CopyIndex) > u8TopDigit) ||
              ((*(u8pTextSource + gu8InitialIndex + u8CopyIndex)) < '0'))
          {
               LCDBuffer[u8CopyIndex].u16Word = BLANK_DIGIT;
          }
          else
          {
               LCDBuffer[u8CopyIndex].u16Word =
               gau16CharactersArray[(*(u8pTextSource + gu8InitialIndex + u8CopyIndex))- '0'];
          }
          u8CopyIndex++;
     }
     gu8InitialIndex++;

     if ((gu8InitialIndex + MAX_DIGITS) >= u8TopDigit)
     {
          gu8InitialIndex = 0;
     }
}

void vfnFPOutPortNibble(unsigned char u8Port,unsigned char u8BPValue)
{
     unsigned char *u8pOutPort;
     
     u8pOutPort = (unsigned char *)(gpau8PortArray[u8Port]);
     
     if((gau8NibblePerDigit[u8Port]) &HIGH_NIBBLE)
     {
          *u8pOutPort |= (u8BPValue <<4);
     }
     else
     {
          *u8pOutPort |= (u8BPValue &0x0F);
     }
}

void vfnFrontPlaneA_Odd(void)
{
     unsigned char u8Digit = 0;
     while(u8Digit < MAX_DIGITS)
     {
          vfnFPOutPortNibble(u8Digit,LCDBuffer[u8Digit]._sMergedSegments.u8BackPlane0);
          u8Digit++;
     }
}

void vfnFrontPlaneA_Even(void)
{
     unsigned char u8Digit = 0;
     while(u8Digit < MAX_DIGITS)
     {
          vfnFPOutPortNibble(u8Digit,(~LCDBuffer[u8Digit]._sMergedSegments.u8BackPlane0));
          u8Digit++;
     }
}

void vfnFrontPlaneB_Odd(void)
{
     unsigned char u8Digit = 0;
     while(u8Digit < MAX_DIGITS)
     {
          vfnFPOutPortNibble(u8Digit,LCDBuffer[u8Digit]._sMergedSegments.u8BackPlane1);
          u8Digit++;
     }
}

void vfnFrontPlaneB_Even(void)
{
     unsigned char u8Digit = 0;
     while(u8Digit < MAX_DIGITS)
     {
          vfnFPOutPortNibble(u8Digit,(~LCDBuffer[u8Digit]._sMergedSegments.u8BackPlane1));
          u8Digit++;
     }
}

void vfnFrontPlaneC_Odd(void)
{
     unsigned char u8Digit = 0;
     while(u8Digit < MAX_DIGITS)
     {
          vfnFPOutPortNibble(u8Digit,LCDBuffer[u8Digit]._sMergedSegments.u8BackPlane2);
          u8Digit++;
     }
}

void vfnFrontPlaneC_Even(void)
{
     unsigned char u8Digit = 0;
     while(u8Digit < MAX_DIGITS)
     {
          vfnFPOutPortNibble(u8Digit,(~LCDBuffer[u8Digit]._sMergedSegments.u8BackPlane2));
          u8Digit++;
     }
}

void vfnFrontPlaneD_Odd(void)
{
     unsigned char u8Digit = 0;
     while(u8Digit < MAX_DIGITS)
     {
          vfnFPOutPortNibble(u8Digit,LCDBuffer[u8Digit]._sMergedSegments.u8BackPlane3);
          u8Digit++;
     }
}

void vfnFrontPlaneD_Even(void)
{
     unsigned char u8Digit = 0;
     while(u8Digit < MAX_DIGITS)
     {
          vfnFPOutPortNibble(u8Digit,(~LCDBuffer[u8Digit]._sMergedSegments.u8BackPlane3));
          u8Digit++;
     }
}

void vfnFrontPlaneContrast(void)
{
     unsigned char u8Digit = 0;
     while(u8Digit < MAX_DIGITS)
     {
          vfnFPOutPortNibble(u8Digit,gu8ContrastToggle);
          u8Digit++;
     }

     gu8ContrastToggle ^= FULL_DIGIT;
     gu8Cycle ^= MAX_PHASE;
}

void (* const pafnFPSequence[])(void) = 
{
     vfnFrontPlaneContrast,                            //ODD sequence
     vfnFrontPlaneA_Odd,          
     vfnFrontPlaneB_Odd,
     vfnFrontPlaneC_Odd,
     vfnFrontPlaneD_Odd,
     vfnFrontPlaneContrast,                            //Even sequence
     vfnFrontPlaneA_Even,
     vfnFrontPlaneB_Even,
     vfnFrontPlaneC_Even,
     vfnFrontPlaneD_Even
};

void vfnLCDDriver(void)
{
     if(gu8LCDTimeElapsed)
     {
          gu8LCDTimeElapsed = 0;

          BACK_PLANE0_DDR = 0;
          BACK_PLANE1_DDR = 0;
          BACK_PLANE2_DDR = 0;
          BACK_PLANE3_DDR = 0;
          
          switch(gu8LCDPhase)
          {
               case 0:
                    BACK_PLANE0_PIN = gu8ContrastToggle;
                    BACK_PLANE0_DDR = 1;
                    BACK_PLANE1_PIN = gu8ContrastToggle;
                    BACK_PLANE1_DDR = 1;
                    BACK_PLANE2_PIN = gu8ContrastToggle;
                    BACK_PLANE2_DDR = 1;
                    BACK_PLANE3_PIN = gu8ContrastToggle;
                    BACK_PLANE3_DDR = 1;
                    break;
                    
               case 1:
                    BACK_PLANE0_PIN = gu8Cycle;
                    BACK_PLANE0_DDR = 1;
                    break;
                    
               case 2:
                    BACK_PLANE1_PIN = gu8Cycle;
                    BACK_PLANE1_DDR = 1;
                    break;
                    
               case 3:
                    BACK_PLANE2_PIN = gu8Cycle;
                    BACK_PLANE2_DDR = 1;
                    break;
                    
               case 4:
                    BACK_PLANE3_PIN = gu8Cycle;
                    BACK_PLANE3_DDR = 1;
                    break;
                    
               default:
                    break;
          }

          LCD_DIGIT1_CLEAR_PORT;
          LCD_DIGIT2_CLEAR_PORT;
          LCD_DIGIT3_CLEAR_PORT;
          LCD_DIGIT4_CLEAR_PORT;
          LCD_DIGIT5_CLEAR_PORT;
          LCD_DIGIT6_CLEAR_PORT;
          LCD_DIGIT7_CLEAR_PORT;
          LCD_DIGIT8_CLEAR_PORT;
          pafnFPSequence[gu8LCDPhase+gu8Cycle]();
     }
}

interrupt LCD_INTERRUPT_VECTOR void LCD_ISR(void)
{
     TIM_CHANNEL_REGISTER;
     CLEAR_TIMER_FLAG;                                      //Clear Interrupt Flag
     
     gu8LCDTimeElapsed = 1;

     if ((++gu8LCDPhase) == MAX_PHASE)
     {
          RELOAD_TIMER_VALUE += g16ContrastValue;            //Reload Constrast Period
          gu8LCDPhase = 0;
     }
     else
     {
          RELOAD_TIMER_VALUE += WAVE_FORM_PERIOD;            //Reload WaveForm Period
     }
}