/////////////////////////////////////////////////////////////////////////////////////////
//
// Sample for Freescale DEMO9S08LC60
//
// This program displays the value of the PTA0 potentiometer on the LEDs
// and on the LCD.
//
// Before to run this example, verify that all jumpers are in their default
// position. See the "Summary of Jumper and Connector Settings" chapter in
// the user's manual.
//
// --------------------------------------------------------------------------------------
//
// CodeWarrior V5.1 for HC(S)08
// Registers Files version 2.87.175
//
// --------------------------------------------------------------------------------------
//
// Copyright (c) 2006 SofTec Microsystems
// http://www.softecmicro.com/
//
/////////////////////////////////////////////////////////////////////////////////////////

#include <hidef.h>      /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "lcd.h"

// Unsecured status flash
const unsigned char flash_security  @0xFFBF = 0xFE;


/////////////////////////////////////////////////////////////////////////////////////////
// PeriphInit
// --------------------------------------------------------------------------------------
// Initializes various registers and peripherals
/////////////////////////////////////////////////////////////////////////////////////////
void PeriphInit(void)
{
  // Disables COP and Enable RESET and BKGD pin
  SOPT1 = 0x13;
  
  #if !CLOCK
    // Selects FEE mode for fBUS = 8 MHz
    // Using crystal low-range oscillator. FLL prescaler factor (P) = 64
    ICGC1 = 0x38;
    // Sets MFD multiplication factor (N) to 4 and RFD divisor (R) to 1
    ICGC2 = 0x20;
    // Waita until FLL frequency is locked
    while (!(ICGS1 & 0x08))
      ;
  #else
    // Selecta FEI mode
    // Sets trimming for fBUS = 8 MHz
    ICGTRM = NVICGTRM;
    // Using internal reference clock. FLL prescaler factor (P) = 64
    ICGC1 = 0x28;
    // Sets MFD multiplication factor (N) to 14 and RFD divisor (R) to 2
    ICGC2 = 0x51;
    // Waits until FLL frequency is locked
    while (!(ICGS1 & 0x08))
      ;
  #endif
  
  //Setup RTI
  SRTISC = 0x47;
  //Setup ADC
  ADCCFG = 0x04;
  ADCSC2 = 0x40;
  ADCSC1 = 0x00;
  APCTL1 = 0x01;
  //Enable ADC Interrupts
  ADCSC1_AIEN = 1;

 // Counter overflow every 1ms
  TPM1MOD = 8000;
  // Stops timer and selects 1 as prescaler divisor
  TPM1SC = 0x00;

  // Initializes LCD Driver Peripheral
  LCDInit();
        
}

/////////////////////////////////////////////////////////////////////////////////////////
// Delay
// --------------------------------------------------------------------------------------
// Delay of multiples of 1ms
/////////////////////////////////////////////////////////////////////////////////////////
void Delay(unsigned char delaytime)
{
  // Selcts fBUS as timer clock source and starts the timer
  TPM1SC = 0x08;
  while(delaytime)
    if(TPM1SC&0x80) 
    {
      delaytime--;
      // Clears TOF flag
      TPM1SC &= ~0x80;
    }
  // Stops the timer  
  TPM1SC = 0x00;
}

/////////////////////////////////////////////////////////////////////////////////////////
// PrintString
// --------------------------------------------------------------------------------------
// Prints a string on the LCD
/////////////////////////////////////////////////////////////////////////////////////////
void PrintString(unsigned char *str)
{
  byte charloc;

  LCDClear();
  for(charloc = 0; str[charloc] != 0; charloc++)
    LCDPutChar(str[charloc], charloc+1);
    
}

/////////////////////////////////////////////////////////////////////////////////////////
// SlideString
// --------------------------------------------------------------------------------------
// Slides strings on the LCD
/////////////////////////////////////////////////////////////////////////////////////////
void SlideString(unsigned char *str, unsigned char delay)
{

  unsigned char rotator, charloc, start;

  for(rotator = 0; str[rotator] != 0; rotator++) 
  {
    Delay(delay);
    LCDClear();
    if(rotator < NUM_DIGITS) 
      start = 0;
    else  
      start = rotator-NUM_DIGITS+1; 
    for(charloc = start; charloc <= rotator; charloc++)
      LCDPutChar(str[charloc], NUM_DIGITS-rotator+charloc);
  }
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// DispHexVal
// --------------------------------------------------------------------------------------
// Displays val variable value in hex on the LCD in location startloc
/////////////////////////////////////////////////////////////////////////////////////////
void DispHexVal(unsigned char val, unsigned char startloc)
{

  unsigned char ch[2];
    
  // High value
  if((val>>4) < 0x0A)
    ch[1] = '0' + (val>>4);
  else
    ch[1] = 'A' + (val>>4) - 0x0A;
  // Low value
  if((val&0x0F) < 0x0A)
    ch[0] = '0' + (val&0x0F);
  else
    ch[0] = 'A' + (val&0x0F) - 0x0A;
  
  LCDPutChar(ch[1], startloc);
  LCDPutChar(ch[0], startloc+1);
  
}

/////////////////////////////////////////////////////////////////////////////////////////
// MAIN
// --------------------------------------------------------------------------------------
// Entry point
/////////////////////////////////////////////////////////////////////////////////////////
void main(void)
{
  
  // Initializes Peripherals
  PeriphInit();
  LCDOnOffState();
  EnableInterrupts;
  
  // Prints a welcome string
  SlideString("ADC TRIGGERED BY RTI     ", 250);
  PrintString("POT = ");
  // Turns on the Freescale logo on the LCD
  LCDWriteSegment(FREESCALE,0);  
  
  for(;;) 
  {
    __asm wait;
  }
  
}

void interrupt 18 ADCInterrrupt(void){

  DispHexVal(ADCRH,5);
  DispHexVal(ADCRL,7);
}