/**************************************************************************
 *
 * Copyright:
 *	MOTOROLA, INC. All Rights Reserved.  
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Motorola, Inc. This 
 *  software is provided on an "AS IS" basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, MOTOROLA 
 *  DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED, INCLUDING 
 *  IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
 *  PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD TO THE 
 *  SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY 
 *  ACCOMPANYING WRITTEN MATERIALS.
 * 
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL MOTOROLA BE LIABLE FOR ANY DAMAGES WHATSOEVER (INCLUDING 
 *  WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS, BUSINESS 
 *  INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER PECUNIARY
 *  LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.   
 * 
 *  Motorola assumes no responsibility for the maintenance and support
 *  of this software
 ***************************************************************************/

/****************************************************************************/
/* FILE NAME: tpu_mcpwm.c                     COPYRIGHT (c) MOTOROLA 2002 	*/
/* VERSION: 1.0                                   All Rights Reserved     	*/
/*                                                                        	*/
/* DESCRIPTION: This file contains the TPU MCPWM functions. These functions */
/* allow you to completely control TPU channels running the MCPWM function. */
/* They provide a simple interface requiring the minimum amount of        	*/
/* configuration by the user.                                             	*/
/*==========================================================================*/
/* HISTORY           ORIGINAL AUTHOR: Mario Ramiro Perez II                 */
/*                                                                        	*/
/* REV      AUTHOR         DATE       DESCRIPTION OF CHANGE                 */
/* ---   -----------     ---------    ---------------------                 */
/* 1.0   Mario R. Perez  24 Sep 02    Initial version of function.          */
/****************************************************************************/

#include "tpu_mcpwm.h"
#include "mpc500_util.h"


/***************************************************************************************
FUNCTION      : tpu_mcpwm_master_init

PURPOSE       : To initialize a TPU channel to run the MCPWM function.

INPUTS NOTES  : This function has 5 parameters:

            *tpu          : This is a pointer to the TPU3 module to use. It is of type 
            				TPU3_tag which is defined in m_tpu3.h.
            channel       : This is the TPU channel number of the MCPWM master channel.  
            				The master must be assigned the lowest channel number of all 
            				MCPWM channels.  
            priority      : This is the priority to assign to the channel.  This 
            				parameter should be assigned a value of:
                            TPU_PRIORITY_HIGH, TPU_PRIORITY_MIDDLE or TPU_PRIORITY_LOW.
			period        : This defines the period of the PWM output in TCR1 timebase
							clocks.	period < $4000. 
			irq_rate 	  :	This determines the frequency of periodic interrupt requests
							to the host CPU. Any 8 bit value is valid.  If the irq_rate 
							= 0 then an interrupt request is generated every 256 PWM 
							periods.
							
EXTRA NOTES  : The MCPWM function Master channel has 1 parameter which is updated by the TPU:

			period_count  :	This parameter is used by the TPU as a PWM period counter for 
							periodic CPU interrupt requests and is automatically reset to 
							zero after each request is generated (when PERIOD_COUNT = 
							IRQ_RATE). Should be initialized to a start value of $FF.							
*******************************************************************************************/
void tpu_mcpwm_master_init(struct TPU3_tag *tpu, UINT8 channel, UINT8 priority, 
			UINT16 period, UINT8 irq_rate )
{
    /* disable channel so it can be configured safely */
    tpu_disable( tpu, channel);

    /* select MCPWM function */
    tpu_func( tpu, channel, TPU_FUNCTION_MCPWM);


    /* Initialize parameter RAM                                     */
    /* - initialize PERIOD & IRQ_RATE to their desired values 		*/
    /* - initialize PERIOD_COUNT to $FF 							*/

    tpu->PARM.R[channel][TPU_MASTER_PERIOD] = period;
    tpu->PARM.R[channel][TPU_MASTER_IRQRATE_PERIODCNT] = (INT16) (irq_rate << 8) | 0xFF;
    tpu->PARM.R[channel][TPU_MASTER_RISETIMEPTR] = (channel << 4) + 4;
    tpu->PARM.R[channel][TPU_MASTER_FALLTIMEPTR] = (channel << 4) + 6;

    /* Initialize channel */
    tpu_hsr(tpu, channel, TPU_MCPWM_INIT_MASTER);

    /* Enable channel by assigning a priority */
    tpu_enable(tpu, channel, priority);
}


/****************************************************************************************
FUNCTION      : tpu_mcpwm_slave_edgemode_init

PURPOSE       : To initialize a TPU channel to run the MCPWM function.

INPUTS NOTES  : This function has 7 parameters:

            *tpu          : This is a pointer to the TPU3 module to use. It is of type 
            				TPU3_tag which is defined in m_tpu3.h
            channel       : This is the TPU channel number of the MCPWM edge mode slave 
            				channel.  This parameter must be assigned a higher value than 
            				the channel number of the associated master channel.   
            priority      : This is the priority to assign to the channel.  This parameter 
            				should be assigned the same value as the priority of the associated 
            				master channel.
			period        : This defines the period of the PWM output in TCR1 clocks.
							period < $4000.  The PERIOD parameters of all slaves must
							have the same value as that of the master that they are 
							referencing.	 
			high_time 	  :	This is specified in TCR1 clocks over the range 0 to PERIOD, 
							representing 0 to 100% duty cycle.
		    high_time_ptr : This contains the TPU memory address of the value to be used for 
		    				the high time of the PWM output waveform.  Any TPU memory location 
		    				can be used, but in order to use the high time value specified by 
		    				the previous parameter, this pointer should be set to (channel * $10) 
		    				+ 2.  For example, if this slave is initialized to run on TPU channel 
		    				7, then high_time_ptr should be initialized to $72 in order to use the 
		    				high_time value specified above..  
			master_channel: This is the TPU channel number of the MCPWM master channel.					                          
*******************************************************************************************/
void tpu_mcpwm_slave_edgemode_init(struct TPU3_tag *tpu, UINT8 channel, UINT8 priority,   
                UINT16 period, UINT16 high_time, UINT8 high_time_ptr, UINT8 master_channel )
{    
    /* disable channel so it can be configured safely */
    tpu_disable( tpu, channel);

    /* select MCPWM function */
    tpu_func( tpu, channel, TPU_FUNCTION_MCPWM);


    /* Initialize parameter RAM												*/
    /* - initialize PERIOD, HIGH_TIME, HIGH_TIME_POINTER & MASTER_CHANNEL	*/
    /*	 to their desired values 											*/
   
    tpu->PARM.R[channel][TPU_SLAVE_PERIOD] = period;
    tpu->PARM.R[channel][TPU_SLAVE_HIGHTIME] = high_time;
    tpu->PARM.R[channel][TPU_SLAVE_HIGHTIME_PTR] = high_time_ptr;
    tpu->PARM.R[channel][TPU_SLAVE_RISETIME_PTR] = (master_channel << 4) + 4;
    tpu->PARM.R[channel][TPU_SLAVE_FALLTIME_PTR] = (master_channel << 4) + 6;
    
    /* Configure the channel measurement mode */
    tpu_hsq(tpu, channel, TPU_MCPWM_EDGE_ALLIGNED_MODE);

    /* Initialize channel */
    tpu_hsr(tpu, channel, TPU_MCPWM_All_OTHERSLAVES);

    /* Enable channel by assigning a priority */
    tpu_enable(tpu, channel, priority);
}


/****************************************************************************************
FUNCTION      : tpu_mcpwm_slavea_centermode_init

PURPOSE       : To initialize a TPU channel to run the MCPWM function.

INPUTS NOTES  : This function has 8 parameters:

            *tpu          : This is a pointer to the TPU3 module to use. It is of type 
            				TPU3_tag which is defined in m_tpu3.h
            channel       : This is the TPU channel number of the MCPWM slave A center-
            				aligned channel.  This parameter must be assigned a higher 
            				value than the channel number of the associated master channel.  
            				In CA mode, the slave A channel of a pair must be assigned a 
            				lower channel number than the slave B channel.   When two CA 
            				mode pairs are used to form an inverter driver (the second pair 
            				has dead time and references the CURRENT_HIGH_TIME parameter of 
            				the first pair), the pair without dead time must be assigned 
            				lower channel numbers than the pair with dead time. 
            priority      : This is the priority to assign the channel.  This parameter should 
            				be assigned the same value as the priority of the associated master 
            				channel.
			period        : This defines the period of the PWM output in TCR1 clocks.
							period < $4000.  The PERIOD parameters of all slaves must have 
							the same value as that of the master that they are referencing	 
			dead_time 	  :	This 8-bit parameter contains the total dead time to be 
							subtracted from the high time of the PWM, specified in TCR1
							clocks. 0 < dead_time < $FF.
		    high_time_ptr : This contains the TPU memory address of the value to be used for 
		    				the	high time of the PWM output waveform.  In order to use the value 
		    				specified by the high time parameter in the slave B center-aligned 
		    				mode initialization call, this pointer should be set to (slave B 
		    				channel * $10) + 1.  For example, if this channel's slave B pair is 
		    				initialized to run on TPU channel 5, then high_time_ptr should be 
		    				initialized to $51 in order to use the high_time value specified in 
		    				the slave B initialization.  When a second CA mode slave pair is being 
		    				used to generate a PWM with dead time, high_time_ptr should be initialized 
		    				to (1st slave B channel * $10) + 3. For example, if the 1st slave B 
		    				channel is initialized to run on TPU channel 5, then high_time_ptr  
		    				should be initialized to $53.   
		    polarity      : This is the polarity of the center mode slave A channel.  This 
		    				parameter should be assigned a value of:INVERTED or NONINVERTED
			master_channel: This is the TPU channel number of the MCPWM master channel.					                          
*********************************************************************************************/
void tpu_mcpwm_slavea_centermode_init(struct TPU3_tag *tpu, UINT8 channel, UINT8 priority,   
                UINT16 period, UINT8 dead_time, UINT8 high_time_ptr, UINT8 polarity, 
                UINT8 master_channel)
{
    UINT8 time_ptr1;
    UINT8 time_ptr2;
    
    /* disable channel so it can be configured safely */
    tpu_disable( tpu, channel);

    /* select MCPWM function */
    tpu_func( tpu, channel, TPU_FUNCTION_MCPWM);
	
	/* select the polarity and assign the correct parameter location*/
	if (polarity == NONINVERTED)
		{
			time_ptr1 = (master_channel << 4) + 4; 
			time_ptr2 = (master_channel << 4) + 6;
		}
	else
		{
			time_ptr1 = (master_channel << 4) + 6;
			time_ptr2 = (master_channel << 4) + 4;
		}	
    
    
    /* Initialize parameter RAM                                             */
    /* - initialize PERIOD, DEAD_TIME, HIGH_TIME_POINTER, RISE_TIME_POINTER */
    /*	 	and FALL_TIME_POINTER to there desired values         			*/
   
    tpu->PARM.R[channel][TPU_SLAVEA_PERIOD] = period;
    tpu->PARM.R[channel][TPU_SLAVEA_DEADTIME_HIGHTIMEPTR] = (INT16) (dead_time << 8) | high_time_ptr;
    tpu->PARM.R[channel][TPU_SLAVEA_TIMEPTR1] = time_ptr1;
    tpu->PARM.R[channel][TPU_SLAVEA_TIMEPTR2] = time_ptr2;

    /* Configure the channel measurement mode */
    tpu_hsq(tpu, channel, TPU_MCPWM_CENTER_SLAVE_A_MODE);

    /* Initialize channel */
    if (polarity == NONINVERTED)
		{
   		tpu_hsr(tpu, channel, TPU_MCPWM_All_OTHERSLAVES);
		}
	else
		{
		tpu_hsr(tpu, channel, TPU_MCPWM_SLAVEA_INVERTED);
		}

    /* Enable channel by assigning a priority */
    tpu_enable(tpu, channel, priority);
}



/****************************************************************************************
FUNCTION      : tpu_mcpwm_slaveb_centermode_init

PURPOSE       : To initialize a TPU channel to run the MCPWM function.

INPUTS NOTES  : This function has 6 parameters:

          *tpu            : This is a pointer to the TPU3 module to use. It is of type 
            				TPU3_tag which is defined in m_tpu3.h.
          channel         : This is the TPU channel number of the MCPWM slave B center-aligned 
          					channel.  This parameter must be assigned a higher value than 
          					the channel number of the associated master channel.  In CA mode, 
          					the slave A channel of a pair must be assigned a lower channel number 
          					than the slave B channel.  
          priority        : This is the priority to assign the channel.  This parameter 
          					should be assigned the same value as the priority of the 
          					associated master channel.
		  high_time 	  :	This 8-bit parameter contains the total dead time to be 
							subtracted from the high time of the PWM, specified in TCR1
							clocks. 0 < dead_time < $FF.
		  polarity        : This is the polarity of the center mode slave B channel.  This 
		    				parameter should be assigned a value of:INVERTED or NONINVERTED
		    				This value should be set to the same polarity as that of the 
		    				slave A channel they are referencing.
		  slave_a_channel : This is the TPU channel number of the MCPWM slave A center-aligned 
		  					channel that the slave B is referencing.					 
*******************************************************************************************/
void tpu_mcpwm_slaveb_centermode_init(struct TPU3_tag *tpu, UINT8 channel, UINT8 priority,   
                UINT16 high_time, UINT8 polarity, UINT8 slave_a_channel)
{

    UINT8 b_time_ptr1;
    UINT8 b_time_ptr2;
    
    /* disable channel so it can be configured safely */
    tpu_disable( tpu, channel);

    /* select MCPWM function */
    tpu_func( tpu, channel, TPU_FUNCTION_MCPWM);

	/* select the polarity and assign the correct parameter location*/
		if (polarity == NONINVERTED)
			{
			b_time_ptr1 = (slave_a_channel << 4) + 4; 
			b_time_ptr2 = (slave_a_channel << 4) + 2;
			}
		else
			{
			b_time_ptr1 = (slave_a_channel << 4) + 2;
			b_time_ptr2 = (slave_a_channel << 4) + 4;
			}	
    /* Initialize parameter RAM                                             */
    /* - initialize HIGH_TIME, B_FALL_TIME_POINTER, B_RISE_TIME_POINTER		*/
    /*	 to their desired values         									*/

    tpu->PARM.R[channel][TPU_SLAVEB_HIGHTIME] = high_time;
    tpu->PARM.R[channel][TPU_SLAVEB_BFALLTIMEPTR] = b_time_ptr1;
    tpu->PARM.R[channel][TPU_SLAVEB_BRISETIMEPTR] = b_time_ptr2;
   
    /* Configure the channel measurement mode */
    tpu_hsq(tpu, channel, TPU_MCPWM_CENTER_SLAVE_B_MODE);

    /* Initialize channel */
    tpu_hsr(tpu, channel, TPU_MCPWM_All_OTHERSLAVES);

    /* Enable channel by assigning a priority */
    tpu_enable(tpu, channel, priority);
}


/****************************************************************************************
FUNCTION      : tpu_mcpwm_update_hightime

PURPOSE       : To update a TPU channel running a MCPWM function.

INPUTS NOTES  : This function has 4 parameters:

            *tpu      : This is a pointer to the TPU3 module to use. It is of type 
            			TPU3_tag which is defined in m_tpu3.h.
            channel   : This is the channel number of the MCPWM channel. 
            high_time : This is specified in TCR1 clocks over the range 0 to PERIOD, 
					  	representing 0 to 100% duty cycle.
			mode 	  : This is the mode of the update hightime function.  This parameter 
						should be assigned a value of : EDGE or CENTER				  
*******************************************************************************************/
void tpu_mcpwm_update_hightime(struct TPU3_tag *tpu, UINT8 channel, UINT16 high_time, UINT8 mode)

{
    /* Initialize parameter RAM  */
    /* - update HIGH_TIME to its desired values */
    
	if (mode == EDGE)
		{
    		tpu->PARM.R[channel][TPU_SLAVE_HIGHTIME] = high_time; 
		}
	else
		{
    		tpu->PARM.R[channel][TPU_SLAVEB_HIGHTIME] = high_time;
		}
	   
}  

