/********************************************************************* 
 * 
 * Copyright: 
 *      MOTOROLA, INC. All Rights Reserved.  
 *  You are hereby granted a copyright license to use, modify, and 
 *  distribute the SOFTWARE so long as this entire notice is 
 *  retained without alteration in any modified and/or redistributed 
 *  versions, and that such modified versions are clearly identified 
 *  as such. No licenses are granted by implication, estoppel or 
 *  otherwise under any patents or trademarks of Motorola, Inc. This 
 *  software is provided on an "AS IS" basis and without warranty. 
 * 
 *  To the maximum extent permitted by applicable law, MOTOROLA 
 *  DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED, INCLUDING 
 *  IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 *  PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD TO THE 
 *  SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY 
 *  ACCOMPANYING WRITTEN MATERIALS. 
 * 
 *  To the maximum extent permitted by applicable law, IN NO EVENT 
 *  SHALL MOTOROLA BE LIABLE FOR ANY DAMAGES WHATSOEVER (INCLUDING 
 *  WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS, BUSINESS 
 *  INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER PECUNIARY 
 *  LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.   
 * 
 *  Motorola assumes no responsibility for the maintenance and support 
 *  of this software 
 ********************************************************************/
/************************************************************************/
/* FILE NAME: tpu_uart.c		COPYRIGHT (c) MOTOROLA 2002	*/
/* VERSION: 1.0				    All Rights Reserved		*/
/*									*/
/* DESCRIPTION: This file contains the TPU UART functions.  These 	*/
/* functions allow you to completely control TPU channels running the	*/
/* UART function.  They provide a simple interface requiring the minimum*/
/* amount of configuration by the user.  The functions are:		*/
/*                                                                      */
/*=====================================================================	*/
/* HISTORY				ORIGINAL AUTHOR: Vernon Goler	*/
/* REV		AUTHOR		DATE	DESCRIPTION OF CHANGE		*/
/* ---		------		----	---------------------		*/
/* 1.0		V. GOLER      15/SEP/02	Initial Version of Function	*/
/************************************************************************/

#include "tpu_uart.h"
#include "mpc500_util.h"

/************************************************************************
FUNCTION:		tpu_uart_init
************************************************************************/

/*

Function:		tpu_uart_transmit_init
Purpose:		To initialize a channel to function as a UART transmitter

Input Notes:	This function has 7 parameters

*tpu - This is a pointer to the TPU3 module to use. It is of type TPU3_tag which is defined in m_tpu3.h
channel - This is the TPU channel number of the UART channel.  This parameter should be assigned a value of 0 to 15.
priority - This is the priority to assign to the channel. This parameter should be assigned a value of                         TPU_PRIORITY_HIGH, TPU_PRIORITY_MIDDLE or TPU_PRIORITY_LOW. The TPU priorities are defined in mpc500_utils.h.
baud_rate - Baud rate is a measure of the number of times per second a signal in a communications channel varies, or makes a transition between states (states being frequencies, voltage levels, or phase angles). One baud is one such change. Thus, a 300-baud modem's signal changes state 300 times each second, while a 600- baud modem's signal changes state 600 times per second. The baud_rate value is the number of TCR1 counts per bit time, and is calculated by the following equation:

# Timer Count Register1 (TCR1) counts / second
-----------------------------------------------------------------
number of transitions (baud)/ second


bits_per_data_word - This is the number of bits to be transmitted in one data word.  This bits_per_data_word commonly has a value of eight, because most serial protocols use 8-bit words.
parity - This is the desired parity.  This parameter should be assigned a value of TPU_UART_NOPARITY,  TPU_UART__EVEN_PARITY, or TPU_UART__ODD_PARITY. The TPU baud rates for the UART function are defined in tpu_uart.h.
nointerrupt_interrupt - This parameter determines whether an interrupt is generated when each data word is transmitted.

*/


void tpu_uart_transmit_init(struct TPU3_tag *tpu, UINT8 channel, 	\
UINT8 priority, INT16 baud_rate, INT16 bits_per_data_word, 		\
UINT8 parity, UINT8 nointerrupt_interrupt)
{

	/* disable the channel so the channel can be configured safely		*/
	tpu_disable(tpu, channel);
	
	/* select the UART function for the channel				*/
	tpu_func(tpu, channel, TPU_FUNCTION_UART);
	
	/* setup the desired baud rate by configurating match_rate param	*/
	tpu->PARM.R[channel][TPU_UART_TRANSMIT_MATCH_RATE] = baud_rate;

	/* set the TDRE flag to indicate that the transmit data reg is empty	*/
	tpu->PARM.R[channel][TPU_UART_TRANSMIT_DATA] = 0x8000;
	
	/* configure the number of data bits per data word.  this number
	represents only the number of data bits, and does not include start
	and stop bits or the parity bit						*/
	tpu->PARM.R[channel][TPU_UART_TRANSMIT_DATA_SIZE] = bits_per_data_word;
	
	/* configure the host sequence bits to set the desired parity		*/	
	tpu_hsq(tpu, channel, parity);
	
	/* interrupt status flag						*/
	tpu_clear_interrupt(tpu, channel);
	
	/* if nointerrupt_interrupt flag is set, enable interrupts		*/
	if(nointerrupt_interrupt) {
		tpu_interrupt_enable(tpu, channel);
	}
	
	/* issue host service request						*/
	tpu_hsr(tpu, channel, TPU_UART_TRANSMIT);
	
	/* enable channel by assigning a priority				*/
	tpu_enable(tpu, channel, priority);
}


/*

Function:		tpu_uart_receive_init
Purpose:		To initialize a channel to function as a UART receiver

Input Notes:	This function has 7 parameters

*tpu - This is a pointer to the TPU3 module to use. It is of type TPU3_tag which is defined in m_tpu3.h
channel - This is the TPU channel number of the UART channel.  This parameter should be assigned a value of 0 to 15.
priority - This is the priority to assign to the channel. This parameter should be assigned a value of                         TPU_PRIORITY_HIGH, TPU_PRIORITY_MIDDLE or TPU_PRIORITY_LOW. The TPU priorities are defined in mpc500_utils.h.
baud_rate - Baud rate is a measure of the number of times per second a signal in a communications channel varies, or makes a transition between states (states being frequencies, voltage levels, or phase angles). One baud is one such change. Thus, a 300-baud modem's signal changes state 300 times each second, while a 600- baud modem's signal changes state 600 times per second. The baud_rate value is the number of TCR1 counts per bit time, and is calculated by the following equation:

# Timer Count Register1 (TCR1) counts / second
-----------------------------------------------------------------
number of transitions (baud)/ second


bits_per_data_word - This is the number of bits to be received in one data word.  This bits_per_data_word commonly has a value of eight, because most serial protocols use 8-bit words.
parity - This is the desired parity.  This parameter should be assigned a value of TPU_UART_NOPARITY,  TPU_UART__EVEN_PARITY, or TPU_UART__ODD_PARITY. The TPU baud rates for the UART function are defined in tpu_uart.h.
nointerrupt_interrupt - This parameter determines whether an interrupt is generated when each data word is received.

*/


void tpu_uart_receive_init(struct TPU3_tag *tpu, UINT8 channel, 	\
UINT8 priority, INT16 baud_rate, INT16 bits_per_data_word, 		\
UINT8 parity, UINT8 nointerrupt_interrupt)
{

	/* disable the channel so the channel can be configured safely		*/
	tpu_disable(tpu, channel);
	
	/* select the UART function for the channel				*/
	tpu_func(tpu, channel, TPU_FUNCTION_UART);
	
	/* setup the desired baud rate by configurating match_rate param	*/
	tpu->PARM.R[channel][TPU_UART_RECEIVE_MATCH_RATE] = baud_rate;

	/* configure the number of data bits per data word.  this number
	represents only the number of data bits, and does not include start
	and stop bits or the parity bit						*/
	tpu->PARM.R[channel][TPU_UART_RECEIVE_DATA_SIZE] = bits_per_data_word;
	
	/* configure the host sequence bits to set the desired parity		*/	
	tpu_hsq(tpu, channel, parity);
	
	/* interrupt status flag						*/
	tpu_clear_interrupt(tpu, channel);
	
	/* if nointerrupt_interrupt flag is set, enable interrupts		*/
	if(nointerrupt_interrupt) {
		tpu_interrupt_enable(tpu, channel);
	}
	
	/* issue host service request						*/
	tpu_hsr(tpu, channel, TPU_UART_RECEIVE);
	
	/* enable channel by assigning a priority				*/
	tpu_enable(tpu, channel, priority);
}

/*

Function:		tpu_uart_write_transmit_data
Purpose:		To write data to TPU UART transmitter channel for data to be 
			Serially shifted out

Input Notes:	This function has 3 parameters

*tpu - This is a pointer to the TPU3 module to use. It is of type TPU3_tag which is defined in m_tpu3.h
channel - This is the TPU channel number of the UART channel. This parameter should be assigned a value of 0 to 15.
transmit_data - This the actual data to be transmitted.  Up to 14 bits of data per data word can be transmitted.

*/



void tpu_uart_write_transmit_data(struct TPU3_tag *tpu, UINT8 channel, \
INT16 transmit_data)
{

	/* clear interrupt status register					*/
	tpu_clear_interrupt(tpu, channel);

	/* write the transmit data reg						*/
	tpu->PARM.R[channel][TPU_UART_TRANSMIT_DATA] = \
	(transmit_data & 0x7fff);
	
}


/*

Function:		tpu_uart_transmit_init
Purpose:		To read received data from TPU UART receive channel,
			and to optionally check for parity and or framing errors

Input Notes:	This function has 5 parameters

*tpu - This is a pointer to the TPU3 module to use. It is of type TPU3_tag which is defined in m_tpu3.h
channel - This is the TPU channel number of the UART channel. This parameter should be assigned a value of 0 to 15.
*receive_data - This is a pointer to the received data.  The calling routine of this function should pass the address of where the received data is to be stored to the receive_data parameter. Up to 14 bits of data per data word can be received.
*parity - If parity is enabled, this parameter is either one or zero depending on whether even or odd parity is enabled.  For even parity this bit is set to a one if the number of ones in the received data word is odd, else the bit is set to zero.  For odd parity this bit is set to a one if the number of ones in the received data word is even, else the bit is set to zero.  The calling routine of this function should pass the address of where the parity parameter is to be stored to *parity. The parity is only valid for each received data word.
*framing_error - The framing_error parameter is set to a one if a framing error is detected.  A framing error occurs when the UART function determines that a stop bit is low instead of high. The calling routine of this function should pass the address of where the framing error parameter is to be stored to *framing_error. The framing error indication is only valid for each received data word.

*/


void tpu_uart_read_receive_data(struct TPU3_tag *tpu, UINT8 channel, \
INT16 *receive_data, UINT8 *parity_error, UINT8 *framing_error)
{

	/* Check for parity error						*/
	if((tpu->PARM.R[channel][TPU_UART_RECEIVE_DATA] & 0x8000) == 0x8000)
		*parity_error = 0x1;
	else
		*parity_error = 0x0;
		
/* Check for framing error						*/
	if((tpu->PARM.R[channel][TPU_UART_RECEIVE_DATA] & 0x4000) == 0x4000)
		*framing_error = 0x1;
	else
		*framing_error = 0x0;
		
/* Strip parity and framing error bits in data word				*/
	*receive_data = (tpu->PARM.R[channel][TPU_UART_RECEIVE_DATA] & 0x3fff);
	
}


/******************************************************************************/












/*****************************************************************************/
/* Motorola reserves the right to make changes without further notice to any */
/* product herein to improve reliability, function, or design. Motorola does */
/* not assume any  liability arising  out  of the  application or use of any */
/* product,  circuit, or software described herein;  neither  does it convey */
/* any license under its patent rights  nor the  rights of others.  Motorola */
/* products are not designed, intended,  or authorized for use as components */
/* in  systems  intended  for  surgical  implant  into  the  body, or  other */
/* applications intended to support life, or  for any  other application  in */
/* which the failure of the Motorola product  could create a situation where */
/* personal injury or death may occur. Should Buyer purchase or use Motorola */
/* products for any such intended  or unauthorized  application, Buyer shall */
/* indemnify and  hold  Motorola  and its officers, employees, subsidiaries, */
/* affiliates,  and distributors harmless against all claims costs, damages, */
/* and expenses, and reasonable  attorney  fees arising  out of, directly or */
/* indirectly,  any claim of personal injury  or death  associated with such */
/* unintended or unauthorized use, even if such claim alleges that  Motorola */
/* was negligent regarding the  design  or manufacture of the part. Motorola */
/* and the Motorola logo* are registered trademarks of Motorola Inc.         */
/* Motorola is an Equal Opportunity/Affirmative Action Employer.             */
/*****************************************************************************/

		
