/**************************************************************************/
/* FILE NAME: tpu_halld.c                     COPYRIGHT (c) MOTOROLA 2002 */
/* VERSION: 1.0                                   All Rights Reserved     */
/*                                                                        */
/* DESCRIPTION: This file contains the TPU HALLD functions. These         */
/* functions allow the user to control control TPU channels running the   */
/* HALLD function.                                                        */
/* The functions included are as follows:                                 */
/* tpu_halld_init                                                         */
/* tpu_halld_enable                                                       */
/* tpu_halld_disable                                                      */
/* tpu_halld_set_direction                                                */
/* tpu_halld_set_state_no_address                                         */
/* tpu_halld_get_state_no                                                 */
/* tpu_halld_get_pinstate                                                 */
/*========================================================================*/
/* HISTORY           ORIGINAL AUTHOR: Ken Terry                           */
/* REV      AUTHOR      DATE       DESCRIPTION OF CHANGE                  */
/* ---   -----------  ---------    ---------------------                  */
/* 1.0   K Terry      14/Aug/02    Initial version of function.           */
/**************************************************************************/
#include "tpu_halld.h"
#include "mpc500_util.h"

/*******************************************************************************
FUNCTION      : tpu_halld_init
PURPOSE       : To initialize two or three TPU channels to run the HALLD function
INPUTS NOTES  : This function has 4 parameters:

                 *tpu    - This is a pointer to the TPU3 module to use. It is of
                           type TPU3_tag which is defined in m_tpu3.h

                 channel - This is the channel number of the primary HALLD
                           channel. The following channels is used for two channel
                           mode and following two channels are used for three 
                           channel mode. 

                 state_no_address - This value defines the STATE_NO_ADDRESS value 
                           in the parameter RAM.It takes the value 0x00xy, where xy
                           represents the address in parameter RAM where the state no.
                           is to be written.
                 
                 mode    - This parameter selects either two or three channel mode

RETURNS NOTES : none

*******************************************************************************/
void tpu_halld_init(struct TPU3_tag *tpu, UINT8 channel, INT16 direction,  
                    INT16 state_no_address, UINT8 mode)
{


UINT8	channel2;
UINT8	channel3;


    channel2 = (channel + 1) & 0xf;
    channel3 = (channel + 2) & 0xf;
    
    /* disable channels so they can be configured safely */
    tpu_disable( tpu, channel);
    tpu_disable( tpu, channel2);
    if (mode==HALLD_THREE_CHANNEL_MODE)
    {
    	tpu_disable(tpu, channel3);
    }
    
        
    /* select HALLD function for all channels */
    tpu_func( tpu, channel, TPU_FUNCTION_HALLD);
    tpu_func( tpu, channel2, TPU_FUNCTION_HALLD);
    if (mode==HALLD_THREE_CHANNEL_MODE)
    {
    	tpu_func(tpu, channel3, TPU_FUNCTION_HALLD);
    }
        
    /* Initialize parameter RAM - use default values for initialization */
    tpu->PARM.R[channel][HALLD_DIRECTION] = HALLD_DIRECTION_0;      // Select direction 0 as default

    tpu->PARM.R[channel][HALLD_STORE_MODE] = (INT16)(mode&0x00ff);   // store mode in channel 1 unused parameter RAM


    /* Configure the first channel as Channel A */
    /* Second channel as Channel B              */
    /* if 3 channel mode selected configure 3rd channel as Channel C) */
    tpu_hsq(tpu, channel, HALLD_CHANNEL_A);
    tpu_hsq(tpu, channel2, HALLD_CHANNEL_B );
    if (mode==HALLD_THREE_CHANNEL_MODE)
    {
    	tpu_hsq(tpu, channel3, HALLD_CHANNEL_C );
    }
    
    /* Initialize channels */
    if (mode==HALLD_THREE_CHANNEL_MODE)
    {
    	tpu_hsr(tpu, channel, HALLD_THREE_CHANNEL_MODE);
    	tpu_hsr(tpu, channel2, HALLD_THREE_CHANNEL_MODE);
    	tpu_hsr(tpu, channel3, HALLD_THREE_CHANNEL_MODE);	
    }
    else
    {
    	tpu_hsr(tpu, channel, HALLD_TWO_CHANNEL_MODE);
    	tpu_hsr(tpu, channel2, HALLD_TWO_CHANNEL_MODE);
    }

    /* set state_no_address */
    if (mode == HALLD_TWO_CHANNEL_MODE) // for 2 channel mode STATE_NO_ADDRESS is stored 
    {                                                // in channel 2    	    	
    tpu->PARM.R[channel2][HALLD_STATE_NO_ADDRESS] = state_no_address;    
    }
    if (mode == HALLD_THREE_CHANNEL_MODE) // for 3 channel mode STATE_NO_ADDRESS is stored 
    {                                                  // in channel 3    	    	
    tpu->PARM.R[channel3][HALLD_STATE_NO_ADDRESS] = state_no_address;    
    }
    

}


/*******************************************************************************
FUNCTION      : tpu_halld_enable
PURPOSE       : To enable intialised TPU channels to run the HALLD function
INPUTS NOTES  : This function has 3 parameters:

                 *tpu    - This is a pointer to the TPU3 module to use. It is of
                           type TPU3_tag which is defined in m_tpu3.h

                 channel - This is the channel number of the primary HALLD
                           channel.
                
                 priority - This parameter should be assigned a value of:
                            TPU_PRIORITY_HIGH, TPU_PRIORITY_MIDDLE or
                            TPU_PRIORITY_LOW. All channels are assigned the same
                            priority. Although this is not necessary for correct
                            operation it simplifies system performance calculations


RETURNS NOTES : none

*******************************************************************************/
void tpu_halld_enable(struct TPU3_tag *tpu, UINT8 channel, UINT8 priority)
{ 
	
UINT8	channel2;
UINT8	channel3;
INT16	mode;	

    channel2 = (channel + 1) & 0xf;
    channel3 = (channel + 2) & 0xf;

     mode = tpu->PARM.R[channel][HALLD_STORE_MODE];
   
    /* Enable channels by assigning a priority to them. */
    /* All channels are assigned the same priority */
    tpu_enable(tpu, channel, priority);
    tpu_enable(tpu, channel2, priority);
    /* Initialize channels */
    if (mode==HALLD_THREE_CHANNEL_MODE)
    {
    	tpu_enable(tpu, channel3, priority);
    }
}


/*******************************************************************************
FUNCTION      : tpu_halld_disable
PURPOSE       : To disable TPU channels setup to run the HALLD function
INPUTS NOTES  : This function has 4 parameters:

                 *tpu    - This is a pointer to the TPU3 module to use. It is of
                           type TPU3_tag which is defined in m_tpu3.h

                 channel - This is the channel number of the primary HALLD
                           channel.


                 mode     - This parameter selects either two or three channel mode

RETURNS NOTES : none

*******************************************************************************/
void tpu_halld_disable(struct TPU3_tag *tpu, UINT8 channel)
{

UINT8	channel2;
UINT8	channel3;
INT16	mode;	

    channel2 = (channel + 1) & 0xf;
    channel3 = (channel + 2) & 0xf;
    
    mode = tpu->PARM.R[channel][HALLD_STORE_MODE];

    /* disable channels selected for HALLD function */
    tpu_disable( tpu, channel);
    tpu_disable( tpu, channel2);
    if (mode==HALLD_THREE_CHANNEL_MODE)
    {
    	tpu_disable(tpu, channel3);
    }
      
}


/*******************************************************************************
FUNCTION      : tpu_halld_set_direction
PURPOSE       : To set the DIRECTION parameter in the HALLD function paramter RAM
INPUTS NOTES  : This function has 3 parameters:

                 *tpu    - This is a pointer to the TPU3 module to use. It is of
                           type TPU3_tag which is defined in m_tpu3.h

                 channel - This is the channel number of the channel A HALLD
                           channel. The following channel is used for two channel
                           mode and following two channels are used for three 
                           channel mode. 
                
                 
                 direction - This value defines the DIRECTION value in the parameter
                             RAM. It has the value 0x0000 or 0x0001.                 	

RETURNS NOTES : none

*******************************************************************************/
void tpu_halld_set_direction(struct TPU3_tag *tpu, UINT8 channel, INT16 direction)
    {   
        tpu->PARM.R[channel][HALLD_DIRECTION] = direction;
    }


/*******************************************************************************
FUNCTION      : tpu_halld_set_state_no_address
PURPOSE       : To set the STATE_NO_ADDRESS value in the HALLD function parameter RAM
INPUTS NOTES  : This function has 3 parameters:

                 *tpu    - This is a pointer to the TPU3 module to use. It is of
                           type TPU3_tag which is defined in m_tpu3.h

                 channel - This is the channel number of the channel A HALLD
                           channel. The following channel is used for two channel
                           mode and following two channels are used for three 
                           channel mode. 

                 state_no_address - This value defines the STATE_NO_ADDRESS value 
                           in the parameter RAM.It takes the value 0x00xy, where xy
                           represents the address in parameter RAM where the state no.
                           is to be written.
          	

RETURNS NOTES : none

*******************************************************************************/
void tpu_halld_set_state_no_address(struct TPU3_tag *tpu, UINT8 channel, INT16 state_no_address)
{ 
	
    INT16 mode;
    
    mode = tpu->PARM.R[channel][HALLD_STORE_MODE];
       
    if (mode == HALLD_TWO_CHANNEL_MODE) // for 2 channel mode STATE_NO_ADDRESS is stored in channel 2  
    {                                                  	    	
        tpu->PARM.R[((channel+1) & 0xf)][HALLD_STATE_NO_ADDRESS] = state_no_address;    
    }
    if (mode == HALLD_THREE_CHANNEL_MODE) // for 3 channel mode STATE_NO_ADDRESS is stored in channel 3 
    {                                                     	    	
        tpu->PARM.R[((channel+2) & 0xf)][HALLD_STATE_NO_ADDRESS] = state_no_address;    
    }
}

/*******************************************************************************
FUNCTION      : tpu_halld_get_state_no
PURPOSE       : returns the STATE_NO_ADDRESS parameter from the TPU HALLD function parameter RAM
INPUTS NOTES  : This function has 2 parameters:

                 *tpu    - This is a pointer to the TPU3 module to use. It is of
                           type TPU3_tag which is defined in m_tpu3.h

                 channel - This is the channel number of the HALLD function 
                           channel A.
                  	

RETURNS NOTES : returns STATE_NO_ADDRESS value as INT16

*******************************************************************************/
INT16 tpu_halld_get_state_no(struct TPU3_tag *tpu, UINT8 channel)
{
    INT16	state_no_address;
    INT16	state_no;
    INT16 mode;
    
    mode = tpu->PARM.R[channel][HALLD_STORE_MODE];


    if ( mode == HALLD_TWO_CHANNEL_MODE) // for 2 channel mode STATE_NO_ADDRESS is stored 
    {                                                // in channel 2    	    	
    	state_no_address = tpu->PARM.R[((channel+1) & 0xf)][HALLD_STATE_NO_ADDRESS];    
    }

    if (mode == HALLD_THREE_CHANNEL_MODE) // for 3 channel mode STATE_NO_ADDRESS is stored 
    {                                                  // in channel 3    	    	
    	state_no_address = tpu->PARM.R[((channel+2) & 0xf)][HALLD_STATE_NO_ADDRESS];
    }    		
    state_no = tpu->PARM.R[((state_no_address >> 4) & 0x000f)][((state_no_address & 0x000f)/2)];
	
    return(state_no);    
}

/*******************************************************************************
FUNCTION      : tpu_halld_get_pinstate
PURPOSE       : returns the PINSTATE parameter from the TPU HALLD function parameter RAM
INPUTS NOTES  : This function has 2 parameters:

                 *tpu    - This is a pointer to the TPU3 module to use. It is of
                           type TPU3_tag which is defined in m_tpu3.h

                 channel - This is the channel number of the HALLD function channel from
                 	   whcih pinstate is to be returned. This can be anyone of the
                 	   two/three HALLD channels.
                  	

RETURNS NOTES : returns PINSTATE value as INT16

*******************************************************************************/
INT16 tpu_halld_get_pin_state(struct TPU3_tag *tpu, UINT8 channel)
{
 	
  	INT16 pinstate;
  			
    	pinstate = tpu->PARM.R[channel][HALLD_PINSTATE];
    
	return(pinstate);    
}

/*********************************************************************
 *
 * Copyright:
 *	MOTOROLA, INC. All Rights Reserved.  
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Motorola, Inc. This 
 *  software is provided on an "AS IS" basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, MOTOROLA 
 *  DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED, INCLUDING 
 *  IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
 *  PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD TO THE 
 *  SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY 
 *  ACCOMPANYING WRITTEN MATERIALS.
 * 
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL MOTOROLA BE LIABLE FOR ANY DAMAGES WHATSOEVER (INCLUDING 
 *  WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS, BUSINESS 
 *  INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER PECUNIARY
 *  LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.   
 * 
 *  Motorola assumes no responsibility for the maintenance and support
 *  of this software
 ********************************************************************/