/*
 * Copyright 2025 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdbool.h>
#include "assert.h"

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "board.h"
#include "ele_nvm_manager.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define RAMFS_MAX_FILES		5

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/
typedef struct
{
    char file_name[32];

	bool used;
    size_t size;
    uint8_t *data;
} RamFS_Ctx_T;

RamFS_Ctx_T ramfs_ctx[RAMFS_MAX_FILES];

/*******************************************************************************
 * Code
 ******************************************************************************/

static int find_free_pos()
{
    for (int i = 0; i < RAMFS_MAX_FILES; i++)
    {
        if (!ramfs_ctx[i].used)
        {
        	return i;
        }
    }
    return -1;
}

static int find_file_by_path(char *path)
{
    for (int i = 0; i < RAMFS_MAX_FILES; i++)
    {
        if (ramfs_ctx[i].used && (strcmp(ramfs_ctx[i].file_name, path) == 0))
        {
        	return i;
        }
    }
    return -1;
}

/* Create the filepath of chunk associated with blob_id_msb, blob_id_lsb and blob_ext */
static int get_chunk_file_path(
    char *path, uint8_t path_buf_sz,
    uint32_t blob_id_msb, uint32_t blob_id_lsb, uint32_t blob_ext)
{
    uint8_t path_len = 0;
    int err = -1;
    char *path_current = path;
    char *path_end = NULL;

    /*  the path will be /blob_ext/blob_id_msb/blob_id_lsb
     */
    path_len = 1 + /* 1 for root path '/' */
               sizeof(blob_id_msb) * 2 + 1 + /* 1 additional byte for / after every name */
               sizeof(blob_id_lsb) * 2 + 1 +
               sizeof(blob_ext) * 2 + 1;

    if (path_buf_sz < path_len)
    {
        PRINTF("Insufficient size of path buffer \r\n");
        return -1;
    }

    path_end = path + path_len; /* For keeping track of free space when appending to the path string */

	snprintf(path, path_len - 1, "/%lx/%lx/%lx",
			 blob_ext, blob_id_msb, blob_id_lsb);

    return 0;
}

/* Create and write to a file identified by passed blob_id */
status_t ramfs_write(uint32_t blob_id_msb, uint32_t blob_id_lsb, uint32_t blob_ext, uint32_t *chunk, size_t chunk_sz)
{
    char path[32];
	int file_idx = 0;
    uint32_t *buffer = NULL;

    PRINTF("Write sd, blob_id_msb = %x, %x, %x  \r\n", blob_id_msb, blob_id_lsb, blob_ext);
    /* Get the filepath of chunk associated with blob ID and blob_ext. */
    if (get_chunk_file_path((char *)path, sizeof(path),
                            blob_id_msb, blob_id_lsb, blob_ext))
    {
        return kStatus_Fail;
    }

    file_idx = find_file_by_path(path);
    if (file_idx < 0)
    {
    	file_idx = find_free_pos();
        if (file_idx < 0)
        {
            return kStatus_Fail;
        }

        buffer = malloc(chunk_sz);
        if (buffer == NULL)
        {
            return kStatus_Fail;
        }
    }
    else
    {
    	if (chunk_sz > ramfs_ctx[file_idx].size)
    	{
    		buffer = realloc(ramfs_ctx[file_idx].data, chunk_sz);
            if (buffer == NULL)
            {
                return kStatus_Fail;
            }
    	}
    }

    memcpy(buffer, chunk, chunk_sz);
    ramfs_ctx[file_idx].data = (uint8_t *)buffer;
    strcpy(ramfs_ctx[file_idx].file_name, path);
    ramfs_ctx[file_idx].size = chunk_sz;
	ramfs_ctx[file_idx].used = true;

    return kStatus_Success;
}

uint32_t *ramfs_read(uint32_t blob_id_msb, uint32_t blob_id_lsb, uint32_t blob_id_ext, uint32_t *chunk, size_t *sz)
{
    char path[32];
	int file_idx = 0;

    PRINTF("Read sd, blob_id msb = 0x%x, lsb: 0x%x, ext: 0x%x \r\n", blob_id_msb, blob_id_lsb, blob_id_ext);
    if (get_chunk_file_path((char *)path, sizeof(path), blob_id_msb, blob_id_lsb, blob_id_ext))
    {
        return NULL;
    }

    file_idx = find_file_by_path(path);
    if (file_idx < 0)
    {
        return NULL;
    }

    /* If buffer for chunk is not passed, allocate based on sie of file */
    if (!chunk)
    {
    	chunk = malloc(ramfs_ctx[file_idx].size);
        *sz   = ramfs_ctx[file_idx].size;
    }

    if (chunk)
    {
        memcpy(chunk, ramfs_ctx[file_idx].data, *sz);
    }

    return chunk;
}

/* Initialize the RamFS for NVM manager */
void ramfs_initialize(void)
{
	memset(ramfs_ctx, 0x00, sizeof(ramfs_ctx));
    for (int i = 0; i < RAMFS_MAX_FILES; i++)
    {
    	ramfs_ctx[i].used = false;
    }
}
