/*
 * Copyright 2025 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/
#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "app.h"

#include "ele_nvm_manager.h"
#include "ele_crypto.h" /* ELE Crypto SW */
#include "fsl_s3mu.h"   /* Messaging unit driver */
#include "ele_ahab_fw.h"
#include "ele_oemprov_fw.h" /* ELE FW, to be placed in bootable container in real world app */

#include "mbedtls/aes.h"


/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define S3MU MU_RT_S3MUA

#define NXP_PROD_MANUFACT_KEY_ID        (0x70000000)

/* Enable to store NVM data into RAM */
#define USE_RAMFS						1

/* Group ID defined by user */
#define USER_KEY_GROUP_ID               (0x45u)

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
status_t sd_file_write(uint32_t blob_id_msb, uint32_t blob_id_lsb, uint32_t blob_ext, uint32_t *chunk, size_t chunk_sz);
uint32_t *sd_file_read(uint32_t blob_id_msb, uint32_t blob_id_lsb, uint32_t blob_id_ext, uint32_t *chunk, size_t *sz);
int sd_fs_initialize(void);

void ramfs_initialize(void);
uint32_t *ramfs_read(uint32_t blob_id_msb, uint32_t blob_id_lsb, uint32_t blob_id_ext, uint32_t *chunk, size_t *sz);
status_t ramfs_write(uint32_t blob_id_msb, uint32_t blob_id_lsb, uint32_t blob_ext, uint32_t *chunk, size_t chunk_sz);

/*******************************************************************************
 * Variables
 ******************************************************************************/
const uint8_t ecc256_pub_key[64] = {
    0x2f, 0xf5, 0x34, 0xf9, 0x7f, 0xd0, 0xed, 0x35,  // 0x0000_0000
    0x56, 0x88, 0x64, 0xd9, 0xd8, 0x74, 0x01, 0xa6,  // 0x0000_0008
    0x8c, 0x46, 0xc1, 0xa5, 0xfd, 0x50, 0x10, 0x26,  // 0x0000_0010
    0x15, 0xb4, 0x72, 0x5e, 0x78, 0x7c, 0xd2, 0x21,  // 0x0000_0018
    0x33, 0xaa, 0x92, 0x7b, 0xc0, 0x3b, 0xc9, 0x52,  // 0x0000_0020
    0xe9, 0x92, 0xcb, 0x2a, 0xaa, 0x08, 0x1a, 0xea,  // 0x0000_0028
    0x2d, 0x53, 0xb9, 0xf5, 0xd0, 0xa9, 0x24, 0x00,  // 0x0000_0030
    0xce, 0x46, 0x2b, 0x02, 0x48, 0x37, 0x90, 0x92,  // 0x0000_0038
};

const uint8_t signed_msg_bin[576] = {
    0x00, 0x40, 0x02, 0x89, 0x02, 0x00, 0x00, 0x00,  // 0x0000_0000
    0x00, 0x00, 0x00, 0x00, 0xb0, 0x00, 0x00, 0x00,  // 0x0000_0008
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_0010
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_0018
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_0020
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_0028
    0x00, 0x00, 0x00, 0x00, 0xe8, 0x97, 0x00, 0x00,  // 0x0000_0030
    0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_0038
    0x00, 0x00, 0x00, 0x00, 0x47, 0x07, 0x00, 0x00,  // 0x0000_0040
    0x78, 0x56, 0x34, 0x12, 0x09, 0x01, 0x02, 0x09,  // 0x0000_0048
    0x2d, 0x00, 0x00, 0x00, 0x00, 0x92, 0x00, 0x01,  // 0x0000_0050
    0x01, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00,  // 0x0000_0058
    0x09, 0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,  // 0x0000_0060
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70,  // 0x0000_0068
    0xf3, 0xdf, 0xec, 0x59, 0xde, 0xfa, 0x67, 0xf3,  // 0x0000_0070
    0x37, 0x0d, 0xff, 0x12, 0x73, 0xfa, 0x12, 0x66,  // 0x0000_0078
    0xfd, 0x6a, 0xc1, 0x9c, 0x06, 0xc8, 0x0d, 0x54,  // 0x0000_0080
    0x98, 0x6b, 0xf6, 0x71, 0x00, 0xd1, 0xe6, 0x7e,  // 0x0000_0088
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_0090
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_0098
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_00A0
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_00A8
    0x00, 0x90, 0x01, 0x90, 0x00, 0x00, 0x10, 0x00,  // 0x0000_00B0
    0x48, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0x0000_00B8
    0xd7, 0x34, 0x01, 0x42, 0xe1, 0x4c, 0x00, 0x27,  // 0x0000_00C0
    0x00, 0x01, 0x00, 0x00, 0x20, 0x00, 0x20, 0x00,  // 0x0000_00C8
    0x56, 0x11, 0x39, 0x06, 0x42, 0x71, 0xdc, 0x8f,  // 0x0000_00D0
    0xd4, 0xa7, 0xd0, 0x2b, 0xc7, 0x16, 0xb3, 0x45,  // 0x0000_00D8
    0xa3, 0xab, 0x56, 0x37, 0xbd, 0x4f, 0xc0, 0xb2,  // 0x0000_00E0
    0x4c, 0x25, 0x25, 0xff, 0x31, 0x29, 0xc5, 0x3b,  // 0x0000_00E8
    0xef, 0x95, 0x11, 0x5d, 0xaf, 0xf5, 0xfb, 0x62,  // 0x0000_00F0
    0x24, 0x1f, 0x74, 0x43, 0xaf, 0x8b, 0x0b, 0x2e,  // 0x0000_00F8
    0x24, 0x91, 0x80, 0xc9, 0xbd, 0x09, 0xc8, 0x61,  // 0x0000_0100
    0x53, 0x6c, 0xe6, 0x0d, 0x41, 0xcc, 0xe5, 0xd1,  // 0x0000_0108
    0xe1, 0x4c, 0x00, 0x27, 0x00, 0x01, 0x00, 0x00,  // 0x0000_0110
    0x20, 0x00, 0x20, 0x00, 0x11, 0x1b, 0x9a, 0xab,  // 0x0000_0118
    0x8d, 0x6e, 0xa1, 0x38, 0x1a, 0x31, 0x42, 0xcb,  // 0x0000_0120
    0x48, 0xdd, 0x1c, 0xda, 0x62, 0xe4, 0x90, 0x68,  // 0x0000_0128
    0x64, 0x29, 0xeb, 0x86, 0x93, 0x7a, 0x5e, 0xee,  // 0x0000_0130
    0x36, 0xfe, 0xbc, 0x8b, 0xb1, 0xd5, 0x32, 0xb8,  // 0x0000_0138
    0xb7, 0xdd, 0xb5, 0x98, 0xb1, 0x61, 0x0a, 0x3a,  // 0x0000_0140
    0x36, 0x9c, 0x69, 0x3f, 0x29, 0x3a, 0x9b, 0x95,  // 0x0000_0148
    0x4b, 0xef, 0x83, 0xf3, 0xa3, 0x9b, 0xe4, 0x4e,  // 0x0000_0150
    0x38, 0x88, 0x79, 0x9f, 0xe1, 0x4c, 0x00, 0x27,  // 0x0000_0158
    0x00, 0x01, 0x00, 0x00, 0x20, 0x00, 0x20, 0x00,  // 0x0000_0160
    0x33, 0xa8, 0xce, 0x0b, 0x04, 0x96, 0x4a, 0xfe,  // 0x0000_0168
    0x19, 0xb2, 0x79, 0x2a, 0xb4, 0x49, 0xbc, 0x63,  // 0x0000_0170
    0x70, 0xec, 0x2e, 0xe8, 0xc2, 0x5a, 0x1b, 0xff,  // 0x0000_0178
    0xf5, 0xba, 0xdc, 0x8b, 0x35, 0xd0, 0x30, 0x90,  // 0x0000_0180
    0x3a, 0x26, 0x76, 0x8c, 0x78, 0x4c, 0x7e, 0xca,  // 0x0000_0188
    0xc4, 0xa6, 0xed, 0x53, 0x01, 0x72, 0xf7, 0x2b,  // 0x0000_0190
    0xf5, 0x91, 0x05, 0x2e, 0x60, 0x32, 0x57, 0x1c,  // 0x0000_0198
    0xf9, 0xa8, 0x66, 0x92, 0x94, 0x41, 0x31, 0x78,  // 0x0000_01A0
    0xe1, 0x4c, 0x00, 0x27, 0x00, 0x01, 0x00, 0x00,  // 0x0000_01A8
    0x20, 0x00, 0x20, 0x00, 0xc8, 0x0a, 0x3e, 0xdc,  // 0x0000_01B0
    0x67, 0x0f, 0x4b, 0x4b, 0x4f, 0x2e, 0xd7, 0x55,  // 0x0000_01B8
    0xb9, 0x6c, 0xde, 0xc6, 0xb2, 0x95, 0x55, 0x38,  // 0x0000_01C0
    0x4d, 0xa5, 0xa8, 0x79, 0xd6, 0x44, 0x48, 0x6f,  // 0x0000_01C8
    0xa4, 0x34, 0x95, 0x7a, 0x11, 0xf2, 0x12, 0x0e,  // 0x0000_01D0
    0xf2, 0xae, 0x80, 0xa1, 0x67, 0x7b, 0xca, 0x1c,  // 0x0000_01D8
    0xe8, 0xa4, 0x04, 0xf0, 0xaa, 0xa6, 0xc9, 0x05,  // 0x0000_01E0
    0xa3, 0xf6, 0xbb, 0xa8, 0x23, 0xe6, 0xa1, 0xd4,  // 0x0000_01E8
    0x94, 0x6b, 0xdb, 0xa8, 0x00, 0x00, 0x00, 0x00,  // 0x0000_01F0
    0x00, 0x48, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x00,  // 0x0000_01F8
    0xba, 0xb1, 0x1a, 0xf2, 0x90, 0x76, 0xdb, 0x07,  // 0x0000_0200
    0x94, 0xb2, 0x0d, 0xe6, 0x52, 0xe5, 0x6a, 0xab,  // 0x0000_0208
    0xab, 0xe9, 0x45, 0xe9, 0xa0, 0xf5, 0xee, 0x12,  // 0x0000_0210
    0x07, 0x60, 0x4a, 0x49, 0x1c, 0x9c, 0xaa, 0x7f,  // 0x0000_0218
    0xc7, 0xb0, 0x27, 0x74, 0x19, 0x48, 0x4b, 0xb7,  // 0x0000_0220
    0x92, 0x21, 0x30, 0x98, 0x3e, 0xc3, 0xea, 0x95,  // 0x0000_0228
    0xa0, 0xa3, 0x61, 0x64, 0x2f, 0x91, 0x6f, 0x00,  // 0x0000_0230
    0xd6, 0x97, 0xe1, 0x78, 0xe9, 0x9c, 0x51, 0x03,  // 0x0000_0238
};

const uint8_t oem_tlv_blob[141] = {
    0x40, 0x15, 0x65, 0x64, 0x67, 0x65, 0x6c, 0x6f,  // 0x0000_0000
    0x63, 0x6b, 0x65, 0x6e, 0x63, 0x6c, 0x61, 0x76,  // 0x0000_0008
    0x65, 0x69, 0x6d, 0x70, 0x6f, 0x72, 0x74, 0x41,  // 0x0000_0010
    0x04, 0x3f, 0xff, 0xff, 0xfe, 0x42, 0x04, 0x84,  // 0x0000_0018
    0xc0, 0xff, 0x00, 0x43, 0x04, 0x00, 0x00, 0x03,  // 0x0000_0020
    0x00, 0x44, 0x02, 0x24, 0x00, 0x45, 0x04, 0x00,  // 0x0000_0028
    0x00, 0x01, 0x00, 0x46, 0x04, 0xc0, 0x02, 0x00,  // 0x0000_0030
    0x01, 0x47, 0x04, 0x00, 0x00, 0x00, 0x00, 0x50,  // 0x0000_0038
    0x04, 0x3f, 0xff, 0xff, 0xff, 0x51, 0x04, 0x00,  // 0x0000_0040
    0x00, 0x00, 0x01, 0x54, 0x04, 0x00, 0x00, 0x00,  // 0x0000_0048
    0x01, 0x55, 0x28, 0xf9, 0x48, 0xef, 0x6b, 0xe6,  // 0x0000_0050
    0x00, 0x7d, 0x17, 0x6d, 0xd4, 0x0e, 0x9d, 0xfb,  // 0x0000_0058
    0xbf, 0x7a, 0x9d, 0x22, 0x65, 0x57, 0x94, 0xe8,  // 0x0000_0060
    0xa1, 0x86, 0xe9, 0x0f, 0x08, 0xa6, 0x2a, 0x87,  // 0x0000_0068
    0x31, 0x68, 0x79, 0x80, 0xde, 0x4b, 0x22, 0xe5,  // 0x0000_0070
    0xc2, 0x9d, 0xc5, 0x5e, 0x10, 0xa9, 0x93, 0x21,  // 0x0000_0078
    0x51, 0x67, 0x55, 0xec, 0x70, 0xd9, 0xed, 0x50,  // 0x0000_0080
    0xb2, 0x01, 0x07, 0x18, 0x7e,  // 0x0000_0088
};

uint32_t sessionID, keyStoreHandleID, nvmStorageID, keyHandleID, cipherHandleID;
/*******************************************************************************
 * Code
 ******************************************************************************/
status_t ELE_APP_setup(const uint8_t *fw,
                       ele_keystore_t *keystoreParam,
                       uint32_t *cipherHandleID,
                       uint32_t *keyHandleID,
                       uint32_t *keyStoreHandleID,
                       uint32_t *nvmStorageID,
                       uint32_t *sessionID)
{
    status_t result = kStatus_Fail;

    do {
        if (ELE_LoadFw(S3MU, fw) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("EdgeLock FW loaded and authenticated successfully.\r\n\r\n");
        }

        /****************** Start RNG ***********************/
        PRINTF("****************** Start RNG ******************************\r\n");
        if (ELE_StartRng(S3MU) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("EdgeLock RNG Start success.\r\n");
        }

        uint32_t trng_state = 0u;
        do
        {
            result = ELE_GetTrngState(S3MU, &trng_state);
        } while (((trng_state & 0xFFu) != kELE_TRNG_ready) &&
                 ((trng_state & 0xFF00u) != kELE_TRNG_CSAL_success << 8u ) &&
                   result == kStatus_Success);

        PRINTF("EdgeLock RNG ready to use.\r\n\r\n");

        /****************** Initialize EdgeLock services ************/
        PRINTF("****************** Initialize EdgeLock services ***********\r\n");
        if (ELE_InitServices(S3MU) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("EdgeLock services initialized successfully.\r\n\r\n");
        }

        /****************** Open EdgeLock session ***********************/
        PRINTF("****************** Open EdgeLock session ******************\r\n");
        if (ELE_OpenSession(S3MU, sessionID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Open session successfully. Session ID: 0x%x\r\n\r\n", *sessionID);
        }
        
        /****************** Open NVM Storage service **************************/
        PRINTF("****************** Open NVM Storage service ****************\r\n");
        if (ELE_OpenNvmStorageService(S3MU, *sessionID, nvmStorageID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Open NVM Storage service successfully. Handle ID: 0x%x\r\n\r\n", *nvmStorageID);
        }
        
        /***************** Import Storage master from NVM ********************/
        result = ELE_StorageMasterImport_From_NVM(S3MU, *nvmStorageID);
        if (result == kStatus_NoData)
        {
            result = kStatus_Success;
        }
        if (result != kStatus_Success)
        {
            PRINTF("ELE_StorageMasterImport_From_NVM failed\n");
            break;
        }

        if (ELE_OpenKeystore(S3MU, *sessionID, keystoreParam, keyStoreHandleID, NULL,
                             0) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Open service and create Key Store successfully. Key Store ID: 0x%x\r\n\r\n", keyStoreHandleID);
        }

        /****************** Key Management Open *******************************/
        PRINTF("****************** Key Management Open ********************\r\n");
        if (ELE_OpenKeyService(S3MU, *keyStoreHandleID, keyHandleID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Open Key management service successfully. Key Handle ID: 0x%x\r\n\r\n", keyHandleID);
        }
        
        /****************** Open cipher service ********************************/
        PRINTF("****************** Open cipher service *********************\r\n");
        if (ELE_OpenCipherService(S3MU, *keyStoreHandleID, cipherHandleID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Open Cipher service successfully. Handle ID: 0x%x\r\n\r\n", *cipherHandleID);
        }
        
    } while (0);
    
    return result;
}

status_t ELE_APP_Cleanup(uint32_t dataStorageID,
                         uint32_t cipherHandleID,
                         uint32_t signHandleID,
                         uint32_t verifyHandleID,
                         uint32_t keyHandleID,
                         uint32_t keyStoreHandleID,
                         uint32_t nvmStorageID,
                         uint32_t sessionID)
{
    /****************** Close Data storage session *************************/
    if (dataStorageID != 0u)
    {
        if (ELE_CloseDataStorage(S3MU, dataStorageID) != kStatus_Success)
        {
            return kStatus_Fail;
        }
        else
        {
            PRINTF("Close data storage session successfully.\r\n\r\n");
        }
    }

    /****************** Close cipher service ***********************/
    if (cipherHandleID != 0u)
    {
        PRINTF("****************** Close cipher service **********************\r\n");
        if (ELE_CloseCipherService(S3MU, cipherHandleID) != kStatus_Success)
        {
            return kStatus_Fail;
        }
        else
        {
            PRINTF("Cipher service closed successfully.\r\n\r\n");
        }
    }

    if (signHandleID != 0u)
    {
        /****************** Close Sign service *********************************/
        PRINTF("****************** Close Sign service ************************\r\n");
        if (ELE_CloseSignService(S3MU, signHandleID) != kStatus_Success)
        {
            return kStatus_Fail;
        }
        else
        {
            PRINTF("Sign service close successfully.\r\n\r\n");
        }
    }

    if (verifyHandleID != 0u)
    {
        /****************** Close Verify service *******************************/
        PRINTF("****************** Close Verify service *******************\r\n");
        if (ELE_CloseVerifyService(S3MU, verifyHandleID) != kStatus_Success)
        {
            return kStatus_Fail;
        }
        else
        {
            PRINTF("Verify service close successfully.\r\n\r\n");
        }
    }

    /****************** Close Key Management Service **********************/
    PRINTF("****************** Close Key Management Service ***********\r\n");
    if (ELE_CloseKeyService(S3MU, keyHandleID) != kStatus_Success)
    {
        return kStatus_Fail;
    }
    else
    {
        PRINTF("Close Key Management Service successfully.\r\n\r\n");
    }

    /****************** Close Key Store  ***********************************/
    PRINTF("****************** Close Key Store ************************\r\n");
    if (ELE_CloseKeystore(S3MU, keyStoreHandleID) != kStatus_Success)
    {
        return kStatus_Fail;
    }
    else
    {
        PRINTF("Close Key Store successfully.\r\n\r\n");
    }

    /****************** Close NVM storage session **************************/
    if (ELE_CloseNvmStorageService(S3MU, nvmStorageID) != kStatus_Success)
    {
        return kStatus_Fail;
    }
    else
    {
        PRINTF("Close NVM storage session successfully.\r\n\r\n");
    }

    /****************** Close EdgeLock session *****************************/
    PRINTF("****************** Close EdgeLock session *****************\r\n");
    if (ELE_CloseSession(S3MU, sessionID) != kStatus_Success)
    {
        return kStatus_Fail;
    }
    else
    {
        PRINTF("Close session successfully.\r\n\r\n");
    }

    return kStatus_Success;
}

/*!
 * @brief Main function
 */
int main(void)
{
    /*
     * This code example demonstrates EdgeLock usage in following steps:
     * 0.  load and authenticate EdgeLock Enclave FW (to be done in secure boot flow in real world app)
     * 1.  Start RNG
     * 2.  Initialize EdgeLock Enclave services
     * 3.  Register NVM Manager
     * 4.  Open EdgeLock session
     * 5.  Open and Create Key Store
     * 6.  Key management service open
     * 7.  Open NVM Storage service
     * 8.  Execute Key exchange command
     * 9.  Import the key using Key import
     * 10. Close keystore, session and services
     * 11. Load ELE Runtime FW and initialize EdgeLock Enclave services
     * 12. Encrypt and decrypt data using ELE AES-ECB or MbedTLS
     * 13. Close keystore, session and services
     * 14. Unregister NVM Manager
     * Note: This example does not close already opened contexts or objects in case of failed command.
     */

    status_t result = kStatus_Fail;
    ele_nvm_manager_t manager = {0};

#if USE_RAMFS
    manager.nvm_read          = ramfs_read;
    manager.nvm_write         = ramfs_write;
#else
    manager.nvm_read          = sd_file_read;
    manager.nvm_write         = sd_file_write;
#endif

    do
    {
        /* HW init */
        BOARD_InitHardware();

#if USE_RAMFS
        ramfs_initialize();
#else
        /* Initialize the filesystem with SD backend */
        sd_fs_initialize();
#endif
        
        PRINTF("EdgeLock Enclave Sub-System oem provsioning example:\r\n\r\n");

        /****************** Load EdgeLock FW message ***********************/
        PRINTF("****************** Load EdgeLock FW ***********************\r\n");
        if (ELE_LoadFw(S3MU, ele_oemprov_fw) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("EdgeLock FW loaded and authenticated successfully.\r\n\r\n");
        }

        /****************** Start RNG ***********************/
        PRINTF("****************** Start RNG ******************************\r\n");
        if (ELE_StartRng(S3MU) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("EdgeLock RNG Start success.\r\n");
        }

        uint32_t trng_state = 0u;
        do
        {
            result = ELE_GetTrngState(S3MU, &trng_state);
        } while (((trng_state & 0xFFu) != kELE_TRNG_ready) &&
                 ((trng_state & 0xFF00u) != kELE_TRNG_CSAL_success << 8u ) &&
                   result == kStatus_Success);

        PRINTF("EdgeLock RNG ready to use.\r\n\r\n");

        /****************** Initialize EdgeLock services ************/
        PRINTF("****************** Initialize EdgeLock services ***********\r\n");
        if (ELE_InitServices(S3MU) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("EdgeLock services initialized successfully.\r\n\r\n");
        }
        
        /****************** Register a NVM Manager ***********************/
        PRINTF("****************** Load EdgeLock NVM Mgr ***********************\r\n");
        if (ELE_Register_NVM_Manager(&manager) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("EdgeLock NVM manager registered.\r\n\r\n");
        }

        /****************** Open EdgeLock session ***********************/
        PRINTF("****************** Open EdgeLock session ******************\r\n");
        if (ELE_OpenSession(S3MU, &sessionID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Open session successfully. Session ID: 0x%x\r\n\r\n", sessionID);
        }

        /****************** Open and Create Key Store  ***********************************/
        PRINTF("****************** Create Key Store ***********************\r\n");
        ele_keystore_t keystoreParam = {0u};

        keystoreParam.id            = 0x12345678u;
        keystoreParam.nonce         = 0xabcdef12u;
        keystoreParam.max_updates   = 0xff;
        keystoreParam.min_mac_check = false;
        keystoreParam.min_mac_len   = 0u;

        if (ELE_CreateKeystore(S3MU, sessionID, &keystoreParam, &keyStoreHandleID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Open service and create Key Store successfully. Key Store ID: 0x%x\r\n\r\n", keyStoreHandleID);
        }

        /****************** Open NVM Storage service **************************/
        PRINTF("****************** Open NVM Storage service ***************\r\n");
        if (ELE_OpenNvmStorageService(S3MU, sessionID, &nvmStorageID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Open NVM Storage service successfully. Handle ID: 0x%x\r\n\r\n", nvmStorageID);
        }

        /****************** Key Management Open *******************************/
        PRINTF("****************** Key Management Open ********************\r\n");
        if (ELE_OpenKeyService(S3MU, keyStoreHandleID, &keyHandleID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Open Key management service successfully. Key Handle ID: 0x%x\r\n\r\n", keyHandleID);
        }
#if 0
        /****************** Export NXP production public Key ******************/
        PRINTF("*********** Export NXP production public Key ***********\r\n");
        /* Output buffer for public key */
        SDK_ALIGN(uint8_t pubKeyNIST[64u], 8u) = {0u};
        uint16_t pubKey_size = 0;

        if (ELE_GeneratePubKey(S3MU, keyStoreHandleID, NXP_PROD_MANUFACT_KEY_ID, (uint32_t *)pubKeyNIST, sizeof(pubKeyNIST), &pubKey_size) !=
            kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Export NXP production public key successfully.\r\n");

            PRINTF("NXP production public key: ");
            for (uint8_t i = 0; i < pubKey_size; i++)
                PRINTF("%02x", pubKeyNIST[i]);
            PRINTF("\r\n\r\n");
        }
#endif
        uint32_t derived_key_ID = 0;
        uint8_t flags = 0;

        result = ELE_ExchangeKey(S3MU,
        						 keyHandleID,
    							 flags,
								 signed_msg_bin, sizeof(signed_msg_bin),
								 NULL, 0,  //user fixed info
								 ecc256_pub_key, sizeof(ecc256_pub_key),
                                 &derived_key_ID);
        if (result != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("ele perform key agreement successfully. Derived key ID: 0x%x\r\n\r\n", derived_key_ID);
        }

		#define USER_KEY_SIZE_IN_BYTES  (kKeySize_AES_256 / 8)
        uint32_t user_key_size = USER_KEY_SIZE_IN_BYTES;
        uint8_t user_key[USER_KEY_SIZE_IN_BYTES] = {
			  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
			  0x0c, 0x0d, 0x0e, 0x0f, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			  0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
        };
        uint32_t AESkeyID = 0x3FFFFFFE;

        result = ELE_ImportKey_HSM(S3MU,
        		                   keyHandleID,
        						   oem_tlv_blob,
								   sizeof(oem_tlv_blob),
                                   false, //auto group
                                   USER_KEY_GROUP_ID,
                                   kImportKeyOption_ELE,
                                   true,
                                   false,
                                   &AESkeyID);
        if (result != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Import key successfully. User key ID: 0x%x\r\n\r\n", AESkeyID);
        }

        if (ELE_DeleteKey(S3MU, keyHandleID, derived_key_ID, false, false) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("OEM_IMPORT_MK_SK deleted successfully. Key Pair ID: 0x%x\r\n\r\n", derived_key_ID);
        }
 
        /****************** Cleanup ********************************************/
        /* Close all services */
        if (ELE_APP_Cleanup(0u, 0u, 0u, 0u, keyHandleID, keyStoreHandleID, nvmStorageID,
                            sessionID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }

        result = ELE_APP_setup(ele_alt_ahab_fw,
                               &keystoreParam,
                               &cipherHandleID,
                               &keyHandleID,
                               &keyStoreHandleID,
                               &nvmStorageID,
                               &sessionID);

        if (result != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            PRINTF("Setup ELE service successfully.\r\n");
        }

        /****************** Crypto AES-EBC ***********************/
        PRINTF("****************** Cipher AES ECB *************************\r\n");

        static uint8_t AesEcbPlain[] = {0xb3, 0x79, 0x77, 0x7f, 0x90, 0x50, 0xe2, 0xa8, 0x18, 0xf2,
                                  0x94, 0x0c, 0xbb, 0xd9, 0xab, 0xa4};

        /* Buffer for ciphertext after running AES-ECB encrypt */
        SDK_ALIGN(uint8_t AesEcbCipher[16u], 8u) = {0u};
        size_t AesEcbCipherSize                  = sizeof(AesEcbCipher);
        
        /* Buffer for plaintext after running AES-ECB decrypt on AesEcbCipher data  */
        AT_NONCACHEABLE_SECTION_INIT(SDK_ALIGN(static uint8_t AesEcbDecrypted[16u], 8u));
        
        ele_hsm_cipher_t AESHsmParam = {0u};
#if 0
        /* Encrypt plain data with ELE engine */
        AESHsmParam.keyID       = AESkeyID;
        AESHsmParam.input       = (uint32_t)AesEcbPlain;
        AESHsmParam.input_size  = sizeof(AesEcbPlain);
        AESHsmParam.output      = (uint32_t)AesEcbCipher;
        AESHsmParam.output_size = &AesEcbCipherSize;
        AESHsmParam.iv          = 0u;
        AESHsmParam.iv_size     = 0u;
        AESHsmParam.alg         = kPermitted_ECB;
        AESHsmParam.mode        = kHSM_Encrypt;

        /* AES-ECB Encrypt */
        if (ELE_Cipher(S3MU, cipherHandleID, &AESHsmParam) == kStatus_Success)
        {
            PRINTF("Output size returned by AES-ECB encryption is : 0x%x\r\n", AesEcbCipherSize);
        }
        else
        {
            result = kStatus_Fail;
            break;
        }
#else
        /* Encrypt plain data with mbedtls SW */
        mbedtls_aes_context aes_ctx;
        mbedtls_aes_init(&aes_ctx);

        mbedtls_aes_setkey_enc(&aes_ctx, user_key, (sizeof(user_key) * 8));

        result = mbedtls_aes_crypt_ecb(&aes_ctx, MBEDTLS_AES_ENCRYPT, AesEcbPlain, AesEcbCipher);
        mbedtls_aes_free(&aes_ctx); 
        if (result != 0)
        {
            result = kStatus_Fail;
            break;
        }
        PRINTF("Output returned by AES-ECB encryption.\r\n\r\n");
#endif
        PRINTF("****************** Decrypt Cipher AES-ECB ******************\r\n");

        AESHsmParam.keyID      = AESkeyID;
        AESHsmParam.input      = (uint32_t)AesEcbCipher;
        AESHsmParam.input_size = sizeof(AesEcbPlain);
        AESHsmParam.output     = (uint32_t)AesEcbDecrypted;
        AESHsmParam.alg        = kPermitted_ECB;
        AESHsmParam.mode       = kHSM_Decrypt;

        /* AES-ECB Decrypt */
        if (ELE_Cipher(S3MU, cipherHandleID, &AESHsmParam) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }
        else
        {
            /* Check decrypted data match the original plain text */
            if (memcmp(AesEcbDecrypted, AesEcbPlain, sizeof(AesEcbPlain)) == 0)
            {
                PRINTF("AES-ECB decrypted data match the original plain text - success.\r\n\r\n");
            }
            else
            {
                result = kStatus_Fail;
                break;
            }
        }

        /* Close all services */
        if (ELE_APP_Cleanup(0u, cipherHandleID, 0u, 0, keyHandleID, keyStoreHandleID, nvmStorageID,
                            sessionID) != kStatus_Success)
        {
            result = kStatus_Fail;
            break;
        }

        /****************** Unregister NVM Manager *****************************/
        PRINTF("****************** Unregister NVM Manager *****************\r\n");
        if (ELE_Unregister_NVM_Manager() != kStatus_Success)
        {
            return kStatus_Fail;
        }
        else
        {
            PRINTF("NVM manager unregistered successfully.\r\n\r\n");
        }

        /****************** END of Example *************************************/
        result = kStatus_Success;

    } while (0);

    if (result == kStatus_Success)
    {
        PRINTF("End of Example with SUCCESS!!\r\n\r\n");
    }
    else
    {
        PRINTF("ERROR: execution of commands on Security Sub-System failed!\r\n\r\n");
    }
    while (1)
    {
    }
}
