/*
 * Copyright 2020-2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include <stdlib.h>

#include "usb_device_config.h"
#include "usb.h"
#include "usb_device.h"

#include "usb_device_class.h"
#include "usb_device_audio.h"
#include "usb_device_descriptor.h"

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#if (defined(FSL_FEATURE_SOC_SYSMPU_COUNT) && (FSL_FEATURE_SOC_SYSMPU_COUNT > 0U))
#include "fsl_sysmpu.h"
#endif /* FSL_FEATURE_SOC_SYSMPU_COUNT */

#if (defined(FSL_FEATURE_SOC_USBPHY_COUNT) && (FSL_FEATURE_SOC_USBPHY_COUNT > 0U))
#include "usb_phy.h"
#endif

#include "pin_mux.h"

#include "GlobalDef.h"

#if EnableCodec==1
	#include "fsl_codec_common.h"
	#include "fsl_codec_adapter.h"
#endif


#include "main.h"
#include "clock_config.h"
#include "board.h"
#include "CircularBuf.h"
#include "SubFunc.h"
#include "AudioDmaIO.h"
#include "Sweep.h"
#include "NxpCm7ASRC.h"
#include "AsrcProcess.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/* Select Audio/Video PLL (786.48 MHz) as sai1 clock source */
#define DEMO_SAI1_CLOCK_SOURCE_SELECT (2U)
/* Clock pre divider for sai1 clock source */
#define DEMO_SAI1_CLOCK_SOURCE_PRE_DIVIDER (0U)
/* Clock divider for sai1 clock source */
#define DEMO_SAI1_CLOCK_SOURCE_DIVIDER (63U)
/* Get frequency of sai1 clock */
#define DEMO_SAI_CLK_FREQ                                                        \
    (CLOCK_GetFreq(kCLOCK_AudioPllClk) / (DEMO_SAI1_CLOCK_SOURCE_DIVIDER + 1U) / \
     (DEMO_SAI1_CLOCK_SOURCE_PRE_DIVIDER + 1U))

#if EnableCodec==1
	/* I2C instance and clock */
	#define DEMO_I2C LPI2C1
	/* Select USB1 PLL (480 MHz) as master lpi2c clock source */
	#define DEMO_LPI2C_CLOCK_SOURCE_SELECT (0U)
	/* Clock divider for master lpi2c clock source */
	#define DEMO_LPI2C_CLOCK_SOURCE_DIVIDER (5U)
	/* Get frequency of lpi2c clock */
	#define DEMO_I2C_CLK_FREQ ((CLOCK_GetFreq(kCLOCK_Usb1PllClk) / 8) / (DEMO_LPI2C_CLOCK_SOURCE_DIVIDER + 1U))
#endif



/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void BOARD_InitHardware(void);
void USB_DeviceClockInit(void);
void USB_DeviceIsrEnable(void);
#if USB_DEVICE_CONFIG_USE_TASK
void USB_DeviceTaskFn(void *deviceHandle);
#endif

/*!
 * @brief USB device callback function.
 *
 * This function handles the usb device specific requests.
 *
 * @param handle          The USB device handle.
 * @param event           The USB device event type.
 * @param param           The parameter of the device specific request.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
usb_status_t USB_DeviceCallback(usb_device_handle handle, uint32_t event, void *param);
extern usb_status_t USB_DeviceMscDiskInit(usb_device_composite_struct_t *deviceComposite);
extern usb_device_class_struct_t g_UsbDeviceCdcVcomConfig;
extern usb_device_class_struct_t g_mscDiskClass;
extern usb_device_class_struct_t g_UsbDeviceAudioClassRecorder;
extern usb_device_class_struct_t g_UsbDeviceAudioClassSpeaker;

extern usb_device_composite_struct_t *g_UsbDeviceComposite;
extern usb_device_composite_struct_t *g_deviceAudioComposite;
extern uint8_t audioFeedBackBuffer[4];

extern uint8_t USB_DeviceMscCardInit(void);
#if (defined(USB_DEVICE_CONFIG_USE_TASK) && (USB_DEVICE_CONFIG_USE_TASK > 0)) && \
    (defined(USB_DEVICE_MSC_USE_WRITE_TASK) && (USB_DEVICE_MSC_USE_WRITE_TASK > 0))
extern void USB_DeviceMscInitQueue(void);
extern void USB_DeviceMscWriteTask(void);
#endif
/*******************************************************************************
 * Variables
 ******************************************************************************/
#if EnableCodec==1
#ifdef CODEC_WM8904_ENABLE
	#define I2S_48KHz
	wm8904_config_t wm8904Config =
	{
		.i2cConfig          = {.codecI2CInstance = BOARD_CODEC_I2C_INSTANCE},
		.recordSource       = kWM8904_RecordSourceLineInput,
		.recordChannelLeft  = kWM8904_RecordChannelLeft2,
		.recordChannelRight = kWM8904_RecordChannelRight2,
		.playSource         = kWM8904_PlaySourceDAC,
		.slaveAddress       = WM8904_I2C_ADDRESS,
		.protocol           = kWM8904_ProtocolI2S,

	#ifdef I2S_48KHz
		.format             = {.sampleRate = kWM8904_SampleRate48kHz, .bitWidth = kWM8904_BitWidth32},
	#endif
	#ifdef I2S_16KHz
		.format             = {.sampleRate = kWM8904_SampleRate16kHz, .bitWidth = kWM8904_BitWidth32},
	#endif
		.master             = true,
	};
	codec_config_t boardCodecConfig = {.codecDevType = kCODEC_WM8960, .codecDevConfig = &wm8904Config};
#endif
#ifdef CODEC_WM8960_ENABLE
	wm8960_config_t wm8960Config =
	{
		.i2cConfig = {.codecI2CInstance = BOARD_CODEC_I2C_INSTANCE, .codecI2CSourceClock = BOARD_CODEC_I2C_CLOCK_FREQ},
		.route     = kWM8960_RoutePlaybackandRecord,
		.rightInputSource = kWM8960_InputDifferentialMicInput2,
		.leftInputSource = kWM8960_InputDifferentialMicInput3,
		.playSource       = kWM8960_PlaySourceDAC,
		.slaveAddress     = WM8960_I2C_ADDR,
		.bus              = kWM8960_BusI2S,
		//.format = {.mclk_HZ = 6144000U, .sampleRate = kWM8960_AudioSampleRate16KHz, .bitWidth = kWM8960_AudioBitWidth16bit},
		.format = {.mclk_HZ =12288000U, .sampleRate = kWM8960_AudioSampleRate48KHz, .bitWidth = kWM8960_AudioBitWidth32bit},
		.master_slave = false,
	};
	codec_config_t boardCodecConfig = {.codecDevType = kCODEC_WM8960, .codecDevConfig = &wm8960Config};
#endif
codec_handle_t codecHandle;
#endif

/* Composite device structure. */
USB_DMA_NONINIT_DATA_ALIGN(USB_DATA_ALIGN_SIZE) usb_device_composite_struct_t g_composite;

/* USB device class information */
usb_device_class_config_struct_t g_CompositeClassConfig[3] =
{
	{
		USB_DeviceCdcVcomCallback,
		(class_handle_t) NULL,
		&g_UsbDeviceCdcVcomConfig,
	},
	{
		USB_DeviceAudioCompositeCallback, 
		(class_handle_t) NULL, 
		&g_UsbDeviceAudioClassRecorder, 
	},
	{
		USB_DeviceAudioCompositeCallback, 
		(class_handle_t) NULL, 
		&g_UsbDeviceAudioClassSpeaker, 
	}
};

/* USB device class configuration information */
usb_device_class_config_list_struct_t g_UsbDeviceCompositeConfigList = 
{ 
	g_CompositeClassConfig,
	USB_DeviceCallback,
	3
};

/*******************************************************************************
 * Code
 ******************************************************************************/
void BOARD_EnableSaiMclkOutput(bool enable)
{
	if (enable)
	{
		IOMUXC_GPR->GPR1 |= IOMUXC_GPR_GPR1_SAI1_MCLK_DIR_MASK;
	} else
	{
		IOMUXC_GPR->GPR1 &= (~IOMUXC_GPR_GPR1_SAI1_MCLK_DIR_MASK);
	}
}

__attribute__((__section__("CodeQuickAccess")))
void USB_OTG1_IRQHandler(void)
{
	DbgPin3Up;
	USB_DeviceEhciIsrFunction(g_composite.deviceHandle);
	DbgPin3Dn;
	/* Add for ARM errata 838869, affects Cortex-M4, Cortex-M4F Store immediate overlapping
	 exception return operation might vector to incorrect interrupt */
	__DSB();
}

void USB_OTG2_IRQHandler(void)
{
	USB_DeviceEhciIsrFunction(g_composite.deviceHandle);
	/* Add for ARM errata 838869, affects Cortex-M4, Cortex-M4F Store immediate overlapping
	 exception return operation might vector to incorrect interrupt */
	__DSB();
}

void USB_DeviceClockInit(void)
{
	usb_phy_config_struct_t phyConfig =
	{
		BOARD_USB_PHY_D_CAL,
		BOARD_USB_PHY_TXCAL45DP,
		BOARD_USB_PHY_TXCAL45DM,
	};

	if (CONTROLLER_ID == kUSB_ControllerEhci0)
	{
		CLOCK_EnableUsbhs0PhyPllClock(kCLOCK_Usbphy480M, 480000000U);
		CLOCK_EnableUsbhs0Clock(kCLOCK_Usb480M, 480000000U);
	} else
	{
		CLOCK_EnableUsbhs1PhyPllClock(kCLOCK_Usbphy480M, 480000000U);
		CLOCK_EnableUsbhs1Clock(kCLOCK_Usb480M, 480000000U);
	}
	USB_EhciPhyInit(CONTROLLER_ID, BOARD_XTAL0_CLK_HZ, &phyConfig);
}

void USB_DeviceIsrEnable(void)
{
	uint8_t irqNumber;

	uint8_t usbDeviceEhciIrq[] = USBHS_IRQS;
	irqNumber = usbDeviceEhciIrq[CONTROLLER_ID - kUSB_ControllerEhci0];

	/* Install isr, set priority, and enable IRQ. */
	NVIC_SetPriority((IRQn_Type) irqNumber, USB_DEVICE_INTERRUPT_PRIORITY);
	EnableIRQ((IRQn_Type) irqNumber);
}
#if USB_DEVICE_CONFIG_USE_TASK
void USB_DeviceTaskFn(void *deviceHandle)
{
    USB_DeviceEhciTaskFunction(deviceHandle);
}
#endif
/*!
 * @brief USB device callback function.
 *
 * This function handles the usb device specific requests.
 *
 * @param handle          The USB device handle.
 * @param event           The USB device event type.
 * @param param           The parameter of the device specific request.
 *
 * @return A USB error code or kStatus_USB_Success.
 */
#define USB_AUDIO_CONTROL_INTERFACE_ALTERNATE_COUNT         (1)
#define USB_AUDIO_RECORDER_STREAM_INTERFACE_ALTERNATE_COUNT (2)
#define USB_AUDIO_SPEAKER_STREAM_INTERFACE_ALTERNATE_COUNT  (2)
#define USB_AUDIO_CONTROL_INTERFACE_ALTERNATE_0             (0)
#define USB_AUDIO_RECORDER_STREAM_INTERFACE_ALTERNATE_0     (0)
#define USB_AUDIO_RECORDER_STREAM_INTERFACE_ALTERNATE_1     (1)
#define USB_AUDIO_SPEAKER_STREAM_INTERFACE_ALTERNATE_0      (0)
#define USB_AUDIO_SPEAKER_STREAM_INTERFACE_ALTERNATE_1      (1)

extern void USB_DeviceAudioSpeakerStatusReset(void);
__attribute__((__section__("CodeQuickAccess")))
usb_status_t USB_DeviceCallback(usb_device_handle handle, uint32_t event, void *param)
{
    usb_status_t error = kStatus_USB_InvalidRequest;
    uint16_t *temp16   = (uint16_t *)param;
    uint8_t *temp8     = (uint8_t *)param;
    uint8_t count      = 0U;

    switch (event)
    {
        case kUSB_DeviceEventBusReset:
        {
            for (count = 0U; count < USB_DEVICE_INTERFACE_COUNT; count++)
            {
                g_composite.currentInterfaceAlternateSetting[count] = 0U;
            }
            /* reset audio speaker status to be the initialized status */
            USB_DeviceAudioSpeakerStatusReset();
			#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
			#else
				/* reset the the last feedback value */
				g_composite.audioUnified.lastFeedbackValue             = 0U;
			#endif
            g_composite.attach               = 0U;
            g_composite.currentConfiguration = 0U;
            error                            = kStatus_USB_Success;
#if (defined(USB_DEVICE_CONFIG_EHCI) && (USB_DEVICE_CONFIG_EHCI > 0U)) || (defined(USB_DEVICE_CONFIG_LPCIP3511HS) && (USB_DEVICE_CONFIG_LPCIP3511HS > 0U))
            /* Get USB speed to configure the device, including max packet size and interval of the endpoints. */
            if (kStatus_USB_Success == USB_DeviceClassGetSpeed(CONTROLLER_ID, &g_composite.speed))
            {
                USB_DeviceSetSpeed(handle, g_composite.speed);
            }
            if (USB_SPEED_HIGH == g_composite.speed)
            {
				#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
					g_composite.audioUnified.currentStreamOutMaxPacketSize = (HS_ISO_OUT_ENDP_PACKET_SIZE);
				#else
					g_composite.audioUnified.currentStreamOutMaxPacketSize =
						(HS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE);
					g_composite.audioUnified.currentFeedbackMaxPacketSize = HS_ISO_FEEDBACK_ENDP_PACKET_SIZE;
				#endif /* USB_DEVICE_AUDIO_USE_SYNC_MODE */

				#if (USB_DEVICE_CONFIG_AUDIO_CLASS_2_0)
					/* high speed and audio 2.0, audio play interval is set by HS EP packet size */
					g_composite.audioUnified.audioPlayTransferSize = HS_ISO_OUT_ENDP_PACKET_SIZE;
					/* use short play buffer size, only use two elements */
					g_composite.audioUnified.audioPlayBufferSize = AUDIO_SPEAKER_UsbDnBufLengthInMs * AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME;
					#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
					#else
						#if defined(USB_DEVICE_CONFIG_LPCIP3511HS) && (USB_DEVICE_CONFIG_LPCIP3511HS > 0U)
							AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, AUDIO_SAMPLING_RATE_TO_16_16_SPECIFIC);
						#else
							AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, AUDIO_SAMPLING_RATE_TO_16_16);
						#endif
					#endif
				#else
					/* high speed and audio 1.0, audio play interval is 1 ms using this play size */
					g_composite.audioUnified.audioPlayTransferSize = AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME;
					/* use the whole play buffer size */
					g_composite.audioUnified.audioPlayBufferSize =  AUDIO_SPEAKER_UsbDnBufLengthInMs * AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME;
					#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
					#else
						AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, AUDIO_SAMPLING_RATE_TO_10_14);
					#endif
				#endif /* USB_DEVICE_CONFIG_AUDIO_CLASS_2_0 */
                g_deviceAudioComposite->audioUnified.speed = USB_SPEED_HIGH;
            }
            else
            {
				#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
            		g_composite.audioUnified.currentStreamOutMaxPacketSize = (FS_ISO_OUT_ENDP_PACKET_SIZE);
				#else
					g_composite.audioUnified.currentStreamOutMaxPacketSize =
						(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE);
					g_composite.audioUnified.currentFeedbackMaxPacketSize = FS_ISO_FEEDBACK_ENDP_PACKET_SIZE;
					AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, AUDIO_SAMPLING_RATE_TO_10_14);
				#endif
                /* full speed, audio play interval is 1 ms using this play size */
                g_composite.audioUnified.audioPlayTransferSize = AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME;
                /* use the whole play buffer size */
                g_composite.audioUnified.audioPlayBufferSize =  AUDIO_SPEAKER_UsbDnBufLengthInMs * AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME;
            }
#else
			#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
				g_composite.audioUnified.currentStreamOutMaxPacketSize = (FS_ISO_OUT_ENDP_PACKET_SIZE);
			#else
				g_composite.audioUnified.currentStreamOutMaxPacketSize =
					(FS_ISO_OUT_ENDP_PACKET_SIZE + AUDIO_OUT_FORMAT_CHANNELS * AUDIO_OUT_FORMAT_SIZE);
				g_composite.audioUnified.currentFeedbackMaxPacketSize = FS_ISO_FEEDBACK_ENDP_PACKET_SIZE;
				AUDIO_UPDATE_FEEDBACK_DATA(audioFeedBackBuffer, AUDIO_SAMPLING_RATE_TO_10_14);
				/* reset the the last feedback value */
				g_deviceAudioComposite->audioUnified.lastFeedbackValue = 0U;
			#endif
            /* full speed, audio play interval is 1 ms using this play size */
            g_composite.audioUnified.audioPlayTransferSize = AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME;
            /* use the whole play buffer size */
            g_composite.audioUnified.audioPlayBufferSize = AUDIO_SPEAKER_UsbDnBufLengthInMs * AUDIO_PLAY_BUFFER_SIZE_ONE_FRAME;
#endif /* USB_DEVICE_CONFIG_EHCI, USB_DEVICE_CONFIG_LPCIP3511HS */

        }
        break;
        case kUSB_DeviceEventSetConfiguration:
            if (0U == (*temp8))
            {
                g_composite.attach               = 0U;
                g_composite.currentConfiguration = 0U;
                error                            = kStatus_USB_Success;
            }
            else if (USB_COMPOSITE_CONFIGURE_INDEX == (*temp8))
            {
                g_composite.attach               = 1U;
                g_composite.currentConfiguration = *temp8;
				USB_DeviceAudioCompositeSetConfigure(g_composite.audioUnified.audioSpeakerHandle, *temp8);
				USB_DeviceAudioCompositeSetConfigure(g_composite.audioUnified.audioRecorderHandle, *temp8);
				USB_DeviceCdcVcomSetConfigure(g_composite.cdcVcom.cdcAcmHandle, *temp8);
                //USB_DeviceHidKeyboardSetConfigure(g_composite.hidKeyboard.hidHandle, *temp8);
                error = kStatus_USB_Success;
            }
            else
            {
            }
            break;
        case kUSB_DeviceEventSetInterface:

            if (g_composite.attach)
            {
                uint8_t interface        = (uint8_t)((*temp16 & 0xFF00U) >> 0x08U);
                uint8_t alternateSetting = (uint8_t)(*temp16 & 0x00FFU);

                switch (interface)
                {
                    case USB_AUDIO_CONTROL_INTERFACE_INDEX:
                        if (alternateSetting < USB_AUDIO_CONTROL_INTERFACE_ALTERNATE_COUNT)
                        {
                            error = kStatus_USB_Success;
                        }
                        break;
                    case USB_AUDIO_RECORDER_STREAM_INTERFACE_INDEX:
                        if (alternateSetting < USB_AUDIO_RECORDER_STREAM_INTERFACE_ALTERNATE_COUNT)
                        {
                            error = USB_DeviceAudioRecorderSetInterface(g_composite.audioUnified.audioRecorderHandle,
                                                                        interface, alternateSetting);
                        }
                        break;
                    case USB_AUDIO_SPEAKER_STREAM_INTERFACE_INDEX:
                        if (alternateSetting < USB_AUDIO_SPEAKER_STREAM_INTERFACE_ALTERNATE_COUNT)
                        {
                            error = USB_DeviceAudioSpeakerSetInterface(g_composite.audioUnified.audioSpeakerHandle,
                                                                       interface, alternateSetting);
							#if defined(USB_DEVICE_AUDIO_USE_SYNC_MODE) && (USB_DEVICE_AUDIO_USE_SYNC_MODE > 0U)
								if (USB_AUDIO_SPEAKER_STREAM_INTERFACE_ALTERNATE_1 == alternateSetting)
								{
									g_composite.audioUnified.stopDataLengthAudioAdjust = 0U;
								}
								else if (USB_AUDIO_SPEAKER_STREAM_INTERFACE_ALTERNATE_0 == alternateSetting)
								{
									g_composite.audioUnified.stopDataLengthAudioAdjust = 1U;
								}
							#else
								/* usb host stops the speaker, so there is no need for feedback */
								if ((1U == g_composite.audioUnified.startPlay) &&  (USB_AUDIO_SPEAKER_STREAM_INTERFACE_ALTERNATE_0 == alternateSetting))
								{
									g_composite.audioUnified.stopFeedbackUpdate = 1U;
								}

								/* usb host start the speaker, discard the feedback for AUDIO_SPEAKER_FEEDBACK_DISCARD_COUNT
								 * times */
								if (USB_AUDIO_SPEAKER_STREAM_INTERFACE_ALTERNATE_1 == alternateSetting)
								{
									g_composite.audioUnified.stopFeedbackUpdate              = 0U;
									g_deviceAudioComposite->audioUnified.feedbackDiscardFlag = 1U;
									g_deviceAudioComposite->audioUnified.feedbackDiscardTimes = AUDIO_SPEAKER_FEEDBACK_DISCARD_COUNT;
								}
							#endif
                        }
                        break;
                    default:
                        break;
                }

                if (kStatus_USB_Success == error)
                {
                    g_composite.currentInterfaceAlternateSetting[interface] = alternateSetting;
                }
            }
            break;
        case kUSB_DeviceEventGetConfiguration:
            if (param)
            {
                *temp8 = g_composite.currentConfiguration;
                error  = kStatus_USB_Success;
            }
            break;
        case kUSB_DeviceEventGetInterface:
            if (param)
            {
                uint8_t interface = (uint8_t)((*temp16 & 0xFF00U) >> 0x08U);
                if (interface < USB_DEVICE_INTERFACE_COUNT)
                {
                    *temp16 = (*temp16 & 0xFF00U) | g_composite.currentInterfaceAlternateSetting[interface];
                    error   = kStatus_USB_Success;
                }
            }
            break;
        case kUSB_DeviceEventGetDeviceDescriptor:
            if (param)
            {
                error = USB_DeviceGetDeviceDescriptor(handle, (usb_device_get_device_descriptor_struct_t *)param);
            }
            break;
        case kUSB_DeviceEventGetConfigurationDescriptor:
            if (param)
            {
                error = USB_DeviceGetConfigurationDescriptor(handle,
                                                             (usb_device_get_configuration_descriptor_struct_t *)param);
            }
            break;
        case kUSB_DeviceEventGetStringDescriptor:
            if (param)
            {
                error = USB_DeviceGetStringDescriptor(handle, (usb_device_get_string_descriptor_struct_t *)param);
            }
            break;

#if (defined(USB_DEVICE_CONFIG_CV_TEST) && (USB_DEVICE_CONFIG_CV_TEST > 0U))
        case kUSB_DeviceEventGetDeviceQualifierDescriptor:
            if (param)
            {
                /* Get device descriptor request */
                error = USB_DeviceGetDeviceQualifierDescriptor(
                    handle, (usb_device_get_device_qualifier_descriptor_struct_t *)param);
            }
            break;
#endif
        default:
            break;
    }

    return error;
}

/*!
 * @brief Application initialization function.
 *
 * This function initializes the application.
 *
 * @return None.
 */
void APPInit(void)
{
	USB_DeviceClockInit();
#if (defined(FSL_FEATURE_SOC_SYSMPU_COUNT) && (FSL_FEATURE_SOC_SYSMPU_COUNT > 0U))
    SYSMPU_Enable(SYSMPU, 0);
#endif /* FSL_FEATURE_SOC_SYSMPU_COUNT */

	g_composite.speed = USB_SPEED_FULL;
	g_composite.attach = 0;
	g_composite.audioUnified.audioSpeakerHandle = (class_handle_t) NULL;
	g_composite.audioUnified.audioRecorderHandle = (class_handle_t) NULL;
	g_composite.cdcVcom.cdcAcmHandle = (class_handle_t) NULL;
	g_composite.deviceHandle = NULL;

	if (kStatus_USB_Success != USB_DeviceClassInit(CONTROLLER_ID, &g_UsbDeviceCompositeConfigList, &g_composite.deviceHandle))
	{
		usb_echo("RT1060: USB device init failed\r\n");
		return;
	} else
	{
		usb_echo("RT1060: USB device init is successful\r\n");
		g_composite.cdcVcom.cdcAcmHandle = g_UsbDeviceCompositeConfigList.config[0].classHandle;
		g_composite.audioUnified.audioRecorderHandle = g_UsbDeviceCompositeConfigList.config[1].classHandle;
		g_composite.audioUnified.audioSpeakerHandle = g_UsbDeviceCompositeConfigList.config[2].classHandle;

		USB_DeviceAudioCompositeInit(&g_composite);
		USB_DeviceCdcVcomInit(&g_composite);
	}

	USB_DeviceIsrEnable();

    /*Add one delay here to make the DP pull down long enough to allow host to detect the previous disconnection.*/
    SDK_DelayAtLeastUs(5000, SDK_DEVICE_MAXIMUM_CPU_CLOCK_FREQUENCY);
	USB_DeviceRun(g_composite.deviceHandle);
}

extern int InitASRCIsSuccessful(void);
extern void TestASRC(void);
#if defined(__CC_ARM) || (defined(__ARMCC_VERSION)) || defined(__GNUC__)
int main(void)
#else
void main(void)
#endif
{
	BOARD_ConfigMPU();

	BOARD_InitPins();
	BOARD_BootClockRUN();
	BOARD_InitDebugConsole();

	PRINTF("RT1060: Start\r\n");

#if EnableCodec==1
    /*Clock setting for LPI2C*/
    CLOCK_SetMux(kCLOCK_Lpi2cMux, DEMO_LPI2C_CLOCK_SOURCE_SELECT);
    CLOCK_SetDiv(kCLOCK_Lpi2cDiv, DEMO_LPI2C_CLOCK_SOURCE_DIVIDER);
#endif

	/*Clock setting for SAI1*/
	CLOCK_SetMux(kCLOCK_Sai1Mux, DEMO_SAI1_CLOCK_SOURCE_SELECT);
	CLOCK_SetDiv(kCLOCK_Sai1PreDiv, DEMO_SAI1_CLOCK_SOURCE_PRE_DIVIDER);
	CLOCK_SetDiv(kCLOCK_Sai1Div, DEMO_SAI1_CLOCK_SOURCE_DIVIDER);

	/*Enable MCLK clock*/
	BOARD_EnableSaiMclkOutput(true);


	//set to 48KHz FsIn as starting --- only 48KHz local I2S Fs is considered in this demo
	//                                          Amplitude(),      FreqBeg,   FreqEnd,  Sweeping Duration,  Fs
	//                                        in between +/- 1      Hz          Hz            Sec          Hz
	InitSineToneGenerator(&SineToneGenerator1,     0.9999f,          0.1f,      24000.1f,      20.0f,        48000);

	#if SetToneGen2To17KHz==1
		InitSineToneGenerator(&SineToneGenerator2,     0.9999f,          17001.0f,   17001.0f,      20.0f,        48000);
	#endif

	#if SetToneGen2To1KHz==1
		InitSineToneGenerator(&SineToneGenerator2,     0.9999f,          1001.0f,   17001.0f,      20.0f,        48000);
	#endif

	PI_Control_SampleAOD_ErrAccumulated=0;
	PreErr=0;

	Kp=KpValueAtFsIn48KHz;
	Ki=KiValueAtFsIn48KHz;
	PI_ErrAccMaxValue=PI_ErrAccMaxValueAtFsIn48KHz;
	ControlPara[0]=AudioDataRateMathing_NxpCM7Asrc;
	OpeningBlink(3);

	if(InitASRCIsSuccessful())
	{
		PRINTF("RT1060: ASRC init is successful.\r\n");
	}else
	{
		PRINTF("RT1060: ASRC init is failed.\r\n");
		while(1) {};
	}


#if EnableCodec==1
    if (CODEC_Init(&codecHandle, &boardCodecConfig))
    {
		PRINTF("RT1060: COEDC init is failed.\r\n");
        assert(false);
    }else
    {
		PRINTF("RT1060: COEDC init is successful.\r\n");
    }

    if (CODEC_SetVolume(&codecHandle, kCODEC_PlayChannelHeadphoneLeft | kCODEC_PlayChannelHeadphoneRight, 70) != kStatus_Success)
    {
        assert(false);
    }
#endif


	InitAudioCircularBuf();

	APPInit();
	InitSai1I2S();
	InitDmaForSaiAudio();

	PRINTF("RT1060: audio IO is initialized\r\n");

	ClearI2SDmaDualTxBuf();
	ClearI2SDmaDualRxBuf();

	EnableSai1TxAndRxDmaRequest(1);


	while (true)
	{
		if(DmaAudioRxBufIsReady)
		{
			DmaAudioRxBufIsReady=0;
		    DbgPin1Up;
				AudioRxTxIOProcess();
		    DbgPin1Dn;
		    USB_DeviceCdcVcomTask();
		}

		#if DoAsrcInUsbIntr==1
		#else
			if(AsrcInputBufIsReady)
			{
				MoveS32SamplesToFloatBuf_6SampleGrouped(AsrcInputAudioBuf, (int *)AsrcInputAudioBuf, 2);	//each frame, there are 6 sample pairs --- 12 samples total
				AsrcProcessAndSaveToCirBuf(AsrcInputAudioBuf);
				AsrcInputBufIsReady=0;
			}
		#endif
	}

}


void UpdateAudioPll(float r)
{
	//Note: in clock_config.c we have:
    //CCM_ANALOG->PLL_AUDIO_NUM = CCM_ANALOG_PLL_AUDIO_NUM_A(1073718624);  	//--- this is -23200
    //CCM_ANALOG->PLL_AUDIO_DENOM = CCM_ANALOG_PLL_AUDIO_DENOM_B(100000);	//--- it makes: 24*(33+(-23200/100000))=786.432MHz
	//and the adjusting range is 32.00000~33.99999 ==》 -2.3%~+3.7%
	int a=(r*32.768f-33)*100000;
	CCM_ANALOG->PLL_AUDIO_NUM = a;
}
#if 0

//Note: if clock_config.c is auto generated again, need to put the following back

#include "GlobalDef.h"

#if LocalI2SFsIs44p1KHz==1
//fout=24*(30 + 10560/100000)=722.5344=11.2896*64 --- this is for 44.1KHz local I2S
CCM_ANALOG->PLL_AUDIO_NUM = CCM_ANALOG_PLL_AUDIO_NUM_A(10560);
CCM_ANALOG->PLL_AUDIO_DENOM = CCM_ANALOG_PLL_AUDIO_DENOM_B(100000);
pllAudio = (CCM_ANALOG->PLL_AUDIO & (~(CCM_ANALOG_PLL_AUDIO_DIV_SELECT_MASK | CCM_ANALOG_PLL_AUDIO_POWERDOWN_MASK))) |
           CCM_ANALOG_PLL_AUDIO_ENABLE_MASK | CCM_ANALOG_PLL_AUDIO_DIV_SELECT(30);
#endif

#if LocalI2SFsIs48KHz==1
//fout=24*(33 - 23200/100000)=786.432=12.288*64 --- this is for 48KHz local I2S
CCM_ANALOG->PLL_AUDIO_NUM = CCM_ANALOG_PLL_AUDIO_NUM_A(1073718624);		//the value 1073718624 stands for -23200
CCM_ANALOG->PLL_AUDIO_DENOM = CCM_ANALOG_PLL_AUDIO_DENOM_B(100000);
pllAudio = (CCM_ANALOG->PLL_AUDIO & (~(CCM_ANALOG_PLL_AUDIO_DIV_SELECT_MASK | CCM_ANALOG_PLL_AUDIO_POWERDOWN_MASK))) |
           CCM_ANALOG_PLL_AUDIO_ENABLE_MASK | CCM_ANALOG_PLL_AUDIO_DIV_SELECT(33);
#endif
#endif

