/*
 * Copyright 2020-2024 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdint.h>
#include <string.h>

#include "GlobalDef.h"
#include "CircularBuf.h"

#include "usb_device_config.h"
#include "usb.h"
#include "usb_device.h"

#include "usb_device_class.h"
#include "usb_device_audio.h"
#include "usb_device_descriptor.h"
#include "audio_unified.h"

T_CircularAudioBuf_S64     UsbDnStrmCirBuf;
T_CircularAudioBuf_MultiCh UsbUpStrmCirBuf;

uint8_t UsbAudioUpStreamingIsStarted;
uint8_t UsbAudioDnStreamingIsStarted;
int32_t AllZeroBuf_48PointsSingleCh_32Bit[AUDIO_IN_FORMAT_CHANNELS*(AUDIO_IN_SAMPLING_RATE_KHZ+1)];		//some more than 48 should be OK

//note: this buffer is to be directly accessed by USB dma, so it should be in TCM
T_MultiCh32BitAudioSample UsbUpStrmCirBuf_TxOutput_DataArea[UsbUpStreamingCirBuf_Len_InSamples + UsbUpStreamingCirBuf_MaxReadLengthInSamples];

//note: this buffer is to be directly accessed by USB dma, so it should be in TCM
long long UsbDnStrmCirBuf_DataArea[AUDIO_SPEAKER_UsbDnBufLengthInMs * AUDIO_OUT_SAMPLING_RATE_KHZ + AudioFrameSizeInSamplePerCh];

//----------------------------------------------------------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------
//circular buffer management --- beg
#if 1		//for folding
//-------------------circular buffer management functions for single ch type -------- beg----
__attribute__((__section__("CodeQuickAccess")))
void CirAudioBuf_WriteSamples(T_CircularAudioBuf *CirBufPtr, uint32_t SampleNumbersToBePut, int32_t *PtrAudioDataSrc)
{
	if ((CirBufPtr->LengthInSamples - (CirBufPtr->PtrWr - CirBufPtr->PtrBufHead) + 1) >= SampleNumbersToBePut)
	{   //no need to cut
		memcpy(CirBufPtr->PtrWr, PtrAudioDataSrc, SampleNumbersToBePut * sizeof(int32_t));

		if (CirBufPtr->PtrWr + SampleNumbersToBePut > CirBufPtr->PtrBufHead + CirBufPtr->LengthInSamples)
			CirBufPtr->PtrWr = CirBufPtr->PtrWr + SampleNumbersToBePut - CirBufPtr->LengthInSamples - 1;
		else
			CirBufPtr->PtrWr += SampleNumbersToBePut;
	} else
	{   //need to cut into 2 parts
	    int32_t l1;
	    int32_t l2;

		l1 = CirBufPtr->LengthInSamples - (CirBufPtr->PtrWr - CirBufPtr->PtrBufHead) + 1;
		l2 = SampleNumbersToBePut - l1;
		memcpy(CirBufPtr->PtrWr, PtrAudioDataSrc, l1 * sizeof(int32_t));

		PtrAudioDataSrc += l1;
		memcpy(CirBufPtr->PtrBufHead, PtrAudioDataSrc, l2 * sizeof(int32_t));
		CirBufPtr->PtrWr = CirBufPtr->PtrBufHead + l2;
	}
}
__attribute__((__section__("CodeQuickAccess")))
int32_t* CirAudioBuf_ReadSamples_GetRdPtr(T_CircularAudioBuf *CirBufPtr, uint32_t SampleNumbersToBeGot)
{
    int32_t *ptrR;
	//make it clear that: before calling this function, it must have been confirmed that there are enough data (>SampleNumbersToBeGot) available after the read pointer
	if ((CirBufPtr->LengthInSamples - (CirBufPtr->PtrRd - CirBufPtr->PtrBufHead) + 1) >= SampleNumbersToBeGot)
	{   //no need to cut
		ptrR = CirBufPtr->PtrRd;
		if (CirBufPtr->PtrRd + SampleNumbersToBeGot > CirBufPtr->PtrBufHead + CirBufPtr->LengthInSamples)
			CirBufPtr->PtrRd = CirBufPtr->PtrRd + SampleNumbersToBeGot - CirBufPtr->LengthInSamples - 1;
		else
			CirBufPtr->PtrRd += SampleNumbersToBeGot;
		return ptrR;
	} else
	{   //need to merge 2 parts together
	    int32_t l1;
	    int32_t l2;

		l1 = CirBufPtr->LengthInSamples - (CirBufPtr->PtrRd - CirBufPtr->PtrBufHead) + 1;
		l2 = SampleNumbersToBeGot - l1;
		memcpy(CirBufPtr->PtrBufHead + CirBufPtr->LengthInSamples + 1, CirBufPtr->PtrBufHead, l2 * sizeof(int32_t));
		ptrR = CirBufPtr->PtrRd;
		CirBufPtr->PtrRd = CirBufPtr->PtrBufHead + l2;
		return ptrR;
	}
}
__attribute__((__section__("CodeQuickAccess")))
uint32_t CirAudioBuf_SpaceOccupiedInSamples(T_CircularAudioBuf *CirBufPtr)
{
	//buf is completely empty
	if (CirBufPtr->PtrRd == CirBufPtr->PtrWr)
		return (0);

	if (CirBufPtr->PtrWr > CirBufPtr->PtrRd)
	{
		return (CirBufPtr->PtrWr - CirBufPtr->PtrRd);
	} else {
		return (CirBufPtr->LengthInSamples - (CirBufPtr->PtrRd - CirBufPtr->PtrWr) + 1);
	}
}
__attribute__((__section__("CodeQuickAccess")))
uint32_t CirAudioBuf_SpaceAvailableInSamples(T_CircularAudioBuf *CirBufPtr)
{
	if (CirBufPtr->PtrRd == CirBufPtr->PtrWr)
		return (CirBufPtr->LengthInSamples);

	if (CirBufPtr->PtrWr > CirBufPtr->PtrRd)
	{
		return (CirBufPtr->LengthInSamples - (CirBufPtr->PtrWr - CirBufPtr->PtrRd));
	} else
	{
		return ((CirBufPtr->PtrRd - CirBufPtr->PtrWr) - 1);
	}
}
__attribute__((__section__("CodeQuickAccess")))
void CirAudioBuf_ClearAllSamples(T_CircularAudioBuf *CirBufPtr)
{
	CirBufPtr->PtrRd = CirBufPtr->PtrBufHead;
	CirBufPtr->PtrWr = CirBufPtr->PtrBufHead;
	memset(CirBufPtr->PtrBufHead, 0, CirBufPtr->LengthInSamples * sizeof(int32_t));
}
//-------------------circular buffer management functions for single ch type -------- end----

//-------------------circular buffer management functions for multiple ch type ------ beg----
__attribute__((__section__("CodeQuickAccess")))
void CirAudioBuf_WriteSamples_MultiCh(T_CircularAudioBuf_MultiCh *CirBufPtr, uint32_t SampleNumbersToBePut, T_MultiCh32BitAudioSample *PtrAudioDataSrc)
{
	if ((CirBufPtr->LengthInSamples - (CirBufPtr->PtrWr - CirBufPtr->PtrBufHead) + 1) >= SampleNumbersToBePut)
	{   //no need to cut
		memcpy(CirBufPtr->PtrWr, PtrAudioDataSrc, SampleNumbersToBePut * sizeof(T_MultiCh32BitAudioSample));

		if (CirBufPtr->PtrWr + SampleNumbersToBePut > CirBufPtr->PtrBufHead + CirBufPtr->LengthInSamples)
			CirBufPtr->PtrWr = CirBufPtr->PtrWr + SampleNumbersToBePut - CirBufPtr->LengthInSamples - 1;
		else
			CirBufPtr->PtrWr += SampleNumbersToBePut;
	} else
	{   //need to cut into 2 parts
	    int32_t l1;
	    int32_t l2;

		l1 = CirBufPtr->LengthInSamples - (CirBufPtr->PtrWr - CirBufPtr->PtrBufHead) + 1;
		l2 = SampleNumbersToBePut - l1;
		memcpy(CirBufPtr->PtrWr, PtrAudioDataSrc, l1 * sizeof(T_MultiCh32BitAudioSample));

		PtrAudioDataSrc += l1;
		memcpy(CirBufPtr->PtrBufHead, PtrAudioDataSrc, l2 * sizeof(T_MultiCh32BitAudioSample));
		CirBufPtr->PtrWr = CirBufPtr->PtrBufHead + l2;
	}
}
__attribute__((__section__("CodeQuickAccess")))
T_MultiCh32BitAudioSample* CirAudioBuf_ReadSamples_GetRdPtr_MultiCh(T_CircularAudioBuf_MultiCh *CirBufPtr, uint32_t SampleNumbersToBeGot)
{
    T_MultiCh32BitAudioSample *ptrR;
	//make it clear that: before calling this function, it must have been confirmed that there are enough data (>SampleNumbersToBeGot) available after the read pointer
	if ((CirBufPtr->LengthInSamples - (CirBufPtr->PtrRd - CirBufPtr->PtrBufHead) + 1) >= SampleNumbersToBeGot)
	{   //no need to cut
		ptrR = CirBufPtr->PtrRd;
		if (CirBufPtr->PtrRd + SampleNumbersToBeGot > CirBufPtr->PtrBufHead + CirBufPtr->LengthInSamples)
			CirBufPtr->PtrRd = CirBufPtr->PtrRd + SampleNumbersToBeGot - CirBufPtr->LengthInSamples - 1;
		else
			CirBufPtr->PtrRd += SampleNumbersToBeGot;
		return ptrR;
	} else
	{   //need to merge 2 parts together
	    int32_t l1;
	    int32_t l2;

		l1 = CirBufPtr->LengthInSamples - (CirBufPtr->PtrRd - CirBufPtr->PtrBufHead) + 1;
		l2 = SampleNumbersToBeGot - l1;
		memcpy(CirBufPtr->PtrBufHead + CirBufPtr->LengthInSamples + 1, CirBufPtr->PtrBufHead, l2 * sizeof(T_MultiCh32BitAudioSample));
		ptrR = CirBufPtr->PtrRd;
		CirBufPtr->PtrRd = CirBufPtr->PtrBufHead + l2;
		return ptrR;
	}
}
__attribute__((__section__("CodeQuickAccess")))
uint32_t CirAudioBuf_SpaceOccupiedInSamples_MultiCh(T_CircularAudioBuf_MultiCh *CirBufPtr)
{
	//buf is completely empty
	if (CirBufPtr->PtrRd == CirBufPtr->PtrWr)
		return (0);

	if (CirBufPtr->PtrWr > CirBufPtr->PtrRd)
	{
		return (CirBufPtr->PtrWr - CirBufPtr->PtrRd);
	} else
	{
		return (CirBufPtr->LengthInSamples - (CirBufPtr->PtrRd - CirBufPtr->PtrWr) + 1);
	}
}
__attribute__((__section__("CodeQuickAccess")))
uint32_t CirAudioBuf_SpaceAvailableInSamples_MultiCh(T_CircularAudioBuf_MultiCh *CirBufPtr)
{
	if (CirBufPtr->PtrRd == CirBufPtr->PtrWr)
		return (CirBufPtr->LengthInSamples);

	if (CirBufPtr->PtrWr > CirBufPtr->PtrRd)
	{
		return (CirBufPtr->LengthInSamples - (CirBufPtr->PtrWr - CirBufPtr->PtrRd));
	} else
	{
		return ((CirBufPtr->PtrRd - CirBufPtr->PtrWr) - 1);
	}
}
__attribute__((__section__("CodeQuickAccess")))
void CirAudioBuf_ClearAllSamples_MultiCh(T_CircularAudioBuf_MultiCh *CirBufPtr)
{
	CirBufPtr->PtrRd = CirBufPtr->PtrBufHead;
	CirBufPtr->PtrWr = CirBufPtr->PtrBufHead;
}
//-------------------circular buffer management functions for multiple ch type ------ end----

//------------------------circular buffer management functions for long long 1ch, or int 2ch --------------------- beg----
//this function doesn't check if there are enough space available. Before calling this function, should call CirAudioBuf_SpaceAvailableInSamples_S64
__attribute__((__section__("CodeQuickAccess")))
void CirAudioBuf_WriteSamples_S64(volatile T_CircularAudioBuf_S64 *CirBufPtr, unsigned int SampleNumbersToBePut, long long *PtrAudioDataSrc)
{
	if ((CirBufPtr->LengthInSamples - (CirBufPtr->PtrWr - CirBufPtr->PtrBufHead) + 1) >= SampleNumbersToBePut)
	{   //no need to cut
		memcpy(CirBufPtr->PtrWr, PtrAudioDataSrc, SampleNumbersToBePut * sizeof(long long));

		if (CirBufPtr->PtrWr + SampleNumbersToBePut > CirBufPtr->PtrBufHead + CirBufPtr->LengthInSamples)
			CirBufPtr->PtrWr = CirBufPtr->PtrWr + SampleNumbersToBePut - CirBufPtr->LengthInSamples - 1;
		else
			CirBufPtr->PtrWr += SampleNumbersToBePut;
	} else
	{   //need to cut into 2 parts
		unsigned int l1;
		unsigned int l2;

		l1 = CirBufPtr->LengthInSamples - (CirBufPtr->PtrWr - CirBufPtr->PtrBufHead) + 1;
		l2 = SampleNumbersToBePut - l1;
		memcpy(CirBufPtr->PtrWr, PtrAudioDataSrc, l1 * sizeof(long long));

		PtrAudioDataSrc += l1;
		memcpy(CirBufPtr->PtrBufHead, PtrAudioDataSrc, l2 * sizeof(long long));
		CirBufPtr->PtrWr = CirBufPtr->PtrBufHead + l2;
	}
}

//this function doesn't check if there are enough samples available. Before calling this function, should call CirAudioBuf_SpaceOccupiedInSamples_S64
__attribute__((__section__("CodeQuickAccess")))
long long* CirAudioBuf_ReadSamples_GetRdPtr_S64(volatile T_CircularAudioBuf_S64 *CirBufPtr, unsigned int SampleNumbersToBeGot)
{
	long long *ptrR;
	//make it clear that: before calling this function, it must have been confirmed that there are enough data (>SampleNumbersToBeGot) available after the read pointer
	if ((CirBufPtr->LengthInSamples - (CirBufPtr->PtrRd - CirBufPtr->PtrBufHead) + 1) >= SampleNumbersToBeGot)
	{   //no need to cut
		ptrR = CirBufPtr->PtrRd;
		if (CirBufPtr->PtrRd + SampleNumbersToBeGot > CirBufPtr->PtrBufHead + CirBufPtr->LengthInSamples)
			CirBufPtr->PtrRd = CirBufPtr->PtrRd + SampleNumbersToBeGot - CirBufPtr->LengthInSamples - 1;
		else
			CirBufPtr->PtrRd += SampleNumbersToBeGot;
		return ptrR;
	} else
	{   //need to merge 2 parts together
	    unsigned int l1;
	    unsigned int l2;

		l1 = CirBufPtr->LengthInSamples - (CirBufPtr->PtrRd - CirBufPtr->PtrBufHead) + 1;
		l2 = SampleNumbersToBeGot - l1;
		memcpy(CirBufPtr->PtrBufHead + CirBufPtr->LengthInSamples + 1, CirBufPtr->PtrBufHead, l2 * sizeof(long long));
		ptrR = CirBufPtr->PtrRd;
		CirBufPtr->PtrRd = CirBufPtr->PtrBufHead + l2;
		return ptrR;
	}
}
__attribute__((__section__("CodeQuickAccess")))
unsigned int CirAudioBuf_SpaceOccupiedInSamples_S64(volatile T_CircularAudioBuf_S64 *CirBufPtr)
{
	//buf is completely empty
	if (CirBufPtr->PtrRd == CirBufPtr->PtrWr)
		return (0);

	if (CirBufPtr->PtrWr > CirBufPtr->PtrRd)
	{
		return (CirBufPtr->PtrWr - CirBufPtr->PtrRd);
	} else {
		return (CirBufPtr->LengthInSamples - (CirBufPtr->PtrRd - CirBufPtr->PtrWr) + 1);
	}
}
__attribute__((__section__("CodeQuickAccess")))
unsigned int CirAudioBuf_SpaceAvailableInSamples_S64(volatile T_CircularAudioBuf_S64 *CirBufPtr)
{
	if (CirBufPtr->PtrRd == CirBufPtr->PtrWr)
		return (CirBufPtr->LengthInSamples);

	if (CirBufPtr->PtrWr > CirBufPtr->PtrRd)
	{
		return (CirBufPtr->LengthInSamples - (CirBufPtr->PtrWr - CirBufPtr->PtrRd));
	} else
	{
		return ((CirBufPtr->PtrRd - CirBufPtr->PtrWr) - 1);
	}
}
__attribute__((__section__("CodeQuickAccess")))
void CirAudioBuf_ClearAllSamples_S64(volatile T_CircularAudioBuf_S64 *CirBufPtr)
{
	CirBufPtr->PtrRd = CirBufPtr->PtrBufHead;
	CirBufPtr->PtrWr = CirBufPtr->PtrBufHead;
}
//------------------------circular buffer management functions for long long 1ch, or int 2ch --------------------- end----


#endif
//circular buffer management --- end
//----------------------------------------------------------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------


void InitAudioCircularBuf(void)
{
	memset(AllZeroBuf_48PointsSingleCh_32Bit,0,sizeof(AllZeroBuf_48PointsSingleCh_32Bit));

    UsbUpStrmCirBuf.LengthInSamples = UsbUpStreamingCirBuf_Len_InSamples;
    UsbUpStrmCirBuf.PtrBufHead = UsbUpStrmCirBuf_TxOutput_DataArea;
    CirAudioBuf_ClearAllSamples_MultiCh(&UsbUpStrmCirBuf);

    UsbDnStrmCirBuf.LengthInSamples = AUDIO_SPEAKER_UsbDnBufLengthInMs * AUDIO_OUT_SAMPLING_RATE_KHZ;
    UsbDnStrmCirBuf.PtrBufHead = UsbDnStrmCirBuf_DataArea;
    CirAudioBuf_ClearAllSamples_S64(&UsbDnStrmCirBuf);
}

